/** @file
  @This file implements the functions to get VMD info.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification Reference:
**/
#include <Uefi.h>
#include <IndustryStandard/Pci22.h>
#include <Register/VmdRegs.h>
#include <Library/VmdInfoLib.h>
#include <Library/BaseLib.h>
#include <Library/DebugLib.h>
#include <Library/IoLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/PciSegmentLib.h>
#include <Register/SaRegsHostBridge.h>
#include <Library/CpuPlatformLib.h>


/**
  GetVmdBusNumber: Get VMD Bus Number

  @retval PCI bus number for VMD
**/
UINT8
GetVmdBusNumber (
  VOID
  )
{
  return (UINT8)VMD_BUS_NUM;
}

/**
  GetVmdDevNumber: Get VMD Dev Number

  @retval PCI dev number for VMD
**/
UINT8
GetVmdDevNumber (
  VOID
  )
{
  return (UINT8)VMD_DEV_NUM;
}

/**
  GetVmdFunNumber: Get VMD Fun Number

  @retval PCI fun number for VMD
**/
UINT8
GetVmdFuncNumber (
  VOID
  )
{
  return (UINT8)VMD_FUN_NUM;
}

/**
  IsVmdEnabled: Check if VMD is enabled by reading the VMD Device Id

  @retval TRUE  if  VMD is enabled
  @retval False if  VMD is not enabled
**/

BOOLEAN
IsVmdEnabled ()
{
  UINT16 DevId;
  DevId = PciSegmentRead16 (
            PCI_SEGMENT_LIB_ADDRESS (
              SA_SEG_NUM,
              VMD_BUS_NUM,
              VMD_DEV_NUM,
              VMD_FUN_NUM,
              PCI_DEVICE_ID_OFFSET
            )
          );
  if (DevId == 0xFFFF) {
    return FALSE;
  }
  return TRUE;
}



/**
  IsVmdLocked: Get the status of VMD lock for critical config space registers of VMD

  @retval TRUE if  VMD lock bit is set
  @retval False if  VMD lock bit is not set
**/
BOOLEAN
IsVmdLocked (
  VOID
  )
{
  if ((PciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, VMD_BUS_NUM, VMD_DEV_NUM, VMD_FUN_NUM, R_VMCONFIG_0_14_0_PCI_VMD_REG_OFFSET)) & BIT0) != 0) {
    return TRUE;
  }
  return FALSE;
}

/**
  IsVmdSupported: Get the status of VMD support bit from capability register.

  @retval TRUE  if  VMD is supported
  @retval False if  VMD is not supported
**/
BOOLEAN
IsVmdSupported (
  VOID
  )
{
  UINT32 Data32;

  if (IsCmlCpu ()) {
    return FALSE;
  }

  Data32 = PciSegmentRead32 (
             PCI_SEGMENT_LIB_ADDRESS (
               SA_SEG_NUM,
               SA_MC_BUS,
               SA_MC_DEV,
               SA_MC_FUN,
               R_SA_MC_CAPID0_B
             )
           );
  if (Data32 & V_SA_MC_CAPID0_B_VMD_DIS) {
    DEBUG ((DEBUG_INFO, "------- Vmd is not supported -----\n"));
    return FALSE;
  }
  return TRUE;
}
