/** @file
  PCH ADR setup library

@copyright
  INTEL CONFIDENTIAL
  Copyright 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
**/

#include <Uefi/UefiBaseType.h>
#include <Library/DebugLib.h>
#include <Library/IoLib.h>
#include <Register/PmcRegs.h>
#include <Register/PmcRegsVer3.h>
#include <AdrConfig.h>
#include <Library/PchAdrLib.h>
#include "PchAdrConfig.h"
#include <Library/PmcLib.h>

/**
  Configures ADR in the PMC depending on policy

  @param PchAdrConfig  PCH ADR configuration structure

**/
VOID
PchAdrConfigurePmc (
  ADR_CONFIG *PchAdrConfig
  )
{
  UINT32 PchPwrmBase;
  UINT32 OrMask;
  UINT32 AndMask;

  PchPwrmBase = PmcGetPwrmBase ();

  //
  // Disable ADR before configuring PMC
  //
  PchAdrEnableAdr (FALSE);

  if (PchAdrConfig->AdrEn == FORCE_ENABLE) {
    OrMask = B_PCH_PMC_MSG_ADR_GEN_CFG_ADR_P2_EN;
    AndMask = (UINT32) ~0;
  } else {
    OrMask = 0;
    AndMask = (UINT32) ~B_PCH_PMC_MSG_ADR_GEN_CFG_ADR_P2_EN;
  }
  MmioAndThenOr32 (
    PchPwrmBase + R_PCH_PMC_MSG_ADR_GEN_CFG,
    AndMask,
    OrMask
   );

  if (PchAdrConfig->AdrEn == FORCE_ENABLE && PchAdrConfig->AdrHostPartitionReset == FORCE_ENABLE) {
    OrMask = B_PCH_PMC_MSG_ADR_EN_HPR_ADR_EN;
    AndMask = (UINT32) ~0;
  } else {
    OrMask = 0;
    AndMask = (UINT32) ~(B_PCH_PMC_MSG_ADR_EN_HPR_ADR_EN);
  }
  MmioAndThenOr32 (
    PchPwrmBase + R_PCH_PMC_MSG_ADR_EN,
    AndMask,
    OrMask
   );
}


/**
  Configures ADR sources in the PMC

  @param PchAdrConfig  PCH ADR configuration structure

**/
VOID
PchAdrConfigureSources (
  ADR_CONFIG *PchAdrConfig
  )
{
  UINT32 PchPwrmBase;

  PchPwrmBase = PmcGetPwrmBase ();

  MmioWrite32 (
    PchPwrmBase + R_PCH_PMC_MSG_ADR_GBLRST_EN_0,
    PchAdrConfig->AdrSrcSel.Value
  );
}

/**
  Configures ADR timer in PMC.

  @param PchAdrConfig  PCH ADR configuration structure
**/
VOID
PchAdrConfigureTimer (
  ADR_CONFIG *PchAdrConfig
  )
{
  UINT32          PchPwrmBase;
  UINT32          OrMask;
  UINT32          AndMask;

  PchPwrmBase = PmcGetPwrmBase ();

  AndMask = ~(UINT32) (B_PCH_PMC_MSG_ADR_GEN_CFG_ADR_P2_TIMER_SCALE |
                       B_PCH_PMC_MSG_ADR_GEN_CFG_ADR_P2_TIMER_TOV |
                       B_PCH_PMC_MSG_ADR_GEN_CFG_ADR_P1_TIMER_SCALE |
                       B_PCH_PMC_MSG_ADR_GEN_CFG_ADR_P1_TIMER_TOV);

  OrMask = (UINT32)((PchAdrConfig->AdrTimer1Val << N_PCH_PMC_MSG_ADR_GEN_CFG_ADR_P1_TIMER_TOV) |
                    (PchAdrConfig->AdrMultiplier1Val << N_PCH_PMC_MSG_ADR_GEN_CFG_ADR_P1_TIMER_SCALE) |
                    (PchAdrConfig->AdrTimer2Val << N_PCH_PMC_MSG_ADR_GEN_CFG_ADR_P2_TIMER_TOV) |
                    (PchAdrConfig->AdrMultiplier2Val << N_PCH_PMC_MSG_ADR_GEN_CFG_ADR_P2_TIMER_SCALE));

  MmioAndThenOr32 (
    PchPwrmBase + R_PCH_PMC_MSG_ADR_GEN_CFG,
    AndMask,
    OrMask
  );
}

/**
  PCH ADR Enable master switch.

  This function enables/disables PCH-side ADR.

  @param EnableAdr  TRUE:  Enables PCH-side ADR
                    FALSE: Disables PCH-side ADR
**/
VOID
PchAdrEnableAdr (
  BOOLEAN  EnableAdr
  )
{
  UINT32         PchPwrmBase;
  UINT32         OrMask;
  UINT32         AndMask;

  OrMask = 0;
  AndMask = (UINT32) ~0;

  PchPwrmBase = PmcGetPwrmBase ();
  if (EnableAdr) {
    OrMask = B_PCH_PMC_MSG_ADR_EN_ADR_FEAT_EN;
  } else {
    AndMask = (UINT32) ~B_PCH_PMC_MSG_ADR_EN_ADR_FEAT_EN;
  }
  MmioAndThenOr32 (
    PchPwrmBase + R_PCH_PMC_MSG_ADR_EN,
    AndMask,
    OrMask
  );
}

/**
  Configures PCH-side ADR.

  This function must always be called by MRC even if ADR is not being enabled.
  This is to allow PCH DFX overrides to work.

**/
VOID
PchAdrConfigure (
  VOID
  )
{
  ADR_CONFIG *PchAdrConfig;

  PchAdrGetConfig (&PchAdrConfig);

  PchAdrConfigurePmc (PchAdrConfig);
  PchAdrConfigureSources (PchAdrConfig);
  if (PchAdrConfig->AdrTimerEn == FORCE_ENABLE) {
    PchAdrConfigureTimer (PchAdrConfig);
  }
  PchAdrClearAdrResetStatus ();

  PchAdrEnableAdr (PchAdrConfig->AdrEn == FORCE_ENABLE);
}


/**
  @brief Reconfigure PCH-side ADR for DDRT DIMMs.

  @param[in] TimerVal Not used, kept for compability with previous projects.

  @return Void.
**/
VOID
PchAdrConfigureForDdrt (
  IN UINT32 TimerVal
  )
{

}


/**
  Clears PCH ADR Reset Status
**/
VOID
PchAdrClearAdrResetStatus (
  VOID
  )
{
  UINT32     PchPwrmBase;

  PchPwrmBase = PmcGetPwrmBase ();
  MmioWrite32 (PchPwrmBase + R_PCH_PMC_MSG_ADR_CTRL_STS, B_PCH_PMC_MSG_ADR_CTRL_STS_ADR_RST_STS);
}

/**
  Gets PCH ADR Reset Status

  @retval TRUE  ADR flow was executed during a prior global reset entry.
  @retval FALSE ADR flow was *not* executed during a prior global reset entry.

**/
BOOLEAN
PchAdrGetAdrResetStatus (
  VOID
  )
{
  UINT32     PchPwrmBase;

  PchPwrmBase = PmcGetPwrmBase ();
  return !!(MmioRead32 (PchPwrmBase + R_PCH_PMC_MSG_ADR_CTRL_STS) & B_PCH_PMC_MSG_ADR_CTRL_STS_ADR_RST_STS);
}
