/** @file
    Source code file for Silicon Init Post Memory module.

@copyright
  INTEL CONFIDENTIAL
  Copyright 2013 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification
**/

#include "SiInit.h"
#include <Library/PerformanceLib.h>
#include <Library/PeiP2sbPrivateLib.h>
#include <Library/PeiSiPolicyOverrideLib.h>
#include <Library/PostCodeLib.h>
#include <Library/IpuInitLib.h>
#include <Library/CpuTraceHubInfoLib.h>
#include <Library/PeiSiSsidLib.h>
#include <Library/PeiHdaInitLib.h>
#include <Library/VoltageRegulatorDomains.h>
#include <Library/VoltageRegulatorCommands.h>
#include <Library/PeiVrPolicyLib.h>
#include <Library/PeiVrLib.h>

#include <Register/IgdRegs.h>

#ifdef FSP_FLAG
#include <Library/FspCommonLib.h>
#endif
#include <Library/SiPolicyLib.h>
#include <Library/MemoryAllocationLib.h>
#include <Ppi/PeiSiDefaultPolicy.h>
#include <Library/PciSegmentLib.h>
#include <Register/GnaRegs.h>
#include <PcieRegs.h>
#include <CpuAccess.h>
#include <SaConfigHob.h>
#include <VtdDataHob.h>
#include <Ppi/SiPolicy.h>
#include <Library/SaInitLib.h>
#include <Library/GnaInitLib.h>
#include <Library/PeiGraphicsInitLib.h>
#include <Library/PeiVtdInitLib.h>
#include <Library/PeiCpuPcieVgaInitLib.h>
#include <Library/PeiCpuPcieRpInitLib.h>
#include <Library/PeiCpuTraceHubLib.h>
#include <TraceHubDataHob.h>
#include <Library/PeiServicesTablePointerLib.h>
#include <Library/ConfigBlockLib.h>
#include <Library/PeiDisplayInitLib.h>
#include <Library/GpioPrivateLib.h>
#include <Library/GpioNativePads.h>
#include <Library/CpuDmiInfoLib.h>
#include <Library/TcssInitLib.h>
#include <Library/PeiVmdInitLib.h>
#include <Library/TwoLmInitLib.h>
#include <Library/PeiDxeSmmTwoLmLib.h>
#include <Library/TelemetryPrivateLib.h>
#include <TelemetryPeiConfig.h>
#include <Library/PeiFusaLib.h>
#include <Library/CpuPlatformLib.h>
#include <Library/PeiHostBridgeInitLib.h>
#include <Library/PeiCpuPciePreMemRpInitLib.h>
#include <Library/PeiFusaE2eCtcLib.h>
#include <Library/MsrFruLib.h>
#include <Library/PeiCpuDmiInitLib.h>

EFI_PEI_PPI_DESCRIPTOR mEndOfSiInit = {
  (EFI_PEI_PPI_DESCRIPTOR_PPI | EFI_PEI_PPI_DESCRIPTOR_TERMINATE_LIST),
  &gEndOfSiInitPpiGuid,
  NULL
};

EFI_PEI_PPI_DESCRIPTOR  mEnablePeiGraphicsPpi = {
  (EFI_PEI_PPI_DESCRIPTOR_PPI | EFI_PEI_PPI_DESCRIPTOR_TERMINATE_LIST),
  &gEnablePeiGraphicsPpiGuid,
  NULL
};

#if FixedPcdGetBool(PcdFspBinaryEnable) == 0
GLOBAL_REMOVE_IF_UNREFERENCED EFI_PEI_NOTIFY_DESCRIPTOR  mSiInitNotifyList[] = {
  {
    EFI_PEI_PPI_DESCRIPTOR_NOTIFY_CALLBACK | EFI_PEI_PPI_DESCRIPTOR_TERMINATE_LIST,
    &gEndOfSiInitPpiGuid,
    SiInitOnEndOfPei
  }
};
#else
GLOBAL_REMOVE_IF_UNREFERENCED EFI_PEI_NOTIFY_DESCRIPTOR  mSiInitNotifyList[] = {
  {
    EFI_PEI_PPI_DESCRIPTOR_NOTIFY_CALLBACK | EFI_PEI_PPI_DESCRIPTOR_TERMINATE_LIST,
    &gEfiEndOfPeiSignalPpiGuid,
    SiInitOnEndOfPei
  }
};
#endif

static EFI_PEI_NOTIFY_DESCRIPTOR  mSiInitPostMemNotifyList[] = {
  {
    EFI_PEI_PPI_DESCRIPTOR_NOTIFY_DISPATCH | EFI_PEI_PPI_DESCRIPTOR_TERMINATE_LIST,
    &gSiPolicyReadyPpiGuid,
    SiInitPostMemOnPolicy
  }
};

/**
  PPI function to install default ConfigBlock Policy PPI.

  @retval EFI_STATUS       - Status from each sub function.
**/
EFI_STATUS
EFIAPI
PeiSiDefaultPolicyInit (
  VOID
  )
{
  EFI_STATUS                            Status;
  EFI_PEI_PPI_DESCRIPTOR                *SiPolicyPpiDesc;
  SI_POLICY_PPI                         *SiPolicyPpi;

  SiPolicyPpi = NULL;
  Status = SiCreateConfigBlocks (&SiPolicyPpi);
  ASSERT_EFI_ERROR (Status);
  if (SiPolicyPpi != NULL) {
    SiPolicyPpiDesc = (EFI_PEI_PPI_DESCRIPTOR *) AllocateZeroPool (sizeof (EFI_PEI_PPI_DESCRIPTOR));
    if (SiPolicyPpiDesc == NULL) {
      ASSERT (FALSE);
      return EFI_OUT_OF_RESOURCES;
    }

    SiPolicyPpiDesc->Flags = EFI_PEI_PPI_DESCRIPTOR_PPI | EFI_PEI_PPI_DESCRIPTOR_TERMINATE_LIST;
    SiPolicyPpiDesc->Guid  = &gSiPolicyPpiGuid;
    SiPolicyPpiDesc->Ppi   = SiPolicyPpi;
    //
    // Install Silicon Policy PPI
    //
    Status = PeiServicesInstallPpi (SiPolicyPpiDesc);
    ASSERT_EFI_ERROR (Status);
  }
  return Status;
}

PEI_SI_DEFAULT_POLICY_INIT_PPI mPeiSiDefaultPolicyInitPpi = {
  PeiSiDefaultPolicyInit
};

static EFI_PEI_PPI_DESCRIPTOR  mPeiSiDefaultPolicyInitPpiList[] = {
  {
    EFI_PEI_PPI_DESCRIPTOR_PPI | EFI_PEI_PPI_DESCRIPTOR_TERMINATE_LIST,
    &gSiDefaultPolicyInitPpiGuid,
    &mPeiSiDefaultPolicyInitPpi
  }
};

typedef union {
  struct {
    UINT32  Low;
    UINT32  High;
  } Data32;
  UINT64 Data;
} UINT64_STRUCT;

/**
  Silicon Init End of PEI callback function. This is the last change before entering DXE and OS when S3 resume.

  @param[in] PeiServices   - Pointer to PEI Services Table.
  @param[in] NotifyDesc    - Pointer to the descriptor for the Notification event that
                             caused this function to execute.
  @param[in] Ppi           - Pointer to the PPI data associated with this function.

  @retval EFI_STATUS       - Always return EFI_SUCCESS
**/
EFI_STATUS
SiInitOnEndOfPei (
  IN EFI_PEI_SERVICES                   **PeiServices,
  IN EFI_PEI_NOTIFY_DESCRIPTOR          *NotifyDesc,
  IN VOID                               *Ppi
  )
{
  EFI_STATUS                  Status;
  SI_POLICY_PPI               *SiPolicy;
  SI_PREMEM_POLICY_PPI        *SiPreMemPolicy;
  SI_CONFIG                   *SiConfig;
  CPU_CONFIG                  *CpuConfig;
  EFI_BOOT_MODE               BootMode;
  HOST_BRIDGE_PEI_CONFIG      *HostBridgePeiConfig;
  GRAPHICS_PEI_PREMEM_CONFIG  *GtPreMemConfig;
  GRAPHICS_PEI_CONFIG         *GtConfig;

  //
  // Get Policy settings through the SiPolicy PPI
  //
  Status = PeiServicesLocatePpi (
             &gSiPolicyPpiGuid,
             0,
             NULL,
             (VOID **) &SiPolicy
             );
  if (Status != EFI_SUCCESS) {
    ASSERT (FALSE);
    return EFI_SUCCESS;
  }

  //
  // Get Si PreMem Policy settings through the SiPreMemPolicy PPI
  //
  Status = PeiServicesLocatePpi (
             &gSiPreMemPolicyPpiGuid,
             0,
             NULL,
             (VOID **) &SiPreMemPolicy
             );
  if (Status != EFI_SUCCESS) {
    ASSERT (FALSE);
    return EFI_SUCCESS;
  }

  Status = GetConfigBlock ((VOID *) SiPolicy, &gSiConfigGuid, (VOID *) &SiConfig);
  ASSERT_EFI_ERROR (Status);
  Status = GetConfigBlock ((VOID *) SiPolicy, &gCpuConfigGuid, (VOID *)&CpuConfig);
  ASSERT_EFI_ERROR(Status);

  Status = GetConfigBlock ((VOID *) SiPolicy, &gHostBridgePeiConfigGuid, (VOID *) &HostBridgePeiConfig);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPreMemPolicy, &gGraphicsPeiPreMemConfigGuid, (VOID *) &GtPreMemConfig);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPolicy, &gGraphicsPeiConfigGuid, (VOID *) &GtConfig);
  ASSERT_EFI_ERROR (Status);


  DEBUG ((DEBUG_INFO, "SiInitOnEndOfPei - Start\n"));

  //
  // Initializes PCH after End of Pei
  //
  PchOnEndOfPei ();
  //
  // Execute before P2SB lock to ensure hiding trace hub thru PSF is valid.
  //
  ConfigureMscForCpuAndPchTraceHub (SiConfig->TraceHubMemBase);
  //
  // Configure P2SB at the end of EndOfPei
  // This must be done before POSTBOOT_SAI programming.
  //
  P2sbLock (SiPolicy);
#ifndef CPU_CFL
  if (CpuConfig->SkipMpInit == 0) {
    //
    // Set BIOS DONE MSR on all Cores
    //
      SetBiosDone ((CONST EFI_PEI_SERVICES **) PeiServices);
  }
#else
  //
  // Set PCH DMI HOSTIA_POSTBOOT_SAI with CFL CPU.
  //
  PchDmiEnablePostBootSai ();
#endif

  //
  // Do necessary PCH configuration just after POSTBOOT_SAI switch
  //
  PchDmiConfigAfterPostBootSai ();

  CpuInitAtEndOfPei ((CONST EFI_PEI_SERVICES **) PeiServices);

  //
  // Set BIOS_RESET_CPL to indicate BIOS initialization completed
  //
  PERF_START_EX (&gPerfSaResetPostMemGuid, NULL, NULL, AsmReadTsc (), 0x4090);

  ///
  /// Set SAPMCTL Register.
  ///
  SetSaPmCtlReg();

  ///
  /// Set BIOS_RESET_CPL
  ///
  SetBiosResetCpl();

  ///
  /// Graphics PM initialization after BIOS_RESET_CPL
  ///
  DEBUG ((DEBUG_INFO, "GraphicsPmInit Start\n"));
  PostCode (0xA63);
  GraphicsPmInit (GtPreMemConfig, GtConfig);

  ///
  /// Initialize PEI Display
  ///
  DEBUG ((DEBUG_INFO, "Initializing Pei Display\n"));
  PostCode (0xA03);
  PeiDisplayInit (GtPreMemConfig, GtConfig);

  ///
  /// Initialize full CD Clock
  ///
  DEBUG ((DEBUG_INFO, "Initializing CD Clock\n"));
  PostCode (0xA65);
  CdClkInit (GtConfig, GtPreMemConfig);

  ///
  /// Print SA PCI space in Debug log.
  ///
  DEBUG ((DEBUG_INFO, "SaPciPrint Start\n"));
  PostCode (0xA64);
  SaPciPrint ();

  PERF_END_EX (&gPerfSaResetPostMemGuid, NULL, NULL, AsmReadTsc (), 0x4091);

  PERF_START_EX (&gPerfHdaPostMemGuid, NULL, NULL, AsmReadTsc (), 0x40D0);
  HdAudioInitOnEndOfPei (SiPolicy, SiPreMemPolicy);
  PERF_END_EX (&gPerfHdaPostMemGuid, NULL, NULL, AsmReadTsc (), 0x40D1);

  //
  // Initialize power management after RESET_CPL at post-memory phase.
  //
  if (CpuConfig->SkipMpInit == 0) {
    PERF_START_EX (&gPerfCpuPowerMgmtGuid, NULL, NULL, AsmReadTsc (), 0x40A0);
    CpuPowerMgmtInit ((CONST EFI_PEI_SERVICES **) PeiServices);
    PERF_END_EX (&gPerfCpuPowerMgmtGuid, NULL, NULL, AsmReadTsc (), 0x40A1);
  }

  MeOnEndOfPei ();
  SaOnEndOfPei ();

  //
  // Build FVI Info HOB in normal boot
  //
  Status = PeiServicesGetBootMode (&BootMode);
  if ((Status == EFI_SUCCESS) && (BootMode != BOOT_ON_S3_RESUME)) {
    BuildFviInfoHob ();
  }

  InitializeSmbiosCpuHobs ();

  DEBUG ((DEBUG_INFO, "SiInitOnEndOfPei - End\n"));
  return EFI_SUCCESS;
}

/**
  Get HybridGraphics subsystem ID.

  @retval  HybridGraphics subsystem ID
**/
STATIC
UINT16
GetHgSsid (
  VOID
  )
{
  EFI_STATUS                    Status;
  SI_PREMEM_POLICY_PPI          *SiPreMemPolicyPpi;
  HYBRID_GRAPHICS_CONFIG        *HgGpioData;
  UINT16                        HgSubSystemId;

  SiPreMemPolicyPpi = NULL;
  HgGpioData = NULL;
  HgSubSystemId = 0;

  Status = PeiServicesLocatePpi (
             &gSiPreMemPolicyPpiGuid,
             0,
             NULL,
             (VOID **) &SiPreMemPolicyPpi
             );

  if (!EFI_ERROR (Status) && (SiPreMemPolicyPpi != NULL)) {
    Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gHybridGraphicsConfigGuid, (VOID *) &HgGpioData);
    if (HgGpioData != NULL) {
      HgSubSystemId = HgGpioData->HgSubSystemId;
    }
  }
  return HgSubSystemId;
}

/**
  Override HG SVID and SSID

  @param[in]      PciDevNum       Pci device number
  @param[in]      PciFuncNum      Pci function number
  @param[in,out]  Svid            Svid value
  @param[in,out]  Ssid            Ssid value

  @retval         TRUE            Silicon overrides the SSID
  @retval         FALSE           Silicon doesn't override the SSID
**/
STATIC
BOOLEAN
HgSsidOverride (
  UINT32                 PciDevNum,
  UINT32                 PciFuncNum,
  UINT16                 *Svid,
  UINT16                 *Ssid
  )
{
  UINT16    HgSsid;

  if ((PciDevNum == IGD_DEV_NUM) && (PciFuncNum == IGD_FUN_NUM)) {
    HgSsid = GetHgSsid ();
    if (HgSsid != 0) {
      *Ssid = HgSsid;
      return TRUE;
    }
  }
  return FALSE;
}

/**
  Slicon Initializes after PostMem phase Policy PPI produced,
  All required polices must be installed before the callback

  @param[in] PeiServices          General purpose services available to every PEIM.
  @param[in] NotifyDescriptor     The notification structure this PEIM registered on install.
  @param[in] Ppi                  SiPolicy PPI.

  @retval EFI_SUCCESS             Succeeds.
  @retval EFI_UNSUPPORTED         The function failed to locate SiPolicy
**/
EFI_STATUS
EFIAPI
SiInitPostMemOnPolicy (
  IN  EFI_PEI_SERVICES             **PeiServices,
  IN  EFI_PEI_NOTIFY_DESCRIPTOR    *NotifyDescriptor,
  IN  VOID                         *Ppi
  )
{
  SI_POLICY_PPI               *SiPolicy;
  SI_PREMEM_POLICY_PPI        *SiPreMemPolicyPpi;
  VOID                        *HobPtr;
  EFI_STATUS                  Status;
  CPU_CONFIG                  *CpuConfig;
  SI_CONFIG                   *SiConfig;
  SA_MISC_PEI_CONFIG          *MiscPeiConfig;
  HOST_BRIDGE_PEI_CONFIG      *HostBridgePeiConfig;
  PCIE_PEI_PREMEM_CONFIG      *PciePeiPreMemConfig;
  CPU_PCIE_CONFIG             *CpuPcieRpConfig;
  GRAPHICS_PEI_PREMEM_CONFIG  *GtPreMemConfig;
  GRAPHICS_PEI_CONFIG         *GtConfig;
  CPU_TRACE_HUB_PREMEM_CONFIG *CpuTraceHubPreMemConfig;
  VTD_CONFIG                  *Vtd;
  GNA_CONFIG                  *GnaConfig;
  MEMORY_CONFIGURATION        *MemConfig;
  TCSS_PEI_PREMEM_CONFIG      *TcssPeiPreMemConfig;
  TCSS_PEI_CONFIG             *TcssPeiConfig;
  VMD_PEI_CONFIG              *VmdPeiConfig;
  TELEMETRY_PEI_CONFIG        *TelemetryPeiConfig;
  TWOLM_PREMEM_CONFIG         *TwoLmPreMemConfig;
  CPU_PCIE_RP_PREMEM_CONFIG   *CpuPcieRpPreMemConfig;
  CPU_POWER_MGMT_TEST_CONFIG  *CpuPowerMgmtTestConfig;

  DEBUG ((DEBUG_INFO, "SiInit () - Start\n"));

  SiPreMemPolicyPpi   = NULL;
  SiPolicy            = NULL;
  TwoLmPreMemConfig   = NULL;
  VmdPeiConfig        = NULL;
  HostBridgePeiConfig = NULL;

  Status = PeiServicesLocatePpi (
             &gSiPreMemPolicyPpiGuid,
             0,
             NULL,
             (VOID **)&SiPreMemPolicyPpi
             );
  ASSERT_EFI_ERROR (Status);

  Status = PeiServicesLocatePpi (
             &gSiPolicyPpiGuid,
             0,
             NULL,
             (VOID **)&SiPolicy
             );
  ASSERT_EFI_ERROR (Status);
  if ((Status != EFI_SUCCESS) || (SiPolicy == NULL)) {
    return EFI_UNSUPPORTED;
  }
  if (IsDekelWaRequired () == TRUE) {
    CpuPcieRpPreMemConfig = NULL;
    Status = GetConfigBlock ((VOID *)SiPreMemPolicyPpi, &gCpuPcieRpPrememConfigGuid, (VOID *)&CpuPcieRpPreMemConfig);
    ASSERT_EFI_ERROR (Status);
  }

  //
  // Configure Fusa Parity configuration
  //
  if (IsFusaSupported()) {
    IopFusaOverrideProgramming (SiPolicy);
    if (!IsPchLinkDmi ()) {
      OpioFusaOverrideProgramming (SiPolicy);
    }
    IgdFusaOverrideProgramming(SiPolicy);
    PsfFusaOverrideProgramming (SiPolicy);
    ///
    /// Lock from OS to modify parity controls and injection registers
    ///
    FusaRegisterLock ();
  }

  Status = GetConfigBlock ((VOID *) SiPolicy, &gCpuConfigGuid, (VOID *) &CpuConfig);
  ASSERT_EFI_ERROR(Status);
  Status = GetConfigBlock ((VOID *) SiPolicy, &gSiConfigGuid, (VOID *) &SiConfig);
  ASSERT_EFI_ERROR(Status);

  Status = GetConfigBlock ((VOID *) SiPolicy, &gSaMiscPeiConfigGuid, (VOID *) &MiscPeiConfig);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPolicy, &gHostBridgePeiConfigGuid, (VOID *) &HostBridgePeiConfig);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gCpuPciePeiPreMemConfigGuid, (VOID *) &PciePeiPreMemConfig);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gVtdConfigGuid, (VOID *) &Vtd);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gGraphicsPeiPreMemConfigGuid, (VOID *) &GtPreMemConfig);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPolicy, &gGraphicsPeiConfigGuid, (VOID *) &GtConfig);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPolicy, &gGnaConfigGuid, (VOID *) &GnaConfig);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gMemoryConfigGuid, (VOID *) &MemConfig);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gCpuTraceHubPreMemConfigGuid, (VOID *) &CpuTraceHubPreMemConfig);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gTcssPeiPreMemConfigGuid, (VOID *) &TcssPeiPreMemConfig);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPolicy, &gTcssPeiConfigGuid, (VOID *) &TcssPeiConfig);
  ASSERT_EFI_ERROR(Status);

  Status = GetConfigBlock ((VOID *) SiPolicy, &gVmdPeiConfigGuid, (VOID *) &VmdPeiConfig);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gTwoLmPreMemConfigGuid, (VOID *)&TwoLmPreMemConfig);
  ASSERT_EFI_ERROR(Status);

  Status = GetConfigBlock ((VOID *) SiPolicy, &gTelemetryPeiConfigGuid, (VOID *) &TelemetryPeiConfig);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPolicy, &gCpuPcieRpConfigGuid, (VOID *)&CpuPcieRpConfig);
  ASSERT_EFI_ERROR(Status);

  Status = GetConfigBlock ((VOID *) SiPolicy, &gCpuPowerMgmtTestConfigGuid, (VOID *) &CpuPowerMgmtTestConfig);
  ASSERT_EFI_ERROR (Status);
  //
  // PSMI Configuration
  //
  ConfigurePsmi ();

  //
  // Cross IP Policy override for specific feature enabling
  //
  PeiSiPolicyOverride (SiPreMemPolicyPpi, SiPolicy);

  HobPtr = BuildGuidDataHob (&gSiConfigHobGuid, SiConfig, sizeof (SI_CONFIG));
  ASSERT (HobPtr != 0);

  ///
  /// SubsystemID programming on Internal Devices.
  ///
  DEBUG ((DEBUG_INFO, "SubsystemID programming on Internal Devices\n"));
  SiProgramSsid (SiPolicy, HgSsidOverride);

  //
  // Perform ME post mem init
  // Call before PchInit to have MbpHob data ready.
  //
  PERF_START_EX (&gPerfMePostMemGuid, NULL, NULL, AsmReadTsc (), 0x40B0);
  MePostMemInit (SiPolicy);
  PERF_END_EX (&gPerfMePostMemGuid, NULL, NULL, AsmReadTsc (), 0x40B1);

  //
  // Initializes PCH after memory services initialized
  //
  PERF_START_EX (&gPerfPchPostMemGuid, NULL, NULL, AsmReadTsc (), 0x4020);
  PchInit (SiPolicy);
  PERF_END_EX (&gPerfPchPostMemGuid, NULL, NULL, AsmReadTsc (), 0x4021);

  // call after PchInit simply because PchInitEntryPointFsp execution later expects the HOB generated from the PchInit.
  // may have the benefit for the LPSS controller initialized which maybe used for MCU communication.
  if (FspDxDiagnosticModeGet())
  {
    FspDxCheck((CONST EFI_PEI_SERVICES **)PeiServices);
    return EFI_SUCCESS; // early return for diagnostic
  }

  //
  // SA Post Mem initialization
  //
  PERF_START_EX (&gPerfSaPostMemGuid, NULL, NULL, AsmReadTsc (), 0x4030);

  DEBUG ((DEBUG_INFO, "PostMem SaInit Entry\n"));
  PostCode (0xA00);

  ///
  /// Check and Enable Panel Power (Vdd Bit)
  ///
  PostMemCheckAndForceVddOn (GtPreMemConfig);

  ///
  /// SA device configuration
  ///
  DEBUG ((DEBUG_INFO, "DeviceConfigure Start\n"));
  PostCode (0xA01);
  DeviceConfigure (HostBridgePeiConfig, GnaConfig);

  ///
  /// SA Register Mirror
  ///
  DEBUG ((DEBUG_INFO, "Sa Register Mirror Start\n"));
  SaRegisterMirror (GtPreMemConfig);

  ///
  /// Install PPI to enable Gfx PEIM
  ///
  if (GtConfig->SkipFspGop == 0) {
    if ((GtConfig->PeiGraphicsPeimInit == 1) && (GtConfig->GraphicsConfigPtr != NULL)) {
      Status = PeiServicesInstallPpi (&mEnablePeiGraphicsPpi);
      ASSERT_EFI_ERROR (Status);
    }
  }

  ///
  /// Initialize SA GNA Device
  ///
  DEBUG ((DEBUG_INFO, "Initializing SA GNA device\n"));
  PostCode (0xA16);
  GnaInit (GnaConfig);

  if (IsDekelWaRequired () == TRUE) {
    ///
    /// Initialize CPU PCIe
    ///
    DEBUG ((DEBUG_INFO, "Initializing CPU PCIe\n"));
    PostCode (0xA30);
    if (CpuPcieRpPreMemConfig != NULL) {
      PciExpressInit (CpuPcieRpPreMemConfig);
    }
  }
  ///
  /// PciExpress PostMem Initialization
  ///
  DEBUG((DEBUG_INFO, "PciExpress PostMem Initialization\n"));
  PostCode(0xA20);
  CpuPcieRpInit (SiPolicy);

  ///
  /// Check if 2LM is enabled
  ///
  if (IsTwoLmEnabled () == TRUE) {
    TwoLmPostMemInit (TwoLmPreMemConfig);
  }

  ///
  /// CPU Trace Hub Initialization
  ///
  DEBUG ((DEBUG_INFO, "Initializing ConfigureCpuTraceHub\n"));
  PostCode (0xA22);
  ConfigureCpuTraceHub (CpuTraceHubPreMemConfig->TraceHub.EnableMode);

  ///
  /// Initializing TCSS Devices
  ///
  DEBUG ((DEBUG_INFO, "Initializing TCSS\n"));
  PostCode (0xA31);
  TcssInit (SiPolicy, TcssPeiPreMemConfig, TcssPeiConfig);

  ///
  /// Update SA HOBs in PostMem
  ///
  DEBUG ((DEBUG_INFO, "UpdateSaHobPostMem Start\n"));
  PostCode (0xA02);

  ///
  /// Update HostBridge Hob in PostMem
  ///
  UpdateHostBridgeHobPostMem (HostBridgePeiConfig);

  ///
  /// Update TraceHub Hob in PostMem
  ///
  UpdateTraceHubHobPostMem (CpuTraceHubPreMemConfig);

  ///
  /// Update CpuPcie Hob in PostMem
  ///
  UpdateCpuPcieHobPostMem (PciePeiPreMemConfig, CpuPcieRpConfig);

  ///
  /// Update SA Hob in PostMem
  ///
  UpdateSaHobPostMem (GtPreMemConfig
  );
  DEBUG ((DEBUG_INFO, "PostMem SA Data HOB updated\n"));
  ///
  /// VMD Initializations if the VMD IP is Supported
  ///
  DEBUG ((DEBUG_INFO, "Initializing VMD\n"));
  PostCode (0xA33);
  VmdInit(VmdPeiConfig);

  ///
  /// PAVP Initialization
  ///
  DEBUG ((DEBUG_INFO, "Initializing Pavp\n"));
  PostCode (0xA32);
  PavpInit (GtConfig, MiscPeiConfig, GtPreMemConfig);

  ///
  /// Program PSMI Registers
  ///
  ProgramPsmiRegs (GtConfig, GtPreMemConfig);

  ///
  /// Configure root port function number mapping
  /// This has to be done before PCI enumeration
  ///
  CpuPcieRpConfigureRpfnMapping (CpuPcieRpConfig);

  ///
  /// Configure CPU CrashLog
  ///
  if ((!TelemetryPeiConfig->CpuCrashLogEnable)) {
    DEBUG ((DEBUG_INFO, "Disable CpuCrashLog\n"));
    PostCode (0xA35);
    CpuCrashLogDisable ();
  }

  ///
  /// Program Edram Mode
  ///
  if (MsrIsEdramEnable ()) {
    DEBUG ((DEBUG_INFO, "ProgramEdramMode Start\n"));
    PostCode (0xA36);
    ProgramEdramMode (HostBridgePeiConfig);
  }

  PERF_END_EX (&gPerfSaPostMemGuid, NULL, NULL, AsmReadTsc (), 0x4031);

  //
  //  IPU Initilization
  //
  DEBUG ((DEBUG_INFO, "Initializing IPU device\n"));
  PostCode (0xA14);
  IpuInit();

#ifdef CPU_CFL
  //
  // SA Security Lock down after all initialization done
  //
  PERF_START_EX (&gPerfSaSecLockPostMemGuid, NULL, NULL, AsmReadTsc (), 0x4050);
  SaSecurityLock ();
  PERF_END_EX (&gPerfSaSecLockPostMemGuid, NULL, NULL, AsmReadTsc (), 0x4051);
#endif

  //
  // Initialize processor features, performance and power management features,
  // BIOS GUARD, and overclocking etc features before RESET_CPL at post-memory phase.
  //
  PERF_START_EX (&gPerfCpuPostMemGuid, NULL, NULL, AsmReadTsc (), 0x4080);
  CpuInit ((CONST EFI_PEI_SERVICES **) PeiServices, SiPolicy);
  PERF_END_EX (&gPerfCpuPostMemGuid, NULL, NULL, AsmReadTsc (), 0x4081);

  DEBUG ((DEBUG_INFO, "ConfigureSvidVrs Start \n"));
  PostCode (0xC23);

  ///
  /// Program platform power and SVID VR's
  ///
  ConfigureSvidVrs (SiPolicy, (UINT16) CpuPowerMgmtTestConfig->ConfigTdpLevel);

  //
  // Perform AMT post mem init
  //
  PERF_START_EX (&gPerfAmtPostMemGuid, NULL, NULL, AsmReadTsc (), 0x40C0);
  AmtPostMemInit ();
  PERF_END_EX (&gPerfAmtPostMemGuid, NULL, NULL, AsmReadTsc (), 0x40C1);
  ///
  /// Initialize Late VT-d
  ///
  DEBUG ((DEBUG_INFO, "Initializing VtdLateInit\n"));
  PostCode (0xA37);
  VtdLateInit (Vtd);
  ///
  /// Update Vtd Hob in PostMem
  ///
  UpdateVtdHobPostMem (Vtd);

#ifndef CPU_CFL
  //
  // SA Security Lock down after all initialization done
  //
  PERF_START_EX (&gPerfSaSecLockPostMemGuid, NULL, NULL, AsmReadTsc (), 0x4050);
  SaSecurityLock ();
  PERF_END_EX (&gPerfSaSecLockPostMemGuid, NULL, NULL, AsmReadTsc (), 0x4051);
#endif

  //
  // Install EndOfPei callback function.
  //
#ifdef FSP_FLAG
  if (GetFspGlobalDataPointer()->FspMode == FSP_IN_DISPATCH_MODE) {
    CopyGuid (mSiInitNotifyList->Guid, &gEfiEndOfPeiSignal2PpiGuid);
  }
#endif

  Status = PeiServicesNotifyPpi (mSiInitNotifyList);
  ASSERT_EFI_ERROR (Status);

  //
  // End of SiInit notification event
  //
#ifndef FSP_FLAG
  Status = PeiServicesInstallPpi (&mEndOfSiInit);
  ASSERT_EFI_ERROR (Status);
#else
  if (GetFspGlobalDataPointer()->FspMode == FSP_IN_DISPATCH_MODE) {
    Status = PeiServicesInstallPpi (&mEndOfSiInit);
    ASSERT_EFI_ERROR (Status);
  }
#endif

  DEBUG ((DEBUG_INFO, "SiInit () - End\n"));

  return EFI_SUCCESS;
}

/**
  Silicon Initializes after memory services initialized

  @param[in] FileHandle           The file handle of the file, Not used.
  @param[in] PeiServices          General purpose services available to every PEIM.

  @retval EFI_SUCCESS             The function completes successfully
**/
EFI_STATUS
EFIAPI
SiInit (
  IN  EFI_PEI_FILE_HANDLE               FileHandle,
  IN CONST EFI_PEI_SERVICES             **PeiServices
  )
{
  EFI_STATUS                Status;

  //
  // Install PostMem phase OnPolicyInstalled callback function.
  //
  Status = PeiServicesNotifyPpi (mSiInitPostMemNotifyList);
  ASSERT_EFI_ERROR (Status);

  //
  // Install a Default Policy initialization PPI
  //
  Status = PeiServicesInstallPpi (mPeiSiDefaultPolicyInitPpiList);
  ASSERT_EFI_ERROR (Status);

  return Status;
}
