/** @file
  The PEIM implements the SA PEI Initialization.

@copyright
  INTEL CONFIDENTIAL
  Copyright 1999 - 2019 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains an 'Intel Peripheral Driver' and is uniquely identified as
  "Intel Reference Module" and is licensed for Intel CPUs and chipsets under
  the terms of your license agreement with Intel or your vendor. This file may
  be modified by the user, subject to additional terms of the license agreement.

@par Specification
**/
#include <Library/PeiServicesLib.h>
#include <Library/IoLib.h>
#include <Library/BaseMemoryLib.h>
#include <Library/DebugLib.h>
#include <Library/HobLib.h>
#include <Ppi/MemoryDiscovered.h>
#include <Ppi/EndOfPeiPhase.h>
#include <Library/PciSegmentLib.h>
#include <CpuRegs.h>
#include <Register/IgdRegs.h>
#include <Register/GnaRegs.h>
#include <PcieRegs.h>
#include <CpuAccess.h>
#include <VtdDataHob.h>
#include <Ppi/SiPolicy.h>
#include <Library/CpuMailboxLib.h>
#include <Library/SaInitLib.h>
#include <Library/PeiOcLib.h>
#include <Library/GnaInitLib.h>
#include <Library/PeiGraphicsInitLib.h>
#include <Library/PeiCpuTraceHubLib.h>
#include <TraceHubDataHob.h>
#include <Library/PeiServicesTablePointerLib.h>
#include <Library/ConfigBlockLib.h>
#include <Library/PeiDisplayInitLib.h>
#include <Library/GpioPrivateLib.h>
#include <Library/GpioNativePads.h>
#include <Library/PostCodeLib.h>
#include <Library/CpuDmiInfoLib.h>
#include <Library/CpuPlatformLib.h>
#include <Library/PeiHostBridgeInitLib.h>
#include <SaConfigHob.h>
#include <Library/CpuPcieRpLib.h>

typedef struct {
  UINT8 DeviceNumber;
  UINT8 FunctionNumber;
  UINT8 SvidRegOffset;
} SA_SVID_SID_INIT_ENTRY;

/*
  Required BIOS configurations for PTM
  System BIOS must set bit 5 of 0x5880 register to lock the configuration.

  System Agent Configuration Locking
  For reliable operation and security, System BIOS must set the following bits:
   1. Lock GGC from writes by setting the B0.D0.F0.R050 [0] = 1b
   2. Lock PAVP settings by setting MCHBAR offset 5500h [0] = 1b
   3. Lock DPR by setting the B0.D0.F0.R05Ch [0] = 1b
   4. Lock ME memory range configuration by setting the B0.D0.F0.R078h [10] = 1b
   5. Lock remap base and limit by setting the B0.D0.F0.R090h [0] = 1b and
      B0.D0.F0.R098h [0] = 1b
   6. Lock TOM by setting the B0.D0.F0.R0A0h [0] = 1b
   7. Lock TOUUD by setting the B0.D0.F0.R0A8h [0] = 1b
   8. Lock BDSM by setting the B0.D0.F0.R0B0h [0] = 1b
   9. Lock BGSM by setting the B0.D0.F0.R0B4h [0] = 1b
   10. Lock TSEG Memory Base by setting the B0.D0.F0.R0B8h [0] = 1b
   11. Lock TOLUD by setting the B0.D0.F0.R0BCh [0] = 1b
   12. Lock Memory controller configuration by setting the MCHBAR offset 50FCh [7:0] = 8Fh
   13. Lock primary channel arbiter weights by setting the MCHBAR offset 7000h [31] = 1b
       MCHBAR offset 77FCh [0] = 1b MCHBAR offset 7BFCh [0] = 1b and MCHBAR offset 6800h [31] = 1b
   14. Lock UMA GFX by setting the MCHBAR offset 6020h [0] = 1b
   15. Lock VTDTRK by setting the MCHBAR offset 63FCh [0] = 1b
   16. Read and write back MCHBAR offset 6030h [31:0].
   17. Read and write back MCHBAR offset 6034h [31:0].
*/
//
// BaseAddr 0 for mSaSecurityRegisters means registers of B0: D0: F0.
//
GLOBAL_REMOVE_IF_UNREFERENCED BOOT_SCRIPT_REGISTER_SETTING mSaSecurityRegistersGen3[] = {
                 {0,  0x0050,  0xFFFFFFFF,  BIT0},
                 {0,  0x005C,  0xFFFFFFFF,  BIT0},
                 {0,  0x0078,  0xFFFFFFFF,  BIT10},
                 {0,  0x0090,  0xFFFFFFFF,  BIT0},
                 {0,  0x0098,  0xFFFFFFFF,  BIT0},
                 {0,  0x00A0,  0xFFFFFFFF,  BIT0},
                 {0,  0x00A8,  0xFFFFFFFF,  BIT0},
                 {0,  0x00B0,  0xFFFFFFFF,  BIT0},
                 {0,  0x00B4,  0xFFFFFFFF,  BIT0},
                 {0,  0x00B8,  0xFFFFFFFF,  BIT0},
                 {0,  0x00BC,  0xFFFFFFFF,  BIT0},
  { (UINTN) FixedPcdGet64 (PcdMchBaseAddress),  0x5500,  0xFFFFFFFF,  BIT0},
  { (UINTN) FixedPcdGet64 (PcdMchBaseAddress),  0x50FC,  0xFFFFFFFF,  0x8F},
  { (UINTN) FixedPcdGet64 (PcdMchBaseAddress),  0x7000,  0xFFFFFFFF,  BIT31},
  { (UINTN) FixedPcdGet64 (PcdMchBaseAddress),  0x77FC,  0xFFFFFFFF,  BIT0},
  { (UINTN) FixedPcdGet64 (PcdMchBaseAddress),  0x7BFC,  0xFFFFFFFF,  BIT0},
  { (UINTN) FixedPcdGet64 (PcdMchBaseAddress),  0x6800,  0xFFFFFFFF,  BIT31},
  { (UINTN) FixedPcdGet64 (PcdMchBaseAddress),  0x6020,  0xFFFFFFFF,  BIT0},
  { (UINTN) FixedPcdGet64 (PcdMchBaseAddress),  0x63FC,  0xFFFFFFFF,  BIT0},
  { (UINTN) FixedPcdGet64 (PcdMchBaseAddress),  0x6030,  0xFFFFFFFF,  0},
  { (UINTN) FixedPcdGet64 (PcdMchBaseAddress),  0x6034,  0xFFFFFFFF,  0},
  { (UINTN) FixedPcdGet64 (PcdMchBaseAddress),  0x5880,  0xFFFFFFFF,  BIT5}
};


/**
  Function to handle SA at end of PEI

  @retval None

**/
EFI_STATUS
EFIAPI
SaOnEndOfPei (
  VOID
  )
{
  UINTN           McD2BaseAddress;
  CPU_GENERATION  CpuGeneration;

  CpuGeneration = GetCpuGeneration();
  if (CpuGeneration != EnumCmlCpu) {
    EndOfPeiCheckAndForceVddOn ();
  }
  SaS3ResumeAtEndOfPei ();

  ///
  /// Clear IGD GttMmAdr, GmAdr and disable Bus Master and Memory Access for 0/2/0
  ///
  McD2BaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, IGD_BUS_NUM, IGD_DEV_NUM, IGD_FUN_NUM, 0);
  PciSegmentWrite32 (McD2BaseAddress + PCI_COMMAND_OFFSET, 0x0);
  PciSegmentWrite32 (McD2BaseAddress + R_SA_IGD_GTTMMADR, 0x0);
  PciSegmentWrite32 (McD2BaseAddress + R_SA_IGD_GTTMMADR + 0x4, 0x0);
  PciSegmentWrite32 (McD2BaseAddress + R_SA_IGD_GMADR, 0x0);
  PciSegmentWrite32 (McD2BaseAddress + R_SA_IGD_GMADR + 4, 0x0);

  return EFI_SUCCESS;
}

///
/// Functions
///
/**
  This function handles SA S3 resume task

  @retval EFI_STATUS       - Always return EFI_SUCCESS
**/
EFI_STATUS
SaS3ResumeAtEndOfPei (
  VOID
  )
{
  EFI_BOOT_MODE               BootMode;
  EFI_STATUS                  Status;
#if FixedPcdGetBool(PcdFspBinaryEnable) == 0
  SI_POLICY_PPI               *SiPolicyPpi;
  HOST_BRIDGE_PEI_CONFIG      *HostBridgePeiConfig;
  BOOLEAN                     SkipPamLock;
#endif

  Status = PeiServicesGetBootMode (&BootMode);
  DEBUG ((DEBUG_INFO, "[SA] BootMode = %X\n", BootMode));
  if ((Status != EFI_SUCCESS) || (BootMode != BOOT_ON_S3_RESUME)) {
    return EFI_SUCCESS;
  }
  DEBUG ((DEBUG_INFO, "SaS3ResumeAtEndOfPei Callback Entry\n"));
  PostCode (0xA70);
  //
  // SA S3 tasks that must be done after S3 Boot Script Restore finished.
  //
#if FixedPcdGetBool(PcdFspBinaryEnable) == 0
  //
  // In FSP S3 resume path, PAM lock is took care by Notify Phase API, so skipped it here.
  //
  SiPolicyPpi = NULL;
  HostBridgePeiConfig = NULL;
  SkipPamLock = FALSE;

  Status = PeiServicesLocatePpi (
             &gSiPolicyPpiGuid,
             0,
             NULL,
             (VOID **) &SiPolicyPpi
             );
  if ((Status == EFI_SUCCESS) && (SiPolicyPpi != NULL)) {
    Status = GetConfigBlock ((VOID *) SiPolicyPpi, &gHostBridgePeiConfigGuid, (VOID *) &HostBridgePeiConfig);
    ASSERT_EFI_ERROR (Status);
    if (Status == EFI_SUCCESS) {
      SkipPamLock  = (BOOLEAN) (UINTN) HostBridgePeiConfig->SkipPamLock;
    }
  }

  if (SkipPamLock == FALSE) {
    DEBUG ((DEBUG_INFO, "S3 PAM_LOCK!!\n"));
    PciSegmentOr32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, 0, 0, 0, R_SA_PAM0), BIT0);
  }
#endif

  DEBUG ((DEBUG_INFO, "SaS3ResumeAtEndOfPei Callback Exit\n"));
  PostCode (0xA7F);
  return EFI_SUCCESS;
}
typedef union {
  struct {
    UINT32  Low;
    UINT32  High;
  } Data32;
  UINT64 Data;
} UINT64_STRUCT;
/**
  This function does SA security lock
**/
VOID
SaSecurityLock (
  VOID
  )
{
  UINT8                      Index;
  UINT64                     BaseAddress;
  UINT32                     RegOffset;
  UINT32                     Data32And;
  UINT32                     Data32Or;
  SI_PREMEM_POLICY_PPI       *SiPreMemPolicyPpi;
  SA_MISC_PEI_PREMEM_CONFIG  *MiscPeiPreMemConfig;
  BOOLEAN                    Device4Enable;
  UINT8                      LockPTMregs;
  HOST_BRIDGE_PEI_CONFIG     *HostBridgePeiConfig;
  EFI_STATUS                 Status;
  SI_POLICY_PPI              *SiPolicyPpi;
  UINT64_STRUCT              DmiBar;
  CPU_GENERATION             CpuGeneration;

  CpuGeneration             = GetCpuGeneration();

  DEBUG ((DEBUG_INFO, "SaSecurityLock Start\n"));
  PostCode (0xA50);

  if (CpuGeneration == EnumCmlCpu) {
    Device4Enable = FALSE;
    LockPTMregs = 1;

    ///
    /// 17.2 System Agent Security Lock configuration
    ///
    Status = PeiServicesLocatePpi (
              &gSiPolicyPpiGuid,
              0,
              NULL,
              (VOID **) &SiPolicyPpi
              );
    if ((Status == EFI_SUCCESS) && (SiPolicyPpi != NULL)) {
      Status = GetConfigBlock ((VOID *) SiPolicyPpi, &gHostBridgePeiConfigGuid, (VOID *) &HostBridgePeiConfig);
      ASSERT_EFI_ERROR (Status);
      if (Status == EFI_SUCCESS) {
        Device4Enable = (UINT8) (UINTN) HostBridgePeiConfig->Device4Enable;
      }
    }
    Status = PeiServicesLocatePpi (
              &gSiPreMemPolicyPpiGuid,
              0,
              NULL,
              (VOID **) &SiPreMemPolicyPpi
              );
    if ((Status == EFI_SUCCESS) && (SiPolicyPpi != NULL)) {
      Status = GetConfigBlock ((VOID *) SiPreMemPolicyPpi, &gSaMiscPeiPreMemConfigGuid, (VOID *) &MiscPeiPreMemConfig);
      if (Status == EFI_SUCCESS) {
        LockPTMregs = MiscPeiPreMemConfig->LockPTMregs;
      }
    }

    DEBUG ((DEBUG_INFO, "Initializing SaSecurityLock\n"));
    for (Index = 0; Index < (sizeof (mSaSecurityRegistersGen3) / sizeof (BOOT_SCRIPT_REGISTER_SETTING)); Index++) {
      BaseAddress = mSaSecurityRegistersGen3[Index].BaseAddr;
      RegOffset   = mSaSecurityRegistersGen3[Index].Offset;
      Data32And   = mSaSecurityRegistersGen3[Index].AndMask;
      Data32Or    = mSaSecurityRegistersGen3[Index].OrMask;
      if (BaseAddress != (UINTN) PcdGet64 (PcdMchBaseAddress)) {
        BaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, SA_MC_DEV, SA_MC_FUN, 0);
      }
      if (Device4Enable == TRUE) {
        if (RegOffset ==  0x50FC) {
          Data32Or = 0x87; ///< unlock bit3 if Device (0,4,0) is enabled.
        }
      }
      if (LockPTMregs == 0) {
        if (RegOffset ==  0x5880) {
          continue;
        }
      }
      if (BaseAddress != (UINTN) PcdGet64 (PcdMchBaseAddress)) {
        PciSegmentAndThenOr32 (BaseAddress + RegOffset, Data32And, Data32Or);
      } else {
        MmioAndThenOr32 ((UINTN)BaseAddress + RegOffset, Data32And, Data32Or);
      }
    }
    ///
    /// Lock processor/chipset BAR registers
    ///
    ///
    /// Set DMI PCIELOCK2.LOCK_GENERAL2
    ///
    DmiBar.Data32.High = PciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, 0, 0, R_SA_DMIBAR + 4));
    DmiBar.Data32.Low  = PciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, 0, 0, R_SA_DMIBAR));
    DmiBar.Data       &= ~BIT0;
    MmioOr32 ((UINTN) DmiBar.Data + R_SA_DMIBAR_PCIELOCK2_OFFSET, BIT0);
  } else {
    DmiBar.Data32.High = PciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, 0, 0, R_SA_DMIBAR + 4));
    DmiBar.Data32.Low  = PciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, 0, 0, R_SA_DMIBAR));
    DmiBar.Data       &= (UINT64) ~BIT0;

    ///
    /// System Agent Security Lock configuration
    ///

    Status = PeiServicesLocatePpi (
              &gSiPolicyPpiGuid,
              0,
              NULL,
              (VOID **) &SiPolicyPpi
              );
    if ((Status == EFI_SUCCESS) && (SiPolicyPpi != NULL)) {
      if (Status == EFI_SUCCESS) {
      }
    }

    if (!IsPchLinkDmi ()) {
      ///
      /// Enable pOPIO Clockgating during POSTBOOT_SAI Transition before setting pOPIO Security Lock Bit
      ///
      MmioOr32 ((UINTN) (DmiBar.Data + R_SA_DMIBAR_OPIO_PHY_CONTROL), N_SA_DMIBAR_OPIO_CLOCK_GATE);
      if (MmioRead32 (TXT_PUBLIC_BASE + 0x200) & BIT31) {
        ///
        /// For ULT/ULX set DMIBAR offset 0xB34 [26] to lockdown OPI debug on production systems
        ///
        MmioOr32 ((UINTN) (DmiBar.Data + R_SA_DMIBAR_OPIO_PHY_CONTROL), N_SA_DMIBAR_OPIO_SECURITY_LOCK_BIT);
      }
    }
  }

  DEBUG ((DEBUG_INFO, "SaSecurityLock End\n"));
  PostCode (0xA5F);
}

/**
This function performs SA internal devices enabling/disabling

@param[in] HostBridgePeiConfig - Instance of HOST_BRIDGE_PEI_CONFIG
@param[in] GnaConfig - Instance of GNA_CONFIG

**/
VOID
DeviceConfigure (
  IN    HOST_BRIDGE_PEI_CONFIG  *HostBridgePeiConfig,
  IN    GNA_CONFIG              *GnaConfig
  )
{
  UINT64         McD0BaseAddress;
  UINT32         DevEn;
  CPU_GENERATION CpuGeneration;

  CpuGeneration   = GetCpuGeneration();
  McD0BaseAddress = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, 0, 0, 0);
  DevEn = PciSegmentRead32 (McD0BaseAddress + R_SA_DEVEN);
  if (CpuGeneration == EnumCmlCpu) {
    ///
    /// Enable/Disable CHAP device (B0,D7,F0).
    ///
    if (HostBridgePeiConfig->ChapDeviceEnable) {
      DevEn |= B_SA_DEVEN_D7EN_MASK;
    } else {
      DevEn &= ~B_SA_DEVEN_D7EN_MASK;
    }
  }
  ///
  /// Enable/Disable Thermal device (B0,D4,F0).
  ///
  if (HostBridgePeiConfig->Device4Enable) {
    DevEn |= B_SA_DEVEN_D4EN_MASK;
  } else {
    DevEn &= ~B_SA_DEVEN_D4EN_MASK;
  }
  ///
  /// Enable/Disable GNA device (B0,D8,F0).
  ///
  if (GnaConfig->GnaEnable) {
    DevEn |= B_SA_DEVEN_D8EN_MASK;
  } else {
    DevEn &= ~B_SA_DEVEN_D8EN_MASK;
  }

  PciSegmentWrite32 (McD0BaseAddress + R_SA_DEVEN, DevEn);
  return;
}
/**
  Print SA PCI space in Debug log.

  @retval None
**/
VOID
SaPciPrint (
  VOID
  )
{
  UINT64 PciBase;
  UINT8  Device;
  UINT8  i;
  UINT8  j;

  for (Device = 0; Device <= 8; Device++) {
    if ((PcdGetBool (PcdSaPciPrint)) || (Device == 0) || (Device == 2)) {
      PciBase = PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, 0, Device, 0, 0);
      if (PciSegmentRead16 (PciBase) != 0xFFFF) {
        DEBUG ((DEBUG_INFO, "\nPrinting PCI space for device %x\n  ", Device));
        for (i = 0; i <= 0xF ; i++) {
          DEBUG ((DEBUG_INFO,"  %2X",i));
        }
        for (i = 0; i <= 0xF; i++) {
          DEBUG ((DEBUG_INFO, "\n%2X", (i * 0x10)));
          for (j = 0; j <= 0xF; j++) {
            DEBUG ((DEBUG_INFO, "  %2X", PciSegmentRead8 (PciBase + (i * 0x10) + j)));
          }
        }
      }
    }
  }

  DEBUG ((DEBUG_INFO, "\n"));
}

/**
  Set SAPMCTL Register.

  @retval None
**/
VOID
SetSaPmCtlReg (
  VOID
  )
{
  EFI_STATUS                  Status;
  UINT64_STRUCT               MchBar;
  UINT32                      Data32And;
  UINT32                      Data32Or;
  SI_POLICY_PPI               *SiPolicyPpi;
  HOST_BRIDGE_PEI_CONFIG      *HostBridgePeiConfig;

  SiPolicyPpi = NULL;

  ///
  /// Get policy settings through the SaPolicy PPI
  ///
  Status = PeiServicesLocatePpi (&gSiPolicyPpiGuid, 0, NULL, (VOID **) &SiPolicyPpi);
  ASSERT_EFI_ERROR (Status);

  Status = GetConfigBlock ((VOID *) SiPolicyPpi, &gHostBridgePeiConfigGuid, (VOID *) &HostBridgePeiConfig);
  ASSERT_EFI_ERROR (Status);

  MchBar.Data32.High = PciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, 0, 0, R_SA_MCHBAR + 4));
  MchBar.Data32.Low  = PciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, 0, 0, R_SA_MCHBAR));
  MchBar.Data       &= (UINT64) ~BIT0;

  Data32And = (UINT32) ~(BIT10 | BIT9);
  Data32Or  = 0x3 << 9;

MmioAndThenOr32 ((UINTN) MchBar.Data + R_SA_MCHBAR_SAPMCTL_OFFSET, Data32And, Data32Or);
}

/**
  This function is to Set BIOS_RESET_CPL bits.

  @retval None
**/
VOID
SetBiosResetCpl (
  VOID
)
{
  UINT64_STRUCT          MchBar;

  MchBar.Data32.High = PciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, 0, 0, R_SA_MCHBAR + 4));
  MchBar.Data32.Low  = PciSegmentRead32 (PCI_SEGMENT_LIB_ADDRESS (SA_SEG_NUM, SA_MC_BUS, 0, 0, R_SA_MCHBAR));
  MchBar.Data       &= (UINT64) ~BIT0;

  DEBUG ((DEBUG_INFO, "Set BIOS_RESET_CPL to indicate all configurations complete\n"));
  PostCode (0xA61);
  MmioOr8 ((UINTN) MchBar.Data + R_SA_MCHBAR_BIOS_RESET_CPL_OFFSET, BIT0 | BIT1);
}

/**
  Update SA Hob in PostMem

  @param[in]  GtConfig                 - Instance of GRAPHICS_PEI_PREMEM_CONFIG

  @retval EFI_SUCCESS
**/
EFI_STATUS
UpdateSaHobPostMem (
  IN       GRAPHICS_PEI_PREMEM_CONFIG  *GtPreMemConfig
)
{
  SA_CONFIG_HOB               *SaConfigHob;

  ///
  /// Locate HOB for SA Config Data
  ///
  SaConfigHob = (SA_CONFIG_HOB *) GetFirstGuidHob (&gSaConfigHobGuid);
  if (SaConfigHob != NULL) {
    SaConfigHob->ApertureSize = (UINT8)(GtPreMemConfig->ApertureSize);
  }

  return EFI_SUCCESS;
}

