### @file
# Tool to generate encoded source file for Catalog usage.
#
# @copyright
#  INTEL CONFIDENTIAL
#  Copyright 2016 - 2019 Intel Corporation.
#
#  The source code contained or described herein and all documents related to the
#  source code ("Material") are owned by Intel Corporation or its suppliers or
#  licensors. Title to the Material remains with Intel Corporation or its suppliers
#  and licensors. The Material may contain trade secrets and proprietary and
#  confidential information of Intel Corporation and its suppliers and licensors,
#  and is protected by worldwide copyright and trade secret laws and treaty
#  provisions. No part of the Material may be used, copied, reproduced, modified,
#  published, uploaded, posted, transmitted, distributed, or disclosed in any way
#  without Intel's prior express written permission.
#
#  No license under any patent, copyright, trade secret or other intellectual
#  property right is granted to or conferred upon you by disclosure or delivery
#  of the Materials, either expressly, by implication, inducement, estoppel or
#  otherwise. Any license under such intellectual property rights must be
#  express and approved by Intel in writing.
#
#  Unless otherwise agreed by Intel in writing, you may not remove or alter
#  this notice or any other notice embedded in Materials by Intel or
#  Intel's suppliers or licensors in any way.
#
#  This file contains 'Framework Code' and is licensed as such under the terms
#  of your license agreement with Intel or your vendor. This file may not be
#  modified, except as allowed by additional terms of your license agreement.
#
# @par Specification Reference:
#
# @par Glossary:
###

import os
import sys
import re
import argparse
import time
try:
  import pickle as Pick
except:
  import cPickle as Pick
import portalocker

__prog__        = 'CatalogEncoder'
__version__     = '%s Version %s' % (__prog__, '0.6')
__copyright__   = 'Copyright (c) 2019, Intel Corporation. All rights reserved.\n'
__description__ = 'The script is used to change DEBUG macro from normal to catalog.\n'

mConfigureTable = {
  "MaxParameter":     6,
  "CurrentLine":      None,
  "CurrentHexNumber": 0x1001,
  "HexNumber":        [],
  "DebugMacro":       [],
  "Line":             [],
  "InputFileContent": '',
  "OutputFile":       None,
  "XmlFile":          None,
  "XmlFileContent":   '',
  "OutputDebugFile":  None
}
mIgnoreList = [
  (
    """
    (
    %                                  # literal "%"
    (?:[-+0 #]{0,5})                   # optional flags
    (?:,|\d+|\*)?                      # width
    (?:\.(?:\d+|\*))?                  # precision
    (?:h|l|ll|w|I|I32|I64)             # size
    [diouxXeEfgGAnpZtr]                # type
    )
    """,
    "%08x"
  ),
  (
    """
    (
    %                                  # literal "%"
    (?:h|l|ll|w|I|I32|I64)             # size
    (?:[-+0 #]{0,5})                   # optional flags
    (?:,|\d+|\*)?                      # width
    (?:\.(?:\d+|\*))?                  # precision
    [diouxXeEfgGAnpZtr]                # type
    )
    """,
    "%08x"
  ),
  (
    """
    (
    %                                  # literal "%"
    (?:[-+0 #]{0,5})                   # optional flags
    (?:,|\d+|\*)?                      # width
    (?:\.(?:\d+|\*))?                  # precision
    [ieEfgGAnZtr]                      # type
    )
    """,
    "%08x"
  ),
  (
    """
    (
    %                                  # literal "%"
    (?:[h\.\*]{0,2})                   # optional flags
    [cCasS]                            # type
    )
    """,
    "%08x"
  ),
  (
    """
    (
    %                                  # literal "%"
    (?:[-+0 #]{0,5})                   # optional flags
    (?:,|\d+|\*)?                      # width
    (?:\.(?:\d+|\*))?                  # precision
    (?:l){1,2}                         # type
    )
    """,
    "%08x"
  )
]

## Set Ignore Parameter that PVT not supported format
#
#   Set Ignore Parameter that PVT not supported format
#
#   @param[in]  Format              Format
#
#   retval  FormatList              Format List
#
def GetFormatList (Format):
  global mIgnoreList
  global mConfigureTable

  #@Todo: It needs to remove after all source code follow correct format ussage.
  Format = Format.replace ("%02:", "%02x").replace ("0x%\\", "0x%x\\")

  MaxParameter      = mConfigureTable['MaxParameter']
  EfiFormat = '''\
  (
  #==================================================
  %                                  # literal "%"
  (?:                                # first option
  (?:[-+0 #]{0,5})                   # optional flags
  (?:,|\d+|\*)?                      # width
  (?:\.(?:\d+|\*))?                  # precision
  (?:h|l|ll|w|I|I32|I64)?            # size
  [diouxXeEfgGAnpZtr]                # type
  ) |                                # OR
  #==================================================
  %                                  # literal "%"
  (?:                                # first option
  (?:h|l|ll|w|I|I32|I64)?            # size
  (?:[-+0 #]{0,5})                   # optional flags
  (?:,|\d+|\*)?                      # width
  (?:\.(?:\d+|\*))?                  # precision
  [diouxXeEfgGAnpZtr]                # type
  ) |                                # OR
  #==================================================
  %                                  # literal "%"
  (?:                                # first option
  (?:[-+0 #]{0,5})                   # optional flags
  (?:,|\d+|\*)?                      # width
  (?:\.(?:\d+|\*))?                  # precision
  (?:l){1,2}                         # type
  ) |                                # OR
  #==================================================
  %                                  # literal "%"
  (?:                                # first option
  (?:[h\.\*]{0,2})                   # optional flags
  [cCasS]                            # type
  ) |                                # OR
  #==================================================
  %%                                 # literal "%%"
  )
  '''
  StartPosition  = 0
  FormatDataList = []
  FormatList     = []

  for Item in re.finditer (EfiFormat, Format, re.X):
    FormatDataList.append (Format[StartPosition : Item.end (1)])
    StartPosition = Item.end ()
  FormatDataList.append (Format[StartPosition:])

  for Index in range (len (FormatDataList)):
    if (FormatDataList[Index] == ''):
      continue
    if (FormatDataList[Index][-1] == '\"'):
      continue
    for (Origin, Modify) in mIgnoreList:
      if (re.search (Origin, FormatDataList[Index], re.X) != None):
        FormatDataList[Index] = re.sub (Origin, Modify, FormatDataList[Index], 0, re.X)
        return (FormatList, False)
        break

  FormatDataListLength = len (FormatDataList)
  for Index in range (0, FormatDataListLength, MaxParameter):
    FormatList.append ('\"' + ''.join (FormatDataList[Index : Index + 6]) + '\"')

  return (FormatList, True)

## Generate new Id for corresponding debug macro
#
#   Generate new Hex Number for corresponding debug macro and update in global list
#
#   @param[in]  DebugMacroMessage          Debug macro message
#   @param[in]  IsSupported                Debug macro format whether supported or not
#
def GenerateId (DebugMacroMessage, IsSupported):
  global mConfigureTable

  mConfigureTable['HexNumber'].append (mConfigureTable['CurrentHexNumber'])
  mConfigureTable['DebugMacro'].append (DebugMacroMessage.strip ())
  if (mConfigureTable['CurrentLine'] != None):
    mConfigureTable['Line'].append ((mConfigureTable['CurrentLine'], IsSupported))
  mConfigureTable['CurrentHexNumber'] += 1

## Parse debug macro and convert parameter list
#
#   Parse debug macro and convert parameter list
#
#   @param[in]  DebugMacro          Debug macro look like DEBUG ((Level, Parameter1, Parameter2, ..., ParameterN))
#
#   retval  DebugBlock              Debug block look like DEBUG ((Level,
#   retval  FormatList              If format parameter flags > Max supported count, it will be splitted.
#   retval  ParameterList           Parameter List look like [Parameter1, Parameter2, ..., ParameterN]
#   retval  IsSupported             Check format whether follow normal usage
#
def ParseDebugMacro (DebugMacro):
  Result        = DebugMacro.split (",", 1)
  FormatList    = []
  ParaList      = []
  IsSupported   = False


  DebugBlock            = Result[0].replace ('\n', '')

  #
  # Get FormatBlock
  #
  FormatBlock           = ""
  Parttern              = re.compile (r"\)\s*\)\s*\;")
  FormatParaBlock       = Parttern.sub ("", Result[1]).replace ('\n', '')
  FormatParaBlockLength = len (FormatParaBlock)
  FormatParaList        = []
  FormatItem            = ''
  Index                 = 0
  while (Index < FormatParaBlockLength):
    if (FormatParaBlock[Index] == '\\'):
      Index += 1
    elif (FormatParaBlock[Index] == '\"'):
      FormatItem += FormatParaBlock[Index]
      Index += 1
      CountC = 1
      IsSkip = False
      while (CountC > 0):
        if (IsSkip == True):
          FormatItem += FormatParaBlock[Index]
          Index += 1
          IsSkip = False
          continue
        elif (IsSkip == False and FormatParaBlock[Index] == '\\'):
          FormatItem += FormatParaBlock[Index]
          Index += 1
          IsSkip = True
          continue
        elif (FormatParaBlock[Index] == '\"'):
          CountC -= 1
        FormatItem += FormatParaBlock[Index]
        Index += 1
    elif (FormatParaBlock[Index] == '('):
      FormatItem += FormatParaBlock[Index]
      Index += 1
      CountL = 1
      while (CountL > 0):
        if (FormatParaBlock[Index] == '('):
          CountL += 1
        elif (FormatParaBlock[Index] == ')'):
          CountL -= 1
        FormatItem += FormatParaBlock[Index]
        Index += 1
    elif (FormatParaBlock[Index] == ','):
      FormatParaList.append (FormatItem)
      FormatItem = ''
      Index += 1
    else:
      FormatItem += FormatParaBlock[Index]
      Index += 1
  FormatParaList.append (FormatItem)

  FormatBlock = FormatParaList[0].strip ()
  if (FormatBlock[0] == '\"'):
    FormatItem = ''
    FormatBlockLength = len (FormatBlock)
    Index             = 0
    while (Index < FormatBlockLength):
      if (FormatBlock[Index] == '\\'):
        Index += 1
      elif (FormatBlock[Index] == '\"'):
        Index += 1
        CountC = 1
        IsSkip = False
        while (CountC > 0):
          if (IsSkip == True):
            FormatItem += FormatBlock[Index]
            Index += 1
            IsSkip = False
            continue
          elif (IsSkip == False and FormatBlock[Index] == '\\'):
            FormatItem += FormatBlock[Index]
            Index += 1
            IsSkip = True
            continue
          elif (FormatBlock[Index] == '\"'):
            CountC -= 1
            Index += 1
            continue
          else:
            FormatItem += FormatBlock[Index]
            Index += 1
      else:
        Index += 1
    (FormatList, IsSupported) = GetFormatList (FormatItem)
  else:
    FormatList.append (FormatBlock)

  ParaList = FormatParaList[1:]
  return (DebugBlock, FormatList, ParaList, IsSupported)

## Convert debug macro to catalog.
#
#   Convert debug macro format from normal to catalog
#
#   @param[in]  DebugMacroNormal          Normal format debug macro
#
#   retval  DebugMacroCatalog             Catalog format debug macro
#
def ConvertDebugMacroToCatalog (DebugMacroNormal):
  global mConfigureTable

  (DebugBlock, FormatList, ParaList, IsSupported) = ParseDebugMacro (DebugMacroNormal)
  if (IsSupported == False):
    return DebugMacroNormal

  MaxParameter      = mConfigureTable['MaxParameter']
  DebugMacroCatalog = ""
  FormatIndex       = 0
  ParaListCount     = len (ParaList)
  for Format in FormatList:
    GenerateId (Format, IsSupported)
    DebugMacroCatalog += DebugBlock + ', NULL'
    if (ParaListCount >= MaxParameter):
      DebugMacroCatalog += (", %d" % (MaxParameter))
      ParaListCount -= 6
    else:
      DebugMacroCatalog += (", %d" % (ParaListCount))

    DebugMacroCatalog += (", 0x%x" % (mConfigureTable['CurrentHexNumber'] - 1))
    for Index in range (FormatIndex * MaxParameter, FormatIndex * MaxParameter + MaxParameter):
      if (Index >= len (ParaList)):
        break
      else:
        DebugMacroCatalog += ", " + ParaList[Index].strip ()
    FormatIndex += 1
    DebugMacroCatalog += "));"

  return DebugMacroCatalog

## Get debug macros from source file.
#
#   Get debug macros list, non debug macros list and IsExistDebugMacro from source file.
#
#   @param[in]  SourceFileContent       Source file content
#
#   retval  IsExistDebugMacro           Source file whether exist debug macro
#   retval  MatchedList                 Array is used to store debug macros
#   retval  UnmatchedList               Array is used to store none debug macros
#
def GetDebugMacro (SourceFileContent):
  P1            = re.compile (r"\bDEBUG[\w_]*\s*\(\s*\(\s*[|\s\w]+,")
  P2            = re.compile (r"\)\s*\)\s*\;")
  S             = 0
  E             = len (SourceFileContent)
  MatchedList   = []
  UnmatchedList = []

  while (S < E):
    G1 = P1.search (SourceFileContent, S)
    if G1:
      S1 = G1.start ()
      G2 = P2.search (SourceFileContent, S1)
      E1 = G2.end ()
      UnmatchedList.append (SourceFileContent[S:S1])
      MatchedList.append (SourceFileContent[S1:E1])
      S = E1
    else:
      UnmatchedList.append (SourceFileContent[S:E])
      break
  IsExistDebugMacro = len (MatchedList) != 0

  return (IsExistDebugMacro, MatchedList, UnmatchedList)

## Get arguments from user.
#
#   retval  Args                    Argument which assign from user
#
def GetArgs ():
  ArgParser = argparse.ArgumentParser (
                         prog = __prog__,
                         description = __copyright__
                         )
  ArgParser.add_argument (
              "-V",
              "--Version",
              action = 'version',
              version = __version__
              )
  ArgParser.add_argument (
              "-I",
              "--InputFile",
              dest = "InputFile",
              help = ("Specify the absolute or relative path of source file."),
              required = True
              )
  ArgParser.add_argument (
              "-O",
              "--OutputFile",
              dest = "OutputFile",
              help = ("Specify the absolute or relative path of encoded source file."),
              required = True
              )
  ArgParser.add_argument (
              "-X",
              "--XmlFile",
              dest = "XmlFile",
              help = ("Specify the absolute or relative path of output Xml file.")
              )
  ArgParser.add_argument (
              "-M",
              "--Mode",
              dest = "Mode",
              help = ("Specify encode mode: HEX. (Default: HEX)"),
              default = "HEX"
              )
  ArgParser.add_argument (
              "-OD",
              "--OutputDebugFile",
              dest = "OutputDebugFile",
              help = ("Specify the absolute or relative path of debug log file.")
              )
  return ArgParser.parse_args ()

## Verifying all arguments is valid
#
#   Verifying all arguments which assign from user is legal
#
#   @param[in]  Args                    Argument which assign from user
#
#   retval  True                        Argument is valid
#   retval  False                       Argument isn't valid
#
def IsArgsValid (Args):
  global mConfigureTable

  if (Args.Mode != None):
    if (Args.Mode != "HEX"):
      print ("-M or --Mode = %s is unsupported.", Args.Mode)
      return False

  if (Args.InputFile != None):
    with open (Args.InputFile, 'r') as File:
      portalocker.lock (File, portalocker.LOCK_EX)
      mConfigureTable['InputFileContent'] = File.read ()

  if (Args.OutputFile != None):
    pass

  if (Args.OutputDebugFile != None):
    while (True):
      try:
        mConfigureTable['OutputDebugFile'] = open (Args.OutputDebugFile, 'a')
        portalocker.lock (mConfigureTable['OutputDebugFile'], portalocker.LOCK_EX)
        break
      except:
        time.sleep (0.01)

  if (Args.XmlFile != None):
    while (True):
      try:
        if (mConfigureTable['XmlFile'] != None):
          mConfigureTable['XmlFile'].close ()
          mConfigureTable['XmlFile'] = None
        if (os.path.isfile (Args.XmlFile) == True):
          mConfigureTable['XmlFile'] = open (Args.XmlFile, 'r+')
          portalocker.lock (mConfigureTable['XmlFile'], portalocker.LOCK_EX)
          mConfigureTable['XmlFileContent'] = mConfigureTable['XmlFile'].read ()
        else:
          with open (os.path.split (__file__)[0] + os.sep + "BIOS_Decoder.xml.template", "r") as File:
            portalocker.lock (File, portalocker.LOCK_EX)
            mConfigureTable['XmlFileContent'] = File.read ()
          mConfigureTable['XmlFile'] = open (Args.XmlFile, 'w+')
          portalocker.lock (mConfigureTable['XmlFile'], portalocker.LOCK_EX)
          mConfigureTable['XmlFile'].write (mConfigureTable['XmlFileContent'])
          pass
        break
      except:
        time.sleep (0.01)

  return True

## Export File
#
#   Export Debug file, Xml file, Target file
#
#   @param[in]  Args                    Argument which assign from user
#   @param[in]  IsExistDebugMacro       Judge whether exist debug macro from Source File
#   @param[in]  TargetFileContent       Target file content
#
def ExportFile (Args, IsExistDebugMacro, TargetFileContent):
  global mConfigureTable

  #
  # Export OutputFile
  #
  if (Args.OutputFile != None) :
    with open (Args.OutputFile, 'w') as File:
      portalocker.lock (File, portalocker.LOCK_EX)
      File.write (TargetFileContent)

  if (IsExistDebugMacro == True):
    #
    # Export OutputDebugFile
    #
    if (mConfigureTable['OutputDebugFile'] != None) :
      SortedDicDataInDebug = ''
      for (HexNumber, DebugMacro, (Line, IsSupported)) in zip (mConfigureTable['HexNumber'], mConfigureTable['DebugMacro'], mConfigureTable['Line']):
        SortedDicDataInDebug += ('{\n'                                               +
                                 '  \"InputFile\":    ' + Args.InputFile     + ',\n' +
                                 '  \"Line\":         ' + '%d' % Line        + ',\n' +
                                 '  \"IsSupported\":  ' + '%d' % IsSupported + ',\n' +
                                 '  \"OutputFile\":   ' + Args.OutputFile    + ',\n' +
                                 '  \"HexNumber\":    ' + '%x' % HexNumber   + ',\n' +
                                 '  \"DebugMacro\":   ' + DebugMacro         + ',\n' +
                                 '},\n'
                                )
      mConfigureTable['OutputDebugFile'].seek (0, os.SEEK_END)
      mConfigureTable['OutputDebugFile'].write (SortedDicDataInDebug)

    #
    # Export XmlFile
    #
    if (mConfigureTable['XmlFile'] != None) :
      SortedDicDataInXml = ''
      for (HexNumber, DebugMacro) in zip (mConfigureTable['HexNumber'], mConfigureTable['DebugMacro']):
        SortedDicDataInXml += '    <Message ID="0x'+ '%x' % HexNumber + '"><![CDATA[' + DebugMacro + "]]></Message>\n"
      if (mConfigureTable['XmlFileContent'] != ''):
        mConfigureTable['XmlFile'].seek (0)
        ReplaceStart = "  </Messages>\n</SvenDictionary>"
        mConfigureTable['XmlFileContent'] = mConfigureTable['XmlFileContent'].replace (ReplaceStart, SortedDicDataInXml + ReplaceStart)
        mConfigureTable['XmlFile'].write (mConfigureTable['XmlFileContent'])

## Set start Hex number
#
#   Set start Hex number if exist xml file from your workspace
#
#   @param[in]  Args       Argument which assign from user
#
def SetStartHexNumber (Args):
  global mConfigureTable

  if (mConfigureTable['XmlFile'] != None):
    XmlFileIdPattern = re.compile (r"<Message ID=\"([0x]*[a-zA-Z0-9]+)\">", re.M)
    XmlFileIdList = XmlFileIdPattern.findall (mConfigureTable['XmlFileContent'])
    if (len (XmlFileIdList) > 0):
      mConfigureTable['CurrentHexNumber'] = eval (XmlFileIdList[-1]) + 1

## The CatalogEncoder.py Entry Point for Application.
#
def Main ():
  global mConfigureTable

  Args = GetArgs ()
  if (IsArgsValid (Args) == False):
    exit (1)

  (IsExistDebugMacro, DebugMacros, UnDebugMacros) = GetDebugMacro (mConfigureTable['InputFileContent'])

  SetStartHexNumber (Args)

  if (IsExistDebugMacro == True):
    TargetFileContent = ""
    Paragraph = 0
    mConfigureTable['CurrentLine'] = 1
    for NoDebugMacroDatum in UnDebugMacros:
      TargetFileContent += NoDebugMacroDatum
      mConfigureTable['CurrentLine'] += NoDebugMacroDatum.count ('\n')
      if (Paragraph < len (DebugMacros)):
        TargetFileContent += ConvertDebugMacroToCatalog (DebugMacros[Paragraph])
        mConfigureTable['CurrentLine'] += DebugMacros[Paragraph].count ('\n')
      Paragraph += 1
  else:
    TargetFileContent = mConfigureTable['InputFileContent']

  ExportFile (Args, IsExistDebugMacro, TargetFileContent)

  if (mConfigureTable['XmlFile'] != None):
    mConfigureTable['XmlFile'].close ()
  if (mConfigureTable['OutputDebugFile'] != None):
    mConfigureTable['OutputDebugFile'].close ()

if __name__ == "__main__":
  Main ()
