/** @file
  DG1 ACPI RTD3 SSDT table for PCIe

@copyright
  INTEL CONFIDENTIAL
  Copyright 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
**/

/// @details
/// Code in this file uses following variables:
/// SCLK: ICC Clock number - optional
/// WAKG: WAKE GPIO pad - optional
/// Below objects should be defined according to the format described in PinDriverLib.asl
/// RSTG: reset pin definition - mandatory
/// PWRG: power GPIO pad - optional
/// WAKP: Flag to indicate that power gating must not be performed if WAKE is enabled - optional
/// @defgroup pcie_scope PCIe Root Port Scope **/

//
// AcpiPinDriverLib imports(from DSDT in platform)
//
External(\PIN.STA, MethodObj)
External(\PIN.ON, MethodObj)
External(\PIN.OFF, MethodObj)
External(\_ADR, MethodObj)
External(\_SB.PC00.GPCB, MethodObj)
//
// GpioLib imports(DSDT)
//
External(\_SB.SHPO, MethodObj)

//
// HSIO lib imports
//
External(\_SB.PSD0, MethodObj)
External(\_SB.PSD3, MethodObj)
//
// External decalarations for optional objects.
// Defined by board specific code.
//
External(WAKG)
External(PWRG)
External(SCLK)
External(WAKP)
External(VCCE)
External(NCNV)
Name(LNRD, 0) // Delay before PERST# assertion in us

// DG Op Region
OperationRegion(DGMR, SystemMemory, DGOP, 0x2000)
Field(DGMR, AnyAcc, NoLock, Preserve)
{
  //
  // OpRegion Header starts at 0x000
  //
  Offset(0x60),
  PCON, 32,  // Platform Configuration start at 0x60h
  //
  // OpRegion Mailbox 1 starts at 0x100
  //
  Offset(0x100),
  //
  // OpRegion Mailbox 2 starts at 0x200
  //
  Offset(0x200),
  //
  // OpRegion Mailbox 3 starts at 0x300
  //
  Offset(0x3C6), // VRSR start at 0x3C6
  VRSR, 1,
}

//
// Update PERST# assertion delay.
// This function will be called from reference code during PCIe _DSM function index 11 evaluation.
// Arg0 - New delay value in microseconds. Max is 10ms
//
// @return Last sucessfully negotiated value in us. 0 if no such value exists.
//
Method(UPRD, 1, Serialized) {
  If(LLessEqual(Arg0, 10000)) {
    // If the value does not exceed the limit
    // Update last negotiated value and calculate the value in ms.
    Store(Arg0, LNRD)
  }
  Return(LNRD)
}


//
// PCIe root port Pci Cfg space to MMIO address
// Converts PCI device address from _ADR method to physical memory address
// Arg0 - _ADR object of the device that will be accessed
//
Method (PCIM, 1, Serialized) {     // Convert _ADR to PCI MMIO address.
  Store (\_SB.PC00.GPCB(), Local0) // MMIO Base address
  Add (Local0, ShiftRight(And(Arg0, 0x001F0000), 1), Local0) // Device no
  Add (Local0, ShiftLeft(And(Arg0, 0x00000007), 12), Local0) // Function no
  Return (Local0)
}


//
// PCIe slot power resource definition
//

  Method(DGON, 0, Serialized) {
  Store (PCIM(_ADR()), Local0)
  OperationRegion(DGCS,SystemMemory,Local0,0x480)
  Field(DGCS,AnyAcc, NoLock, Preserve)
  {
    Offset(0),
    VDID, 32,
    Offset(0x50), // LCTL - Link Control Register
    L0SE, 1,      // 0, L0s Entry Enabled
    Offset(0x52), // LSTS - Link Status Register
    , 13,
    LASX, 1,      // 0, Link Active Status
    Offset(0x5A), // SLSTS[7:0] - Slot Status Register
    , 3,
    PDCX, 1,      // 3, Presence Detect Changed
    , 2,
    PDSX, 1,      // 6, Presence Detect State
    , 1,
    Offset(0x60), // RSTS - Root Status Register
    , 16,
    PSPX, 1,      // 16,  PME Status
    Offset(0xD8), // 0xD8, MPC - Miscellaneous Port Configuration Register
    , 30,
    HPEX, 1,      // 30,  Hot Plug SCI Enable
    PMEX, 1,      // 31,  Power Management SCI Enable
    Offset(0xE0), // 0xE0, SPR - Scratch Pad Register
    , 7,
    NCB7, 1,                   // Non-Sticky Scratch Pad Bit (NSCB)[7]
    Offset(0xE2), // 0xE2, RPPGEN - Root Port Power Gating Enable
    , 2,
    L23E, 1,      // 2,   L23_Rdy Entry Request (L23ER)
    L23R, 1,       // 3,   L23_Rdy to Detect Transition (L23R2DT)
    Offset(0x420), // 0x420, PCIEPMECTL (PCIe PM Extension Control)
    , 30,
    DPGE, 1,       // PCIEPMECTL[30]: Disabled, Detect and L23_Rdy State PHY Lane Power Gating Enable (DLSULPPGE):
  }

  If (LEqual (VDID, 0xFFFFFFFF)) {
    Return()
  }

  //
  // Turn on slot power
  //
  PWON()

  //
  // Trigger L2/L3 ready exit flow in rootport - transition link to Detect
  //

  If(LNotEqual(NCNV,0x1)) {
    Return()
  }

  /// Clear DLSULPPGE, then set L23_Rdy to Detect Transition  (L23R2DT)
  Store(0, DPGE)
  Store(1, L23R)
  Store(0, Local0)
  /// Wait for transition to Detect
  While(L23R) {
    If(Lgreater(Local0, 4))
    {
      Break
    }
    Sleep(16)
    Increment(Local0)
  }

  Store(0,NCNV)
  /// Once in Detect, wait up to 124 ms for Link Active (typically happens in under 70ms)
  /// Worst case per PCIe spec from Detect to Link Active is:
  /// 24ms in Detect (12+12), 72ms in Polling (24+48), 28ms in Config (24+2+2+2+2)
  Store(1, DPGE)
  Store(0, Local0)
  While(LEqual(LASX,0)) {
    If(Lgreater(Local0, 8))
    {
      Break
    }
    Sleep(16)
    Increment(Local0)
  }


  //
  // Notify RTD3 Exit to EC
  //
  If (\ECON) {
    \_SB.PC00.LPCB.H_EC.ECMD (0x3B)
  }
}

Method(DGOF, 0, Serialized) {
  Store (PCIM(_ADR()), Local0)

  OperationRegion(DGCS,SystemMemory,Local0,0x480)
  Field(DGCS,AnyAcc, NoLock, Preserve)
  {
    Offset(0),
    VDID, 32,
    Offset(0x50), // LCTL - Link Control Register
    L0SE, 1,      // 0, L0s Entry Enabled
    Offset(0x52), // LSTS - Link Status Register
    , 13,
    LASX, 1,      // 0, Link Active Status
    Offset(0x5A), // SLSTS[7:0] - Slot Status Register
    , 3,
    PDCX, 1,      // 3, Presence Detect Changed
    , 2,
    PDSX, 1,      // 6, Presence Detect State
    , 1,
    Offset(0x60), // RSTS - Root Status Register
    , 16,
    PSPX, 1,      // 16,  PME Status
    Offset(0xD8), // 0xD8, MPC - Miscellaneous Port Configuration Register
    , 30,
    HPEX, 1,      // 30,  Hot Plug SCI Enable
    PMEX, 1,      // 31,  Power Management SCI Enable
    Offset(0xE0),    // 0xE0, SPR - Scratch Pad Register
    , 7,
    NCB7, 1,                   // Non-Sticky Scratch Pad Bit (NSCB)[7]
    Offset(0xE2), // 0xE2, RPPGEN - Root Port Power Gating Enable
    , 2,
    L23E, 1,      // 2,   L23_Rdy Entry Request (L23ER)
    L23R, 1,       // 3,   L23_Rdy to Detect Transition (L23R2DT)
    Offset(0x420), // 0x420, PCIEPMECTL (PCIe PM Extension Control)
    , 30,
    DPGE, 1,       // PCIEPMECTL[30]: Disabled, Detect and L23_Rdy State PHY Lane Power Gating Enable (DLSULPPGE):
  }

  If (LEqual (VDID, 0xFFFFFFFF)) {
    Return()
  }

  //
  // Notify RTD3 Entry to EC
  //
  If (\ECON) {
    \_SB.PC00.LPCB.H_EC.ECMD (0x3A)
  }
  //
  // Trigger L2/L3 ready entry flow in rootport
  //

  Store(1, L23E)
  Sleep(16)
  Store(0, Local0)
  While(L23E) {
    If(Lgreater(Local0, 4))
    {
      Break
    }
    Sleep(16)
    Increment(Local0)
  }
  Store(1,NCNV)
  //
  // Turn off slot power
  //
  PWOF()
  }

// Turn on power to PCIe Slot
// Since this method is also used by the remapped devices to turn on power to the slot
// this method should not make any access to the PCie config space.
Method(PWON, 0, Serialized) {
  //
  // Restore power to the PHY
  //
  If(CondRefOf(\_SB.PSD0)) {
    \_SB.PSD0(SLOT)
  }

  // Turn ON Power for PCIe Slot
  If(CondRefOf(PWRG)) {
    \PIN.ON(PWRG)
  }

  If(CondRefOf(VCCE)) {
    \PIN.OFF(VCCE)
  }

  Sleep(PEP0)

  If(CondRefOf(SCLK)) {
    SPCO(SCLK, 1)
  }

  // De-Assert Reset Pin
  \PIN.OFF(RSTG)
}

// Turn off power to PCIe Slot
// Since this method is also used by the remapped devices to turn off power to the slot
// this method should not make any access to the PCIe config space.
Method(PWOF, 0, Serialized) {
  // Assert Reset Pin after the delay passed from the bus driver
  // Reset pin is mandatory for correct PCIe RTD3 flow
  Divide(LNRD, 1000, Local0, Local1)
  Sleep(Local1)
  \PIN.ON(RSTG)

  // Enable PHY power gating
  // This must be done after the device has been put in reset
  //
  If(CondRefOf(\_SB.PSD3)) {
    \_SB.PSD3(SLOT)
  }

  //
  // On RTD3 entry, BIOS will instruct the PMC to disable source clocks.
  // This is done through sending a PMC IPC command.
  //
  If(CondRefOf(SCLK)) {
    SPCO(SCLK, 0)
  }


  // By default to skip power off
  Store(0, Local0)

  // Power OFF for Slot
  If(CondRefOf(PWRG)) {
    // If WAKP is defined this slot does not supply device with auxilary power and we have to keep primary power
    // to allow for WAKE. If WAKP is not equal to 0 and WKEN has been enabled do not disable the power.
    If(CondRefOf (WAKP)) {

      If(LOr(LEqual(WAKP, 0), LNot(WKEN))) {
        Store(1, Local0)  // Handle power off
      } Else {
        Store(0, Local0)  // Skip power off
      }
    } Else {
      // If WAKP has not been defined we can safely disable power.
      Store(1, Local0)
    }
  }


  If (LEqual(Local0, 1)) {
    // BIOS supports VRAM self refersh and OS/driver enabled self refersh
    If (LAnd((LEqual(And(0x1800, PCON), 0x1800)), VRSR)) {
      \PIN.ON(PWRG)
      if(CondRefOf(VCCE)) {
        \PIN.ON(VCCE)
      }
    }
    Else {
      // Either BIOS does not support self-refresh or OS/driver does not enabled self-refresh
      \PIN.OFF(PWRG)
    }
  }

  // enable WAKE
  If(CondRefOf (WAKG)) {
    If(LAnd(LNotEqual(WAKG, 0), WKEN)) {
      \_SB.SHPO(WAKG, 0)
    }
  }
}
