/** @file
  Source code file for the Boot Mode Library

@copyright
  INTEL CONFIDENTIAL
  Copyright 2018 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor.This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
**/

#include <Base.h>
#include <PiPei.h>
#include <Library/BaseLib.h>
#include <Library/PcdLib.h>
#include <Library/DebugLib.h>
#include <Library/IoLib.h>
#include <Library/PeiServicesLib.h>
#include <BootState.h>
#include <Ppi/Capsule.h>
#include <SetupVariable.h>
#include <Library/PmcLib.h>
#include <Register/PmcRegs.h>
#include <FastBootDataDef.h>
#include <FastBootExceptionInfoHob.h>
#include <CmosMap.h>
#include <Library/CmosAccessLib.h>
#include <Library/PeiServicesTablePointerLib.h>
#include <Library/GpioNativeLib.h>
#include <Library/GpioLib.h>
#include <PlatformBoardConfig.h>
#include <Library/PeiBootModeLib.h>
#include "PeiBootModeLibInternal.h"
#include <PlatformBoardId.h>
#include <PlatformBoardType.h>
#include <PlatformNvRamHookLib.h>

#define FAST_BOOT_OR_MASK       0x01

GLOBAL_REMOVE_IF_UNREFERENCED EFI_PEI_PPI_DESCRIPTOR mPpiListNonS3BootMode = {
  (EFI_PEI_PPI_DESCRIPTOR_PPI | EFI_PEI_PPI_DESCRIPTOR_TERMINATE_LIST),
  &gEfiPeiBootInNonS3ModePpiGuid,
  NULL
};

///
/// Priority of our boot modes, highest priority first
///
GLOBAL_REMOVE_IF_UNREFERENCED EFI_BOOT_MODE mBootModePriority[] = {
  BOOT_IN_RECOVERY_MODE,
  BOOT_WITH_DEFAULT_SETTINGS,
  BOOT_ON_FLASH_UPDATE,
  BOOT_ON_S3_RESUME,
  BOOT_ON_S4_RESUME,
  BOOT_WITH_MINIMAL_CONFIGURATION,
  BOOT_ASSUMING_NO_CONFIGURATION_CHANGES,
  BOOT_WITH_FULL_CONFIGURATION_PLUS_DIAGNOSTICS,
  BOOT_WITH_FULL_CONFIGURATION,
  BOOT_ON_S5_RESUME,
  BOOT_SPECIAL_MASK
};

EFI_STATUS
EFIAPI
PreMemUpdateBootMode (
  IN OUT EFI_BOOT_MODE          *BootMode
  );

/**
  Update boot mode if it is in capsule update boot mode.

  @param[in,out] BootMode         Current System BootMode.

  @retval EFI_SUCCESS             If we have a capsule available.
  @retval EFI_NOT_FOUND           No capsule detected.

**/
EFI_STATUS
EFIAPI
PreMemUpdateCapsuleBootMode (
  IN OUT EFI_BOOT_MODE          *BootMode
  )
{
  PEI_CAPSULE_PPI               *Capsule;
  EFI_STATUS                    Status;
  EFI_PEI_SERVICES              **PeiServices = NULL;

  DEBUG((DEBUG_INFO, "PeiBootModeLib PreMemUpdateCapsuleBootMode PeiServicesGetBootMode = 0x%x\n", *BootMode));

  Status = PeiServicesLocatePpi (
             &gPeiCapsulePpiGuid,
             0,
             NULL,
             (VOID **) PeiServices
             );
  if (EFI_ERROR(Status)) {
    return EFI_NOT_FOUND;
  }

  ///
  /// Determine if we're in capsule update mode
  ///
  Status = PeiServicesLocatePpi (
             &gPeiCapsulePpiGuid,
             0,
             NULL,
             (VOID **) &Capsule
             );
  if (Status == EFI_SUCCESS) {
    Status = Capsule->CheckCapsuleUpdate((EFI_PEI_SERVICES **) PeiServices);
    if (Status == EFI_SUCCESS) {
      PrioritizeBootMode(BootMode, BOOT_ON_FLASH_UPDATE);
    }
  }
  if (*BootMode != BOOT_ON_FLASH_UPDATE) {
    Status = PreMemUpdateBootMode (BootMode);
  }
  DEBUG((DEBUG_INFO, "PeiBootModeLib PreMemUpdateCapsuleBootMode PeiServicesSetBootMode = 0x%x\n", *BootMode));
  return Status;
}

/**
  Check fast boot is enabled or not

  @retval  FastBootEnabledStatus  TRUE means fast boot is enabled
                                  FALSE means fast boot is disabled
**/
BOOLEAN
EFIAPI
IsFastBootEnabled (
  VOID
  )
{
  EFI_STATUS                      Status;
  EFI_PEI_READ_ONLY_VARIABLE2_PPI *PeiReadOnlyVarPpi;
  UINTN                           VarSize;
  SETUP_DATA                      SystemConfiguration;
  BOOLEAN                         FastBootEnabledStatus;

  FastBootEnabledStatus = FALSE;
  Status = PeiServicesLocatePpi (
             &gEfiPeiReadOnlyVariable2PpiGuid,
             0,
             NULL,
             (VOID **) &PeiReadOnlyVarPpi
             );
  if (Status == EFI_SUCCESS) {
    VarSize = sizeof (SETUP_DATA);
    Status = PeiReadOnlyVarPpi->GetVariable (
                                  PeiReadOnlyVarPpi,
                                  L"Setup",
                                  &gSetupVariableGuid,
                                  NULL,
                                  &VarSize,
                                  &SystemConfiguration
                                  );
    if (Status == EFI_SUCCESS) {
      if (SystemConfiguration.FastBoot != 0) {
        FastBootEnabledStatus = TRUE;
      }
    }
  }

  return FastBootEnabledStatus;
}

/**
  Update boot mode

  @param[in,out] BootMode             Current System BootMode.

  @retval EFI_SUCCESS                 The boot mode was returned successfully.

**/

EFI_STATUS
EFIAPI
PreMemUpdateBootMode (
  IN OUT EFI_BOOT_MODE          *BootMode
  )
{
  EFI_PEI_READ_ONLY_VARIABLE2_PPI   *VariableServices;
  UINTN                             VarSize;
  BOOLEAN                           IsFastBootEnable;
  EFI_STATUS                        Status;
  UINT8                             FastBootRegData;
  FAST_BOOT_EXCEPTION_INFO_HOB      *FastBootExceptionInfoHob;

  DEBUG ((DEBUG_INFO, "PeiBootModeLib PreMemUpdateBootMode Entry\n"));
  FastBootRegData = 0;

  IsFastBootEnable = IsFastBootEnabled ();
  if (IsFastBootEnable) {
    if (IsPreviousFastBootCompleted () && !IsTimeOutOccurred () && !IsPowerButtonOverride ()) {
        PrioritizeBootMode (BootMode, BOOT_WITH_MINIMAL_CONFIGURATION);
    } else {
      Status = PeiServicesCreateHob (
                 EFI_HOB_TYPE_GUID_EXTENSION,
                 sizeof (FAST_BOOT_EXCEPTION_INFO_HOB),
                 (VOID **) &FastBootExceptionInfoHob
                 );
      if (!EFI_ERROR (Status)) {
        FastBootExceptionInfoHob->Header.Name = gFastBootExceptionInfoHobGuid;
        FastBootExceptionInfoHob->FbExceptionType = ExceptionType2;
        FastBootExceptionInfoHob->FbExceptionCategory = BootFailure;
      }
      DEBUG ((DEBUG_INFO, "PeiBootModeLib Exception has occurred. Prioritizing Boot mode to BOOT_WITH_FULL_CONFIGURATION\n"));
      PrioritizeBootMode (BootMode, BOOT_WITH_FULL_CONFIGURATION);
    }
  } else {
    if ((*BootMode != BOOT_ON_S5_RESUME) && IsBootWithNoChange ()) {
      PrioritizeBootMode (BootMode, BOOT_ASSUMING_NO_CONFIGURATION_CHANGES);
    } else {
      PrioritizeBootMode (BootMode, BOOT_WITH_FULL_CONFIGURATION);
    }
  }

  ///
  /// FastBoot - set boot progress bit to indicate boot is in progress
  /// bit will be cleared before booting to OS/Shell/BIOS Setup or ResetSystem() been triggered
  ///
  if (IsFastBootEnable) {
    if (*BootMode != BOOT_ON_S3_RESUME) {
      FastBootRegData = GetFastBootFlagStatus ();
      FastBootRegData |= FAST_BOOT_OR_MASK;
      UpdateFastBootFlagStatus (FastBootRegData);
    }
  }

  if (*BootMode != BOOT_ON_S3_RESUME) {
    ///
    /// If not in S3 mode, signal it.
    ///
    Status = PeiServicesInstallPpi (&mPpiListNonS3BootMode);
    ASSERT_EFI_ERROR (Status);
  }

  if (IsFastBootEnable) {
    Status = PeiServicesLocatePpi (
               &gEfiPeiReadOnlyVariable2PpiGuid,
               0,
               NULL,
               (VOID **) &VariableServices
               );
    VarSize = 0;
    if (!RETURN_ERROR (Status)) {
      Status = VariableServices->GetVariable (
                                   VariableServices,
                                   L"ConOut",
                                   &gEfiGlobalVariableGuid,
                                   NULL,
                                   &VarSize,
                                   NULL
                                   );
      if (Status == EFI_NOT_FOUND) {
        PrioritizeBootMode (BootMode, BOOT_WITH_FULL_CONFIGURATION);
      }
    }
  }

  DEBUG ((DEBUG_INFO, "PeiBootModeLib PreMemUpdateBootMode Exit "));
  return EFI_SUCCESS;
}

/**
  Detect boot mode

  @retval EFI_BOOT_MODE  Current System BootMode.
**/

EFI_BOOT_MODE
EFIAPI
DetectBootMode (
  VOID
  )
{
  EFI_STATUS                      Status;
  EFI_BOOT_MODE                   BootMode;
  EFI_PEI_READ_ONLY_VARIABLE2_PPI *VariableServices;

  Status = PeiServicesLocatePpi (
             &gEfiPeiReadOnlyVariable2PpiGuid,
             0,
             NULL,
             (VOID **) &VariableServices
             );

  PMC_SLEEP_STATE                 SleepType;
#if FixedPcdGetBool (PcdSetupEnable) == 1
  SETUP_DATA                      Setup;
  SA_SETUP                        SaSetup;
  ME_SETUP                        MeSetup;
  CPU_SETUP                       CpuSetup;
  PCH_SETUP                       PchSetup;
  UINT8                           InitSetupFlag;
  UINTN                           DataSize;
#endif
  Status = PeiServicesGetBootMode (&BootMode);
  ASSERT_EFI_ERROR(Status);
  DEBUG((DEBUG_INFO, "PeiBootModeLib DetectBootMode PeiServicesGetBootMode BootMode= 0x%x \n", BootMode));

  Status = IsRecoveryMode ();
  if (Status == EFI_SUCCESS) {
    BootMode = BOOT_IN_RECOVERY_MODE;
    DEBUG((DEBUG_INFO, "PeiBootModeLib IsRecoveryMode BOOT_IN_RECOVERY_MODE"));
    //mPpiListRecoveryBootMode is installed by MinPlatform
  }

  SleepType = PmcGetSleepTypeAfterWake ();
  DEBUG((DEBUG_INFO, "PeiBootModeLib PmcGetSleepTypeAfterWake SleepType= %d \n", SleepType));
  if (SleepType != PmcNotASleepState) {
    if (!PmcIsPowerFailureDetected () && PmcIsRtcBatteryGood()) {
      switch (SleepType) {
        case PmcS3SleepState:
          PrioritizeBootMode (&BootMode, BOOT_ON_S3_RESUME);
          DEBUG((DEBUG_INFO, "PeiBootModeLib PmcS3SleepState BootMode= 0x%x \n", BootMode));
          break;
        case PmcS4SleepState:
          PrioritizeBootMode (&BootMode, BOOT_ON_S4_RESUME);
          DEBUG((DEBUG_INFO, "PeiBootModeLib PmcS4SleepState BootMode= 0x%x \n", BootMode));
          break;
        case PmcS5SleepState:
          PrioritizeBootMode (&BootMode, BOOT_ON_S5_RESUME);
          DEBUG((DEBUG_INFO, "PeiBootModeLib PmcS5SleepState BootMode= 0x%x \n", BootMode));
          break;
        default:
          break;
      }
    }
  } else {
      ///
      /// Clear Wake Status and Sleep Type
      ///
      PmcClearWakeStatus ();
      PmcSetSleepState (PmcInS0State);
    }
    if (!PmcIsRtcBatteryGood ()) {
      DEBUG ((DEBUG_ERROR, "RTC Power failure !! Could be due to a weak or missing battery, BootMode BOOT_WITH_DEFAULT_SETTINGS.\n"));
      BootMode = BOOT_WITH_DEFAULT_SETTINGS;
    }

  ///
  /// Check whether Setup Variable does exist to know the first boot or not.
  ///
#if FixedPcdGetBool (PcdSetupEnable) == 1

  DataSize = sizeof (SETUP_DATA);
  Status  = VariableServices->GetVariable (VariableServices, L"Setup", &gSetupVariableGuid, NULL, &DataSize, &Setup);
  if (Status == EFI_NOT_FOUND) {
    BootMode = BOOT_WITH_DEFAULT_SETTINGS;
  }
  if (Status != EFI_NOT_FOUND) {
    DataSize = sizeof (SA_SETUP);
    Status = VariableServices->GetVariable (VariableServices, L"SaSetup", &gSaSetupVariableGuid, NULL, &DataSize, &SaSetup);
    if (Status == EFI_NOT_FOUND) {
      BootMode = BOOT_WITH_DEFAULT_SETTINGS;
    }
  }
  if (Status != EFI_NOT_FOUND) {
    DataSize = sizeof (ME_SETUP);
    Status = VariableServices->GetVariable (VariableServices, L"MeSetup", &gMeSetupVariableGuid, NULL, &DataSize, &MeSetup);
    if (Status == EFI_NOT_FOUND) {
      BootMode = BOOT_WITH_DEFAULT_SETTINGS;
    }
  }
  if (Status != EFI_NOT_FOUND) {
    DataSize = sizeof (CPU_SETUP);
    Status = VariableServices->GetVariable (VariableServices, L"CpuSetup", &gCpuSetupVariableGuid, NULL, &DataSize, &CpuSetup);
    if (Status == EFI_NOT_FOUND) {
      BootMode = BOOT_WITH_DEFAULT_SETTINGS;
    }
  }
  if (Status != EFI_NOT_FOUND) {
    DataSize = sizeof (PCH_SETUP);
    Status = VariableServices->GetVariable (VariableServices, L"PchSetup", &gPchSetupVariableGuid, NULL, &DataSize, &PchSetup);
    if (Status == EFI_NOT_FOUND) {
      BootMode = BOOT_WITH_DEFAULT_SETTINGS;
    }
  }
  if (Status != EFI_NOT_FOUND) {
    DataSize = sizeof (InitSetupFlag);
    Status = VariableServices->GetVariable (VariableServices, L"InitSetupVariable", &gSetupVariableGuid, NULL, &DataSize, &InitSetupFlag);
    if (Status == EFI_NOT_FOUND) {
      DEBUG((DEBUG_INFO, "PeiBootModeLib detects the first boot \n"));
      BootMode = BOOT_WITH_DEFAULT_SETTINGS;
    }
  }
#endif
  DEBUG((DEBUG_ERROR, "PeiBootModeLib after Setup Variables check, BootMode = 0x%x\n", BootMode));

#if FixedPcdGetBool (PcdCapsuleEnable) == 1
  ///
  /// Update BootMode
  ///
  Status = PreMemUpdateCapsuleBootMode(&BootMode);
#else
  Status = PreMemUpdateBootMode (&BootMode);
#endif
  DEBUG((DEBUG_INFO, "PeiBootModeLib Exit of DetectBootModeLib, BootMode =0x%x\n", BootMode));
  return BootMode;
}

/**
  Check Boot Flag Status to determine if previous boot was successful
  Boot Progress bit will be cleared before booting to OS/Shell/BIOS Setup
  or ResetSystem() been triggered while Fastboot enabled.

  @retval  TRUE   Previous Fast Boot was success.
  @retval  FALSE  Previous Fast Boot wasn't success.
**/

BOOLEAN
EFIAPI
IsPreviousFastBootCompleted (
  VOID
  )
{
  EFI_STATUS                      Status;
  UINT8                           PreviousFastBootCompletionFlag;
  BOOLEAN                         BootState;
  UINTN                           DataSize;
  EFI_PEI_READ_ONLY_VARIABLE2_PPI *PeiVar;

  DEBUG((DEBUG_INFO, "PeiBooModeLib IsPreviousFastBootCompleted Entry\n"));
  Status = PeiServicesLocatePpi (
             &gEfiPeiReadOnlyVariable2PpiGuid,
             0,
             NULL,
             (VOID **) &PeiVar
             );
  ASSERT_EFI_ERROR (Status);

  DEBUG((DEBUG_INFO, "PeiBootModeLib PeiServicesLocatePpi Status = %r\n", Status));
  ///
  /// Get last Boot State Variable to confirm that it is not a first boot.
  ///
  DataSize = sizeof (BOOLEAN);
  Status = PeiVar->GetVariable (
                     PeiVar,
                     BOOT_STATE_VARIABLE_NAME,
                     &gBootStateGuid,
                     NULL,
                     &DataSize,
                     &BootState
                     );

  if (EFI_ERROR (Status)) {
    return FALSE;
  }
  PreviousFastBootCompletionFlag = 0;
  PreviousFastBootCompletionFlag = GetFastBootFlagStatus ();
  PreviousFastBootCompletionFlag &= FAST_BOOT_OR_MASK;

  if(PreviousFastBootCompletionFlag){
    DEBUG((DEBUG_INFO, "IsPreviousFastBootCompleted() returns FALSE as previous boot is still in progress \n"));
    return FALSE;
  }
  DEBUG ((DEBUG_INFO, "IsPreviousFastBootCompleted() return TRUE as previoius boot is successfully completed.\n"));
  return TRUE;
}

/**
  Query PCH to determine if Intrusion Detection set

  @retval  TRUE   No Change.
  @retavl  FALSE  Change.
**/
BOOLEAN
EFIAPI
IsBootWithNoChange (
  VOID
  )
{
  EFI_STATUS                      Status;
  UINT16                          TcoBase;
  UINT16                          Tco2Status;
  BOOLEAN                         BoxOpen;
  BOOLEAN                         BootState;
  UINTN                           DataSize;
  EFI_PEI_READ_ONLY_VARIABLE2_PPI *PeiVar;

  DEBUG((DEBUG_INFO, "PeiBootModeLib IsBootWithNoChange Entry \n"));
  Status = PeiServicesLocatePpi (
             &gEfiPeiReadOnlyVariable2PpiGuid,
             0,
             NULL,
             (VOID **) &PeiVar
             );
  ASSERT_EFI_ERROR (Status);

  ///
  /// Get last Boot State Variable From NVRAM
  ///
  DataSize = sizeof (BOOLEAN);
  Status = PeiVar->GetVariable (
                     PeiVar,
                     BOOT_STATE_VARIABLE_NAME,
                     &gBootStateGuid,
                     NULL,
                     &DataSize,
                     &BootState
                     );
  if (EFI_ERROR (Status)) {
    // If there's no variable, just report the state of the hardware
    BootState = TRUE;
  }

  ///
  /// Read the TCO registers
  ///
  TcoBase     = (UINT16)PcdGet16 (PcdTcoBaseAddress);
  Tco2Status  = IoRead16 (TcoBase + R_TCO_IO_TCO2_STS);

  ///
  /// This is the state of the hardware
  ///
  BoxOpen = (BOOLEAN) (Tco2Status & B_TCO_IO_TCO2_STS_INTRD_DET);
  if (BoxOpen) {
    ///
    /// Clear the bit for next boot.
    ///
    Tco2Status |= B_TCO_IO_TCO2_STS_INTRD_DET;
    IoWrite16 (TcoBase + R_TCO_IO_TCO2_STS, Tco2Status);

    ///
    /// Since it was OPEN, return that it cannot be in "no config change boot"
    ///
    DEBUG ((DEBUG_INFO, "PeiBootModeLib BoxOpen \n"));
    return FALSE;
  } else {
    ///
    /// Since it was CLOSED, return that can be "no config change boot"
    /// Need variable to know that memory was tested AT LEAST one time!
    ///
    if (!BootState) {
      DEBUG ((DEBUG_INFO, "PeiBootModeLib BootState TRUE\n"));
      return TRUE;
    } else {
      DEBUG ((DEBUG_INFO, "PeiBootModeLib BootState FALSE\n"));
      return FALSE;
    }
  }
}

/**
  Given the current boot mode, and a proposed new boot mode, determine
  which has priority. If the new boot mode has higher priority, then
  make it the current boot mode.

  @param[in][out] CurrentBootMode Pointer to current planned boot mode
  @param[in][out] NewBootMode     Proposed boot mode.

  @retval         EFI_NOT_FOUND   If either boot mode is not recognized.
  @retval         EFI_SUCCESS     If both boot mode values were
                                  recognized and were processed.
**/
EFI_STATUS
EFIAPI
PrioritizeBootMode (
  IN OUT EFI_BOOT_MODE    *CurrentBootMode,
  IN EFI_BOOT_MODE        NewBootMode
  )
{
  UINT32 CurrentIndex;
  UINT32 NewIndex;

  DEBUG((DEBUG_INFO, "PeiBootModeLib PrioritizeBootMode Entry \n"));
  ///
  /// Find the position of the current boot mode in our priority array
  ///
  for ( CurrentIndex = 0;
        CurrentIndex < sizeof (mBootModePriority) / sizeof (mBootModePriority[0]);
        CurrentIndex++) {
    if (mBootModePriority[CurrentIndex] == *CurrentBootMode) {
      break;
    }
  }
  if (CurrentIndex >= sizeof (mBootModePriority) / sizeof (mBootModePriority[0])) {
    return EFI_NOT_FOUND;
  }
  ///
  /// Find the position of the new boot mode in our priority array
  ///
  for ( NewIndex = 0;
        NewIndex < sizeof (mBootModePriority) / sizeof (mBootModePriority[0]);
        NewIndex++) {
    if (mBootModePriority[NewIndex] == NewBootMode) {
      ///
      /// If this new boot mode occurs before the current boot mode in the
      /// priority array, then take it.
      ///
      if (NewIndex < CurrentIndex) {
        *CurrentBootMode = NewBootMode;
        DEBUG((DEBUG_INFO, "PeiBootModeLib NewIndex < CurrentIndex, Update  PrioritizeBootMode CurrentBootMode: 0x%x \n", *CurrentBootMode));
    }
    else {
       DEBUG((DEBUG_INFO, "PeiBootModeLib NewIndex > CurrentIndex Exit PrioritizeBootMode CurrentBootMode 0x%x \n", *CurrentBootMode));
    }
      return EFI_SUCCESS;
    }
  }
  DEBUG((DEBUG_INFO, "PeiBootModeLib PrioritizeBootMode Exit\n"));
  return EFI_NOT_FOUND;
}

/**
  Check if this system boot is due to 4sec power button override has occurred

  @retval  TRUE   Power Button Override occurred in last system boot
  @retval  FALSE  Power Button Override didnt occur
**/
BOOLEAN
EFIAPI
IsPowerButtonOverride (
  VOID
  )
{
  if (PmcIsPowerButtonOverrideDetected ()) {
    DEBUG ((DEBUG_INFO, "IsPowerButtonOverride TRUE\n"));
    return TRUE;
  }
  return FALSE;
}

/**
  Check Fast Boot 30-second WDT (TCO Timer) timeout has occurred

  @retval  TRUE   Timeout has occurred
  @retval  FALSE  Timeout didnt occur
**/
BOOLEAN
IsTimeOutOccurred (
  VOID
  )
{
  if (IoRead8 (PcdGet16 (PcdTcoBaseAddress) + R_TCO_IO_TWDS) == FAST_BOOT_WDT_RESET_SIGNATURE) {
    IoWrite8 (PcdGet16 (PcdTcoBaseAddress) + R_TCO_IO_TWDS, 0);
    DEBUG ((DEBUG_INFO, "IsTimeOutOccurred TRUE\n"));
    return TRUE;
  }
  return FALSE;
}

/**
  Check if the value is of ODD parity.

  @param[in]  Value  The 8 bits value

  @retval     TRUE   The value is of ODD parity
  @retval     FALSE  The value is not of ODD parity
**/
BOOLEAN
CheckOddParity (
  IN  UINT8 Value
  )
{
  UINT8                           OddNum;
  UINT8                           Index;

  OddNum = 0;
  for (Index = 0; Index < 8; Index++) {
    if (Value & (1 << Index)) {
      OddNum++;
    }
  }
  return (BOOLEAN) (OddNum % 2 != 0);
}

/**
  Detect recovery mode

  Detect Jumper to determine if recovery mode is set

  @retval  EFI_SUCCESS       System in Recovery Mode
  @retval  EFI_UNSUPPORTED   System doesn't support Recovery Mode
  @retval  EFI_NOT_FOUND     System is not in Recovery Mode
**/

EFI_STATUS
EFIAPI
IsRecoveryMode (
  VOID
  )
{
  BOOLEAN               RecJumperFlag;
  UINT32                BitMask;
  EFI_STATUS            Status;
  UINT32                RecoveryModeGpio;
  PCD64_BLOB            Pcd64;

  if (PcdGet8 (PcdBoardType) == BoardTypeSv ||
      PcdGet8 (PcdBoardType) == BoardTypeRvpErb ||
      PcdGet8 (PcdPlatformFlavor) == FlavorUpServer) {
    return RETURN_UNSUPPORTED; /* No Recovery Jumper*/
  }

  RecoveryModeGpio = 0;
  Pcd64.Blob = PcdGet64 (PcdRecoveryModeGpio);
  if (Pcd64.Blob != 0 && Pcd64.BoardGpioConfig.Type == BoardGpioTypePch) {
    Status = GpioGetInputValue (Pcd64.BoardGpioConfig.u.Pin, &RecoveryModeGpio);
    ASSERT_EFI_ERROR (Status);
    if (EFI_ERROR (Status)) {
      return Status;
    }
  } else {
    DEBUG ((DEBUG_ERROR, "Invalid Recovery GPIO, PcdRecoveryModeGpio = 0x%x\n", PcdGet64 (PcdRecoveryModeGpio)));
    return EFI_NOT_FOUND;
  }

  BitMask = BIT0;
  RecJumperFlag = ((BOOLEAN)(!(RecoveryModeGpio & BitMask)));
  if (RecJumperFlag) {
    DEBUG ((DEBUG_INFO, "RecJumperFlag SET\n"));
    return EFI_SUCCESS;
  } else {
    DEBUG ((DEBUG_INFO, "RecJumperFlag NOT SET\n"));
    return EFI_NOT_FOUND;
  }
}
