/** @file

@copyright
  INTEL CONFIDENTIAL
  Copyright 2018 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
**/

#include <Uefi.h>
#include <Library/HobLib.h>
#include <Library/PcdLib.h>
#include <Library/UefiBootServicesTableLib.h>
#include <Library/DxeUpdatePlatformInfoLib.h>
#include <Library/SetupInitLib.h>
#include <MemInfoHob.h>
#include <PlatformBoardConfig.h>
#include <PlatformBoardId.h>
#include <Pins/GpioPinsVer2Lp.h>
#include <PcieRegs.h>
#include <Register/PchRegs.h>
#include <Library/PciSegmentLib.h>
#include <Library/MemoryAllocationLib.h>
#include <Library/BoardConfigLib.h>
#include <Library/CnviLib.h>
#include <TigerLakeUBoardConfigPatchTable.h>
#include <Library/CpuPlatformLib.h>
#include <Protocol/RetimerCapsuleUpdate.h>

/**
   Retimer Platform Specific Data
**/

RETIMER_PLATFORM_DATA                                mRetimerPlatformDataTableTglU[MAX_TBT_RETIMER_DEVICE];

STATIC   TBT_PD_RETIMER_PLATFORM_DATA                mRetimerPlatformDataTableTglUThreeTbtPdRetimer[] =
{
  { TbtPdRetimerType, TRUE, 0, 0, 0, 0x00, 0x00, 0x00 },
  { TbtPdRetimerType, TRUE, 1, 0, 0, 0x00, 0x00, 0x00 },
  { TbtPdRetimerType, TRUE, 2, 1, 0, 0x00, 0x00, 0x00 }
};

STATIC   TBT_PD_RETIMER_PLATFORM_DATA                mRetimerPlatformDataTableTglUTwoTbtPdRetimer[] =
{
  { TbtPdRetimerType, TRUE, 0, 0, 0, 0x00, 0x00, 0x00 },
  { TbtPdRetimerType, TRUE, 1, 0, 0, 0x00, 0x00, 0x00 }
};


EFI_STATUS
PatchConfigurationDataInit (
  IN CONFIG_PATCH_STRUCTURE  *ConfigPatchStruct,
  IN UINTN                   ConfigPatchStructSize
  );

VOID
TglUInitPatchConfigurationData (
  VOID
  )
{
  UINT16    BoardId;
  BoardId = PcdGet16(PcdBoardId);
  UINTN     ConfigPatchStructSize;
  CONFIG_PATCH_STRUCTURE *PatchTable;
  UINTN                  PatchTableSize;
  CPU_STEPPING    CpuSteppingId;
  CPU_GENERATION  CpuGeneration;
  CPU_FAMILY      CpuFamily;


  // Reading CPU information for stepping speicfic patch action
  CpuSteppingId  = GetCpuStepping ();
  CpuGeneration  = GetCpuGeneration ();
  CpuFamily      = GetCpuFamily();
  PatchTable     = NULL;
  PatchTableSize = 0;

  switch (BoardId) {
    case BoardIdTglUDdr4:
      ConfigPatchStructSize = SIZE_OF_TABLE (mTigerlakeUDdr4RvpConfigPatchStruct, CONFIG_PATCH_STRUCTURE);
      UpdatePatchTable (&PatchTable, &PatchTableSize, mTigerlakeUDdr4RvpConfigPatchStruct, ConfigPatchStructSize);
      break;
    case BoardIdTglULp4Type4:
    case BoardIdTglULp4Aep:
      ConfigPatchStructSize = SIZE_OF_TABLE (mTigerlakeULp4RvpConfigPatchStruct, CONFIG_PATCH_STRUCTURE);
      UpdatePatchTable (&PatchTable, &PatchTableSize, mTigerlakeULp4RvpConfigPatchStruct, ConfigPatchStructSize);
      break;
    case BoardIdTglULp4Dg1Aep:
      ConfigPatchStructSize = SIZE_OF_TABLE (mTigerlakeULp4Dg1ConfigPatchStruct, CONFIG_PATCH_STRUCTURE);
      UpdatePatchTable (&PatchTable, &PatchTableSize, mTigerlakeULp4Dg1ConfigPatchStruct, ConfigPatchStructSize);
      break;
    case BoardIdTglULp4Gcs:
      ConfigPatchStructSize = SIZE_OF_TABLE (mTigerlakeULp4GcsRvp, CONFIG_PATCH_STRUCTURE);
      UpdatePatchTable (&PatchTable, &PatchTableSize, mTigerlakeULp4GcsRvp, ConfigPatchStructSize);
      break;
    case BoardIdTglULp5Type4:
      ConfigPatchStructSize = SIZE_OF_TABLE (mTigerLakeULp5ConfigPatchStruct, CONFIG_PATCH_STRUCTURE);
      UpdatePatchTable (&PatchTable, &PatchTableSize, mTigerLakeULp5ConfigPatchStruct, ConfigPatchStructSize);
      break;
    default:
      break;
  }

  if (IsTglPch () && IsPchLp () && (PchStepping () == PCH_Z0)) {
    ConfigPatchStructSize = SIZE_OF_TABLE (mTglPchZ0ConfigPatchStruct, CONFIG_PATCH_STRUCTURE);
    UpdatePatchTable (&PatchTable, &PatchTableSize, mTglPchZ0ConfigPatchStruct, ConfigPatchStructSize);
  }

  if (IsTglPch () && IsPchLp () && (PchStepping () < PCH_B0)) {
    ConfigPatchStructSize = SIZE_OF_TABLE (mTglPchA0ConfigPatchStruct, CONFIG_PATCH_STRUCTURE);
    UpdatePatchTable (&PatchTable, &PatchTableSize, mTglPchA0ConfigPatchStruct, ConfigPatchStructSize);
  }

  // Patch for Ax specific SaSetupConfig updates
  if (CpuGeneration == EnumTglCpu && CpuFamily == EnumCpuTglUltUlx && CpuSteppingId == EnumTglA0) {
    ConfigPatchStructSize = SIZE_OF_TABLE (mTglAxSaSetupConfigPatchStruct, CONFIG_PATCH_STRUCTURE);
    UpdatePatchTable (&PatchTable, &PatchTableSize, mTglAxSaSetupConfigPatchStruct, ConfigPatchStructSize);
  }

  // SAGV to be disabled for Lp4 on Ax stepping
  if ((CpuGeneration == EnumTglCpu && CpuFamily == EnumCpuTglUltUlx && CpuSteppingId == EnumTglA0)\
       && (BoardId != BoardIdTglUDdr4) && (BoardId != BoardIdTglUDdr4Sv)) {
    ConfigPatchStructSize = SIZE_OF_TABLE (mTglAxMrcSetupConfigPatchStruct, CONFIG_PATCH_STRUCTURE);
    UpdatePatchTable (&PatchTable, &PatchTableSize, mTglAxMrcSetupConfigPatchStruct, ConfigPatchStructSize);
  }

/**
  WA to enable BT audio offload for integrated bluetooth TGL LP A0/Z0
**/
  if (CnviIsPresent () && (IsTglPch () && IsPchLp () && ((PchStepping () == PCH_Z0) || (PchStepping () == PCH_A0)))) {
    ConfigPatchStructSize = SIZE_OF_TABLE (mTglUA0ConfigPatchStruct, CONFIG_PATCH_STRUCTURE);
    UpdatePatchTable (&PatchTable, &PatchTableSize, mTglUA0ConfigPatchStruct, ConfigPatchStructSize);
  }

#if FixedPcdGet8(PcdEmbeddedEnable) == 0x1
  ConfigPatchStructSize = SIZE_OF_TABLE (mTigerlakeEmbeddedConfigPatchStruct, CONFIG_PATCH_STRUCTURE);
  UpdatePatchTable (&PatchTable, &PatchTableSize, mTigerlakeEmbeddedConfigPatchStruct, ConfigPatchStructSize);
#endif

  PatchConfigurationDataInit (PatchTable, PatchTableSize);
  FreePatchTable (&PatchTable, &PatchTableSize);
}

STATIC
EFI_STATUS
BoardHookPlatformSetup (
  VOID * Content
  )
{
  ((SETUP_VOLATILE_DATA *)Content)->PlatId = PcdGet16 (PcdBoardId);

  return RETURN_SUCCESS;
}

/**
 Init Misc Platform Board Config Block.

 @param[in]  VOID

 @retval VOID
**/
VOID
TglUBoardMiscInit (
  VOID
)
{
  PcdSet64S (PcdFuncBoardHookPlatformSetupOverride, (UINT64) (UINTN) BoardHookPlatformSetup);
}

VOID
TglUBoardSmbiosInit (
  VOID
  )
{
  PcdSet64S (PcdSmbiosFabBoardName, (UINTN) PcdGetPtr (PcdBoardName));
  //
  // @todo : Update Slot Entry Table for all the supported boards using below PCD.
  //  This PCD follows SYSTEM_SLOT_ENTRY structure in \Include\SmbiosPlatformInfoDefinition.h
  //PcdSet64S (PcdSmbiosMainSlotEntry, NULL);
}

VOID
TglUUpdateDimmPopulation (
  VOID
  )
{
  MEMORY_INFO_DATA_HOB    *MemInfo;
  UINT8                   Slot0;
  UINT8                   Slot1;
  UINT8                   Slot2;
  UINT8                   Slot3;
  CONTROLLER_INFO         *ControllerInfo;
  EFI_HOB_GUID_TYPE       *GuidHob;

  GuidHob = NULL;
  MemInfo = NULL;
  GuidHob = GetFirstGuidHob (&gSiMemoryInfoDataGuid);
  ASSERT (GuidHob != NULL);
  if (GuidHob != NULL) {
    MemInfo = (MEMORY_INFO_DATA_HOB *) GET_GUID_HOB_DATA (GuidHob);
  }
  if (MemInfo != NULL) {
    if ( PcdGet8 (PcdPlatformFlavor) == FlavorDesktop ||
         PcdGet8 (PcdPlatformFlavor) == FlavorUpServer ||
         PcdGet8 (PcdPlatformFlavor) == FlavorWorkstation
        ) {
        ControllerInfo = &MemInfo->Controller[0];
        Slot0 = ControllerInfo->ChannelInfo[0].DimmInfo[0].Status;
        Slot1 = ControllerInfo->ChannelInfo[0].DimmInfo[1].Status;
        Slot2 = ControllerInfo->ChannelInfo[1].DimmInfo[0].Status;
        Slot3 = ControllerInfo->ChannelInfo[1].DimmInfo[1].Status;

      //
      // Channel 0          Channel 1
      // Slot0   Slot1      Slot0   Slot1      - Population            AIO board
      // 0          0          0          0          - Invalid        - Invalid
      // 0          0          0          1          - Valid          - Invalid
      // 0          0          1          0          - Invalid        - Valid
      // 0          0          1          1          - Valid          - Valid
      // 0          1          0          0          - Valid          - Invalid
      // 0          1          0          1          - Valid          - Invalid
      // 0          1          1          0          - Invalid        - Invalid
      // 0          1          1          1          - Valid          - Invalid
      // 1          0          0          0          - Invalid        - Valid
      // 1          0          0          1          - Invalid        - Invalid
      // 1          0          1          0          - Invalid        - Valid
      // 1          0          1          1          - Invalid        - Valid
      // 1          1          0          0          - Valid          - Valid
      // 1          1          0          1          - Valid          - Invalid
      // 1          1          1          0          - Invalid        - Valid
      // 1          1          1          1          - Valid          - Valid
      //
      if ((Slot0 && (Slot1 == 0)) || (Slot2 && (Slot3 == 0))) {
        PcdSetBoolS (PcdDimmPopulationError, TRUE);
      }
    }
  }
}

/**
  Enable Tier2 GPIO Sci wake capability.

  @retval EFI_SUCCESS   The function completed successfully.
**/
EFI_STATUS
TglUTier2GpioWakeSupport (
  VOID
  )
{
  BOOLEAN Tier2GpioWakeEnable;

  Tier2GpioWakeEnable = FALSE;

  //
  // Root port #03: M.2 WLAN
  //
  if (IsPcieEndPointPresent (2)) {
   Tier2GpioWakeEnable = TRUE;
  }
  PcdSetBoolS (PcdGpioTier2WakeEnable, Tier2GpioWakeEnable);

  return EFI_SUCCESS;
}

/**
  Copy Tcss Retimer Platform Data.

  @param[in|out]  DestinationTableCurrentIndex   The pointer to the destination buffer Current Index.
  @param[in]      SourceTable                    The pointer to the source buffer of the memory copy.
  @param[in]      SourceTableSize                The number of bytes to copy from SourceBuffer to DestinationBuffer.
  @param[in]      SourceTableType                Source Table Buffer Type

  @retval     EFI_SUCCESS            The function completed successfully.
  @retval     EFI_INVALID_PARAMETER  Invalid Parameter Passed.
**/

EFI_STATUS
TglUCopyTcssRetimerPlatformData (
  IN OUT UINT32                               *DestinationTableCurrentIndex,
  IN  VOID                                    *SourceTable,
  IN  UINT32                                  SourceTableSize,
  IN  RETIMER_TYPE                            SourceTableType
  )
{
  UINT32                                      TableIndex;
  UINT32                                      SourceTableElementSize;
  UINT32                                      SourceTableElementCount;

  SourceTableElementSize = 0;

  if (DestinationTableCurrentIndex == NULL) {
    return EFI_INVALID_PARAMETER;
  }

  if (SourceTable == NULL) {
    return EFI_INVALID_PARAMETER;
  }

  if (SourceTableSize == 0) {
    return EFI_INVALID_PARAMETER;
  }

  if ((*DestinationTableCurrentIndex) >= MAX_TBT_RETIMER_DEVICE) {
    // Return EFI_INVALID_PARAMETER as Invalid Destination Table Current Index is passed
    return EFI_INVALID_PARAMETER;
  }
  if (SourceTableType == TbtPdRetimerType) {
    SourceTableElementSize = sizeof (TBT_PD_RETIMER_PLATFORM_DATA);
  } else if (SourceTableType == NonTbtI2cRetimerType) {
    SourceTableElementSize = sizeof (NON_TBT_I2C_RETIMER_PLATFORM_DATA);
  } else {
    // Return EFI_INVALID_PARAMETER as Invalid Source Table Type is passed
    return EFI_INVALID_PARAMETER;
  }

  SourceTableElementCount = SourceTableSize / SourceTableElementSize;

  if (((*DestinationTableCurrentIndex) + SourceTableElementCount) > MAX_TBT_RETIMER_DEVICE) {
    // Return EFI_INVALID_PARAMETER as Destination Buffer Size is small and Array boundary condition will be reached
    return EFI_INVALID_PARAMETER;
  }

  for (TableIndex = 0; TableIndex < SourceTableElementCount; TableIndex++) {
    if (((RETIMER_PLATFORM_DATA*)SourceTable)->RetimerType == SourceTableType) { // Only add entry to Dastination Table if Source Table Type Match
      gBS->CopyMem (&mRetimerPlatformDataTableTglU[(*DestinationTableCurrentIndex)], SourceTable, SourceTableElementSize);
      *DestinationTableCurrentIndex += 1;
    }
    SourceTable = (VOID*) ((UINTN)SourceTable + SourceTableElementSize);
  }

  return EFI_SUCCESS;
}

/**
  Init Tcss Retimer Platform Data Pcd.

  @param[in]  BoardId           An unsigned integrer represent the board id.

  @retval     EFI_SUCCESS       The function completed successfully.
**/

EFI_STATUS
TglUInitTcssRetimerPlatformDataPcd (
  VOID
)
{
  UINT32                               RetimerDataTableIndex;
  EFI_STATUS                           Status = EFI_SUCCESS;

  DEBUG((DEBUG_INFO, "TglUInitTcssRetimerPlatformDataPcd \n"));
  //
  // Map Retimer and PD Controller Available on Board
  //
  RetimerDataTableIndex = 0; // Initialize Destination Table Index to 0 to Start from Begining
  Status = TglUCopyTcssRetimerPlatformData (&RetimerDataTableIndex, (VOID *) mRetimerPlatformDataTableTglUThreeTbtPdRetimer, sizeof(mRetimerPlatformDataTableTglUThreeTbtPdRetimer), TbtPdRetimerType);
  if (EFI_ERROR (Status)) {
    DEBUG((DEBUG_INFO, "mRetimerPlatformDataTableTglThreeTbtPdRetimer fail. \n"));

    // Updating Mapping of Retimer and PD Controller as Retimer Device is not Available
    RetimerDataTableIndex = 0; // Initialize Destination Table Index to 0 to Start from Begining
    Status = TglUCopyTcssRetimerPlatformData (&RetimerDataTableIndex, (VOID *) mRetimerPlatformDataTableTglUTwoTbtPdRetimer, sizeof(mRetimerPlatformDataTableTglUTwoTbtPdRetimer), TbtPdRetimerType);

    if (EFI_ERROR (Status)) {
      DEBUG((DEBUG_INFO, "mRetimerPlatformDataTableTglUTwoTbtPdRetimer fail. \n"));
    }
  }

  if ( !EFI_ERROR (Status)) {
    // Saving Retimer Platform Data Table into PCD
    PcdSet64S(PcdBoardRetimerDataTablePtr, (UINTN)mRetimerPlatformDataTableTglU);
    PcdSet32S(PcdBoardRetimerDataTableEntryCount, RetimerDataTableIndex);
  }

  return Status;
}

/**
  A hook for board-specific initialization after PCI enumeration.

  @retval EFI_SUCCESS   The board initialization was successful.
**/
EFI_STATUS
EFIAPI
TglUBoardInitAfterPciEnumeration (
  VOID
  )
{
  DEBUG ((DEBUG_INFO, "TglUBoardInitAfterPciEnumeration\n"));

  TglUInitPatchConfigurationData ();

  TglUInitTcssRetimerPlatformDataPcd ();

  TglUTier2GpioWakeSupport ();

  TglUBoardMiscInit();

  TglUBoardSmbiosInit ();

  UpdatePlatformInfo ();

  return EFI_SUCCESS;
}

/**
  A hook for board-specific functionality for the ReadyToBoot event.

  @retval EFI_SUCCESS   The board initialization was successful.
**/
EFI_STATUS
EFIAPI
TglUBoardInitReadyToBoot (
  VOID
  )
{
  DEBUG ((DEBUG_INFO, "TglUBoardInitReadyToBoot\n"));

  TglUUpdateDimmPopulation ();

  return EFI_SUCCESS;
}

/**
  A hook for board-specific functionality for the ExitBootServices event.

  @retval EFI_SUCCESS   The board initialization was successful.
**/
EFI_STATUS
EFIAPI
TglUBoardInitEndOfFirmware (
  VOID
  )
{
  DEBUG ((DEBUG_INFO, "TglUBoardInitEndOfFirmware\n"));

  return EFI_SUCCESS;
}
