/** @file

@copyright
  INTEL CONFIDENTIAL
  Copyright 2017 - 2020 Intel Corporation.

  The source code contained or described herein and all documents related to the
  source code ("Material") are owned by Intel Corporation or its suppliers or
  licensors. Title to the Material remains with Intel Corporation or its suppliers
  and licensors. The Material may contain trade secrets and proprietary and
  confidential information of Intel Corporation and its suppliers and licensors,
  and is protected by worldwide copyright and trade secret laws and treaty
  provisions. No part of the Material may be used, copied, reproduced, modified,
  published, uploaded, posted, transmitted, distributed, or disclosed in any way
  without Intel's prior express written permission.

  No license under any patent, copyright, trade secret or other intellectual
  property right is granted to or conferred upon you by disclosure or delivery
  of the Materials, either expressly, by implication, inducement, estoppel or
  otherwise. Any license under such intellectual property rights must be
  express and approved by Intel in writing.

  Unless otherwise agreed by Intel in writing, you may not remove or alter
  this notice or any other notice embedded in Materials by Intel or
  Intel's suppliers or licensors in any way.

  This file contains a 'Sample Driver' and is licensed as such under the terms
  of your license agreement with Intel or your vendor. This file may be modified
  by the user, subject to the additional terms of the license agreement.

@par Specification Reference:
**/

#include <PiPei.h>
#include <Library/DebugLib.h>
#include <Library/PcdLib.h>

#include <Library/SiliconInitLib.h>
#include <Library/PreSiliconEnvDetectLib.h>
#include <Library/BoardConfigLib.h>

#include "GpioTableTglYPostMem.h"
#include <Pins/GpioPinsVer1Lp.h>
#include <Library/PeiServicesLib.h>
#include <Library/GpioLib.h>
#include <PlatformBoardId.h>
#include <PlatformBoardConfig.h>
#include <UsbTypeC.h>

/**
  Tigerlake Y boards configuration init function for PEI post memory phase.

  @retval EFI_SUCCESS             The function completed successfully.
**/
EFI_STATUS
EFIAPI
TglYGpioTableInit (
  VOID
  )
{
  EFI_STATUS        Status;
  GPIO_INIT_CONFIG  *GpioTable;
  UINT16    BoardId;
  //
  // GPIO Table Init, Update PostMem GPIO table to PcdBoardGpioTable
  //
  Status = EFI_SUCCESS;
  GpioTable = NULL;
  //
  // GPIO Table Init
  //
  BoardId = PcdGet16(PcdBoardId);

  switch (BoardId) {
    case BoardIdTglYLp4Type4Crb:
    case BoardIdTglYExsLp4Type4:
      GpioTable = mGpioTableTglYLp4Crb;
      break;
    case BoardIdTglYLp4Type4:
    case BoardIdTglYLp5Type4:
      GpioTable = mGpioTableTglYLp4;
      break;
    case BoardIdTglYLp4Aep:
      GpioTable = mGpioTableTglYLp4Aep;
      break;
  }
  if (GpioTable != NULL) {
    ConfigureGpioTabletoPCD (GpioTable, POST_MEM);
  } else {
    DEBUG ((DEBUG_INFO, "PostMem GpioTable is NULL\n"));
  }

  return Status;
}

/**
  Touch panel GPIO init function for PEI post memory phase.
**/
VOID
TglYTouchPanelGpioInit (
  VOID
  )
{
  UINT16    BoardId;
  BoardId = PcdGet16(PcdBoardId);

  switch (BoardId) {
    case BoardIdTglYLp4Type4:
    case BoardIdTglYLp4Aep:
    case BoardIdTglYLp5Type4:
      PcdSet32S (PcdBoardGpioTableTouchPanel1,  (UINTN) mTglYTouchPanel1GpioTable);
      PcdSet16S (PcdBoardGpioTableTouchPanel1Size,  sizeof(mTglYTouchPanel1GpioTable)/ sizeof(GPIO_INIT_CONFIG));
      PcdSet32S (PcdBoardGpioTableTouchPanel2,  (UINTN) mTglYTouchPanel2GpioTable);
      PcdSet16S (PcdBoardGpioTableTouchPanel2Size,  sizeof(mTglYTouchPanel2GpioTable)/ sizeof(GPIO_INIT_CONFIG));
      break;
    case BoardIdTglYLp4Type4Crb:
    case BoardIdTglYExsLp4Type4:
      PcdSet32S (PcdBoardGpioTableTouchPanel1,  (UINTN) mTglYCrbTouchPanel1GpioTable);
      PcdSet16S (PcdBoardGpioTableTouchPanel1Size,  sizeof(mTglYCrbTouchPanel1GpioTable)/ sizeof(GPIO_INIT_CONFIG));
      PcdSet32S (PcdBoardGpioTableTouchPanel2,  (UINTN) mTglYTouchPanel2GpioTable);
      PcdSet16S (PcdBoardGpioTableTouchPanel2Size,  sizeof(mTglYTouchPanel2GpioTable)/ sizeof(GPIO_INIT_CONFIG));
      break;
  }
}

/**
  HDA VerbTable init function for PEI post memory phase.
**/
VOID
TglYHdaVerbTableInit (
  VOID
  )
{
  HDAUDIO_VERB_TABLE *VerbTable;
  HDAUDIO_VERB_TABLE *VerbTable2;

  VerbTable  = NULL;
  VerbTable2 = NULL;

  DEBUG ((DEBUG_INFO, "HDA: Init default verb tables (Realtek ALC700 and ALC701)\n"));
  VerbTable = &HdaVerbTableAlc711;

  PcdSet32S (PcdHdaVerbTable, (UINT32) VerbTable);
  PcdSet32S (PcdHdaVerbTable2, (UINT32) VerbTable2);

  PcdSet32S (PcdDisplayAudioHdaVerbTable, (UINT32) &HdaVerbTableDisplayAudio);

  // TGL POR Codecs - Realtek ALC700, ALC701, ALC274 (external - connected via HDA header)
  PcdSet32S (PcdCommonHdaVerbTable1, (UINT32) &HdaVerbTableAlc711);

  PcdSet32S (PcdCommonHdaVerbTable2, (UINT32) &HdaVerbTableAlc701);

  PcdSet32S (PcdCommonHdaVerbTable3, (UINT32) &HdaVerbTableAlc274);

  return;
}

/**
  Misc. init function for PEI post memory phase.
**/
VOID
TglYBoardMiscInit (
  VOID
  )
{
  UINT16    BoardId;
  BoardId = PcdGet16(PcdBoardId);
  PcdSet8S (PcdSataPortsEnable0, 0x1);
  PcdSetBoolS (PcdSataLedEnable, FALSE);
  PcdSetBoolS (PcdVrAlertEnable, FALSE);
  PcdSetBoolS (PcdPchThermalHotEnable, FALSE);
  PcdSetBoolS (PcdMemoryThermalSensorGpioCPmsyncEnable, FALSE);
  PcdSetBoolS (PcdMemoryThermalSensorGpioDPmsyncEnable, FALSE);

  switch (BoardId) {
    case BoardIdTglYLp4Type4:
    case BoardIdTglYExsLp4Type4:
    case BoardIdTglYLp4Type4Crb:
      //
      // For PCIe Slot1 - x4 Connector
      //
      PcdSet32S (PcdPcieSlot1WakeGpioPin, GPIO_VER2_LP_GPP_F5);
      PcdSet8S (PcdPcieSlot1RootPort, 5);
      //
      // For PCIe Slot2 - x1 Connector
      //
      PcdSet32S (PcdPcieSlot2WakeGpioPin, GPIO_VER2_LP_GPP_F4);
      PcdSet32S (PcdPchPCIeSlot2PwrEnableGpioNo, GPIO_VER2_LP_GPP_H2);
      PcdSet32S (PcdPchPCIeSlot2RstGpioNo, GPIO_VER2_LP_GPP_F10);
      PcdSetBoolS (PcdPchPCIeSlot2PwrEnableGpioPolarity, PIN_GPIO_ACTIVE_HIGH);
      PcdSetBoolS (PcdPchPCIeSlot2RstGpioPolarity, PIN_GPIO_ACTIVE_LOW);
      PcdSet8S (PcdPcieSlot2RootPort, 8);

      //
      // PCH M.2 SSD and Sata port
      //
      PcdSet32S (PcdPchSsd1PwrEnableGpioNo, GPIO_VER2_LP_GPP_B16);  // PCH M.2 SSD power enable gpio pin
      PcdSet32S (PcdPchSsd1RstGpioNo, GPIO_VER2_LP_GPP_H0);  // PCH M.2 SSD reset gpio pin
      PcdSetBoolS (PcdPchSsd1PwrEnableGpioPolarity, PIN_GPIO_ACTIVE_HIGH);  // PCH M.2 SSD power enable gpio pin polarity
      PcdSetBoolS (PcdPchSsd1RstGpioPolarity, PIN_GPIO_ACTIVE_LOW);  // PCH M.2 SSD reset gpio pin polarity
      PcdSet32S (PcdPchSataPortPwrEnableGpioNo, GPIO_VER2_LP_GPP_B16);  // Sata port power pin
      PcdSetBoolS (PcdPchSataPortPwrEnableGpioPolarity, PIN_GPIO_ACTIVE_HIGH);  // Sata port power pin polarit
      //
      // Touch Pad and Panels 0 & 1
      //
      PcdSet32S (PcdTouchpadIrqGpio, GPIO_VER2_LP_GPP_C8);  // Touch Pad Interrupt Pin
      PcdSetBoolS (PcdTouchpadIrqGpioPolarity, PIN_GPIO_ACTIVE_LOW);  // Touch Pad Interrupt pin polarity
      PcdSet32S (PcdTouchpanelIrqGpio, GPIO_VER2_LP_GPP_E17);  // Touch 0 Interrupt Pin
      PcdSet32S (PcdTouchpanelPwrEnableGpio, GPIO_VER2_LP_GPP_H23);  // Touch 0 power enable pin
      PcdSet32S (PcdTouchpanelRstGpio, GPIO_VER2_LP_GPP_E6);  // Touch 0 reset pin
      PcdSetBoolS (PcdTouchpanelIrqGpioPolarity, PIN_GPIO_ACTIVE_HIGH);  // Touch 0 Interrupt Pin polarity
      PcdSetBoolS (PcdTouchpanelPwrEnableGpioPolarity, PIN_GPIO_ACTIVE_HIGH);  // Touch 0 power enable pin polarity
      PcdSetBoolS (PcdTouchpanelRstGpioPolarity, PIN_GPIO_ACTIVE_HIGH);  // Touch 0 reset pin polarity
      PcdSet32S (PcdTouchpanel1IrqGpio, GPIO_VER2_LP_GPP_F18);  // Touch 1 Interrupt Pin
      PcdSet32S (PcdTouchpanel1PwrEnableGpio, GPIO_VER2_LP_GPP_F7);  // Touch 1 power enable pin
      PcdSet32S (PcdTouchpanel1RstGpio, GPIO_VER2_LP_GPP_F17);  // Touch 1 reset pin
      PcdSetBoolS (PcdTouchpanel1IrqGpioPolarity, PIN_GPIO_ACTIVE_HIGH);  // Touch 1 Interrupt Pin polarity
      PcdSetBoolS (PcdTouchpanel1PwrEnableGpioPolarity, PIN_GPIO_ACTIVE_HIGH);  // Touch 1 power enable pin polarity
      PcdSetBoolS (PcdTouchpanel1RstGpioPolarity, PIN_GPIO_ACTIVE_HIGH);  // Touch 1 reset pin polarit
      //
      // WLAN
      //
      PcdSet32S(PcdWlanWakeGpio, GPIO_VER2_LP_GPP_C23);  // WiFi wake pin
      PcdSet8S (PcdWlanRootPortNumber, 3);
      break;
    case BoardIdTglYLp5Type4:
      //
      // For PCIe Slot1 - x4 Connector
      //
      PcdSet32S (PcdPcieSlot1WakeGpioPin, GPIO_VER2_LP_GPP_F5);
      PcdSet8S (PcdPcieSlot1RootPort, 5);
      //
      // For PCIe Slot2 - x1 Connector
      //
      PcdSet32S (PcdPcieSlot2WakeGpioPin, GPIO_VER2_LP_GPP_F4);
      PcdSet32S (PcdPchPCIeSlot2PwrEnableGpioNo, GPIO_VER2_LP_GPP_H17);
      PcdSet32S (PcdPchPCIeSlot2RstGpioNo, GPIO_VER2_LP_GPP_F10);
      PcdSetBoolS (PcdPchPCIeSlot2PwrEnableGpioPolarity, PIN_GPIO_ACTIVE_HIGH);
      PcdSetBoolS (PcdPchPCIeSlot2RstGpioPolarity, PIN_GPIO_ACTIVE_HIGH);
      PcdSet8S (PcdPcieSlot2RootPort, 8);

      //
      // PCH M.2 SSD and Sata port
      //
      PcdSet32S (PcdPchSsd1PwrEnableGpioNo, GPIO_VER2_LP_GPP_B16);  // PCH M.2 SSD power enable gpio pin
      PcdSet32S (PcdPchSsd1RstGpioNo, GPIO_VER2_LP_GPP_H0);  // PCH M.2 SSD reset gpio pin
      PcdSetBoolS (PcdPchSsd1PwrEnableGpioPolarity, PIN_GPIO_ACTIVE_HIGH);  // PCH M.2 SSD power enable gpio pin polarity
      PcdSetBoolS (PcdPchSsd1RstGpioPolarity, PIN_GPIO_ACTIVE_HIGH);  // PCH M.2 SSD reset gpio pin polarity
      PcdSet32S (PcdPchSataPortPwrEnableGpioNo, GPIO_VER2_LP_GPP_B16);  // Sata port power pin
      PcdSetBoolS (PcdPchSataPortPwrEnableGpioPolarity, PIN_GPIO_ACTIVE_HIGH);  // Sata port power pin polarit
      //
      // Touch Pad and Panels 0 & 1
      //
      PcdSet32S (PcdTouchpadIrqGpio, GPIO_VER2_LP_GPP_C8);  // Touch Pad Interrupt Pin
      PcdSetBoolS (PcdTouchpadIrqGpioPolarity, PIN_GPIO_ACTIVE_LOW);  // Touch Pad Interrupt pin polarity
      PcdSet32S (PcdTouchpanelIrqGpio, GPIO_VER2_LP_GPP_E17);  // Touch 0 Interrupt Pin
      PcdSet32S (PcdTouchpanelPwrEnableGpio, GPIO_VER2_LP_GPP_H23);  // Touch 0 power enable pin
      PcdSet32S (PcdTouchpanelRstGpio, GPIO_VER2_LP_GPP_E6);  // Touch 0 reset pin
      PcdSetBoolS (PcdTouchpanelIrqGpioPolarity, PIN_GPIO_ACTIVE_HIGH);  // Touch 0 Interrupt Pin polarity
      PcdSetBoolS (PcdTouchpanelPwrEnableGpioPolarity, PIN_GPIO_ACTIVE_HIGH);  // Touch 0 power enable pin polarity
      PcdSetBoolS (PcdTouchpanelRstGpioPolarity, PIN_GPIO_ACTIVE_HIGH);  // Touch 0 reset pin polarity
      PcdSet32S (PcdTouchpanel1IrqGpio, GPIO_VER2_LP_GPP_F18);  // Touch 1 Interrupt Pin
      PcdSet32S (PcdTouchpanel1PwrEnableGpio, GPIO_VER2_LP_GPP_F7);  // Touch 1 power enable pin
      PcdSet32S (PcdTouchpanel1RstGpio, GPIO_VER2_LP_GPP_F17);  // Touch 1 reset pin
      PcdSetBoolS (PcdTouchpanel1IrqGpioPolarity, PIN_GPIO_ACTIVE_HIGH);  // Touch 1 Interrupt Pin polarity
      PcdSetBoolS (PcdTouchpanel1PwrEnableGpioPolarity, PIN_GPIO_ACTIVE_HIGH);  // Touch 1 power enable pin polarity
      PcdSetBoolS (PcdTouchpanel1RstGpioPolarity, PIN_GPIO_ACTIVE_HIGH);  // Touch 1 reset pin polarit
      //
      // WLAN
      //
      PcdSet32S(PcdWlanWakeGpio, GPIO_VER2_LP_GPP_C23);  // WiFi wake pin
      PcdSet8S (PcdWlanRootPortNumber, 3);
      break;
    case BoardIdTglYLp4Aep:
      PcdSet32S (PcdPcieSlot2WakeGpioPin, GPIO_VER2_LP_GPP_F4);
      PcdSet32S (PcdTouchpadIrqGpio, GPIO_VER2_LP_GPP_C8);  // Touch Pad Interrupt Pin
      PcdSetBoolS (PcdTouchpadIrqGpioPolarity, PIN_GPIO_ACTIVE_LOW);  // Touch Pad Interrupt pin polarity
      PcdSet32S (PcdTouchpanelIrqGpio, GPIO_VER2_LP_GPP_E17);  // Touch 0 Interrupt Pin
      PcdSet32S (PcdTouchpanelRstGpio, GPIO_VER2_LP_GPP_E6);  // Touch 0 reset pin
      PcdSetBoolS (PcdTouchpanelIrqGpioPolarity, PIN_GPIO_ACTIVE_HIGH);  // Touch 0 Interrupt Pin polarity
      PcdSetBoolS (PcdTouchpanelRstGpioPolarity, PIN_GPIO_ACTIVE_HIGH);  // Touch 0 reset pin polarity
      break;
  }

  return;
}

/**
  Tbt init function for PEI post memory phase.
**/
VOID
TglYBoardTbtInit (
  VOID
  )
{
  PcdSet8S (PcdCpuUsb30OverCurrentPinPort0, USB_OC_SKIP);
  PcdSet8S (PcdCpuUsb30OverCurrentPinPort1, USB_OC_SKIP);
  PcdSet8S (PcdCpuUsb30OverCurrentPinPort2, USB_OC_SKIP);
  PcdSet8S (PcdCpuUsb30OverCurrentPinPort3, USB_OC_SKIP);
  PcdSet8S (PcdCpuUsb30OverCurrentPinPort4, USB_OC_SKIP);
  PcdSet8S (PcdCpuUsb30OverCurrentPinPort5, USB_OC_SKIP);

  //
  // Clear ALL port setting by default.
  //
  PcdSet8S (PcdCpuUsb30PortEnable, 0);

  if (PcdGet8 (PcdEcEspiFlashSharingMode) == 1) {
    PcdSet8S (PcdCpuUsb30OverCurrentPinPort0, 0);
    PcdSet8S (PcdCpuUsb30OverCurrentPinPort1, 1);
    PcdSet8S (PcdCpuUsb30OverCurrentPinPort2, 2);
    PcdSet8S (PcdCpuUsb30PortEnable, 0x07);
  } else {
    PcdSet8S (PcdCpuUsb30OverCurrentPinPort0, 0);
    PcdSet8S (PcdCpuUsb30OverCurrentPinPort1, 0);
    PcdSet8S (PcdCpuUsb30OverCurrentPinPort2, 3);
    PcdSet8S (PcdCpuUsb30PortEnable, 0x07);
  }

  PcdSet8S (PcdITbtRootPortNumber, 3);
}

/**
   Security GPIO init function for PEI post memory phase.
**/
VOID
TglYBoardSecurityInit (
  VOID
  )
{
  GPIO_PAD  GpioPad;
  UINT32    GpioIrqNumber;

  GpioPad = (GPIO_PAD) GPIO_VER2_LP_GPP_C14;
  GpioGetPadIoApicIrqNumber (GpioPad, &GpioIrqNumber);

  DEBUG ((DEBUG_INFO, "TPM Interrupt Number: %d\n", GpioIrqNumber));
  PcdSet32S (PcdTpm2CurrentIrqNum, GpioIrqNumber);
}

/**
  Board Specific Init for PEI post memory phase.
**/
VOID
PeiTglYBoardSpecificInitPostMemNull (
  VOID
  )
{
}

/**
  Board's PCD function hook init function for PEI post memory phase.
**/
VOID
TglYBoardFunctionInit (
  VOID
)
{
  PcdSet32S (PcdFuncPeiBoardSpecificInitPostMem, (UINTN) PeiTglYBoardSpecificInitPostMemNull);
}

/**
  PMC-PD solution enable init lib
**/
VOID
TglYBoardPmcPdInit (
  VOID
  )
{
  PcdSetBoolS (PcdBoardPmcPdEnable, 1);
}

/**
  TypeC init function for before silicon init.
**/
VOID
TglYBoardTypeCPortMapping (
  VOID
  )
{
  UINT16                          BoardId;

  BoardId = PcdGet16 (PcdBoardId);

  switch (BoardId) {
    case BoardIdTglYLp4Type4Crb:
    case BoardIdTglYLp4Type4:
    case BoardIdTglYExsLp4Type4:
    case BoardIdTglYLp4Aep:
    case BoardIdTglYLp5Type4:
      PcdSetBoolS (PcdUsbTypeCSupport, TRUE);
      // Number of ports supported
      PcdSet8S (PcdTypeCPortsSupported, 3);
      // Type C port 1 mapping
      PcdSet8S (PcdUsbTypeCPort1, 0);
      PcdSet8S (PcdUsbTypeCPort1Pch, 1);
      PcdSet8S (PcdUsbCPort1Properties, (0 | USB_TYPEC_CPU << 1 | SPLIT_SUPPORTED));
      // Type C port 2 mapping
      PcdSet8S (PcdUsbTypeCPort2, 1);
      PcdSet8S (PcdUsbTypeCPort2Pch, 5);
      PcdSet8S (PcdUsbCPort2Properties, (0 | USB_TYPEC_CPU << 1 | SPLIT_SUPPORTED));
      // Type C port 3 mapping
      PcdSet8S (PcdUsbTypeCPort3, 2);
      PcdSet8S (PcdUsbTypeCPort3Pch, 6);
      PcdSet8S (PcdUsbCPort3Properties, (0 | USB_TYPEC_CPU << 1 | SPLIT_SUPPORTED));
      break;
    default:
      PcdSetBoolS (PcdUsbTypeCSupport, FALSE);
      break;
  }
}

/**
  Configure GPIO, TouchPanel, HDA, PMC, TBT etc.

  @retval  EFI_SUCCESS   Operation success.
**/
EFI_STATUS
EFIAPI
TglYBoardInitBeforeSiliconInit (
  VOID
  )
{
  TglYGpioTableInit ();
  TglYTouchPanelGpioInit ();
  TglYHdaVerbTableInit ();
  TglYBoardMiscInit ();
  TglYBoardTbtInit ();
  TglYBoardFunctionInit ();
  TglYBoardSecurityInit ();
  TglYBoardPmcPdInit ();
  TglYBoardTypeCPortMapping ();
  GpioInit (PcdGetPtr (PcdBoardGpioTable));
  ///
  /// Do Late PCH init
  ///
  LateSiliconInit ();

  return EFI_SUCCESS;
}

VOID
TglYBoardSpecificGpioInitPostMem (
  VOID
  )
{
  UINT16    BoardId;
  BoardId = PcdGet16(PcdBoardId);

  PcdSet64S (PcdXhciAcpiTableSignature, SIGNATURE_64 ('x', 'h', '_', 't', 'y', 'd', 'd', '4'));

  //
  //Modify Preferred_PM_Profile field based on Board SKU's. Default is set to Mobile
  //
  PcdSet8S (PcdPreferredPmProfile, EFI_ACPI_2_0_PM_PROFILE_MOBILE);
  if (PcdGet8 (PcdPlatformFlavor) == FlavorUpServer) {
    PcdSet8S (PcdPreferredPmProfile, EFI_ACPI_2_0_PM_PROFILE_ENTERPRISE_SERVER);
  }

  //
  // Assign FingerPrint, Gnss, Bluetooth & TouchPanel relative GPIO.
  //
  switch (BoardId) {
    case BoardIdTglYLp4Type4Crb:
    case BoardIdTglYLp4Type4:
    case BoardIdTglYExsLp4Type4:
    case BoardIdTglYLp5Type4:
      PcdSet32S (PcdFingerPrintIrqGpio,         GPIO_VER2_LP_GPP_E3);  // Finger Print Sensor Interrupt Pin
      PcdSet32S (PcdTouchpanelIrqGpio,          GPIO_VER2_LP_GPP_E17); // Touch Interrupt Pin
      PcdSet32S (PcdTouchpadIrqGpio,            GPIO_VER2_LP_GPP_C8);  // Force (Touch) Pad Interrupt Pin
      PcdSet32S (PcdHdaI2sCodecIrqGpio,         GPIO_VER2_LP_GPP_C12); // Audio I2S Codec IRQ Pin
      PcdSet8S  (PcdHdaI2sCodecI2cBusNumber,    0);                    // Audio I2S Codec conntected to I2C0
      PcdSet32S (PcdWwanModemBaseBandResetGpio, GPIO_VER2_LP_GPP_C10); // WWAN/Modem Base Band Reset pin
      PcdSet8S  (PcdBleUsbPortNumber,           2);                    // Bluetooth USB Port
      //
      // Configure GPIOs for BT modules - UART or USB
      //
      PcdSet32S (PcdBtRfKillGpio, GPIO_VER2_LP_GPP_A13); // Bluetooth RF-KILL# pin
      if (PcdGet8 (PcdDiscreteBtModule) == 2) {          // Only for BT Over UART Selection
        PcdSet32S (PcdBtIrqGpio, GPIO_VER2_LP_GPP_H19);  // Bluetooth IRQ Pin
      }
      break;
    case BoardIdTglYLp4Aep:
      PcdSet32S (PcdFingerPrintIrqGpio,         GPIO_VER2_LP_GPP_E3);  // Finger Print Sensor Interrupt Pin
      PcdSet32S (PcdTouchpanelIrqGpio,          GPIO_VER2_LP_GPP_E17); // Touch Interrupt Pin
      PcdSet32S (PcdTouchpadIrqGpio,            GPIO_VER2_LP_GPP_C8);  // Force Pad Interrupt Pin
      PcdSet32S (PcdHdaI2sCodecIrqGpio,         0);                    // Audio I2S Codec IRQ Pin
      PcdSet8S  (PcdHdaI2sCodecI2cBusNumber,    0);                    // Audio I2S Codec conntected to I2C0
      PcdSet32S (PcdWwanModemBaseBandResetGpio, GPIO_VER2_LP_GPP_C10); // WWAN/Modem Base Band Reset pin
      PcdSet8S  (PcdBleUsbPortNumber,           2);                    // Bluetooth USB Port
      //
      // Configure GPIOs for BT modules - UART or USB
      //
      PcdSet32S (PcdBtRfKillGpio, GPIO_VER2_LP_GPP_A13); // Bluetooth RF-KILL# pin
      break;
  }

  PcdSet64S (PcdBoardRtd3TableSignature, SIGNATURE_64 ('T', 'g', 'l', 'Y', '_', 'R', 'v', 'p'));
}

VOID
TglYInitCommonPlatformPcdPostMem (
  VOID
  )
{
  UINT16                          BoardId;
  BoardId = PcdGet16(PcdBoardId);

  PcdSet32S (PcdEcSmiGpio, GPIO_VER2_LP_GPP_E7);
  PcdSet32S (PcdEcLowPowerExitGpio,GPIO_VER2_LP_GPP_E8);
  PcdSetBoolS (PcdPssReadSN, TRUE);
  PcdSet8S (PcdPssI2cSlaveAddress, 0x6E);
  PcdSet8S (PcdPssI2cBusNumber, 0x05);
  PcdSetBoolS (PcdSpdAddressOverride, FALSE);

  //
  // Battery Present
  // Real & Virtual battery is need to supported in all except Desktop
  //
  PcdSet8S (PcdBatteryPresent, BOARD_REAL_BATTERY_SUPPORTED | BOARD_VIRTUAL_BATTERY_SUPPORTED);
  //
  // Real Battery 1 Control & Real Battery 2 Control
  //
  PcdSet8S (PcdRealBattery1Control, 1);
  PcdSet8S (PcdRealBattery2Control, 2);

  //
  // Sky Camera Sensor
  //
  PcdSetBoolS (PcdMipiCamSensor, FALSE);

  //
  // H8S2113 SIO
  //
  PcdSetBoolS (PcdH8S2113SIO, FALSE);

  //
  // NCT6776F COM, SIO & HWMON
  //
  PcdSetBoolS (PcdNCT6776FCOM, FALSE);
  PcdSetBoolS (PcdNCT6776FSIO, FALSE);
  PcdSetBoolS (PcdNCT6776FHWMON, FALSE);
  //
  // ZPODD
  //
  PcdSet8S (PcdZPoddConfig, 0);
  //
  // SMC Runtime Sci Pin
  // EC will use eSpi interface to generate SCI
  //
  PcdSet32S (PcdSmcRuntimeSciPin, 0x00);
  //
  // Convertable Dock Support
  // Not supported only for S & H SKU's
  PcdSetBoolS (PcdConvertableDockSupport, TRUE);
  //
  // Ec Hotkey F3, F4, F5, F6, F7 and F8 Support
  //
  PcdSet8S (PcdEcHotKeyF3Support, TRUE);
  PcdSet8S (PcdEcHotKeyF4Support, TRUE);
  PcdSet8S (PcdEcHotKeyF5Support, TRUE);
  PcdSet8S (PcdEcHotKeyF6Support, TRUE);
  PcdSet8S (PcdEcHotKeyF7Support, TRUE);
  PcdSet8S (PcdEcHotKeyF8Support, TRUE);

  //
  // Virtual Button Volume Up & Done Support
  // Virtual Button Home Button Support
  // Virtual Button Rotation Lock Support
  //
  PcdSetBoolS (PcdVirtualButtonVolumeUpSupport, TRUE);
  PcdSetBoolS (PcdVirtualButtonVolumeDownSupport, TRUE);
  PcdSetBoolS (PcdVirtualButtonHomeButtonSupport, TRUE);
  PcdSetBoolS (PcdVirtualButtonRotationLockSupport, TRUE);
  //
  // Slate Mode Switch Support
  //
  PcdSetBoolS (PcdSlateModeSwitchSupport, TRUE);
  //
  // Virtual Gpio Button Support
  //
  PcdSetBoolS (PcdVirtualGpioButtonSupport, TRUE);
  //
  // Ac Dc Auto Switch Support
  //
  PcdSetBoolS (PcdAcDcAutoSwitchSupport, TRUE);

  //
  // Pm Power Button Gpio Pin
  //
  PcdSet32S (PcdPmPowerButtonGpioPin, GPIO_VER2_LP_GPD3);
  //
  // Acpi Enable All Button Support
  //
  PcdSetBoolS (PcdAcpiEnableAllButtonSupport, TRUE);
  //
  // Acpi Hid Driver Button Support
  //
  PcdSetBoolS (PcdAcpiHidDriverButtonSupport, TRUE);
}

/**
  Board init for PEI after Silicon initialized

  @retval  EFI_SUCCESS   Operation success.
**/
EFI_STATUS
EFIAPI
TglYBoardInitAfterSiliconInit (
  VOID
  )
{
  DEBUG ((DEBUG_INFO, "TglYBoardInitAfterSiliconInit \n"));
  TglYBoardSpecificGpioInitPostMem ();
  TglYInitCommonPlatformPcdPostMem ();

  return EFI_SUCCESS;
}
