# *************************************************************************
#                                                                        **
#                  I N T E L ( MWG ) P R O P R I E T A R Y          	 **
#                                                                        **
#     COPYRIGHT (c)  2012 BY  INTEL  CORPORATION.  ALL RIGHTS            **
#     RESERVED.   NO  PART  OF THIS PROGRAM  OR  PUBLICATION  MAY        **
#     BE  REPRODUCED,   TRANSMITTED,   TRANSCRIBED,   STORED  IN  A      **
#     RETRIEVAL SYSTEM, OR TRANSLATED INTO ANY LANGUAGE OR COMPUTER      **
#     LANGUAGE IN ANY FORM OR BY ANY MEANS, ELECTRONIC, MECHANICAL,      **
#     MAGNETIC,  OPTICAL,  CHEMICAL, MANUAL, OR OTHERWISE,  WITHOUT      **
#     THE PRIOR WRITTEN PERMISSION OF INTEL  CORPORATION.                **
#                                                                        **
# *************************************************************************

# *************************************************************************
#  File: common.ps1
# 
#  Description: Script common to start, stop tracing logs
# 
# *************************************************************************
$global:UartHardwareId = "*INT33E*"
$global:UsbHardwareIds = @("*PID_07DA*", "*PID_07DC*", "*PID_0A2A*", "*PID_0A2B*","*PID_0A5A*", "*PID_0A5B*", "*PID_0AA7*", "*PID_0025*", "*PID_0AAA*")
$global:EhciControllers = @("*1E26*","*9C26*")
$global:XhciControllers = @("*9C31*","*A12F*","*9CB1*")

$global:IntelTraceOptions = @("audio", "ibtbpl", "ibtver", "ibtusb", "ibtuart", "ixbtusb", "ibtsiva", "usbbl", "iusb", "lpssuart", "uart", "ibtfudrv", "usb3win7", "ibtusc")
$global:MsftTraceOptions = @("msft", "msusb", "msuart", "hid", "hci", "acpi", "sercx2")
$global:PreDefinedTraceOptions = @("sx", "yb", "bsod", "bluetooth", "moto")
$global:MiscTraceOptions = @("boot", "firmware", "install")
$global:DefaultTraces = @("ibtusb", "ibtuart")
$global:CmdTraceValues = $global:IntelTraceOptions + $global:MsftTraceOptions + $global:PreDefinedTraceOptions + $global:MiscTraceOptions
$global:Win7SpecificTraces = @("iusb", "usb3win7", "moto")

$global:IntelAdapterName = "Not Found"
$global:MotAdapterName = "Not Found"
$global:UserName = [Environment]::UserName
$global:DefaultLogFolderPath = [Environment]::GetFolderPath("Desktop") + "\Intel_Bluetooth_Traces"
$global:LogFilePath = ""
$global:SessionState = "Enabled"
$global:ComputerName = (Get-Item env:\Computername).Value
$global:DefaultDriverVersion = "0.0.0.0"
$global:DriverVersion = $global:DefaultDriverVersion
$global:IntelBTStack = $false
$global:MotorolaBTStack = $false
$global:IntelAudioStack = $false
$global:IXTraceStarted = $false
$global:HostController = ""
$global:OSVersion = ""
$global:XHCI = $false
$global:EHCI = $false
$global:USB = $false
$global:OptionXml = $null
$global:OptionsFile = "Options.xml"
$global:SummaryFile = "Summary.log"
$global:LinkKeyFile = "LinkKeys.log"
$global:format = ""
$global:content = ""
$global:OTPVer = ""
$global:DeviceId = ""
$global:HardwareId = ""
$global:BtInfName = ""
$global:LogPostFix = ""
$global:Status = ""
$global:HCIswitch1 = "-c"
$global:HCIcommand = "7cfc25"
$global:temp = ""
$global:OSVersion = ""
$global:OSName = ""
$global:GenericDevice = $false
$global:YbOrSxTracesWin7 = @("ibtusb","usb3win7")
$global:BluetoothTraces = @("ibtusb")
##### The default value for the file size is 128MB
$global:DefaultLogFileSize= 128
##### The default value for the Buffer size is 512KB
$global:DefaultMaxBufferSize= 512
#### Following are the default values in KB for the min/max number of Event Trace Session buffers.
$global:DefaultEventTraceSessionBuffersMin= 128
$global:DefaultEventTraceSessionBuffersMax= 640
$global:IsCircularMode = $true;

# FW over bulk tracing
$global:TraceMap = New-Object -TypeName byte[] -ArgumentList 32
$global:ProvideriBtFw = ".\providers\intel\ibtfw.guid"
$global:LogFileNameiBtFw = "ibtfw.etl"
[System.Byte[]] $global:TraceMapDisable = 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
[System.Byte[]] $global:TraceMapDefault = 0xFE,0x83,0x02,0x80,0x04,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
[System.Byte[]] $global:TraceMapCoex = 0xFE,0x81,0x00,0x00,0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00
[System.Byte[]] $global:TraceMapProfile = 0xFE,0x81,0x00,0x00,0x04,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00
[System.Byte[]] $global:TraceMapSx = 0xFE,0x81,0x02,0x80,0x05,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00


#
# Tool version
# 
# Must update (3rd part) every time a fix is made
# 

$global:Version = "Version 2.1.13"

function IsWin7()
{
    $major = $global:OSVersion.Major
    $minor = $global:OSVersion.Minor

    if ($major -eq 6 -and $minor -eq 1) 
    {
        return $true
    }

    return $false
}

function IsWin8()
{
    $major = $global:OSVersion.Major
    $minor = $global:OSVersion.Minor

    if ($major -eq 6 -and $minor -eq 2) 
    {
        return $true
    }

    return $false
}

function IsWin81()
{
    $major = $global:OSVersion.Major
    $minor = $global:OSVersion.Minor

    if ($major -eq 6 -and $minor -eq 3) 
    {
        return $true
    }

    return $false
}

function IsWin10()
{
    $major = $global:OSVersion.Major
    $minor = $global:OSVersion.Minor

    if ($major -eq 10 -and $minor -eq 0) 
    {
        return $true
    }

    return $false
}

function IsRS1()
{
    $major = $global:OSVersion.Major
    $minor = $global:OSVersion.Minor
    $build = $global:OSVersion.Build

    if ($major -eq 10 -and $minor -eq 0 -and $build -le 14393) 
    {
        return $true
    }

    return $false
}

function IsRS2()
{
    $major = $global:OSVersion.Major
    $minor = $global:OSVersion.Minor
    $build = $global:OSVersion.Build

    if ($major -eq 10 -and $minor -eq 0 -and $build -gt 14393 -and $build -lt 16170) 
    {
        return $true
    }

    return $false
}

function IsRS3()
{
    $major = $global:OSVersion.Major
    $minor = $global:OSVersion.Minor
    $build = $global:OSVersion.Build
    $rev   = $global:OSVersion.Revision

    if ($major -eq 10 -and $minor -eq 0 -and $build -gt 16170 -and $build -lt 17017) 
    {
        return $true
    }

    return $false
}

function IsTh1()
{
    $major = $global:OSVersion.Major
    $minor = $global:OSVersion.Minor
    $build = $global:OSVersion.Build

    if ($major -eq 10 -and $minor -eq 0 -and $build -le 10240) 
    {
        return $true
    }

    return $false
}

function IsTh2()
{
    $major = $global:OSVersion.Major
    $minor = $global:OSVersion.Minor
    $build = $global:OSVersion.Build
    $rev   = $global:OSVersion.Revision

    if ($major -eq 10 -and $minor -eq 0 -and $build -le 10586) 
    {
        return $true
    }

    return $false
}


function GetOSVersion() 
{
    $global:OSVersion = [Environment]::OSVersion.version

    $major = $global:OSVersion.Major
    $minor = $global:OSVersion.Minor
    $build = $global:OSVersion.Build
    $rev   = $global:OSVersion.Revision
	 
	if ($major -eq 10 -and $minor -eq 0 -and $build -le 14393)
    {
        $global:OSName = "Win10 RS1"
    }
	
    elseif ($major -eq 10 -and $minor -eq 0 -and $build -gt 14393 -and $build -lt 16170)  
    {
        $global:OSName = "Win10 RS2"
    }
	
    elseif($major -eq 10 -and $minor -eq 0 -and $build -gt 16170 -and $build -lt 17017)  
    {
        $global:OSName = "Win10 RS3"
    }
	
	elseif($major -eq 10 -and $minor -eq 0 -and $build -ge 17017 -and $build -lt 17600)  
    {
        $global:OSName = "Win10 RS4"
    }

    elseif($major -eq 10 -and $minor -eq 0 -and $build -ge 17600 -and $build -lt 17800)  
    {
        $global:OSName = "Win10 RS5"
    }
	
	elseif ($major -eq 10 -and $minor -eq 0 -and $build -le 10240)
    {
        $global:OSName = "Win10 TH1"
    }
	
	elseif ($major -eq 10 -and $minor -eq 0 -and $build -le 10586)
    {
        $global:OSName = "Win10 TH2"
    }
	
    elseif ($major -eq 6 -and $minor -eq 1) 
    {
        $global:OSName = "Win7"
    }
	
    elseif ($major -eq 6 -and $minor -eq 2) 
    {
        $global:OSName = "Win8"
    }
	
    elseif ($major -eq 6 -and $minor -eq 3) 
    {
        $global:OSName = "Win8.1"
    }
	
    elseif ($major -eq 10 -and $minor -eq 0) 
    {
        $global:OSName = "Win10"
    }
	
	$OS    = "OS version: $global:OSName, Minor: $minor, Major: $major, Build: $build, Revision: $rev"
	return $OS
}

function ProbeHostController($Hid)
{
    if ($Hid -match "PID_(?<pid>.*)\\") 
    {
        $product = "*" + $matches['pid'] + "*"
        
        $Controller = gwmi win32_usbcontrollerdevice | where-object { $_.Dependent -like $product }
        
        $USBController = gwmi win32_USBController

        # If there are multiple controllers and device IDs, DeviceID will be null
        # Therefore, we will grab the DeviceID from each of the individual controllers
        # and test whether or not they match.
        if ($USBController -ne $null -and $USBController.DeviceID -eq $null)
        {
            foreach ($obj in $USBController)
            {
                $USBControllerDevID = $obj.DeviceID.split("\\")
                $USBControllerHWID = $USBControllerDevID[1]
                if ($Controller.Antecedent -like "*$USBControllerHWID*")
                {
                    $USBController = $obj
                    break
                }
            }
        }
        # Otherwise, there is only one controller and one DeviceID
        else
        {
            $USBControllerDevID = $USBController.DeviceID.split("\\")
            $USBControllerHWID = $USBControllerDevID[1]
        }
        
        # At first, we check the USBController 
        if ($Controller.Antecedent -like "*$USBControllerHWID*") 
        {
            if ($USBController.Description -like "*xhci*" -or $USBController.Description -like "*eXtensible*")
            {
                LogMessage("Host controller is XHCI")
                $global:XHCI = $true
            }
            elseif ($USBController.Description -like "*ehci*" -or $USBController.Description -like "*enhanced*")
            {
                LogMessage("Host controller is EHCI")
                $global:EHCI = $true
            }	
            else {
                LogMessage("Cannot determine the type of controller. Use XHCI for default")
                $global:XHCI = $true
            }
        }
        else 
        {
            # Use brute force way to detect.
            foreach ($cid in $global:XhciControllers)
            {
                write-host "CID XHCI: $cid"
                if ($Controller.Antecedent -like $cid) 
                {
                    LogMessage("Host controller is XHCI")
                    $global:XHCI = $true
                    break
                }
            }
            
            if (!$global:XHCI) 
            {
                foreach ($cid in $global:EhciControllers)
                {
                    if ($Controller.Antecedent -like $cid) 
                    {
                        $global:EHCI = $true
                        LogMessage("Host controller is EHCI")
                        break  
                    }
                }
            }
        }
    }
    # if none of this works, assume XHCI
    if ($global:XHCI -eq $false -and $global:EHCI -eq $false)
    {
        LogMessage("Assuming Host controller is XHCI")
        $global:XHCI = $true
    }
}

function ProbeHardwareAndSoftware()
{
    write-host "Probing hardware & Software..."
    
    $bt = $null
    $wifi = $null

    #
    #   This query will return active devices only (e.g. if Bluetooth chips were swapped, then this query will return current Bluetooth device)
    #   I.e. there will be no more than one Bluetooth adapter in the list
    #
    $devices = Get-WmiObject win32_pnpsigneddriver -namespace root\cimv2 | Where-Object {$_.DeviceClass -eq "NET" -or $_.DeviceClass -like "*BLUETOOTH*" }

    foreach ($device in $devices) 
    {
        # 
        #   We are not interested in device nodes on BTH\* and BTHENUM\* buses
        #
        if ($device.DeviceClass -like "*BLUETOOTH*" -or $device.DeviceId -like "*BTH*") 
        {
           $bt = $device
        }

        if ($device.DeviceClass -eq "NET" -and $device.Manufacturer -eq "Intel Corporation" -and $device.DeviceName -like "*Wireless*") 
        {
            $wifi = $device
        }
    }
      
    if ($bt)
    {
        $global:DeviceId = $bt.DeviceID
        $global:HardwareId = $bt.HardwareID
        $global:PnPDescription =  $bt.Description
        $global:DriverVersion = $bt.DriverVersion
        $global:BtInfName = $bt.InfName
		
		# BCPS-4208
		# Originally, ibttracetool logs iBTSerialBus\UART_H4 in Summary.log. This caused an issue with decoding using ibt-parse
		# as it could not find a way to map to the right SKU. We have modified the code to log the HW ID of UART bus enumerator to get the HW ID that can be mapped
		# to a specific SKU.
		if($global:HardwareId -like "*iBTSerialBus\UART*") 
        {
			$uart_bus_enumerator = gwmi win32_pnpsigneddriver -namespace root\cimv2 | Where-Object {$_.DeviceID -like $global:UartHardwareId}
			if($uart_bus_enumerator -ne $null) 
            {
                $uart_bus_enumerator_hw_id = $uart_bus_enumerator.HardwareID
                $global:BtInfName = $uart_bus_enumerator.InfName
                LogMessage("Found $global:PnPDescription($uart_bus_enumerator_hw_id)")
            } 
		} else 
        {
            LogMessage("Found $global:PnPDescription($global:HardwareId)")
            LogMessage("USB location: " + $bt.Location)        
        }
            
        if ($wifi)
        {
            $wifiDriver = $wifi.DriverVersion
            $wifiName = $wifi.Description
            LogMessage("Found " + $wifiName + " v" + $wifiDriver)
        }

        try
        {
            $infContent = Get-Content $([io.path]::combine($env:SystemRoot, 'inf', $global:BtInfName)) -ReadCount 0
        }
        catch
        {
            #   Do nothing
        }

        if ($infContent -match 'btmhsf.sys')
        {
            $global:MotorolaBTStack = $true  

            LogMessage("Found Intel/Motorola Bluetooth stack (" + $bt.DriverVersion + ")")
        }
        elseif ($bt.Manufacturer -like "Intel*")
        {
            $global:IntelBTStack = $true

            LogMessage("Found Intel Bluetooth Stack with Firmware update support (" + $bt.DriverVersion + ")")
        }
        elseif ($bt.Manufacturer -like "Microsoft*")
        {
            $global:GenericDevice = $true

            LogMessage("Found Generic Microsoft Stack (" + $bt.DriverVersion + ")")
            Write-Host-Warning "Intel logs might be empty in the end. Please verify that you have proper Bluetooth device and Bluetooth Stack installed."
        }
        else 
        {
            LogMessage("Found 3rd Party Stack (" + $bt.Manufacturer + ", Version " + $bt.DriverVersion + "). Intel logs might be empty in the end because of this.")

            # It is still OK to collect MS logs while 3rd party is operating, but no Intel logs will be collected.
            Write-Host-Warning "Intel logs might be empty in the end. Please verify that you have proper Bluetooth device and Bluetooth Stack installed."
        }
    }
    else 
    {
        LogMessage("Failed to detect any active Bluetooth Host stack")
        LogMessage("Bluetooth radio might be physically disconnected from computer (e.g. Hardware RF Kill was performed, Bluetooth was disabled in BIOS, etc.)")

        #
        #   Unfortunately, there is no built-in PowerShell/WMI way to check for disconnected devices.
        #   The following code might be updated if 3rd party modules with Setup API / Windows Installer API will be integrated
        #
        #   As for now, let's try to see if there is ibtusb/ibtuart in System32
        #

        Write-Host "Checking presence of ibt drivers in System32..."

        $ibtusb_driver = $null
        $ibtuart_driver = $null
        $ibtusb_path = Get-DriversPath("ibtusb.sys")
        $ibtuart_path = Get-DriversPath("ibtuart.sys")
        
        try { $ibtusb_driver = [System.Diagnostics.FileVersionInfo]::GetVersionInfo($ibtusb_path).FileVersion }
        catch {}

        try { $ibtuart_driver = [System.Diagnostics.FileVersionInfo]::GetVersionInfo($ibtuart_path).FileVersion }
        catch {}
        
        if ($ibtusb_driver -ne $null -and $ibtuart_driver -ne $null)
        {
            LogMessage ("Found ibtusb.sys (" + $ibtusb_driver + ") and ibtuart.sys (" + $ibtuart_driver + ")  in System32\Drivers")
            LogMessage ("ibtusb.sys version is selected for log file naming.")
            
            $global:DriverVersion = $ibtusb_driver
        }
        elseif ($ibtusb_driver -ne $null)
        {
            LogMessage ("Found ibtusb.sys (" + $ibtusb_driver + ") in System32\Drivers.")
            LogMessage ("Assume this driver will be used when Bluetooth will be enabled")

            $global:DriverVersion = $ibtusb_driver
        }
        elseif ($ibtuart_driver -ne $null)
        {
            LogMessage ("Found ibtuart.sys (" + $ibtuart_driver + ") in System32\Drivers.")
            LogMessage ("Assume this driver will be used when Bluetooth will be enabled")

            $global:DriverVersion = $ibtuart_driver
        }
        else
        {
            LogMessage ("Failed to detect any Intel Bluetooth drivers. Logs might be empty.").
            Write-Host-Warning "Failed to detect any Bluetooth on the system. Please verify if it is expected for this test case."

            $global:DriverVersion = $global:DefaultDriverVersion
        }  
    }
}

function UpdateSummaryFilePath()
{
    $temp = "$global:LogFilePath\$global:SummaryFile"
    $global:SummaryFile = $temp
}

function LogMessage($text)
{
    Add-Content $global:SummaryFile $text
    write-host $text
}

function StartMsUsbLogging()
{
    $OS_Win10 = IsWin10
    if ($OS_Win10 -eq $true)
    {
        logman create trace autosession\msusb_Boot -o "$LogFilePath\msusb_boot.etl" -pf .\providers\msft\msusb3.guid -ets -ft 00:30 -nb $DefaultEventTraceSessionBuffersMin $DefaultEventTraceSessionBuffersMax -max $logFileSize -a -ct system > $null
        logman create trace msusb -o "$LogFilePath\msusb.etl"  -pf .\providers\msft\msusb3.guid -ft 00:30 -f bincirc -nb $DefaultEventTraceSessionBuffersMin $DefaultEventTraceSessionBuffersMax -max $logFileSize -ets > $null
    }
}

function StartiBtFuDriverLogging()
{
    $DriverVer = $null
    
    try 
    {
		$fudrv = Get-DriversPath("ibtfudrv.sys")
        $DriverVer = [System.Diagnostics.FileVersionInfo]::GetVersionInfo($fudrv).FileVersion
    }
    catch
    {
        # Driver doesn't exist! 
    }
  
    # start tracing only if driver was found
    if (![string]::IsNullOrEmpty($DriverVer))
    {
        logman create trace ibtfudrv -o "$LogFilePath\ibtfudrv-$DriverVer-%d.etl" -pf .\providers\intel\ibtfudrv.guid -mode newfile -nb $DefaultEventTraceSessionBuffersMin $DefaultEventTraceSessionBuffersMax -bs $DefaultMaxBufferSize -max $logFileSize -ets > $null
        logman create trace autosession\ibtfudrv_Boot -o "$LogFilePath\ibtfudrv-$DriverVer_boot.etl" -pf .\providers\intel\ibtfudrv.guid -ets -nb $DefaultEventTraceSessionBuffersMin $DefaultEventTraceSessionBuffersMax -bs $DefaultMaxBufferSize -max $logFileSize -a -ct system > $null
    }
}


function StartiBtUsbBootloaderLogging()
{
    logman create trace ibtusbbl -o "$LogFilePath\ibtusbbl.etl" -pf .\providers\intel\ibtusbbl.guid -nb $DefaultEventTraceSessionBuffersMin $DefaultEventTraceSessionBuffersMax -bs $DefaultMaxBufferSize -max $logFileSize -ets > $null
}

function StopiBtUsbBootloaderLogging()
{
    logman stop ibtusbbl -ets > $null
}

function StartIXDriverLogging()
{
    if ($global:IXTraceStarted -eq $false) 
    {
        logman create trace ixbtusb -o "$LogFilePath\ixbt-$DriverVersion-%d.etl" -pf .\providers\intel\ixbtusb.guid -mode newfile -nb $DefaultEventTraceSessionBuffersMin $DefaultEventTraceSessionBuffersMax -bs $DefaultMaxBufferSize -max $logFileSize -ets > $null
		logman create trace autosession\ixbtusb_Boot -o "$LogFilePath\ixbt-$DriverVersion.etl" -pf .\providers\intel\ixbtusb.guid -ets -nb $DefaultEventTraceSessionBuffersMin $DefaultEventTraceSessionBuffersMax -bs $DefaultMaxBufferSize -max $logFileSize -ct system > $null
        $global:IXTraceStarted = $true
    }
}


function StartUsb3Win7Logging()
{
    $env:SESSION_NAME='USB3WIN7'
    $env:PROVIDERS=".\providers\usb3win7\guids.ctl"

    .\tools\tracelog.exe -stop $env:SESSION_NAME  > $null
    .\tools\tracelog.exe -start $env:SESSION_NAME -ls -guid $env:PROVIDERS -flag 0X00FFFFFF -b 128 -min 1024 -cir 1000 -guid $env:PROVIDERS -f "$LogFilePath\usb3win7.etl" -level 0x9 > $null
}


function StopUsb3Win7Logging()
{
    $env:SESSION_NAME='USB3WIN7'
    .\tools\tracelog.exe  -stop $env:SESSION_NAME > $null 
    .\tools\tracelog -remove $env:SESSION_NAME > $null
}


function SetRegKeys()
{

	$ModuleNames = @("BTHUSB", "BTHPORT", "BTHENUM", "BTHSERV", "BTHLEENUM", "BTHMINI", "HIDBTH", "BTHUART", "BTHPRINT", "BTHMODEM", "RFCOMM", "BTHA2DP", "BTHHFENUM", "BTHHFAUD", "BTHHFHID")
	
	foreach ($Module in $ModuleNames)
	{
		$ModulePath = "HKLM:\System\CurrentControlSet\Services\$Module\Parameters"
	
		if (!(Test-path $ModulePath))
		{
			New-Item -path $ModulePath -force | Out-Null
			New-ItemProperty -path $ModulePath -name "VerboseOn" -value "0x1" -PropertyType DWORD -force | Out-Null
			New-ItemProperty -path $ModulePath -name "LogPages" -value "0xFF" -PropertyType DWORD -force | Out-Null
		}
		
		else
		{
			New-ItemProperty -path $ModulePath -name "VerboseOn" -value "0x1" -PropertyType DWORD -force | Out-Null
			New-ItemProperty -path $ModulePath -name "LogPages" -value "0xFF" -PropertyType DWORD -force | Out-Null		
		} 
	}
}


#
# StartMotorolaCommonLogging 
#
# Main entry point to start logging for Motorola components
#
function StartMotorolaCommonLogging () 
{   
	logman create trace autosession\BTMPP_BOOT_Trace -o "$LogFilePath\moto-log-boot.etl" -pf ".\providers\motorola\providers-common.guid" -mode "EVENT_TRACE_ADDTO_TRIAGE_DUMP | EVENT_TRACE_NO_PER_PROCESSOR_BUFFERING | EVENT_TRACE_FILE_MODE_APPEND" -nb $DefaultEventTraceSessionBuffersMin $DefaultEventTraceSessionBuffersMax -bs $DefaultMaxBufferSize -max $logFileSize -ets -ct system -ft 1 > $null
	logman create trace BTMPP_Common_KM_Trace  -o "$LogFilePath\log_km_common_%d.etl" -pf ".\providers\motorola\providers-drivers.guid" -mode "EVENT_TRACE_ADDTO_TRIAGE_DUMP | EVENT_TRACE_NO_PER_PROCESSOR_BUFFERING |EVENT_TRACE_FILE_MODE_NEWFILE" -nb $DefaultEventTraceSessionBuffersMin $DefaultEventTraceSessionBuffersMax -bs $DefaultMaxBufferSize -max $logFileSize -ets -ct system -ft 1 > $null
    logman create trace BTMPP_Common_UM_Trace  -o "$LogFilePath\log_um_common_%d.etl" -pf ".\providers\motorola\providers-app.guid" -mode "EVENT_TRACE_ADDTO_TRIAGE_DUMP | EVENT_TRACE_NO_PER_PROCESSOR_BUFFERING |EVENT_TRACE_FILE_MODE_NEWFILE" -nb $DefaultEventTraceSessionBuffersMin $DefaultEventTraceSessionBuffersMax -bs $DefaultMaxBufferSize -max $logFileSize -ets -ct system -ft 1 > $null   
}


#
# StopMotorolaCommonLogging 
#
# Main entry point to stop logging for Motorola components
#
function StopMotorolaCommonLogging ()
{
    logman stop autosession\BTMPP_BOOT_Trace -ets > $null
    logman delete autosession\BTMPP_BOOT_Trace -ets > $null

    logman stop BTMPP_Common_UM_Trace -ets > $null
    logman delete BTMPP_Common_UM_Trace -ets > $null
        
	logman stop BTMPP_Common_KM_Trace -ets > $null
    logman delete BTMPP_Common_KM_Trace -ets > $null
}


function StartDriverLogging([string]$Option)
{
	$Guid = ""

	if ($global:IntelTraceOptions -contains $Option)
	{

		$Guid = ".\providers\intel\$Option.guid"

	}
	
	elseif ($global:MsftTraceOptions -contains $Option)
	{
		if ($Option -like "msft")
		{
			$Guid = ".\providers\msft\msft.guid"
			SetRegKeys
		}
		else{
			$Guid = ".\providers\msft\$Option.guid"
		}
	}
	
	
	if (![string]::IsNullOrEmpty($Guid))
	{
		$OptionCollectorName = $Option
		$BootCollectorName = $Option + "_Boot"

		logman stop $OptionCollectorName -ets > $null
		logman delete $OptionCollectorName -ets > $null
        logman stop $BootCollectorName -ets > $null
		logman delete autosession\$BootCollectorName -ets > $null
        if($global:IsCircularMode)
        {
            logman create trace $OptionCollectorName -o "$LogFilePath\$Option-$DriverVersion.etl" -pf $Guid -max $logFileSize -config .\providers\LogmanArgs.txt > $null
		    logman create trace autosession\$BootCollectorName -o "$LogFilePath\$Option-$DriverVersion-boot.etl"  -pf $Guid -max $logFileSize -config .\providers\LogmanArgsBoot.txt > $null
        }
        else
        {
            logman create trace $OptionCollectorName -o "$LogFilePath\$Option-$DriverVersion.etl" -pf $Guid -config .\providers\LogmanArgsNonCirc.txt > $null
		    logman create trace autosession\$BootCollectorName -o "$LogFilePath\$Option-$DriverVersion-boot.etl"  -pf $Guid -config .\providers\LogmanArgsBootNonCirc.txt > $null
        }
	}
}


function StopDriverLogging($Option)
{
	$OptionCollectorName = $Option
	$BootCollectorName = $Option + "_Boot"
	
	logman stop $OptionCollectorName -ets > $null
    logman delete $OptionCollectorName -ets > $null
	logman stop $BootCollectorName -ets > $null
    logman delete autosession\$BootCollectorName -ets > $null
}


function BTLogSystemInfo($LogFilePath) 
{
    Systeminfo | out-file "$LogFilePath\systeminfo.txt"
}

function BTLogTraceToolVersionInfo()
{
   $global:version > "$LogFilePath\Summary.txt"
}

function BTGetInstalledSoftwareInfo($LogFilePath) 
{
   gwmi -class win32_product | Where-Object {$_.Vendor -like "*Intel*"} | out-file "$LogFilePath\SoftwareInfo.txt"
}

function CreateSessionFolder($UserPath) 
{
    $date = Get-Date -Format u
    $date = $date -replace ' ','-'
    $date = $date -replace '\/','-'
    $date = $date -replace ':','-'

    $LogFolder = "$env:COMPUTERNAME-$date"

    $LocalPath = "$UserPath\$LogFolder"
    New-Item -type Directory -Force -path $LocalPath

    return [string]$LocalPath.Trim()
}

function EnableFirmwareTrace($Option)
{
	$found = $FALSE
	
	if ($Option -like "*coex*") {
		$found = $TRUE
		TraceMapAppend($TraceMapCoex)
	}
	
	if ($Option -like "*profile*") {
		$found = $TRUE
		TraceMapAppend($TraceMapProfile)
	}

	if ($Option -like "*sx*") {
		$found = $TRUE
		TraceMapAppend($TraceMapSx)
	}

	if (!$found) {
		TraceMapAppend($TraceMapDefault)
	}

	$traceMapStr = [System.BitConverter]::ToString($TraceMap).replace('-','')
	$traceMapStr = "7CFC250180000100" + $traceMapStr
	.\tools\tracelog.exe -start iBtFw -guid $global:ProvideriBtFw -f "$global:LogFilePath\$global:LogFileNameiBtFw" > $null
	.\tools\hcitool.exe -c $traceMapStr > $null
}

function DisableFirmwareTrace($Option)
{
	$traceMapStr = [System.BitConverter]::ToString($TraceMapDisable).replace('-','')
	$traceMapStr = "7CFC250180000100" + $traceMapStr
	.\tools\hcitool.exe -c $traceMapStr > $null
	.\tools\tracelog.exe -stop iBtFw > $null
}

function ReadOtpVersion($filePath)
{
    # Test to make sure hcitool exists
    if (Test-Path .\tools\hcitool.exe)
    {
        $temp = .\tools\hcitool.exe -c 05fc00
        if($temp -ne $null)
        {
            $deviceType = $temp.Substring(14, 2)
            $deviceType = [Convert]::ToInt32("$deviceType", 16)

            # For RAM Bluetooth modules
            if($deviceType -ge [int]11)
            {
                $temp1 = .\tools\hcitool.exe -c 8CFC027501
                $temp2 = .\tools\hcitool.exe -c 8CFC027601

                # Get the OTP format version
                $global:format = $temp1.Substring(16)
                
                # Get the OTP content version
                $global:content =  $temp2.Substring(16)

                # Determine the LnPSfP OTP version
                if(($format -eq "0B") -and ($content -eq "05"))
                {
                    $global:OTPVer = "0x0A50"
                }
                elseif(($format -eq "0C") -and ($content -eq "01"))
                {
                    $global:OTPVer = "0x0A52"
                }
                elseif(($format -eq "0C") -and ($content -eq "05"))
                {
                    $global:OTPVer = "0x0A55"
                }
                elseif(($format -eq "0C") -and ($content -eq "07"))
                {
                    $global:OTPVer = "0x0A56"
                }
                else
                {
                    $global:OTPVer = "Unknown"
                }

                # Determine whether to print to file or print to console
                if(-Not [string]::IsNullOrEmpty($filePath))
                {
                    $file = New-Item -type file $filePath -Force
                    if(Test-Path $filePath)
                    {
                        Add-Content $filePath "$global:PnPDescription($global:HardwareId)"
                        Add-Content $filePath "`n$global:format (format), $global:content (content), OTP Version $global:OTPVer"
                        return
                    }				
                }
                Write-Host "`n$global:PnPDescription($global:HardwareId)"
                Write-Host "$global:format (format), $global:content (content), OTP Version $global:OTPVer"
            }
            # For ROM Bluetooth modules
            else
            {
                $temp1 = .\tools\hcitool.exe -c 11FC020100 # Enable Manufacturer Mode
                $temp2 = .\tools\hcitool.exe -c 9AFC0101

                # Get the OTP format version
                $global:format = $temp2.Substring(16, 2)

                # Get the OTP content version
                $global:content = $temp2.Substring(18, 2)
                $temp1 = .\tools\hcitool.exe -c 11FC020000 # Disable Manufacturer Mode

                # Determine whether to print to file or print to console
                if(-Not [string]::IsNullOrEmpty($filePath))
                {
                    $file = New-Item -type file $filePath -Force
                    if(Test-Path $filePath)
                    {
                        Add-Content $filePath "$global:PnPDescription($global:HardwareId)"
                        Add-Content $filePath "`n$global:format (format), $global:content (content)"
                        return
                    }
                }
                Write-Host "`n$global:PnPDescription($global:HardwareId)"
                Write-Host "$global:format (format), $global:content (content)"
            }
        }
        else
        {
            Write-Host "`nEither failed to read OTP version or the Bluetooth device is not recognized!"
        }
    }
    else
    {
        Write-Error "Error: hcitool.exe path not found"
    }
}

function SetHCICommand($Option)
{
    # Set Enable/Disable Tracing
    $temp = "{0:x2}" -f [int]$Status
    $HCIcommand = $HCIcommand + $temp

    # Set Output Interface
    #	0001H - VS-HCI TL Event Queue (Pushed through HCI_Intel_FW_Trace_BinaryEvent)
    #	0002H - Memory Storage (Pulled through HCI_Intel_FW_Trace_Read_BufferCommand)
    #	0004H - Emulated I2C (for future use)
    #	0008H - Debug UART FIFO (for future use)
    #	0010H - AHB Debug Trace
    if ($Option -like "*oi=mem_store*")
    {
        $temp = "{0:x2}" -f 2
        $HCIcommand = $HCIcommand + $temp
    }
    ElseIf ($Option -like "*oi=i2c*")
    {
        $temp = "{0:x2}" -f 4
        $HCIcommand = $HCIcommand + $temp
    }
    ElseIf ($Option -like "*oi=uart*")
    {
        $temp = "{0:x2}" -f 8
        $HCIcommand = $HCIcommand + $temp
    }
    ElseIf ($Option -like "*oi=ahb*")
    {
        $temp = "{0:x2}" -f 16
        $HCIcommand = $HCIcommand + $temp
    }
    else
    {
        $temp = "{0:x2}" -f 1
        $HCIcommand = $HCIcommand + $temp
    }

    # Set Timestamp
    if ($Option -like "*at=yes*")
    {
        $temp = "{0:x4}" -f 1
        $HCIcommand = $HCIcommand + $temp
    }
    else
    {
        $temp = "{0:x4}" -f 0
        $HCIcommand = $HCIcommand + $temp
    }

    # Set Overflow Mode
    if ($Option -like "*om=dc_new*")
    {
        $temp = "{0:x2}" -f 1
        $HCIcommand = $HCIcommand + $temp
    }
    else
    {
        $temp = "{0:x2}" -f 0
        $HCIcommand = $HCIcommand + $temp
    }

    # Set Range Values
    #	Byte[0:3] - TL/TLD Traces - 0x00 to 0x1F
    #	Byte[4:7] - HCI Traces - 0x20 to 0x3F
    #	Byte[8:11] - LM Traces - 0x40 to 0x5F
    #	Byte[12:15] - LEC Traces - 0x60 to 0x7F
    #	Byte[16:19] - LLC Traces - 0x80 to 0x9F
    #	Byte[20:23] - SP Traces - 0xA0 to 0xBF
    #	Byte[24:27] - LC Traces - 0xC0 to 0xDF
    #	Byte[28:31] - BC Traces - 0xE0 to 0xFF
    $trace_on = "{0:x8}" -f 4294967295
    $trace_off = "{0:x8}" -f 0
    # Set specified Trace Selection Patterns (can specify more than one)
    if($Option -like "*tsp=*")
    {
        # Set TL/TLD Trace Selection Pattern
        if($Option -like "*tsp=tl/tld*")
        {
            $HCIcommand = $HCIcommand + $trace_on
        }
        else
        {
            $HCIcommand = $HCIcommand + $trace_off
        }
        # Set HCI Trace Selection Pattern
        if($Option -like "*tsp=hci*")
        {
            $HCIcommand = $HCIcommand + $trace_on
        }
        else
        {
            $HCIcommand = $HCIcommand + $trace_off
        }
        # Set LM Trace Selection Pattern
        if($Option -like "*tsp=lm*")
        {
            $HCIcommand = $HCIcommand + $trace_on
        }
        else
        {
            $HCIcommand = $HCIcommand + $trace_off
        }
        # Set LEC Trace Selection Pattern
        if($Option -like "*tsp=lec*")
        {
            $HCIcommand = $HCIcommand + $trace_on
        }
        else
        {
            $HCIcommand = $HCIcommand + $trace_off
        }
        # Set LLC Trace Selection Pattern
        if($Option -like "*tsp=llc*")
        {
            $HCIcommand = $HCIcommand + $trace_on
        }
        else
        {
            $HCIcommand = $HCIcommand + $trace_off
        }
        # Set SP Trace Selection Pattern
        if($Option -like "*tsp=sp*")
        {
            $HCIcommand = $HCIcommand + $trace_on
        }
        else
        {
            $HCIcommand = $HCIcommand + $trace_off
        }
        # Set LC Trace Selection Pattern
        if($Option -like "*tsp=lc*")
        {
            $HCIcommand = $HCIcommand + $trace_on
        }
        else
        {
            $HCIcommand = $HCIcommand + $trace_off
        }
        # Set BC Trace Selection Pattern
        if($Option -like "*tsp=bc*")
        {
            $HCIcommand = $HCIcommand + $trace_on
        }
        else
        {
            $HCIcommand = $HCIcommand + $trace_off
        }
    }
    # Set default Trace Selection Pattern as HCI
    else
    {
        $HCIcommand = $HCIcommand + $trace_off + $trace_on + $trace_off + $trace_off + $trace_off + $trace_off + $trace_off + $trace_off
    }

    # Print HCI Command
	$message = "HCI command response: $HCIcommand"
	$firmwareFile = "$LogFilePath\firmware.txt"
	
    Write-Host $message
	Add-Content $firmwareFile $message

    # Execute the HCI Command
    if (Test-Path .\tools\hcitool.exe) 
    {
		& .\tools\hcitool.exe $HCIswitch1 $HCIcommand
    }
    else
    {
        Write-Error "Error: hcitool.exe path not found"
		Add-Content $firmwareFile $message
    }
}

function Get-RegistryValue($path, $name)
{
    $key = Get-Item -LiteralPath $path -ErrorAction SilentlyContinue

    if ($key) {
        $key.GetValue($name, $null)
    }
}

#
# This function uses runspaces for runnning tasks concurrently
# Bump the maxthreads every time a new task is added
# Order the tasks in the order of time they take to execute.
# The task that task longer must be on the top.
# Do not run a task concurrently if it takes few 100ms
#

function GetTriageLogs([string]$path) 
{
	
    $maxthreads = 7

    $rspool = [runspacefactory]::CreateRunspacePool(1, $maxthreads)
    
    $rspool.Open()

    $jobs = @()
    $waits = @()  

    #
    # Get system info
    #
    $GetSystemInfo = 
    {
        Param ($path)
        $name = [string]$path + "\systeminfo.txt" 
        $file = New-Item -type file $name -Force 
        Systeminfo | out-file $file
    }
	

    $pst = [powershell]::Create().AddScript($GetSystemInfo).AddArgument($path)
    $pst.RunspacePool = $rspool

    $handle = $pst.BeginInvoke()

    $job = "" | Select-Object Handle, Thread
    $job.Handle = $handle
    $job.Thread = $pst
    $jobs += $job
    $waits += $handle.AsyncWaitHandle  

    #
    # Get event logs
    #
    
    $evtpath = Get-SystemPath("winevt")
    $GetEventLogs = 
    {
        Param ($path, $evtpath)
        $dest = [string]$path + "\Event logs"
		$evtlogspath = $evtpath + "\Logs\*"
        New-Item $dest -type directory -Force > $null
        Copy-Item -Path $evtlogspath -Destination $dest -recurse -Force > $null
    }

    $pst = [powershell]::Create().AddScript($GetEventLogs).AddArgument($path).AddArgument($evtpath)
    $pst.RunspacePool = $rspool

    $handle = $pst.BeginInvoke()

    $job = "" | Select-Object Handle, Thread
    $job.Handle = $handle
    $job.Thread = $pst
    $jobs += $job
    $waits += $handle.AsyncWaitHandle 	
	
    #
    # Get setup logs from c:\windows\inf
    #
    $GetSetupLogs = 
    {
        Param ($path)
        $dest = [string]$path + "\Setup logs"
		$setuplogspath = $env:SystemRoot + "\inf\setup*.log"
        New-Item $dest -type directory -Force > $null
        gwmi Win32_BootConfiguration 
        copy-item -Path $setuplogspath -Destination $dest -recurse -Force > $null

        $msilogspath = $env:temp + "\iProInstLogs"

        If (Test-Path $msilogspath) {
            copy-item -Path $msilogspath -Destination $dest -recurse -Force  > $null
        }
		
		$dpinstlogpath = $env:SystemRoot + "\DPINST.log"

        If (Test-Path $dpinstlogpath) {
            copy-item -Path $dpinstlogpath -Destination $dest -recurse -Force  > $null
        }
    }

    $pst = [powershell]::Create().AddScript($GetSetupLogs).AddArgument($path)
    $pst.RunspacePool = $rspool

    $handle = $pst.BeginInvoke()

    $job = "" | Select-Object Handle, Thread
    $job.Handle = $handle
    $job.Thread = $pst
    $jobs += $job
    $waits += $handle.AsyncWaitHandle

    #
	# Get IFR key content
	#
	$name = [string]$path + "\ifr.log" 
	$key = 'HKLM:\SOFTWARE\Intel\Bluetooth'
	$logs = Get-RegistryValue $key "ifr" > $null$env
	if ($logs -ne $null)
	{
		$file = New-Item -type file $name -Force
		$logs | out-file $file
	}

    #
    # Get drivers folder content
    #
    $driversFolder = Get-SystemPath("\drivers\")
	

    $GetDriverFolderContent = 
    {
        Param ($path, $folder)

        $name = [string]$path + "\drivers.txt" 
		$content = Get-ChildItem $folder
		
        $file = New-Item -type file $name -Force
        $content | out-file $file
    }

    $pst = [powershell]::Create().AddScript($GetDriverFolderContent).AddArgument($path).AddArgument($driversFolder)
    $pst.RunspacePool = $rspool

    $handle = $pst.BeginInvoke()

    $job = "" | Select-Object Handle, Thread
    $job.Handle = $handle
    $job.Thread = $pst
    $jobs += $job
    $waits += $handle.AsyncWaitHandle  

    #
    # Get all third-party drivers
    #
    $GetPnputilContent = 
    {
        Param ($path)
        $name = [string]$path + "\Setup logs\pnputil.txt" 
		$file = New-Item -type file $name -Force
		
		pnputil.exe /e > $file
    }

    $pst = [powershell]::Create().AddScript($GetPnputilContent).AddArgument($path)
    $pst.RunspacePool = $rspool

    $handle = $pst.BeginInvoke()

    $job = "" | Select-Object Handle, Thread
    $job.Handle = $handle
    $job.Thread = $pst
    $jobs += $job
    $waits += $handle.AsyncWaitHandle
	
    #
    # Get installed software list
    #
    $GetInstalledSofware = 
    {
        Param ($path)
        $name = [string]$path + "\SoftwareInfo.txt" 
		
		try
		{
			$file = New-Item -type file $name -Force
		}
		
		catch 
		{
			# Do nothing 
		}
		
		$OS = [Version](Get-ItemProperty -Path "$($Env:Windir)\System32\hal.dll" -ErrorAction SilentlyContinue).VersionInfo.FileVersion.Split()[0]
		
		if ($OS.major -eq 10 -and $OS.minor -eq 0)
		{
			$KeyPath = "HKLM:\Software\WOW6432Node\Microsoft\Windows\CurrentVersion\Uninstall\*"
		}
		
		else 
		{
			$KeyPath = "HKLM:\Software\Microsoft\Windows\CurrentVersion\Uninstall\*"
		}
				
		Get-ItemProperty  $KeyPath |  Select-Object DisplayName, DisplayVersion, Publisher, InstallDate | 
	Format-Table -AutoSize | Out-String -Width 1024 | Out-File $file
	
    }

    $pst = [powershell]::Create().AddScript($GetInstalledSofware).AddArgument($path)
    $pst.RunspacePool = $rspool

    $handle = $pst.BeginInvoke()

    $job = "" | Select-Object Handle, Thread
    $job.Handle = $handle
    $job.Thread = $pst
    $jobs += $job
    $waits += $handle.AsyncWaitHandle  



    #
    # Get Bluetooth folder content
    #
    $dirpath = $null

    If ([IntPtr]::Size -eq 8) 
    {
        $dirpath = ${env:ProgramFiles(x86)} + "\Intel\Bluetooth"        
    }else
    {
        $dirpath = ${env:ProgramFiles}  + "\Intel\Bluetooth"
    }

    If (Test-Path $dirpath) 
    {
        $name = [string]$path + "\btdir.txt" 
		
		try
		{
			$file = New-Item -type file $name -Force 
		}
		
		catch
		{
			# Do nothing 
		}
		
        Get-ChildItem -recurse $dirpath | 
            Format-Table -Autosize -GroupBy @{ Expression = { [System.IO.Path]::GetDirectoryName($_.FullName) } ; Name = 'Directory' } -Property Mode,LastWriteTime,Length,Name,@{Label='FileVersion'; Expression = { $_.VersionInfo.FileVersion } } | 
            Out-String -Width 1024 | 
            Out-File $file
    }
	
	#
    # Get Windows update logs
    #
    $GetWindowsUpdateLogs = 
    {
        Param ($path)
        $dest = [string]$path + "\Setup logs\Panther Logs"
		New-Item $dest -type directory -Force > $null
		$pantherpath = $env:SystemRoot + "\Panther\*"
		if(Test-Path $pantherpath)
		{
			Copy-Item -Path $pantherpath -Destination $dest -recurse -Force 
		}
		
		$dest = [string]$path + "\Setup logs\Windows Update Logs"
		New-Item $dest -type directory -Force > $null
		$windowsupdatespath = $env:SystemRoot + "\Logs\WindowsUpdate\*"
        if(Test-Path $windowsupdatespath)
		{
			Copy-Item -Path $windowsupdatespath -Destination $dest -recurse -Force 
		}	
    }

    $pst = [powershell]::Create().AddScript($GetWindowsUpdateLogs ).AddArgument($path)
    $pst.RunspacePool = $rspool

    $handle = $pst.BeginInvoke()

    $job = "" | Select-Object Handle, Thread
    $job.Handle = $handle
    $job.Thread = $pst
    $jobs += $job
    $waits += $handle.AsyncWaitHandle
	

    #
    # Get all registry keys related to the path HKLM\SOFTWARE\Microsoft\Windows NT\CurrentVersion
    #
    $name = [String]$path + "\RegistryKeys.txt"
	try
	{
		$file = New-Item -type file $name -Force
	}
	
	catch
	{
		# Do nothing 
	}
	
   Get-ItemProperty "HKLM:\SOFTWARE\Microsoft\Windows NT\CurrentVersion" | Out-File $file
    
    
    While (@($jobs | Where-Object {$_.Handle -ne $Null}).count -gt 0)  
    {
        ForEach ($job in $($jobs | Where-Object {$_.Handle -ne $null -and $_.Handle.IsCompleted -eq $True}))
        {
            $job.Thread.EndInvoke($job.Handle)
            $job.Thread.Dispose()
            $job.Thread = $Null
            $job.Handle = $Null
        }
    }

    $rspool.Close() | Out-Null
    $rspool.Dispose() | Out-Null
} 

function GetLogOptions()
{
    $xml = $null

    if (Test-Path $global:OptionsFile -PathType Leaf) 
    {
        Try 
        {
            $xml = [xml](Get-Content $global:OptionsFile)
        }
        Catch
        {
        }
    }

    return $xml
}

function LogFileInfo($filename, $path, $etl_filename)
{
     try 
    { 
        $versioninfo = [System.Diagnostics.FileVersionInfo]::GetVersionInfo($path)
        LogMessage ("--- Version of " + $filename + " in " + $path + " after the test " + $versioninfo.FileVersion )
		if ($versioninfo.Comments -ne $null)
		{	
			LogMessage ("--- Build Tic of " + $filename + " in " + $path + " after the test " + $versioninfo.Comments )
		}
		else{
			LogMessage ("--- Build Tic information can not be found ")
		}

		Get-ChildItem $global:LogFilePath | Where-Object {$_.Name -like "*" + $etl_filename + "-$global:DefaultDriverVersion*"} | foreach {
			Rename-Item -Path $_.FullName -NewName $_.Name.Replace($global:DefaultDriverVersion, $versioninfo.FileVersion)
		}
    }
    catch {}

}


function DoCleanup()
{
    #
    #   Record versions of the drivers and ibtsiva (after the test) in Summary log
    #   Rename ETL's if only version was not available during the start. 
    #
    
    $originalInfPath = Get-OriginalInfPath($global:BtInfName)

    if($originalInfPath)

    {
        $originalInfPath = $originalInfPath | Split-Path
    }

    $driverEtl = @{
        'ibtusb.sys'  = 'ibtusb'
        'ibtuart.sys' = 'ibtuart'
        'ibtsiva.exe' = 'ibtsiva'
    }

    $driverEtl.Keys | ForEach-Object {
        $drvPath = Get-DriversPath($_)
        $versioninfo = ""

        if([System.IO.File]::Exists($drvPath)) {
            $versioninfo = [System.Diagnostics.FileVersionInfo]::GetVersionInfo($drvPath).FileVersion
        }

        # Set the driver path to windows driver store in case if it doesn't
        # exist in system32/drivers or version doesn't match
        
        if(($versioninfo -ne $global:DriverVersion) -and $originalInfPath)
        {
            $drvPath = Join-Path -Path $originalInfPath -ChildPath $_
        }

        LogFileInfo -filename $_ -path $drvPath -etl_filename $driverEtl.Item($_)
    }
}


function CheckFiles([string]$path)
{
 	$CurrentDir = Get-Location
	Set-Location -literalpath $path 
	$CheckValue = $false
 	$FileName = "ETLs summary.txt"
	$SearchString = "Lost"
	
	try 
	{
		tracerpt.exe *.etl -summary $FileName > $null
        if($LastExitCode -ne 0) {
             return
        }
		[string]$Line = Select-String -Pattern $SearchString -path $FileName -CaseSensitive 

		$Pos = $Line.IndexOf("Lost")

		[string]$LostEvents = $Line.Substring($Pos + 4)
		[int]$LostEvents = $LostEvents.replace(' ','')
			
		if ($LostEvents -ne 0) 
		{
			$CheckValue = $true
		}
		
		Remove-Item $global:LogFilePath/$FileName -force
	}
	
	catch
	{
		# Do nothing
	}
	
	if($CheckValue -eq $true)
	{
		$Message = "Some log files may have not been collected properly. We suggest you run the tracetool again to make sure logs are collected properly.`n"
		$Colors = @("Yellow","Red")
		$StartLeft = [System.Console]::CursorLeft
		$StartTop  = [System.Console]::CursorTop
		$Color = 0
		for($i = 0; $i -le 10; $i++) 
		{
			[System.Console]::CursorLeft = $startLeft
			[System.Console]::CursorTop  = $startTop
			if($Color -eq 0)
			{
				$Color = 1
			}
			else
			{
				$Color = 0
			}
			Write-Host "`r$Message`r" -ForegroundColor $Colors[$Color]
			Start-Sleep -Seconds 1
		}
	}
	Set-Location -literalpath $CurrentDir
}


function CheckOption([string]$Option)
{
    # Do not check if option is empty
    if(-Not $Option)
    {
        return
    }

	if(($global:CmdTraceValues -contains $Option))
	{
		if( (-Not (IsWin7)) -and ($global:Win7SpecificTraces -contains $Option) )
		{
			Write-Host "User entered invalid tracing option. Tracing option '$Option' is only available for Win7. See Usage.txt for list of commands!."-errorAction SilentlyContinue -ForegroundColor "Red"
			break
		}
        
        if($Option -eq "boot")
        {
            Write-Host-Warning "User entered boot option. This option is deprecated and enabled by default."
        }

        if($Option -eq "install")
        {
            Write-Host-Warning "User entered install option. This option is deprecated and enabled by default."
        }
	}
	else
	{
		Write-Host "User entered invalid tracing option. Please enter a valid tracing option! See usage.txt for list of commands." -errorAction SilentlyContinue -ForegroundColor "Red"
		break
	}
}

#
# Get driver IFR log
#
function CollectIFRLogs()
{

	#Get the OSVersion
	$global:OSVersion = [Environment]::OSVersion.version
			   
	#GetIfr script only works on Win10 RS1+
	if($global:OSVersion.Major -ge 10 -and $global:OSVersion.Build -ge 14393)
	{
		$dest = [string]$path + "\IFR Logs"
		New-Item $dest -Type directory -Force > $null
				   
		#USB driver IFR
		if($global:UsbHardwareIds -contains $global:HardwareId)
		{
			$IfrLog = ".\ibtusb.etl"
			$KmdfIfrLog = ".\KmdfIfrForibtusb.etl"
			$GetIfrPath =  ".\scripts\GetIfr.ps1 -service ibtusb -IncludeKmdfIfr"
			Start-Process powershell -Argument $GetIfrPath -NoNewWindow -Wait > $null
			Move-Item -Path $IfrLog -Destination $dest -Force > $null
			Move-Item -Path $KmdfIfrLog -Destination $dest -Force > $null
			Rename-Item -Path $dest + "\ibtusb.etl" -NewName "ibtusbIFR.etl"
			Rename-Item -Path $dest + "\KmdfIfrForibtusb.etl" -NewName "ibtusbKmdfIfr.etl"
		}

		#UART driver IFR
		elseif($global:UartHardwareId -contains $global:HardwareId)
		{
			#TODO: uart IFR support
		}
		
		#PCIe driver IFR etc.
		else
		{
			#TODO: Extend to other drivers
		}
	}  
}

function Get-DriversPath([string]$path) {

    return Get-SystemPath(Join-Path -Path "drivers" -ChildPath $path)
}

function Get-SystemPath([string]$path) {
	$sysnative_path = $Env:SystemRoot + "\sysnative\" + $path
	if(Test-Path $sysnative_path) {
		return $sysnative_path
	}
	
	return $Env:SystemRoot + "\system32\" + $path
}

function TraceMapAppend($Map)
{
	for($idx = 0; $idx -lt 32; ++$idx)
	{
		$TraceMap[$idx] = $TraceMap[$idx] -bor $Map[$idx]
	}
}

function Get-OriginalInfPath($InfName)
{
    $path = ""

    if ($InfName)
    {
        try
        {
            $path = Get-WindowsDriver -Online -Driver $InfName | Select-Object -first 1 -ExpandProperty OriginalFileName
        }
        catch 
        {
            # do nothing. return empty string
        }
    }
    
    return $path
}

<#
# Retrieves the .inf file of a present driver and returns tracing option for that driver.
# Returns empty string if nothing matched
#>
function Find-ProviderOption()
{
    $option = ""
    $infToOption = @{
        'ibtusb'    = 'ibtusb'
        'ibtuart'   = 'ibtuart'
        'ibtusbix'  = 'ixbtusb'
        'ibtusbbl'  = 'usbbl'
        'ibtfudrv'  = 'ibtfudrv'
    }

    $originalInfPath = Get-OriginalInfPath($global:BtInfName)

    if($originalInfPath)
    {
        # Match .inf name to trace option
        $infName = [IO.Path]::GetFileNameWithoutExtension($originalInfPath)
        $option = $infToOption.Item($infName)
    }

    return $option
}

function Write-Host-Warning($str)
{
    Write-Host -ForegroundColor "Yellow" $str
}

function Write-Host-Success($str)
{
    Write-Host -ForegroundColor "Green" $str
}

function Write-Host-Danger($str)
{
    Write-Host -ForegroundColor "Red" $str
}
# SIG # Begin signature block
# MIIliQYJKoZIhvcNAQcCoIIlejCCJXYCAQExDzANBglghkgBZQMEAgEFADB5Bgor
# BgEEAYI3AgEEoGswaTA0BgorBgEEAYI3AgEeMCYCAwEAAAQQH8w7YFlLCE63JNLG
# KX7zUQIBAAIBAAIBAAIBAAIBADAxMA0GCWCGSAFlAwQCAQUABCCp6twQiAX6tmqq
# C8KHQ3pw3FhMikO8Ya37NBJ48uORHKCCIJwwggPuMIIDV6ADAgECAhB+k+v7fMZO
# WepLmnfUBvw7MA0GCSqGSIb3DQEBBQUAMIGLMQswCQYDVQQGEwJaQTEVMBMGA1UE
# CBMMV2VzdGVybiBDYXBlMRQwEgYDVQQHEwtEdXJiYW52aWxsZTEPMA0GA1UEChMG
# VGhhd3RlMR0wGwYDVQQLExRUaGF3dGUgQ2VydGlmaWNhdGlvbjEfMB0GA1UEAxMW
# VGhhd3RlIFRpbWVzdGFtcGluZyBDQTAeFw0xMjEyMjEwMDAwMDBaFw0yMDEyMzAy
# MzU5NTlaMF4xCzAJBgNVBAYTAlVTMR0wGwYDVQQKExRTeW1hbnRlYyBDb3Jwb3Jh
# dGlvbjEwMC4GA1UEAxMnU3ltYW50ZWMgVGltZSBTdGFtcGluZyBTZXJ2aWNlcyBD
# QSAtIEcyMIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAsayzSVRLlxwS
# CtgleZEiVypv3LgmxENza8K/LlBa+xTCdo5DASVDtKHiRfTot3vDdMwi17SUAAL3
# Te2/tLdEJGvNX0U70UTOQxJzF4KLabQry5kerHIbJk1xH7Ex3ftRYQJTpqr1SSwF
# eEWlL4nO55nn/oziVz89xpLcSvh7M+R5CvvwdYhBnP/FA1GZqtdsn5Nph2Upg4XC
# YBTEyMk7FNrAgfAfDXTekiKryvf7dHwn5vdKG3+nw54trorqpuaqJxZ9YfeYcRG8
# 4lChS+Vd+uUOpyyfqmUg09iW6Mh8pU5IRP8Z4kQHkgvXaISAXWp4ZEXNYEZ+VMET
# fMV58cnBcQIDAQABo4H6MIH3MB0GA1UdDgQWBBRfmvVuXMzMdJrU3X3vP9vsTIAu
# 3TAyBggrBgEFBQcBAQQmMCQwIgYIKwYBBQUHMAGGFmh0dHA6Ly9vY3NwLnRoYXd0
# ZS5jb20wEgYDVR0TAQH/BAgwBgEB/wIBADA/BgNVHR8EODA2MDSgMqAwhi5odHRw
# Oi8vY3JsLnRoYXd0ZS5jb20vVGhhd3RlVGltZXN0YW1waW5nQ0EuY3JsMBMGA1Ud
# JQQMMAoGCCsGAQUFBwMIMA4GA1UdDwEB/wQEAwIBBjAoBgNVHREEITAfpB0wGzEZ
# MBcGA1UEAxMQVGltZVN0YW1wLTIwNDgtMTANBgkqhkiG9w0BAQUFAAOBgQADCZuP
# ee9/WTCq72i1+uMJHbtPggZdN1+mUp8WjeockglEbvVt61h8MOj5aY0jcwsSb0ep
# rjkR+Cqxm7Aaw47rWZYArc4MTbLQMaYIXCp6/OJ6HVdMqGUY6XlAYiWWbsfHN2qD
# IQiOQerd2Vc/HXdJhyoWBl6mOGoiEqNRGYN+tjCCBKMwggOLoAMCAQICEA7P9DjI
# /r81bgTYapgbGlAwDQYJKoZIhvcNAQEFBQAwXjELMAkGA1UEBhMCVVMxHTAbBgNV
# BAoTFFN5bWFudGVjIENvcnBvcmF0aW9uMTAwLgYDVQQDEydTeW1hbnRlYyBUaW1l
# IFN0YW1waW5nIFNlcnZpY2VzIENBIC0gRzIwHhcNMTIxMDE4MDAwMDAwWhcNMjAx
# MjI5MjM1OTU5WjBiMQswCQYDVQQGEwJVUzEdMBsGA1UEChMUU3ltYW50ZWMgQ29y
# cG9yYXRpb24xNDAyBgNVBAMTK1N5bWFudGVjIFRpbWUgU3RhbXBpbmcgU2Vydmlj
# ZXMgU2lnbmVyIC0gRzQwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCi
# Yws5RLi7I6dESbsO/6HwYQpTk7CY260sD0rFbv+GPFNVDxXOBD8r/amWltm+YXkL
# W8lMhnbl4ENLIpXuwitDwZ/YaLSOQE/uhTi5EcUj8mRY8BUyb05Xoa6IpALXKh7N
# S+HdY9UXiTJbsF6ZWqidKFAOF+6W22E7RVEdzxJWC5JH/Kuu9mY9R6xwcueS51/N
# ELnEg2SUGb0lgOHo0iKl0LoCeqF3k1tlw+4XdLxBhircCEyMkoyRLZ53RB9o1qh0
# d9sOWzKLVoszvdljyEmdOsXF6jML0vGjG/SLvtmzV4s73gSneiKyJK4ux3DFvk6D
# Jgj7C72pT5kI4RAocqrNAgMBAAGjggFXMIIBUzAMBgNVHRMBAf8EAjAAMBYGA1Ud
# JQEB/wQMMAoGCCsGAQUFBwMIMA4GA1UdDwEB/wQEAwIHgDBzBggrBgEFBQcBAQRn
# MGUwKgYIKwYBBQUHMAGGHmh0dHA6Ly90cy1vY3NwLndzLnN5bWFudGVjLmNvbTA3
# BggrBgEFBQcwAoYraHR0cDovL3RzLWFpYS53cy5zeW1hbnRlYy5jb20vdHNzLWNh
# LWcyLmNlcjA8BgNVHR8ENTAzMDGgL6AthitodHRwOi8vdHMtY3JsLndzLnN5bWFu
# dGVjLmNvbS90c3MtY2EtZzIuY3JsMCgGA1UdEQQhMB+kHTAbMRkwFwYDVQQDExBU
# aW1lU3RhbXAtMjA0OC0yMB0GA1UdDgQWBBRGxmmjDkoUHtVM2lJjFz9eNrwN5jAf
# BgNVHSMEGDAWgBRfmvVuXMzMdJrU3X3vP9vsTIAu3TANBgkqhkiG9w0BAQUFAAOC
# AQEAeDu0kSoATPCPYjA3eKOEJwdvGLLeJdyg1JQDqoZOJZ+aQAMc3c7jecshaAba
# tjK0bb/0LCZjM+RJZG0N5sNnDvcFpDVsfIkWxumy37Lp3SDGcQ/NlXTctlzevTcf
# Q3jmeLXNKAQgo6rxS8SIKZEOgNER/N1cdm5PXg5FRkFuDbDqOJqxOtoJcRD8HHm0
# gHusafT9nLYMFivxf1sJPZtb4hbKE4FtAC44DagpjyzhsvRaqQGvFZwsL0kb2yK7
# w/54lFHDhrGCiF3wPbRRoXkzKy57udwgCRNx62oZW8/opTBXLIlJP7nPf8m/PiJo
# Y1OavWl0rMUdPH+S4MO8HNgEdTCCBUwwggM0oAMCAQICEzMAAAA12NVZWwZxQSsA
# AAAAADUwDQYJKoZIhvcNAQEFBQAwfzELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldh
# c2hpbmd0b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
# b3Jwb3JhdGlvbjEpMCcGA1UEAxMgTWljcm9zb2Z0IENvZGUgVmVyaWZpY2F0aW9u
# IFJvb3QwHhcNMTMwODE1MjAyNjMwWhcNMjMwODE1MjAzNjMwWjBvMQswCQYDVQQG
# EwJTRTEUMBIGA1UEChMLQWRkVHJ1c3QgQUIxJjAkBgNVBAsTHUFkZFRydXN0IEV4
# dGVybmFsIFRUUCBOZXR3b3JrMSIwIAYDVQQDExlBZGRUcnVzdCBFeHRlcm5hbCBD
# QSBSb290MIIBIjANBgkqhkiG9w0BAQEFAAOCAQ8AMIIBCgKCAQEAt/caM+byAAQt
# OeBOW+0fvGwPzbX6I7bO3psRM5ekKUx9k5+9SryT7QMa44/P5W1QWtaXKZRagLBJ
# etsulf24yr83OC0ePpFBrXBWx/BPP+gynnTKyJBU6cZfD3idmkA8Dqxhql4Uj56H
# oWpQ3NeaTq8Fs6ZxlJxxs1BgCscTnTgHhgKo6ahpJhiQq0ywTyOrOk+E2N/On+Fp
# b7vXQtdrROTHre5tQV9yWnEIN7N5ZaRZoJQ39wAvDcKSctrQOHLbFKhFxF0qfbe0
# 1sTurM0TRLfJK91DACX6YblpalgjEbenM49WdVn1zSnXRrcKK2W200JvFbK4e/vv
# 6V1T1TRaJwIDAQABo4HQMIHNMBMGA1UdJQQMMAoGCCsGAQUFBwMDMBIGA1UdEwEB
# /wQIMAYBAf8CAQIwHQYDVR0OBBYEFK29mHo0tCb3+sQmVO8DveAky1QaMAsGA1Ud
# DwQEAwIBhjAfBgNVHSMEGDAWgBRi+wohW39DbhHaCVRQa/XSlnHxnjBVBgNVHR8E
# TjBMMEqgSKBGhkRodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9k
# dWN0cy9NaWNyb3NvZnRDb2RlVmVyaWZSb290LmNybDANBgkqhkiG9w0BAQUFAAOC
# AgEANiui8uEzH+ST9/JphcZkDsmbYy/kcDeY/ZTse8/4oUJG+e1qTo00aTYFVXoe
# u62MmUKWBuklqCaEvsG/Fql8qlsEt/3RwPQCvijt9XfHm/469ujBe9OCq/oUTs8r
# z+XVtUhAsaOPg4utKyVTq6Y0zvJD908s6d0eTlq2uug7EJkkALxQ/Xj25SOoiZST
# 97dBMDdKV7fmRNnJ35kFqkT8dK+CZMwHywG2CcMu4+gyp7SfQXjHoYQ2VGLy7BUK
# yOrQhPjx4Gv0VhJfleD83bd2k/4pSiXpBADxtBEOyYSe2xd99R6ljjYpGTptbEZL
# 16twJCiNBaPZ1STy+KDRPII51KiCDmk6gQn8BvDHWTOENpMGQZEjLCKlpwErULQo
# rttGsFkbhrObh+hJTjkLbRTfTAMwHh9fdK71W1kDU+yYFuDQYjV1G0i4fRPleki4
# d1KkB5glOwabek5qb0SGTxRPJ3knPVBzQUycQT7dKQxzscf7H3YMF2UE69JQEJJB
# SezkBn02FURvib9pfflNQME6mLagfjHSta7K+1PVP1CGzV6TO21dfJo/P/epJViE
# 3RFJAKLHyJ433XeObXGL4FuBNF1Uusz1k0eIbefvW+Io5IAbQOQPKtF/IxVlWqyZ
# lEM/RlUm1sT6iJXikZqjLQuF3qyM4PlncJ9xeQIx92GiKcQwggV0MIIEXKADAgEC
# AhAnZu5W60nzjqvXcKL8hN4iMA0GCSqGSIb3DQEBDAUAMG8xCzAJBgNVBAYTAlNF
# MRQwEgYDVQQKEwtBZGRUcnVzdCBBQjEmMCQGA1UECxMdQWRkVHJ1c3QgRXh0ZXJu
# YWwgVFRQIE5ldHdvcmsxIjAgBgNVBAMTGUFkZFRydXN0IEV4dGVybmFsIENBIFJv
# b3QwHhcNMDAwNTMwMTA0ODM4WhcNMjAwNTMwMTA0ODM4WjCBhTELMAkGA1UEBhMC
# R0IxGzAZBgNVBAgTEkdyZWF0ZXIgTWFuY2hlc3RlcjEQMA4GA1UEBxMHU2FsZm9y
# ZDEaMBgGA1UEChMRQ09NT0RPIENBIExpbWl0ZWQxKzApBgNVBAMTIkNPTU9ETyBS
# U0EgQ2VydGlmaWNhdGlvbiBBdXRob3JpdHkwggIiMA0GCSqGSIb3DQEBAQUAA4IC
# DwAwggIKAoICAQCR6FSS0gpWsawNJN3Fz0RndJkrN6N9I3AAcbxT38T6KhKPS38Q
# Vr2fcHK3YX/JSw8Xpz3jsARh7v8Rl8f0hj4K+j5c+ZPmNHrZFGvnnLOFoIJ6dq9x
# kNfs/Q36nGz637CC9BR++b7Epi9Pf5l/tfxnQ3K9DADWietrLNPtj5gcFKt+5eNu
# /Nio5JIk2kNrYrhV/erBvGy2i/MOjZrkm2xpmfh4SDBF1a3hDTxFYPwyllEnvGfD
# yi62a+pGx8cgoLEfZd5ICLqkTqnyg0Y3hOvozIFIQ2dOciqbXL1MGyiKXCJ7tKuY
# 2e7gUYPDCUZObT6Z+pUX2nwzV0E8jVHtC7ZcryxjGt9XyD+86V3Em69FmeKjWiS0
# uqlWPc9vqv9JWL7wqP/0uK3pN/u6uPQLOvnoQ0IeidiEyxPx2bvhiWC4jChWrBQd
# nArncevPDt09qZahSL0896+1DSJMwBGB7FY79tOi4lu3sgQiUpWAk2nojkxl8ZED
# LXB0AuqLZxUpaVICu9ffUGpVRr+goyhhf3DQw6KqLCGqR84onAZFdr+CGCe01a60
# y1Dma/RMhnEw6abfFobg2P9A3fvQQoh/ozM6LlweQRGBY84YcWsr7KaKtzFcOmpH
# 4MN5WdYgGq/yapiqcrxXStJLnbsQ/LBMQeXtHT1eKJ2czL+zUdqnR+WEUwIDAQAB
# o4H0MIHxMB8GA1UdIwQYMBaAFK29mHo0tCb3+sQmVO8DveAky1QaMB0GA1UdDgQW
# BBS7r34CPfqm8TyEjq3uOJjs2TIy1DAOBgNVHQ8BAf8EBAMCAYYwDwYDVR0TAQH/
# BAUwAwEB/zARBgNVHSAECjAIMAYGBFUdIAAwRAYDVR0fBD0wOzA5oDegNYYzaHR0
# cDovL2NybC51c2VydHJ1c3QuY29tL0FkZFRydXN0RXh0ZXJuYWxDQVJvb3QuY3Js
# MDUGCCsGAQUFBwEBBCkwJzAlBggrBgEFBQcwAYYZaHR0cDovL29jc3AudXNlcnRy
# dXN0LmNvbTANBgkqhkiG9w0BAQwFAAOCAQEAZL+D8V+ahdDNuKEpVw3oWvfR6T7y
# dgRu8VJwux48/00NdGrMgYIl08OgKl1M9bqLoW3EVAl1x+MnDl2EeTdAE3f1tKwc
# 0DurFxLW7zQYfivpedOrV0UMryj60NvlUJWIu9+FV2l9kthSynOBvxzz5rhuZhEF
# sx6ULX+RlZJZ8UzOo5FxTHxHDDsLGfahsWyGPlyqxC6Cy/kHlrpITZDylMipc6Lr
# Bnsjnd6i801Vn3phRZgYaMdeQGsj9Xl674y1a4u3b0b0e/E9SwTYk4BZWuBBJB2y
# jxVgWEfb725G/RX12V+as9vYuORAs82XOa6Fux2OvNyHm9Gm7/E7bxA4bzCCBeow
# ggPSoAMCAQICE1YAAAidKVOniKW4iGkAAAAACJ0wDQYJKoZIhvcNAQELBQAwczEL
# MAkGA1UEBhMCVVMxCzAJBgNVBAgTAkNBMRQwEgYDVQQHEwtTYW50YSBDbGFyYTEa
# MBgGA1UEChMRSW50ZWwgQ29ycG9yYXRpb24xJTAjBgNVBAMTHEludGVsIEV4dGVy
# bmFsIElzc3VpbmcgQ0EgN0IwHhcNMTgxMTE5MTk0OTQzWhcNMjAxMTE4MTk0OTQz
# WjB/MQswCQYDVQQGEwJVUzELMAkGA1UECBMCQ0ExFDASBgNVBAcTC1NhbnRhIENs
# YXJhMRowGAYDVQQKExFJbnRlbCBDb3Jwb3JhdGlvbjExMC8GA1UEAxMoSW50ZWwo
# UikgV2lyZWxlc3MgQ29ubmVjdGl2aXR5IFNvbHV0aW9uczCCASIwDQYJKoZIhvcN
# AQEBBQADggEPADCCAQoCggEBAPobAwB8a0o/aofFGLcWheioa0oUydsWBdJt3jV/
# MEkXPllWhmeQtPaZDT4kG1YTG+rhdIx0Cu83m2P4PWgeqi2ExYY+fLAFdAIqhc7v
# J+P3Pd6IABwPLKtuSL6AlR6YYOTPN6VkcMYfSMG6PNU5KP6ksxAPa980J5EGsYmc
# oXFpzoCZZfVmfP6/VNFYIK8OILD9FFvRoJdwG8IYYaJvB0DnPBkQjqrHjcLAQMqb
# WjwjkvYo2caZIsims67kvfy/EmsZmQxj4Cfzq0H6LibuObSz6uRa5Sd6wEnUBQrr
# ucb8fI3/MdOBxmUzON5os55iPo64qFABa8u12uOEI+OFdfECAwEAAaOCAWkwggFl
# MB0GA1UdDgQWBBQZ0bJy899nFIWDp6F74fnraqKKJDAfBgNVHSMEGDAWgBSywGem
# Vo0neRB0w/XFpYlm1vEuNjA3BgNVHR8EMDAuMCygKqAohiZodHRwOi8vcGtpLmlu
# dGVsLmNvbS9jcmwvSW50ZWxDQTdCLmNybDBmBggrBgEFBQcBAQRaMFgwMgYIKwYB
# BQUHMAKGJmh0dHA6Ly9wa2kuaW50ZWwuY29tL2NydC9JbnRlbENBN0IuY3J0MCIG
# CCsGAQUFBzABhhZodHRwOi8vT0NTUC5pbnRlbC5jb20vMAwGA1UdEwEB/wQCMAAw
# CwYDVR0PBAQDAgeAMD0GCSsGAQQBgjcVBwQwMC4GJisGAQQBgjcVCIbDjHWEmeVR
# g/2BKIWOn1OCkcAJZ4WJ9mSF6ZJQAgFkAgESMBMGA1UdJQQMMAoGCCsGAQUFBwMD
# MBMGA1UdIAQMMAowCAYGZ4EMAQQBMA0GCSqGSIb3DQEBCwUAA4ICAQC4U0jdi5hd
# 9+q1kP9uS+fFkw7ZjBaGnJ8fWDwJLMEdJy3bO4PBR+J9R/lTYAjkbCOY2LRjYwej
# NULVAeiMsLsLqaTHi3Gz704/naCveOHXJi5MHKob4W4I+ShuBVg3UGoQ3sQU7BTj
# xGANuWmav6Gd4Tins5+vlKXNC1e6nBVJjsEXZrsgDZeOW69WohmZklV4jlCEMssq
# vk/5+yiL7ON5Rv6L9khPjpEBSoYWeE8xptyte4OESB296VnL5mu/H6M2hmet4U5Q
# cz28s+Fl/cZaafHEPS9sf5AffdU0rZakem70p+ngxhlABXNdZ6YcedLGRqcYD1ZI
# hbocidZ9D4u3YMhbDQcjbzS42WH9xX8V5pkuydneBUz8m6XYcmHhU3QkwqRYzoaC
# fYOzxgVkrunvjQ7R7g7tYrrf27eImhxrCd1Pp1cHfnVHnt49Nt7IbKIlS15Vj8ph
# Ili52ag1W8TwdDr4HphbKN7FsPrg36z6S9+FbvRi7XpFSULPdy9QaWsmosfMPhdz
# zujhJKDTNTSwj1m5kIvEAvzdI++65z/NbKw9VPANumlaYFw5x4NXOJdjgq4xbsrH
# J0BVZTVhvMHe55hrg7uwyGLow+dAhKKKiF9IRH5TKHN/epIdgMxSzXJOLdLIXvdu
# C7gRJ7OEAbPredxzYQYQLW+ONT53wxZSMjCCB0kwggUxoAMCAQICEAabXpkncoTI
# dn8TaKfesPMwDQYJKoZIhvcNAQEMBQAwgYUxCzAJBgNVBAYTAkdCMRswGQYDVQQI
# ExJHcmVhdGVyIE1hbmNoZXN0ZXIxEDAOBgNVBAcTB1NhbGZvcmQxGjAYBgNVBAoT
# EUNPTU9ETyBDQSBMaW1pdGVkMSswKQYDVQQDEyJDT01PRE8gUlNBIENlcnRpZmlj
# YXRpb24gQXV0aG9yaXR5MB4XDTE1MTAyODAwMDAwMFoXDTIxMDYxNzIzNTk1OVow
# czELMAkGA1UEBhMCVVMxCzAJBgNVBAgTAkNBMRQwEgYDVQQHEwtTYW50YSBDbGFy
# YTEaMBgGA1UEChMRSW50ZWwgQ29ycG9yYXRpb24xJTAjBgNVBAMTHEludGVsIEV4
# dGVybmFsIElzc3VpbmcgQ0EgN0IwggIiMA0GCSqGSIb3DQEBAQUAA4ICDwAwggIK
# AoICAQC6Wm/xXxoLVBASupDWousOXqw9zhWFmkoRymR/IoppFlfPIDokgNe74FPc
# qUvPbIGxvhqiAeBFe2wMZtRPj24i7fNSmKJ6NTeKbOy4MYZJZynHtLOIcNCZHivE
# EvfuHvO8JI4RhgOT0nkL567cWjGpoueACfpaklFnMO87sfSx/U6rGmgl59tVYHN7
# Fb8ATJBCnilmD3o8432gQpmQnjVcgxzlSN9BfLOe6Zcqw/oDRiOpJsJl8ZXiV02m
# qoCziVG3MpZDJfQvkPgOzM94dHUNri+doaAL/B1NI8aGtt1hx6a/lxHtFQxA4/3l
# EC2ds8Hv7etkeReMBq5pkutGAyrUIdFZ+vw26ZcWRnkoM6NZKxn0FeZwPEqJssk+
# jXdI1Ap8rEXlRClcE/PITY4mnvBUhQDtn+vuz+ITwbqt06rVRK20dB6rAN0Hcl6c
# prSQ4/ZGoQBUV8gmF+NIc9GYaNGRrKw6duxL4zaFkxliwCFIaAuKxZQlHSjVAtQH
# 6f5QU4ORE2xhAAf8QPeXzY8yew16NwwIo/vsXvqNc+O6xJLQcfT7HGGVbVdbHmYt
# aiB+VTC7rIlfynHtXBOOBoUgckbMIf0OYxDfHEc/EtL5UEsEG99v0sybjlcFIJWv
# Y1aLIVAJz5Xvnzb3K3v/gHFQr31Lmp7Q90QYsGAalKOKMAie6wIDAQABo4IBxDCC
# AcAwHwYDVR0jBBgwFoAUu69+Aj36pvE8hI6t7jiY7NkyMtQwHQYDVR0OBBYEFLLA
# Z6ZWjSd5EHTD9cWliWbW8S42MA4GA1UdDwEB/wQEAwIBhjASBgNVHRMBAf8ECDAG
# AQH/AgEAMD4GA1UdJQQ3MDUGCCsGAQUFBwMDBggrBgEFBQcDBAYIKwYBBQUHAwgG
# CisGAQQBgjcKAwwGCSsGAQQBgjcVBTAiBgNVHSAEGzAZMA0GCyqGSIb4TQEFAWkB
# MAgGBmeBDAEEATBMBgNVHR8ERTBDMEGgP6A9hjtodHRwOi8vY3JsLmNvbW9kb2Nh
# LmNvbS9DT01PRE9SU0FDZXJ0aWZpY2F0aW9uQXV0aG9yaXR5LmNybDBxBggrBgEF
# BQcBAQRlMGMwOwYIKwYBBQUHMAKGL2h0dHA6Ly9jcnQuY29tb2RvY2EuY29tL0NP
# TU9ET1JTQUFkZFRydXN0Q0EuY3J0MCQGCCsGAQUFBzABhhhodHRwOi8vb2NzcC5j
# b21vZG9jYS5jb20wNQYDVR0eBC4wLKAqMAuBCWludGVsLmNvbTAboBkGCisGAQQB
# gjcUAgOgCwwJaW50ZWwuY29tMA0GCSqGSIb3DQEBDAUAA4ICAQA1uwPqzJtgGhPQ
# dVKOgJVFTp6/bsC7ZKrDbrECHUZeL+gvSMyEEPetmTv/+oVoKbDTfDHiGrR7wWbi
# pTvHKRiYNa5jAahFIJVh2xBNuQ1r05lkzl+LuGwTRqBuWg0+55DrtzGhIfWN3jt7
# aTbxCAC5qr8cVmFW18ySPynU2WvYIi8OVvVq0UbogI85epI8Z0i34voZDzdn4t8p
# LQKqQygurixGQiS+bbtqiEmmTCDf5WVP+uHBvnHV+F71nWaSsjtk4eiurJlVF73b
# G9+gk08/VvI7g9XSt8EIWlJAQuM+kSD3NbSR8E3hNGlIecDtMMmTGoTVchmPbYA5
# 9FmrIBbY+f9wJiN77MUAMyJ8PSA67bQovHqBDOcLwT98MAxOULhnD9dkF7fDxSCF
# yo/O1SYqElS5/yL4qCc8yg6FNxTuAuUvZhViY4dqXs8p07iReLdhchd7wRmmGAgi
# 2tCRJfYGCQkmsC2sgIh0M1/H4ETBMJl22HexRwHvaZIr7a5YKWOgNY7kHbcE8do6
# sjKAsci88OcPcQB6MzoG6KTYednZU82b/rJoW4iEhWsHcdBPkwoHYAM0CNJzvxQa
# 3+PHBBstmZ6THJWzh5hCWhyRY1I5io9KKsJMe3BpOjzx+y//DgqHlOQBas+btB+j
# DqnqKtyvK4xEAf06WH0yeKIZ1cl0xTGCBEMwggQ/AgEBMIGKMHMxCzAJBgNVBAYT
# AlVTMQswCQYDVQQIEwJDQTEUMBIGA1UEBxMLU2FudGEgQ2xhcmExGjAYBgNVBAoT
# EUludGVsIENvcnBvcmF0aW9uMSUwIwYDVQQDExxJbnRlbCBFeHRlcm5hbCBJc3N1
# aW5nIENBIDdCAhNWAAAInSlTp4iluIhpAAAAAAidMA0GCWCGSAFlAwQCAQUAoHww
# EAYKKwYBBAGCNwIBDDECMAAwGQYJKoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYK
# KwYBBAGCNwIBCzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkEMSIEIKQhtrbe
# 4MseucBypY3VwYzwJlKDbeJqnjry6TFIihl2MA0GCSqGSIb3DQEBAQUABIIBAMYN
# vu0uJc2ODenj2uWn7LbRElMBlsRWgUDzDoKFLylfrQtF298eYfladknF9RiisS6k
# 694M18RY4Wzs/iwBckIRezO2mrr4gOSEM1j7OoNuPCThjyKKcl3qUnE1eZ3WzBLk
# utU6WDuWCwpllgtKnhWtaGKB94/MRwo/fE8OsNhHztB0ogd5Fo32PM8i7WF+vQ4z
# JxXEOGaRUkwBvqFsbBamdWf5sJEmF89+Qk4p+HGKxq3RZmLV2DbyvCtefXD9jsLf
# LOeOxvkO1SWS2PLo4sXcxTFFOqravO/KXHw0a+oI28OQqhZHjQ5KeB5UfR8KwZdL
# Pf0MZNyLL317R7vB1OqhggILMIICBwYJKoZIhvcNAQkGMYIB+DCCAfQCAQEwcjBe
# MQswCQYDVQQGEwJVUzEdMBsGA1UEChMUU3ltYW50ZWMgQ29ycG9yYXRpb24xMDAu
# BgNVBAMTJ1N5bWFudGVjIFRpbWUgU3RhbXBpbmcgU2VydmljZXMgQ0EgLSBHMgIQ
# Ds/0OMj+vzVuBNhqmBsaUDAJBgUrDgMCGgUAoF0wGAYJKoZIhvcNAQkDMQsGCSqG
# SIb3DQEHATAcBgkqhkiG9w0BCQUxDxcNMTkwNzEyMDUwMjM2WjAjBgkqhkiG9w0B
# CQQxFgQUdo80GG57lwHxs+qsevvmfHYfFw0wDQYJKoZIhvcNAQEBBQAEggEAdrv9
# 80CfHRJVd3aonwxAFlb9k4jXU2FDtenBeIqgx038RvSwpnd2pFt9zR9CVERl9IbK
# gHjnq5NN2olj720e41VrO+h9PCtIRy4I5tphw03oxr0IF8+EHJ7nWITJu8tACXQ2
# ltYbKEpUDpDV0ZfFTd9EYkibXnQ/qNizrfiAo0tT6cmjqeaOX9bTWClhzYQsN/5w
# XzkjhHpJ3JC73e2pDL1SdyGBxM5kYQLXzWJ7BsNPNJJzZy+ebyqBlZi4L1aMduKO
# To4UB+QUqJH7xdZX7u3S3TT+vmrU8I3R7tjJo0sgF4AFeNGbmONK9PS0hEGG/tDN
# QMmdKDRM7CGG8PVG1g==
# SIG # End signature block
