include(CMakeParseArguments)

#################################################################################################
# Description:
#   Sets up the proper CMake commands for copying the given file to the destination directory.
#   At build time, the file is copied to <build directory>/DESTINATION_DIR.
#   At installation time, the file is copied to <installation directory>/DESTINATION_DIR
#
# Required Parameters:
#   SOURCE_DIR        The directory where the file is to be copied from
#   DESTINATION_DIR   The directory (relative to the build and installation directories) where the file should be copied to
#   PACKAGE_COMPONENT The CPack component that this file should go in
#
#   One of:
#      TARGET          The CMake target to which this operation should be attached as a POST_BUILD step
#      DEPENDENCY_LIST A CMake list to which a unique output file will be added. Use this file dependency to add
#                      a list of CopyFile operations as a dependency to a custom CMake target
#
#   One (or more) of:
#      FILENAME         The name of the file to copy (cannot be specified along with DEBUG_FILENAME or RELEASE_FILENAME)
#      DEBUG_FILENAME   The name of the file to copy during Debug builds (cannot be specified along with FILENAME)
#      RELEASE_FILENAME The name of the file to copy during Release builds (cannot be specified along with FILENAME)
#
# Optional Parameters:
#   DESTINATION_FILENAME The name of the destination file (if it needs to be different from the original name of the file)
#
# Example:
#   CopyFile(SOURCE_DIR "${OPENIPC_ROOT}"
#            DEBUG_FILENAME "TestDebug.txt"
#            RELEASE_FILENAME "TestRelease.txt"
#            TARGET "MyExecutable"
#            DESTINATION_DIR "Bin"
#            DESTINATION_FILENAME "Test.txt"
#            PACKAGE_COMPONENT openipc)
#
#   This will copy ${OPENIPC_ROOT}/TestDebug.txt (or ${OPENIPC_ROOT}/TestRelease.txt for release builds)
#   to <build directory>/Bin/Test.txt and <installation directory>/Bin/Test.txt and will package it
#   in the "openipc" CPack component. Copying it to the build directory will be done as a POST_BUILD
#   step for the "MyExecutable" project
#
# Important Considerations:
#   - When TARGET is specified, the copying operation is specified as a POST_BUILD step to the given target.
#     This means that it will only be copied again when the project is built. If the project is up-to-date,
#     the file will not be copied. So whenever you change the original file, you'll need to rebuild this project.
#
#   - When DEPENDENCY_LIST is specified the copying operation is performed with every build, if the file has changed
#
#   - The file that is installed is the copy that was put in the build directory. Therefore any changes that are made
#     to the file in the build directory will be reflected in the installation directory if, and only if, TARGET was
#     specified. See the above two considerations for an explanation
#
#################################################################################################
function(CopyFile)
    # 1. Parse the arguments
    set(Options )
    set(SingleValueArgs TARGET SOURCE_DIR FILENAME DEBUG_FILENAME RELEASE_FILENAME DESTINATION_DIR DESTINATION_FILENAME PACKAGE_COMPONENT DEPENDENCY_LIST)
    set(MultiValueArgs )
    cmake_parse_arguments(COPYFILE "${Options}" "${SingleValueArgs}" "${MultiValueArgs}" ${ARGN})

    # 2. Figure out if all the required arguments have been specified and no conflicting arguments are there

    if(COPYFILE_FILENAME AND (COPYFILE_DEBUG_FILENAME OR COPYFILE_RELEASE_FILENAME))
        message(FATAL_ERROR "Cannot use FILENAME with DEBUG_FILENAME or RELEASE_FILENAME: ${ARGN}")
    endif()

    if(NOT (COPYFILE_FILENAME OR COPYFILE_DEBUG_FILENAME OR COPYFILE_RELEASE_FILENAME))
        message(FATAL_ERROR "At least one of FILENAME, DEBUG_FILENAME, or RELEASE_FILENAME needs to be specified: ${ARGN}")
    endif()

    if(NOT (COPYFILE_TARGET OR COPYFILE_DEPENDENCY_LIST))
        message(FATAL_ERROR "At least one of TARGET or DEPENDENCY_LIST needs to be specified: ${ARGN}")
    endif()

    if(NOT COPYFILE_SOURCE_DIR)
        message(FATAL_ERROR "SOURCE_DIR needs to be specified: ${ARGN}")
    endif()

    if(NOT COPYFILE_DESTINATION_DIR)
        message(FATAL_ERROR "DESTINATION_DIR needs to be specified: ${ARGN}")
    endif()

    if(NOT COPYFILE_PACKAGE_COMPONENT)
        message(FATAL_ERROR "PACKAGE_COMPONENT needs to be specified: ${ARGN}")
    endif()

    # 3. Determine the source and destination filenames

    set(SOURCE_FILENAME "${COPYFILE_FILENAME}")

    if(COPYFILE_DEBUG_FILENAME AND COPYFILE_RELEASE_FILENAME)
        set(SOURCE_FILENAME "$<$<CONFIG:Debug>:${COPYFILE_DEBUG_FILENAME}>$<$<CONFIG:Release>:${COPYFILE_RELEASE_FILENAME}>")
    elseif(COPYFILE_DEBUG_FILENAME)
        set(SOURCE_FILENAME "${COPYFILE_DEBUG_FILENAME}")
    elseif(COPYFILE_RELEASE_FILENAME)
        set(SOURCE_FILENAME "${COPYFILE_RELEASE_FILENAME}")
    endif()

    if(NOT COPYFILE_DESTINATION_FILENAME)
        set(COPYFILE_DESTINATION_FILENAME "${SOURCE_FILENAME}")
    endif()

    # 4. Come up with the command-line for copying the file

    # If only one of DEBUG_FILENAME or RELEASE_FILENAME is specified, we need somehow not do anything for the configuration that was not specified
    # For some reason, i couldn't figure out a way to render the command harmless on the opposite configuration. So instead, we change the output directory
    # to be the "dumpster" folder.
    set(FULL_OUTPUT_PATH "${OPENIPC_OUTPUT_DIRECTORY}/${COPYFILE_DESTINATION_DIR}/${COPYFILE_DESTINATION_FILENAME}")
    if(COPYFILE_DEBUG_FILENAME AND NOT COPYFILE_RELEASE_FILENAME)
        set(FULL_OUTPUT_PATH $<$<CONFIG:Debug>:${FULL_OUTPUT_PATH}>$<$<CONFIG:Release>:${OPENIPC_OUTPUT_DIRECTORY}/dumpster/${COPYFILE_DESTINATION_FILENAME}>)
    elseif(COPYFILE_RELEASE_FILENAME AND NOT COPYFILE_DEBUG_FILENAME)
        set(FULL_OUTPUT_PATH $<$<CONFIG:Release>:${FULL_OUTPUT_PATH}>$<$<CONFIG:Debug>:${OPENIPC_OUTPUT_DIRECTORY}/dumpster/${COPYFILE_DESTINATION_FILENAME}>)
    endif()

    set(COMMAND_TO_RUN copy_if_different "${COPYFILE_SOURCE_DIR}/${SOURCE_FILENAME}" ${FULL_OUTPUT_PATH})

    # 5. Add the necessary CMake command
    if(COPYFILE_TARGET)
        add_custom_command(TARGET ${COPYFILE_TARGET}
                           POST_BUILD
                           COMMAND ${CMAKE_COMMAND} -E ${COMMAND_TO_RUN}
                           DEPENDS "${COPYFILE_SOURCE_DIR}/${SOURCE_FILENAME}"
                           COMMENT "Copying ${COPYFILE_SOURCE_DIR}/${SOURCE_FILENAME}"
                           VERBATIM)
    elseif(COPYFILE_DEPENDENCY_LIST)
        # We need to generate a unique name for the dummy output
        # Get the dependency list's length
        list(LENGTH ${COPYFILE_DEPENDENCY_LIST} LIST_LENGTH)

        set(DummyOutputFilename "NotAThing${COPYFILE_DEPENDENCY_LIST}${LIST_LENGTH}.txt")

        # Now append this dummy output file to the dependency list. Note that we're inside
        # a CMake function, so this will only affect the list inside this function's scope.
        # To affect the parent scope, we use the "set()" command with the PARENT_SCOPE option
        list(APPEND ${COPYFILE_DEPENDENCY_LIST} "${DummyOutputFilename}")
        set(${COPYFILE_DEPENDENCY_LIST} "${${COPYFILE_DEPENDENCY_LIST}}" PARENT_SCOPE)

        add_custom_command(OUTPUT ${DummyOutputFilename}
                           COMMAND ${CMAKE_COMMAND} -E ${COMMAND_TO_RUN}
                           COMMAND ${CMAKE_COMMAND} -E touch ${DummyOutputFilename}
                           DEPENDS "${COPYFILE_SOURCE_DIR}/${SOURCE_FILENAME}"
                           COMMENT "Copying ${COPYFILE_SOURCE_DIR}/${SOURCE_FILENAME}"
                           VERBATIM)
    endif()

    # 6. Install the file
    if(COPYFILE_FILENAME)
        install(FILES "${OPENIPC_OUTPUT_DIRECTORY}/${COPYFILE_DESTINATION_DIR}/${COPYFILE_DESTINATION_FILENAME}"
                DESTINATION ${DPL_OPENIPC_ROOT}/${COPYFILE_DESTINATION_DIR} COMPONENT ${COPYFILE_PACKAGE_COMPONENT})
    else()
        if(COPYFILE_DEBUG_FILENAME)
            install(FILES "${OPENIPC_OUTPUT_DIRECTORY}/${COPYFILE_DESTINATION_DIR}/${COPYFILE_DESTINATION_FILENAME}"
                    CONFIGURATIONS Debug
                    DESTINATION ${DPL_OPENIPC_ROOT}/${COPYFILE_DESTINATION_DIR} COMPONENT ${COPYFILE_PACKAGE_COMPONENT})
        endif()
        if(COPYFILE_RELEASE_FILENAME)
            install(FILES "${OPENIPC_OUTPUT_DIRECTORY}/${COPYFILE_DESTINATION_DIR}/${COPYFILE_DESTINATION_FILENAME}"
                    CONFIGURATIONS Release RelWithDebInfo MinSizeRel
                    DESTINATION ${DPL_OPENIPC_ROOT}/${COPYFILE_DESTINATION_DIR} COMPONENT ${COPYFILE_PACKAGE_COMPONENT})
        endif()
    endif()
endfunction(CopyFile)

#################################################################################################
# Description:
#   Sets up the proper CMake commands for copying the given directory to the destination directory.
#   At build time, the directory is copied to <build directory>/DESTINATION_DIR.
#   At installation time, the directory is copied to <installation directory>/DESTINATION_DIR
#
# Required Parameters:
#   SOURCE_DIR        The directory that contains the directory to copy
#   DESTINATION_DIR   The directory (relative to the build and installation directories) where the directory should be copied to
#   PACKAGE_COMPONENT The CPack component that this directory should go in
#   SOURCE_DIRNAME    The name of the directory to copy
#
#   One of:
#      TARGET          The CMake target to which this operation should be attached as a POST_BUILD step
#      DEPENDENCY_LIST A CMake list to which a unique output file will be added. Use this file dependency to add
#                      a list of CopyDirectory operations as a dependency to a custom CMake target
#
# Optional Parameters:
#   DESTINATION_DIRNAME The name of the directory after it's copied
#
# Example:
#   CopyDirectory(SOURCE_DIR "${OPENIPC_ROOT}"
#                 SOURCE_DIRNAME "TestFolder"
#                 TARGET "MyExecutable"
#                 DESTINATION_DIR "Bin"
#                 DESTINATION_DIRNAME "AwesomeFolder"
#                 PACKAGE_COMPONENT openipc)
#
#   This will copy ${OPENIPC_ROOT}/TestFolder to <build directory>/Bin/AwesomeFolder
#   and <installation directory>/Bin/AwesomeFolder and will package it
#   in the "openipc" CPack component. Copying it to the build directory will be done as a POST_BUILD
#   step for the "MyExecutable" project
#
# Important Considerations:
#   - When TARGET is specified, the copying operation is specified as a POST_BUILD step to the given target.
#     This means that it will only be copied again when the project is built. If the project is up-to-date,
#     the directory will not be copied. So whenever you change the original directory, you'll need to rebuild this project.
#
#   - When DEPENDENCY_LIST is specified the copying operation is performed with every build
#
#   - The directory that is installed is the copy that was put in the build directory. Therefore any changes that are made
#     to the directory in the build directory will be reflected in the installation directory if, and only if, TARGET was
#     specified. See the above two considerations for an explanation
#
#################################################################################################
function(CopyDirectory)
    # 1. Parse the arguments
    set(Options )
    set(SingleValueArgs TARGET SOURCE_DIR SOURCE_DIRNAME DESTINATION_DIR DESTINATION_DIRNAME PACKAGE_COMPONENT DEPENDENCY_LIST)
    set(MultiValueArgs )
    cmake_parse_arguments(COPYDIR "${Options}" "${SingleValueArgs}" "${MultiValueArgs}" ${ARGN})

    # 2. Figure out if all the required arguments have been specified and no conflicting arguments are there

    if(NOT COPYDIR_SOURCE_DIR)
        message(FATAL_ERROR "SOURCE_DIR needs to be specified: ${ARGN}")
    endif()

    if(NOT COPYDIR_SOURCE_DIRNAME)
        message(FATAL_ERROR "SOURCE_DIRNAME needs to be specified: ${ARGN}")
    endif()

    if(NOT COPYDIR_DESTINATION_DIR)
        message(FATAL_ERROR "DESTINATION_DIR needs to be specified: ${ARGN}")
    endif()

    if(NOT (COPYDIR_TARGET OR COPYDIR_DEPENDENCY_LIST))
        message(FATAL_ERROR "At least one of TARGET or DEPENDENCY_LIST needs to be specified: ${ARGN}")
    endif()

    if(NOT COPYDIR_PACKAGE_COMPONENT)
        message(FATAL_ERROR "PACKAGE_COMPONENT needs to be specified: ${ARGN}")
    endif()

    # 3. Determine the destination dirname

    set(DESTINATION_DIRNAME "${COPYDIR_SOURCE_DIRNAME}")
    if(COPYDIR_DESTINATION_DIRNAME)
        set(DESTINATION_DIRNAME "${COPYDIR_DESTINATION_DIRNAME}")
    endif()

    # 4. Come up with the command-line for copying the directory
    set(OUTPUT_DIR "${OPENIPC_OUTPUT_DIRECTORY}/${COPYDIR_DESTINATION_DIR}/${DESTINATION_DIRNAME}")
    set(COMMAND_TO_RUN copy_directory "${COPYDIR_SOURCE_DIR}/${COPYDIR_SOURCE_DIRNAME}" "${OUTPUT_DIR}")

    if(WIN32)
        set(REMOVE_READONLY_ATTR attrib -R /S /D)
    elseif(UNIX)
        set(REMOVE_READONLY_ATTR chmod -R u+w)
    endif()

    # 5. Add the necessary CMake command
    if(COPYDIR_TARGET)
        add_custom_command(TARGET ${COPYDIR_TARGET}
                           POST_BUILD
                           COMMAND ${CMAKE_COMMAND} -E ${COMMAND_TO_RUN}
                           COMMAND ${REMOVE_READONLY_ATTR} ${OUTPUT_DIR}/*
                           DEPENDS "${COPYDIR_SOURCE_DIR}/${COPYDIR_SOURCE_DIRNAME}"
                           COMMENT "Copying ${COPYDIR_SOURCE_DIR}/${COPYDIR_SOURCE_DIRNAME} to ${OUTPUT_DIR}")
    elseif(COPYDIR_DEPENDENCY_LIST)
        # We need to generate a unique name for the dummy output
        # Get the dependency list's length
        list(LENGTH ${COPYDIR_DEPENDENCY_LIST} LIST_LENGTH)

        set(DummyOutputFilename "NotAThing${COPYDIR_DEPENDENCY_LIST}${LIST_LENGTH}.txt")

        # Now append this dummy output file to the dependency list. Note that we're inside
        # a CMake function, so this will only affect the list inside this function's scope.
        # To affect the parent scope, we use the "set()" command with the PARENT_SCOPE option
        list(APPEND ${COPYDIR_DEPENDENCY_LIST} "${DummyOutputFilename}")
        set(${COPYDIR_DEPENDENCY_LIST} "${${COPYDIR_DEPENDENCY_LIST}}" PARENT_SCOPE)

        add_custom_command(OUTPUT ${DummyOutputFilename}
                           COMMAND ${CMAKE_COMMAND} -E ${COMMAND_TO_RUN}
                           COMMAND ${REMOVE_READONLY_ATTR} ${OUTPUT_DIR}/*
                           DEPENDS "${COPYDIR_SOURCE_DIR}/${COPYDIR_SOURCE_DIRNAME}"
                           COMMENT "Copying ${COPYDIR_SOURCE_DIR}/${COPYDIR_SOURCE_DIRNAME} to ${OUTPUT_DIR}")
    endif()

    # 6. Install the directory
    install(DIRECTORY "${OUTPUT_DIR}"
            DESTINATION ${DPL_OPENIPC_ROOT}/${COPYDIR_DESTINATION_DIR} COMPONENT ${COPYDIR_PACKAGE_COMPONENT})
endfunction(CopyDirectory)

#################################################################################################
# Description:
#   Adds the given files and/or directories to the list of things that need to be present in a source release.
#   The given paths are relative to the caller's current directory
#
# Required Parameters:
#   One (or more) of:
#      FILES              The list of files that need to be shipped in a source release
#      DIRECTORIES        The list of directories that need to be shipped in a source release
#      SYSTEM_DIRECTORIES The list of directories that contain system-specific directories (i.e. "windows", "linux", or "darwin")
#
# Optional Parameters:
#   DESTINATION_DIR The directory (relative to the source directory) where these files and/or directories
#                   should go
#   EXCLUDE_PATTERN A CMake pattern that should be used to exclude files from directories
#                   (Only applied to the directories given to the DIRECTORIES parameter)
#
# Example:
#   ShipSources(FILES "myfile.cpp" "myotherfile.h" "CMakeLists.txt"
#               DIRECTORIES "Docs")
#
#################################################################################################
function(ShipSources)
    # 1. Parse the arguments
    set(Options )
    set(SingleValueArgs DESTINATION_DIR EXCLUDE_PATTERN)
    set(MultiValueArgs FILES DIRECTORIES SYSTEM_DIRECTORIES)
    cmake_parse_arguments(PARAM "${Options}" "${SingleValueArgs}" "${MultiValueArgs}" ${ARGN})

    # 2. Validate the options
    if(NOT (PARAM_FILES OR PARAM_DIRECTORIES OR PARAM_SYSTEM_DIRECTORIES))
        message(FATAL_ERROR "At least one of FILES, DIRECTORIES, or SYSTEM_DIRECTORIES needs to be specified: ${ARGN}")
    endif()

    if(PARAM_EXCLUDE_PATTERN AND (NOT PARAM_DIRECTORIES))
        message(WARNING "EXCLUDE_PATTERN was specified, whereas DIRECTORIES was not specified. EXCLUDE_PATTERN will have no effect: ${ARGN}")
    endif()

    # 3. Get the full path to the root source directory (this is the perforce workspace directory)
    get_filename_component(MAIN_SOURCE_PATH "${MAIN_ROOT}" REALPATH)
    set(MAIN_SOURCE_PATH "${MAIN_SOURCE_PATH}/")
    string(LENGTH "${MAIN_SOURCE_PATH}" MAIN_SOURCE_PATH_LENGTH)

    set(FILE_COPY_COMMAND
"
file(COPY \"@SOURCE_FILE@\"
     DESTINATION \"\${SOURCE_RELEASE_DIRECTORY}/@OUTPUT_DIRECTORY@\"
     NO_SOURCE_PERMISSIONS
     )
"
    )

    set(DIR_COPY_COMMAND
"
file(COPY \"@SOURCE_DIRECTORY@\"
     DESTINATION \"\${SOURCE_RELEASE_DIRECTORY}/@OUTPUT_DIRECTORY@\"
     NO_SOURCE_PERMISSIONS
     @EXTRA_PARAMETERS@)
"
    )

    # 4. Parse the given list of files
    if(PARAM_FILES)
        foreach(PARAM_FILE IN LISTS PARAM_FILES)
            # Get the absolute path to the file
            get_filename_component(PARAM_FILE_ABS_PATH "${PARAM_FILE}" REALPATH)

            if(PARAM_DESTINATION_DIR)
                set(DESTINATION_DIR ${PARAM_DESTINATION_DIR})
            else()
                # Make sure it's inside the root source directory
                string(FIND "${PARAM_FILE_ABS_PATH}" "${MAIN_SOURCE_PATH}" IS_RELATIVE)
                if(NOT ("${IS_RELATIVE}" STREQUAL "0"))
                    message(FATAL_ERROR "Cannot process \"${PARAM_FILE}\" because it is not in the project tree")
                endif()

                get_filename_component(PARAM_DIR_ABS_PATH "${PARAM_FILE_ABS_PATH}" DIRECTORY)
                set(PARAM_DIR_ABS_PATH "${PARAM_DIR_ABS_PATH}/")

                # Get the path to the directory relative to the root source directory
                string(SUBSTRING "${PARAM_DIR_ABS_PATH}" ${MAIN_SOURCE_PATH_LENGTH} -1 DESTINATION_DIR)
            endif()

            # Get the name of the file
            get_filename_component(PARAM_FILE_NAME "${PARAM_FILE}" NAME)

            if(NOT DESTINATION_DIR)
                set(DESTINATION_DIR ".")
            endif()

            set(SOURCE_FILE ${PARAM_FILE_ABS_PATH})
            set(OUTPUT_DIRECTORY ${DESTINATION_DIR})

            string(CONFIGURE "${FILE_COPY_COMMAND}" COPY_COMMAND @ONLY)
            set_property(GLOBAL APPEND PROPERTY "SOURCE_RELEASE_COMMANDS" "${COPY_COMMAND}")
            set_property(GLOBAL APPEND PROPERTY "SOURCE_RELEASE_INPUTS" "${SOURCE_FILE}")
            set_property(GLOBAL APPEND PROPERTY "SOURCE_RELEASE_OUTPUTS" "${OUTPUT_DIRECTORY}/${PARAM_FILE_NAME}")
        endforeach()
    endif()

    if(PARAM_DIRECTORIES)
        foreach(PARAM_DIRECTORY IN LISTS PARAM_DIRECTORIES)
            # Get the absolute path to the directory
            get_filename_component(PARAM_DIRECTORY_ABS_PATH "${PARAM_DIRECTORY}" REALPATH)

            if(PARAM_DESTINATION_DIR)
                set(DESTINATION_DIR ${PARAM_DESTINATION_DIR})
            else()
                # Make sure it's inside the root source directory
                string(FIND "${PARAM_DIRECTORY_ABS_PATH}" "${MAIN_SOURCE_PATH}" IS_RELATIVE)
                if(NOT ("${IS_RELATIVE}" STREQUAL "0"))
                    message(FATAL_ERROR "Cannot process \"${PARAM_DIRECTORY}\" because it is not in the project tree")
                endif()

                get_filename_component(PARAM_PARENT_DIR_ABS_PATH "${PARAM_DIRECTORY_ABS_PATH}" DIRECTORY)
                set(PARAM_PARENT_DIR_ABS_PATH "${PARAM_PARENT_DIR_ABS_PATH}/")

                # Get the path to the directory relative to the root source directory
                string(SUBSTRING "${PARAM_PARENT_DIR_ABS_PATH}" ${MAIN_SOURCE_PATH_LENGTH} -1 DESTINATION_DIR)
            endif()

            # Get the name of the directory
            get_filename_component(PARAM_DIRECTORY_NAME "${PARAM_DIRECTORY}" NAME)

            if(NOT DESTINATION_DIR)
                set(DESTINATION_DIR ".")
            endif()

            if(PARAM_EXCLUDE_PATTERN)
                set(EXTRA_PARAMETERS "PATTERN \"${PARAM_EXCLUDE_PATTERN}\" EXCLUDE")
            endif()

            set(SOURCE_DIRECTORY ${PARAM_DIRECTORY_ABS_PATH})
            set(OUTPUT_DIRECTORY ${DESTINATION_DIR})

            string(CONFIGURE "${DIR_COPY_COMMAND}" COPY_COMMAND @ONLY)
            set_property(GLOBAL APPEND PROPERTY "SOURCE_RELEASE_COMMANDS" "${COPY_COMMAND}")
            set_property(GLOBAL APPEND PROPERTY "SOURCE_RELEASE_INPUTS" "${SOURCE_DIRECTORY}")
            set_property(GLOBAL APPEND PROPERTY "SOURCE_RELEASE_OUTPUTS" "${OUTPUT_DIRECTORY}/${PARAM_DIRECTORY_NAME}")

            unset(EXTRA_PARAMETERS)
        endforeach()
    endif()

    if(PARAM_SYSTEM_DIRECTORIES)
        foreach(PARAM_SYSTEM_DIRECTORY IN LISTS PARAM_SYSTEM_DIRECTORIES)
            if(NOT CONFIG_MULTIPLATFORM_SOURCE_RELEASE)
                ShipSources(DIRECTORIES "${PARAM_SYSTEM_DIRECTORY}/${SYSTEM_NAME}")
            else()
                ShipSources(DIRECTORIES "${PARAM_SYSTEM_DIRECTORY}/darwin"
                                        "${PARAM_SYSTEM_DIRECTORY}/linux"
                                        "${PARAM_SYSTEM_DIRECTORY}/windows")
            endif()
        endforeach()
    endif()
endfunction(ShipSources)

#################################################################################################
# Description:
#   Creates a project (called GenerateSourceRelease) that, when built, will copy all the sources
#   that were given to ShipSources() to the given directory.
#   This project is explicitly designed not to be built when the whole solution is built.
#   The user will have to build it manually
#
# Required Parameters:
#   SOURCE_RELEASE_DIRECTORY The directory where the sources should be copied to
#
# Example:
#   DefineSourceReleaseProject("C:/Awesomeness")
#
#################################################################################################
function(DefineSourceReleaseProject SOURCE_RELEASE_DIRECTORY)
    get_property(SOURCE_RELEASE_COMMANDS GLOBAL PROPERTY "SOURCE_RELEASE_COMMANDS")
    get_property(SOURCE_RELEASE_OUTPUTS  GLOBAL PROPERTY "SOURCE_RELEASE_OUTPUTS")
    get_property(SOURCE_RELEASE_INPUTS   GLOBAL PROPERTY "SOURCE_RELEASE_INPUTS")

    # Remove duplicates
    list(LENGTH SOURCE_RELEASE_COMMANDS NUM_COMMANDS)
    math(EXPR MAX_INDEX "${NUM_COMMANDS} - 1")
    foreach(INDEX RANGE ${MAX_INDEX})
        list(GET SOURCE_RELEASE_COMMANDS ${INDEX} CURRENT_COMMAND)
        list(FIND FILTERED_SOURCE_RELEASE_COMMANDS "${CURRENT_COMMAND}" FOUND_INDEX)

        if("${FOUND_INDEX}" LESS "0")
            list(GET SOURCE_RELEASE_OUTPUTS ${INDEX} CURRENT_OUTPUT)
            list(GET SOURCE_RELEASE_INPUTS  ${INDEX} CURRENT_INPUT)

            list(APPEND FILTERED_SOURCE_RELEASE_COMMANDS "${CURRENT_COMMAND}")
            list(APPEND FILTERED_SOURCE_RELEASE_OUTPUTS  "${CURRENT_OUTPUT}")
            list(APPEND FILTERED_SOURCE_RELEASE_INPUTS   "${CURRENT_INPUT}")
        endif()
    endforeach()

    file(WRITE ${CMAKE_BINARY_DIR}/GenerateSourceRelease.cmake ${FILTERED_SOURCE_RELEASE_COMMANDS})

    foreach(SOURCE_RELEASE_OUTPUT IN LISTS FILTERED_SOURCE_RELEASE_OUTPUTS)
        list(APPEND ACTUAL_OUTPUTS "${SOURCE_RELEASE_DIRECTORY}/${SOURCE_RELEASE_OUTPUT}")
    endforeach()

    add_custom_command(OUTPUT ${ACTUAL_OUTPUTS}
                       COMMAND ${CMAKE_COMMAND} -DSOURCE_RELEASE_DIRECTORY=${SOURCE_RELEASE_DIRECTORY} -P ${CMAKE_BINARY_DIR}/GenerateSourceRelease.cmake
                       DEPENDS ${FILTERED_SOURCE_RELEASE_INPUTS}
                       COMMENT ""
                       VERBATIM)

    add_custom_target(GenerateSourceRelease DEPENDS ${ACTUAL_OUTPUTS})

endfunction(DefineSourceReleaseProject)

macro(IncludeIngredient ingredient)
    include("${MAIN_ROOT}/CMakeIncludes/recipes/ingredients/${ingredient}.cmake")
    ShipSources(FILES "${MAIN_ROOT}/CMakeIncludes/recipes/ingredients/${ingredient}.cmake")
endmacro(IncludeIngredient)

macro(ShipLicenseFile LicenseFileName)
    set(FileName "${LicenseFileName}")
    if(NOT "${FileName}" STREQUAL "")
        message("Checking: ${OPENIPC_ROOT}/${FileName}")
        if(NOT EXISTS "${OPENIPC_ROOT}/${FileName}")
            set(FileName "LICENSE")
            message("Checking: ${OPENIPC_ROOT}/${FileName}")
            if(NOT EXISTS "${OPENIPC_ROOT}/${FileName}")
                message(FATAL_ERROR "Cannot find the specified license file")
            endif()
        endif()

        set(TemporaryLicenseLocation "${CMAKE_BINARY_DIR}/OpenIPC/LICENSE")
        configure_file("${OPENIPC_ROOT}/${FileName}" "${TemporaryLicenseLocation}" COPYONLY)
        ShipSources(FILES "${TemporaryLicenseLocation}" DESTINATION_DIR "OpenIPC")
        install(FILES "${TemporaryLicenseLocation}"
                DESTINATION "${DPL_OPENIPC_ROOT}"
                COMPONENT openipc)
    endif()
endmacro(ShipLicenseFile)

# Macro defined for precompiled headers
MACRO(ADD_PRECOMPILED_HEADER PrecompiledHeader PrecompiledSource SourcesVar)
    if(WIN32)
        GET_FILENAME_COMPONENT(PrecompiledBasename ${PrecompiledHeader} NAME_WE)
        set(PrecompiledBinary "${CMAKE_CURRENT_BINARY_DIR}/${CMAKE_BUILD_TYPE}/${PrecompiledBasename}.pch")
        set(Sources ${${SourcesVar}})

        SET_SOURCE_FILES_PROPERTIES(${Sources}
                                    PROPERTIES COMPILE_FLAGS "/Yu\"${PrecompiledHeader}\" /FI\"${PrecompiledHeader}\" /Fp\"${PrecompiledBinary}\""
                                            OBJECT_DEPENDS "${PrecompiledBinary}")

        SET_SOURCE_FILES_PROPERTIES(${PrecompiledSource}
                                    PROPERTIES COMPILE_FLAGS "/Yc\"${PrecompiledHeader}\" /Fp\"${PrecompiledBinary}\""
                                            OBJECT_OUTPUTS "${PrecompiledBinary}")
    endif()

    # add the PCH source file to the project's sources
    set(${SourcesVar} ${${SourcesVar}} ${PrecompiledSource})
ENDMACRO(ADD_PRECOMPILED_HEADER)

MACRO(INSTALL_PDB TargetName)
    if(WIN32)
        get_target_property(INSTALL_PDB_TARGET_OUTPUT_NAME ${TargetName} OUTPUT_NAME)

        if (NOT INSTALL_PDB_TARGET_OUTPUT_NAME)
            set(INSTALL_PDB_TARGET_OUTPUT_NAME ${TargetName})
        endif()

        # First copy the PDB file to a directory that the install() directive can access
        ADD_CUSTOM_COMMAND(
            TARGET ${TargetName}
            POST_BUILD
            COMMAND ${CMAKE_COMMAND} -E copy_if_different
            "$<TARGET_FILE_DIR:${TargetName}>/${INSTALL_PDB_TARGET_OUTPUT_NAME}.pdb"
            "${CMAKE_CURRENT_BINARY_DIR}/${INSTALL_PDB_TARGET_OUTPUT_NAME}.pdb"
        )
        INSTALL(FILES ${CMAKE_CURRENT_BINARY_DIR}/${INSTALL_PDB_TARGET_OUTPUT_NAME}.pdb DESTINATION "DebugSymbols" COMPONENT debug_symbols)
    endif()
ENDMACRO(INSTALL_PDB)

MACRO(UNCRUSTIFY TargetName)
    if(WIN32 AND UNCRUSTIFY_FOUND)
        set(CSOURCES ${ARGN})
        list(FILTER CSOURCES INCLUDE REGEX "(h|hpp|cpp)$") 
        add_custom_command(
            TARGET ${TargetName}
            PRE_BUILD
            COMMAND ${UNCRUSTIFY_CMD} -c "${OPENIPC_ROOT}/uncrustify.cfg" --replace --no-backup ${CSOURCES}
            WORKING_DIRECTORY "${CMAKE_CURRENT_SOURCE_DIR}"
        )
    endif()
ENDMACRO(UNCRUSTIFY)

MACRO(COG TargetName)
    if(WIN32)
        set(CSOURCES ${ARGN})
        list(FILTER CSOURCES INCLUDE REGEX "(h|hpp|cpp)$")
        add_custom_command(
            TARGET ${TargetName}
            PRE_BUILD
            COMMAND ${PYTHON_EXECUTABLE} -c "import sys;print(r'${OPENIPC_ROOT}/Tools/Cog');sys.path.append(r'${OPENIPC_ROOT}/Tools/Cog');from cogapp import __main__" -r ${CSOURCES}
            WORKING_DIRECTORY "${CMAKE_CURRENT_SOURCE_DIR}"
        )
    endif()
ENDMACRO(COG)