/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
///
/// @brief Declaration of functions for managing mutexes in a cross-platform
///        manner.
///
/// These are the functions that the threadmutex.cpp module uses.  They are
/// defined here so that other modules, such as _HandleCollection.h, can also
/// use mutexes.
///
/// @warning
/// These functions are to be used only inside the InternalUtils library!
///
//////////////////////////////////////////////////////////////////////////////

#ifndef ___PRIVATEMUTEX_H__
#define ___PRIVATEMUTEX_H__

#include "threadtypes.h"

namespace InternalUtilsPrivate
{

    //////////////////////////////////////////////////////////////////////////
    //  Function: _CreateMutex
    //
    /// @brief Creates a thread synchronization object.
    ///
    /// This helper function creates a thread synchronization object in such
    /// a way that is operating system-agnostic (that is, controlled by
    /// conditional macros during compilation).
    ///
    /// Although the term "mutex" is used for "mutually exclusive", each
    /// operating system will use the most appropriate synchronization object.
    ///
    /// @par Assumptions:
    ///   The operating system supports thread synchronization objects.
    ///
    /// @par Side Effects:
    ///   A synchronization object is created.
    ///
    /// @param ppSyncObject
    ///     Returns a pointer to the synchronization object.
    ///
    /// @return (int):
    ///     Returns 0 if successful; otherwise, returns an error code (errno).
    //////////////////////////////////////////////////////////////////////////
    int _CreateMutex(void** ppSyncObject);


    //////////////////////////////////////////////////////////////////////////
    //  Function: _DestroyMutex
    //
    /// @brief Destroys a thread synchronization object.
    ///
    /// This helper function destroys a thread synchronization object that
    /// was created by the _CreateMutex() function.  After the object is
    /// destroyed, it can no longer be used.
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   A synchronization object is destroyed.
    ///
    /// @param pSyncObject
    ///     A pointer to the synchronization object to be destroyed.
    ///
    /// @return (int):
    ///     Returns 0 if successful; otherwise, returns an error code (errno).
    //////////////////////////////////////////////////////////////////////////
    int _DestroyMutex(void* pSyncObject);


    //////////////////////////////////////////////////////////////////////////
    //  Function: _LockMutex
    //
    /// @brief Acquire a lock on the specified thread synchronization object,
    ///        with an optional timeout.
    ///
    /// This helper function waits the specified number of milliseconds to
    /// acquire a lock on the specified thread synchronization object.  If the
    /// timeout is InternalUtils::TIMEOUT_INFINITE, then this function
    /// will block indefinitely until the lock is acquired.
    ///
    /// If the lock is successfully acquired, call the _UnlockMutex() function
    /// to release the lock.  Calling _LockMutex() increments an internal
    /// counter on the lock so a thread will not deadlock itself acquiring a
    /// lock it already has.  There must be a corresponding call to
    /// _UnlockMutex() for each call to _LockMutex().
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   If the timeout period has not expired, the lock is acquired.
    ///
    /// @param pSyncObject
    ///     A pointer to the synchronization object to lock.
    /// @param milliseconds
    ///     The number of milliseconds to wait to acquire the lock.  Pass
    ///     InternalUtils::TIMEOUT_INFINITE to wait forever.
    ///
    /// @return (int):
    ///     Returns 0 if successful; otherwise, returns an error code (errno).
    ///     Returns ETIMEDOUT if the lock was not acquired within the timeout
    ///     period.
    //////////////////////////////////////////////////////////////////////////
    int _LockMutex(void* pSyncObject, InternalUtils::TimeoutDelay milliseconds);


    //////////////////////////////////////////////////////////////////////////
    //  Function: _UnlockMutex
    //
    /// @brief Release a previously acquired lock on a thread synchronization
    ///        object.
    ///
    /// This helper function releases a lock on a thread synchronization
    /// object that was previously acquired with the _LockMutex() function.
    /// For each call to _LockMutex(), there must be a call to _UnlockMutex().
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   The lock is released.
    ///
    /// @param pSyncObject
    ///     A pointer to the synchronization object to release.
    ///
    /// @return (int):
    ///     Returns 0 if successful; otherwise, returns an error code (errno).
    //////////////////////////////////////////////////////////////////////////
    int _UnlockMutex(void* pSyncObject);


    //////////////////////////////////////////////////////////////////////////
    //  Function: _TryLockMutex
    //
    /// @brief Try to acquire a lock on the specified thread synchronization
    ///        object.
    ///
    /// This helper function tries to acquire a lock on a thread
    /// synchronization object.  If successful, the object is locked as though
    /// _LockMutex() was called.  If unsuccessful, an error code is returned
    /// and the lock is not acquired.
    ///
    /// If the lock is successfully acquired, call the _UnlockMutex() function
    /// to release the lock.
    ///
    /// If a thread calls _TryLockMutex() on a thread synchronization object
    /// it has already acquired, the call is successful and the internal
    /// counter on the object is incremented (just like _LockMutex()).  The
    /// thread must call _UnlockMutex() once for each call to _TryLockMutex().
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   If the lock is available, the lock is acquired.
    ///
    /// @param pSyncObject
    ///     A pointer to the synchronization object to try to lock.
    ///
    /// @return (int):
    ///     Returns 0 if successful; otherwise, returns an error code (errno).
    ///     Returns EBUSY if the lock was not acquired (the lock was taken by
    ///     another thread).
    //////////////////////////////////////////////////////////////////////////
    int _TryLockMutex(void* pSyncObject);

} // end namespace InternalUtilsPrivate

#endif // ___PRIVATEMUTEX_H__
