/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
///
/// @brief Declaration of functions that are operating system-specific in
///        their implementation for accessing dynamic libraries.
///
//////////////////////////////////////////////////////////////////////////////

#ifndef __ACCESSDLLS_H__
#define __ACCESSDLLS_H__

#include <string>

//vvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvvv
// Handle the differences between Windows and Linux for loading and using
// dynamic libraries.  Note this code is build with Unicode in Windows but
// in ANSI in Linux.
#ifdef _MSC_VER

#include <wtypes.h>

// *********** WINDOWS-SPECIFIC CODE ***********
typedef HMODULE IPC_DLL_HANDLE;       ///< Handle to a loaded DLL file.
//typedef LPCTSTR IPC_DLL_NAME_TYPE;    ///< Represents the name of a DLL.
typedef LPCSTR  IPC_PROC_NAME_TYPE;   ///< Represents the name of a function in a DLL.

#else

// *********** LINUX-SPECIFIC CODE ***********
typedef void*       IPC_DLL_HANDLE;     ///< Handle to a loaded DLL file.
typedef const char* IPC_PROC_NAME_TYPE; ///< Represents the name of a function in a DLL.

#endif
//^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^^


namespace InternalUtils
{

    //////////////////////////////////////////////////////////////////////////
    //  Function: getdllerrormessage
    //
    /// @brief Retrieve the error message describing the reason for the last
    ///        failure to load a DLL/shared library.
    ///
    /// This function converts the last error caused during loading a library
    /// file into human-readable text.  This is specific to library load
    /// errors.
    ///
    /// On Windows, the returned string includes the error code at the
    /// beginning of the string ("(0xNNN) Text of error message.").
    ///
    /// On Linux, no error code is available.
    ///
    /// @par Assumptions:
    ///   An error occurred during the load of a DLL/shared library.
    ///
    /// @par Side Effects:
    ///   On Linux, the last error message associated with loading a library
    ///   is cleared.
    ///
    /// @par Parameters:
    ///   None.
    ///
    /// @return (std::string): A string containing the error message.
    //////////////////////////////////////////////////////////////////////////
    std::string getdllerrormessage();


    //////////////////////////////////////////////////////////////////////////
    //  Function: loaddll
    //
    /// @brief Load a dynamic library given its name.
    ///
    /// This function hides the details of loading a DLL/shared library.
    ///
    /// @note
    ///   For every successful call to loaddll(), a corresponding call must
    ///   be made to freedll().
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   The specified library is loaded.
    ///
    /// @param[in] dllName
    ///     The name of the DLL to load.  If this is NULL, the returned handle
    ///     is NULL but no error conditions is set.
    ///
    /// @return (IPC_DLL_HANDLE): The handle to the loaded library.  If NULL is
    ///                           returned, an error occurred.
    //////////////////////////////////////////////////////////////////////////
    IPC_DLL_HANDLE loaddll(
        const std::string& dllName);
    
    
    //////////////////////////////////////////////////////////////////////////
    //  Function: freedll
    //
    /// @brief Releases a specified dynamic library handle.
    ///
    /// This function frees the specified dynamic library.
    ///
    /// @note
    ///   For every successful call to loaddll(), a corresponding call must
    ///   be made to freedll().
    ///
    /// @par Assumptions:
    ///   The library was previously loaded with loaddll().
    ///
    /// @par Side Effects:
    ///   The library handle is released.
    ///
    /// @param[in] libraryHandle
    ///     The handle to the library to release.  If this is NULL, nothing
    ///     happens.
    ///
    /// @return (bool): true if the DLL was unloaded successfully; otherwise,
    ///                 returns false if there was an error.
    //////////////////////////////////////////////////////////////////////////
    bool freedll(
        IPC_DLL_HANDLE libraryHandle);


    //////////////////////////////////////////////////////////////////////////
    //  Function: getprocedurefromdll
    //
    /// @brief Retrieves a procedure from a dynamic library so it can be called.
    ///
    /// This function retrieves a pointer to a function from the specified
    /// dynamic library.  The returned pointer is of type void* and needs to
    /// be cast to an appropriate function pointer in order to be called.
    ///
    /// @warning
    ///   If the @a libraryHandle and/or @a procHandle parameters are NULL,
    ///   this function returned NULL but no error condition is set.
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   None.
    ///
    /// @param[in] libraryHandle
    ///     The handle to the library to access.  Cannot be NULL.
    ///
    /// @param[in] procName
    ///     Name of the function to get.  Cannot be NULL.
    ///
    /// @return (void*): A pointer to the desired function.  Returns NULL if
    ///                  the function is not found.
    //////////////////////////////////////////////////////////////////////////
    void* getprocedurefromdll(
        IPC_DLL_HANDLE     libraryHandle,
        IPC_PROC_NAME_TYPE procName);


    //////////////////////////////////////////////////////////////////////////
    //  Function: getcurrentmodulepath
    //
    /// @brief Retrieves the path of the current module that called this
    ///        function.
    ///
    /// This function retrieves the path and filename of the dynamic library or
    /// executable that has linked to this static library and called this
    /// function.
    ///
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   None.
    ///
    /// @par Parameters
    ///   None.
    ///
    /// @return (std::string): The path to the dynamic library or executable.
    //////////////////////////////////////////////////////////////////////////
    std::string getcurrentmodulepath();

} // end InternalUtils namespace


#endif // __ACCESSDLLS_H__
