/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
///
/// @brief Implementation of functions associated with file operations.
///
/// Functions:
///  - InternalUtils::openfile()
///  - InternalUtils::closefile()
///  - InternalUtils::getfilesize()
///  - InternalUtils::writefile()
///  - InternalUtils::readfile()
///  - InternalUtils::flushfile()
///
//////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include <errno.h>

#ifdef _WIN32
    #include <codecvt>
#endif

#include "fileops.h"

namespace InternalUtils
{

    //////////////////////////////////////////////////////////////////////////
    //  Function: openfile
    //
    //////////////////////////////////////////////////////////////////////////
    int openfile(FILE** pstream, const char* filename, const char* mode)
    {
        int err = EINVAL;
        if (pstream != NULL && filename != NULL && mode != NULL)
        {
            *pstream = NULL;

#ifdef _WIN32
            std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>, wchar_t> converter;
            err = _wfopen_s(pstream, converter.from_bytes(filename).c_str(), converter.from_bytes(mode).c_str());
#else
            err = 0;
            FILE *pFile = fopen(filename, mode);
            if (pFile != NULL)
            {
                *pstream = pFile;
            }
            else
            {
                err = errno;
            }
#endif
        }

        return err;
    }


    //////////////////////////////////////////////////////////////////////////
    //  Function: openfile_unbuffered
    //
    //////////////////////////////////////////////////////////////////////////
    int openfile_unbuffered(FILE** pstream, const char* filename, const char* mode)
    {
        int err = EINVAL;
        if (pstream != NULL && filename != NULL && mode != NULL)
        {
            *pstream = NULL;

            FILE *pFile = NULL;
#ifdef _WIN32
            std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>, wchar_t> converter;
            err = _wfopen_s(&pFile, converter.from_bytes(filename).c_str(), converter.from_bytes(mode).c_str());
            if (err == 0)

#else

            err = 0;
            pFile = fopen(filename, mode);
            if (pFile != NULL)
#endif
            {
                err = setvbuf(pFile, NULL, _IONBF, 0);
                if (err == 0)
                {
                    *pstream = pFile;
                }
                else
                {
                    fclose(pFile);
                }
            }
            else
            {
                err = errno;
            }

        }

        return err;
    }


    //////////////////////////////////////////////////////////////////////////
    //  Function: closefile
    //
    //////////////////////////////////////////////////////////////////////////
    int closefile(FILE* stream)
    {
        int err = EINVAL;
        
        if (stream != NULL)
        {
            err = fclose(stream);
        }

        return err;
    }


    //////////////////////////////////////////////////////////////////////////
    //  Function: getfilesize
    //
    //////////////////////////////////////////////////////////////////////////
    int getfilesize(FILE* stream, long* pSize)
    {
        int err = EINVAL;

        if (stream != NULL && pSize != NULL)
        {
            fpos_t filepos;

            err = fgetpos(stream, &filepos);
            if (err == 0)
            {
                err = fseek(stream, 0, SEEK_END);
                if (err == 0)
                {
                    *pSize = ftell(stream);
                }
                (void)fsetpos(stream, &filepos);
            }
        }

        return err;
    }


    //////////////////////////////////////////////////////////////////////////
    //  Function: writefile
    //
    //////////////////////////////////////////////////////////////////////////
    int writefile( const void *buffer, size_t size, size_t count, FILE *stream)
    {
        int err = EINVAL;

        if (buffer != NULL && stream != NULL)
        {
            err = 0;
            size_t len = fwrite(buffer, size, count, stream);
            if (len < count)
            {
                err = errno;
            }
        }

        return err;
    }


    //////////////////////////////////////////////////////////////////////////
    //  Function: readfile
    //
    //////////////////////////////////////////////////////////////////////////
    int readfile(void *buffer, size_t size, size_t count, FILE* stream)
    {
        int err = EINVAL;

        if (buffer != NULL && stream != NULL)
        {
            err = 0;
            size_t len = fread(buffer, size, count, stream);
            if (len < count)
            {
                if (!feof(stream))
                {
                    err = errno;
                }
            }
        }

        return err;
    }


    //////////////////////////////////////////////////////////////////////////
    //  Function: readline
    //
    //////////////////////////////////////////////////////////////////////////
    int readline(char *buffer, size_t maxSize, FILE* stream)
    {
        int err = EINVAL;

        if (buffer != NULL && stream != NULL && maxSize != 0)
        {
            buffer[0] = '\0';
            err = 0;
            char* pText = fgets(buffer, static_cast<int>(maxSize), stream);
            if (pText == NULL)
            {
                err = EOF;
                if (ferror(stream))
                {
                    err = errno;
                }
            }
        }

        return err;
    }


    //////////////////////////////////////////////////////////////////////////
    //  Function: flushfile
    //
    //////////////////////////////////////////////////////////////////////////
    int flushfile(FILE* stream)
    {
        int err = EINVAL;

        if (stream != NULL)
        {
            err = fflush(stream);
        }

        return err;
    }

} // end namespace InternalUtils
