/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
///
/// @brief Implementation of a function to locate a dynamic library.
///
//////////////////////////////////////////////////////////////////////////////



#include "stdafx.h"
#include "fileexists.h"
#include "paths.h"
#include "getenvvar.h"
#include "locateimpl.h"
#include "findfile.h"
#include "createmodulerelativepath.h"

#ifdef _WIN32
    #include <codecvt>
#else
    #include <mutex>
    #include <dirent.h>
	#include <fnmatch.h>
#endif

namespace InternalUtils
{
	std::string findadjacentfile(const char* moduleName)
	{
        // Check if IPC is registered
        std::string dllName = createmodulerelativepath(moduleName);

        if (!fileexists(dllName.c_str()))
        {
            dllName.clear();
        }

		return dllName;
	}

	std::string findopenipcfile(const char* moduleName)
	{
        // Check if IPC is registered
        std::string dllName = locateopenipc();
        if (!dllName.empty())
        {
            dllName = appendtail(dllName.c_str(), moduleName);
            if (!fileexists(dllName.c_str()))
            {
                dllName.clear();
            }
        }

		return dllName;
	}

	std::string finddalfile(const char* moduleName)
	{
        // Check if DAL is registered
        std::string dllName = locatedal();
        if (!dllName.empty())
        {
            dllName = appendtail(dllName.c_str(), moduleName);
            if (!fileexists(dllName.c_str()))
            {
                dllName.clear();
            }
        }

		return dllName;
	}

    std::string findmodule(const char* moduleName)
    {
        // First, look in the current directory for the DLL.
		std::string dllName = findadjacentfile(moduleName);

		if (dllName.empty())
		{
			// Now look in the working folder
			dllName = appendtail(".", moduleName);
			if (!fileexists(dllName.c_str()))
			{
				// Okay, see if we have an alternative location for the binaries
				// (used exclusively in development and debugging).  We want to
				// look here first in case the developer working on the IPC IPC API
				// is working with updated binaries.
				dllName = getenvvar("IPC_PATH");
				if (!dllName.empty())
				{
					dllName = appendtail(dllName.c_str(), moduleName);
					if (!fileexists(dllName.c_str()))
					{
						dllName.clear();
					}
				}

				if (dllName.empty())
				{
					// Okay, let's look in the IPC's registered location for the DLL.
					// This is the normal situation for a client application that
					// uses the IPC.
					dllName = findopenipcfile(moduleName);
				}

				if (dllName.empty())
				{
					// Okay, let's look in the IPC's registered location for the DLL.
					// This is the normal situation for a client application that
					// uses the IPC.
					dllName = finddalfile(moduleName);
				}

				if (dllName.empty())
				{
					// Finally, let the operating system find the DLL.
					// Return just the base file name and let the operating system
					// sort things out if it can.
					dllName = removeextension(moduleName);
#ifndef _WIN32
					// If running under Linux, remove the "lib" prefix as well.
					if (dllName.find("lib") == 0)
					{
						dllName = dllName.substr(3);
					}
#endif
				}
			}
		}

        return dllName;
    }
	
	std::vector<std::string> findfiles(const std::string& directory, const std::string& filespec)
	{
		std::vector<std::string> files;

#if defined(_WIN32)
		WIN32_FIND_DATAW data;
        std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>, wchar_t> converter;

		std::string search = appendtail(directory.c_str(), filespec.c_str());

		HANDLE h = FindFirstFileW(converter.from_bytes(search).c_str(), &data);
		if (h != INVALID_HANDLE_VALUE) 
		{
			do
			{
				files.push_back(appendtail(directory.c_str(), converter.to_bytes(data.cFileName).c_str()));
			} while(FindNextFileW(h, &data));
	
			FindClose(h);
		}
#else
			static std::mutex _findFilesLock;
			static std::string pattern;

			{
				std::lock_guard<std::mutex> lock(_findFilesLock);

				pattern = filespec;

				struct dirent** entryList = NULL;
				auto filter = [](const struct dirent * dent)->int {
					return fnmatch(pattern.c_str(), dent->d_name, FNM_PATHNAME) == 0;
				};

				int numEntries = scandir(directory.c_str(), &entryList, filter, alphasort);
				if (numEntries >= 0)
				{
					for (int index = 0; index < numEntries; ++index)
					{
						files.push_back(appendtail(directory.c_str(), entryList[index]->d_name));
						free(entryList[index]);
					}
					free(entryList);
				}
			}
#endif // defined(_WIN32)

		return files;
	}
} // end namespace InternalUtils
