/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
///
/// @brief Implementation of functions to access a logger object from a dynamic
///        library.
///
//////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "findfile.h"
#include "accessdll.h"
#include "getlogger.h"

namespace // anonymous
{
    typedef IIPCLogger* (*GetIPCLoggerFn)(const char*, const char*);
    typedef void        (*ReleaseIPCLoggerFn)(IIPCLogger*);

    GetIPCLoggerFn     _GetIPCLogger     = NULL;
    ReleaseIPCLoggerFn _ReleaseIPCLogger = NULL;

    bool _isInitialized = false;

    IPC_DLL_HANDLE _dllModule = NULL;

    //////////////////////////////////////////////////////////////////////////
    //  Function: _GetLoggerLibraryName
    //
    // Retrieve the name of the logger DLL.
    //////////////////////////////////////////////////////////////////////////
    const char* _GetLoggerLibraryName()
    {
#ifdef _MSC_VER

#  if defined(_M_X64)
        const char* dallogger_Name = "IpcApiLogging_x64.dll";
#  else
        const char* dallogger_Name = "IpcApiLogging_x86.dll";
#  endif

#else
        // *********** UNIX-SPECIFIC CODE ***********
#ifndef __SIZEOF_POINTER__
#error __SIZEOF_POINTER__ not defined.  Cannot decide between 32-bit and 64-bit compilation.
#endif

#  if defined(__APPLE__)
#    if defined(__SIZEOF_POINTER__) && __SIZEOF_POINTER__ == 8
        const char* dallogger_Name = "libIpcApilogging_x64.dylib";
#    else
        const char* dallogger_Name = "libIpcApilogging_x86.dylib";
#    endif
#  else
#    if defined(__SIZEOF_POINTER__) && __SIZEOF_POINTER__ == 8
        const char* dallogger_Name = "libIpcApilogging_x64.so";
#    else
        const char* dallogger_Name = "libIpcApilogging_x86.so";
#    endif
#  endif
#endif
        return dallogger_Name;
    }


    // Initialize access to the logger dynamic library by loading the library
    // and getting the functions needed to get and release a logger.
    //
    // Returns true if the logger was successfully initialized; otherwise,
    // returns false.
    bool _Initialize()
    {
        if (!_isInitialized)
        {
            if (_dllModule == NULL)
            {
                const char* dllName = _GetLoggerLibraryName();
                std::string dllPath = InternalUtils::findmodule(dllName);
                _dllModule = InternalUtils::loaddll(dllPath.c_str());
            }

            if (_dllModule != NULL)
            {
                _GetIPCLogger = (GetIPCLoggerFn)InternalUtils::getprocedurefromdll(
                                                                        _dllModule,
                                                                        "GetIPCLogger");

                _ReleaseIPCLogger = (ReleaseIPCLoggerFn)InternalUtils::getprocedurefromdll(
                                                                        _dllModule,
                                                                        "ReleaseIPCLogger");
                if (_GetIPCLogger != NULL && _ReleaseIPCLogger != NULL)
                {
                    _isInitialized = true;
                }
            }
        }

        return _isInitialized;
    }

} // end namespace anonymous



namespace InternalUtils
{

    //////////////////////////////////////////////////////////////////////////
    //  Function: getlogger
    //
    //////////////////////////////////////////////////////////////////////////
    IIPCLogger* getlogger(
        const char* moduleName,
        const char* logfilename)
    {
        IIPCLogger* pLogger = NULL;

        if (_Initialize())
        {
            pLogger = _GetIPCLogger(moduleName, logfilename);
        }

        return pLogger;
    }


    //////////////////////////////////////////////////////////////////////////
    //  Function: releaselogger
    //
    //////////////////////////////////////////////////////////////////////////
    void releaselogger(
        IIPCLogger* pLogger)
    {
        if (_Initialize())
        {
            _ReleaseIPCLogger(pLogger);
        }
    }

} // end namespace InternalUtils
