/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
///
/// @brief Implementation of the CLockObject class that provides a simple
///        way of protecting resources in a multi-threaded environment.
///
//////////////////////////////////////////////////////////////////////////////

#ifndef __LOCKOBJECT_H__
#define __LOCKOBJECT_H__

#include <string.h>
#include <mutex>
#include <chrono>

#define MAX_LOCK_TIMEOUT 15000 /** Timeout in milliseconds for waiting for a lock to be acquired. */

namespace InternalUtils
{

    //////////////////////////////////////////////////////////////////////////////
    //  class CLockObject    : 
    //
    /// @brief Represents a semaphore-style lock that can be used to protect
    ///        data objects from simultaneous access across multiple threads.
    ///
    /// This class hides the operating system-specific details of using an unnamed
    /// semaphore-style lock.
    ///
    /// @par Usage:
    ///   Create an instance of this class at the top of a scope where access to
    ///   a data object needs to be protected.  Call Lock() to acquire the lock.
    ///   When done with the lock, call the Release() method to release the lock.
    ///
    /// @par Example:
    /// @code
    /// ClockObject m_lockObject;
    ///
    /// void SomeFunction()
    /// {
    ///     if (m_lockObject.Lock())
    ///     {
    ///         // Access the family jewels here
    ///         m_lockObject.Release();
    ///     }
    /// }
    /// @endcode
    //////////////////////////////////////////////////////////////////////////////
    class CLockObject
    {
    private:
        std::timed_mutex m_mutex;
        std::unique_lock<std::timed_mutex> m_lock;
        std::string      m_lastOperation;
        std::string      m_lastErrorMessage;

    public:

        //////////////////////////////////////////////////////////////////////////
        //  Method: CLockObject
        //
        /// @brief (Default) Class constructor
        ///
        /// @par Parameters:
        ///   None.
        //////////////////////////////////////////////////////////////////////////
        CLockObject()
            : m_lock(m_mutex, std::defer_lock)
        {
        }


        //////////////////////////////////////////////////////////////////////////
        //  Method: ~CLockObject
        //
        /// @brief Class destructor
        ///
        /// The mutex is destroyed and, as a side effect, it is released.
        /// Prefer to call the Release() method explicitly before the lock object
        /// is destroyed.
        ///
        //////////////////////////////////////////////////////////////////////////
        ~CLockObject()
        {
        }



        //////////////////////////////////////////////////////////////////////////
        //  Method: Lock
        //
        /// @brief Acquire a lock on this object.
        ///
        /// If the lock is held by some other entity, this method will block until
        /// the lock is released or until the lock timeout expires.
        ///
        /// @warning
        /// For every call to Lock(), there must be a corresponding call to Release().
        ///
        /// @par Assumptions:
        ///   None.
        ///
        /// @par Side Effects:
        ///   The lock is acquired.
        ///
        /// @par Parameters:
        ///   None.
        ///
        /// @return (bool):
        ///     true if the lock was acquired; otherwise, returns false,
        ///     indicating there was an error in acquiring a lock.
        //////////////////////////////////////////////////////////////////////////
        bool Lock()
        {
            m_lastOperation = "Acquiring lock";
            m_lastErrorMessage.clear();

            bool isLocked = false;
            try
            {
                isLocked = m_lock.try_lock_for(std::chrono::milliseconds(MAX_LOCK_TIMEOUT));
                if(!isLocked)
                {
                    m_lastErrorMessage = "Timed out while trying to acquire the lock";
                }
            }
            catch(std::system_error& ex)
            {
                m_lastErrorMessage = ex.what();
            }

            return isLocked;
        }


        //////////////////////////////////////////////////////////////////////////
        //  Method: Release
        //
        /// @brief Release the lock.
        ///
        /// Release the lock on the mutex.
        ///
        /// @warning
        /// For every call to Lock(), there must be a corresponding call to Release().
        ///
        /// @par Assumptions:
        ///   None.
        ///
        /// @par Side Effects:
        ///   The lock is released.
        ///
        /// @par Parameters:
        ///   None.
        ///
        /// @par return(bool):
        ///     true if the lock was released; otherwise, returns false,
        ///     indicating there was an error in releasing the lock.
        ///   
        //////////////////////////////////////////////////////////////////////////
        bool Release()
        {
            m_lastOperation = "Releasing lock";
            m_lastErrorMessage.clear();

            bool isUnlocked = false;
            try
            {
                m_lock.unlock();
                isUnlocked = true;
            }
            catch(std::system_error& ex)
            {
                m_lastErrorMessage = ex.what();
            }

            return isUnlocked;
        }


        //////////////////////////////////////////////////////////////////////////
        //  Method: GetLastErrorMesage
        //
        /// @brief Retrieve a human-readable error message for the last error
        ///        that occurred.
        ///
        /// If an error occurred, call this method to retrieve the text of the
        /// error.  For example, if Lock() returns false.
        /// If the last operation did not generate an error, this method returns
        /// an empty string.
        ///
        /// @par Assumptions:
        ///   None.
        ///
        /// @par Side Effects:
        ///   None.
        ///
        /// @par Parameters:
        ///   None.
        ///
        /// @return (std::string):
        ///     A string containing a description of the last error that occurred.
        ///     If no error occurred, returns an empty string.
        //////////////////////////////////////////////////////////////////////////
        std::string GetLastErrorMesage()
        {
            std::stringstream output;

            if (!m_lastErrorMessage.empty())
            {
                output << m_lastOperation << ": " << m_lastErrorMessage;
            }

            return output.str();
        }
    };

} // end namespace InternalUtils

#endif // __LOCKOBJECT_H__
