/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
///
/// @brief Implementation of the strstri() and wcsstri() functions.
///
/// These utility functions provide a platform-independent way of searching a
/// string for another string in a case-insensitive fashion.
///
///////////////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "strstri.h"

namespace InternalUtils
{

    ///////////////////////////////////////////////////////////////////////////
    // function strstri
    //
    // Search for narrow string s2 in narrow string s1.
    ///////////////////////////////////////////////////////////////////////////
    const char* strstri(
        const char* s1,
        const char* s2)
    {
        // Guard against null pointers (strstr() does not do this).
        if (s1 == NULL || s2 == NULL)
        {
            return (NULL);
        }

        // If string to search for is empty, return beginning of str1.
        if (*s2 == '\0')
        {
            return (s1);
        }

        // Loop over all characters in string to search
        while (*s1 != '\0')
        {
            int c2 = toupper(*s2);
            // Find first occurrence of second string in string 1
            while (*s1 != '\0' && toupper(*s1) != c2)
            {
                s1++;
            }

            if (*s1 == '\0')
            {
                break;
            }
            const char* sc1;
            const char* sc2;

            // Now that the first character of string 2 might have been
            // found, compare subsequent characters of string 2 to see
            // if we have an exact match.
            for (sc1 = s1, sc2 = s2; ; )
            {
                if (*++sc2 == '\0')
                {
                    // We have reached the end of the string to search for
                    // so we must have a match.  Return pointer into s1
                    // to the found string.
                    return (s1);
                }
                else
                {
                    ++sc1;

                    if (toupper(*sc1) != toupper(*sc2))
                    {
                        // a character in string 2 doesn't match so start
                        // the whole thing over with the next character
                        // in string 1.
                        break;
                    }
                }
            }
            ++s1;
        }
        // If we get here, there is no match so return NULL.
        return (NULL);
    }


    ///////////////////////////////////////////////////////////////////////////
    // function wcsstri
    //
    // Search for wide string s2 in wide string s1.
    ///////////////////////////////////////////////////////////////////////////
    const wchar_t* wcsstri(
        const wchar_t* s1,
        const wchar_t* s2)
    {
        // Guard against null pointers (strstr() does not do this).
        if (s1 == NULL || s2 == NULL)
        {
            return (NULL);
        }

        // If string to search for is empty, return beginning of str1.
        if (*s2 == L'\0')
        {
            return (s1);
        }

        // Loop over all characters in string to search
        while (*s1 != L'\0')
        {
            wint_t c2 = towupper(*s2);
            // Find first occurrence of second string in string 1
            while (*s1 != L'\0' && towupper(*s1) != c2)
            {
                s1++;
            }

            if (*s1 == L'\0')
            {
                break;
            }
            const wchar_t* sc1;
            const wchar_t* sc2;

            // Now that the first character of string 2 might have been
            // found, compare subsequent characters of string 2 to see
            // if we have an exact match.
            for (sc1 = s1, sc2 = s2; ; )
            {
                if (*++sc2 == L'\0')
                {
                    // We have reached the end of the string to search for
                    // so we must have a match.  Return pointer into s1
                    // to the found string.
                    return (s1);
                }
                else
                {
                    ++sc1;

                    if (towupper(*sc1) != towupper(*sc2))
                    {
                        // a character in string 2 doesn't match so start
                        // the whole thing over with the next character
                        // in string 1.
                        break;
                    }
                }
            }
            ++s1;
        }
        // If we get here, there is no match so return NULL.
        return (NULL);
    }

} // end namespace InternalUtils
