/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
///
/// @brief Declaration of the event functions that handle signaling
///        across threads.
///
//////////////////////////////////////////////////////////////////////////////

#ifndef __THREADEVENT_H__
#define __THREADEVENT_H__

#include "threadtypes.h"

namespace InternalUtils
{
    /// Represents an invalid mutex handle to waitthreadevent().
    const int EINVAL_MUTEX = 1024;

    //////////////////////////////////////////////////////////////////////////
    //  Function: createthreadevent
    //
    /// @brief Creates a thread signaling object.
    ///
    /// This helper function creates a thread signaling object in such a way
    /// is operating system-agnostic (that is, controlled by conditional
    /// macros during compilation).
    ///
    /// A thread signaling object is used by one thread to signal another
    /// thread waiting for an event that the event has occurred.
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   A signaling object is created.
    ///
    /// @param pThreadEventHandle
    ///     Returns the handle to the signaling object.
    ///
    /// @return (int):
    ///     Returns 0 if successful; otherwise, returns an error code (errno).
    //////////////////////////////////////////////////////////////////////////
    int createthreadevent(ThreadEventHandle* pThreadEventHandle);


    //////////////////////////////////////////////////////////////////////////
    //  Function: destroythreadevent
    //
    /// @brief Destroys a thread signaling object.
    ///
    /// This helper function destroys a thread signaling object that
    /// was created by the createthreadevent() function.  After the object is
    /// destroyed, it can no longer be used.
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   A signaling object is destroyed.
    ///
    /// @param threadEventHandle
    ///     The handle to the signaling object to be destroyed.
    ///
    /// @return (int):
    ///     Returns 0 if successful; otherwise, returns an error code (errno).
    //////////////////////////////////////////////////////////////////////////
    int destroythreadevent(ThreadEventHandle threadEventHandle);


    //////////////////////////////////////////////////////////////////////////
    //  Function: waitthreadevent
    //
    /// @brief Wait on the specified thread signaling object, with an optional
    ///        timeout.
    ///
    /// This helper function waits the specified number of milliseconds for
    /// the thread signaling object to be signaled by another thread.  If the
    /// timeout is TIMEOUT_INFINITE, then this function will block
    /// indefinitely until the signaling object is signaled.
    ///
    /// The specified thread synchronization object is used with the thread
    /// signaling object.  The thread synchronization is released while
    /// waiting for the signaling object to be signaled and reacquired when
    /// the signaling object is signaled or the timeout has occurred.
    ///
    /// @warning
    /// The thread synchronization object in the @ mutexHandle parameter
    /// must be locked before calling waitthreadevent().  The object will
    /// always be left in a locked state after waitthreadevent() returns.
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   If the timeout period has not expired, the thread signaling object
    ///   has been signaled.
    ///
    /// @param threadEventHandle
    ///     The handle to the signaling object to wait on.
    /// @param mutexHandle
    ///     The handle of the synchronization object to use with the signaling
    ///     object.
    /// @param milliseconds
    ///     The number of milliseconds to wait for the signal.  Pass
    ///     TIMEOUT_INFINITE to wait forever.
    ///
    /// @return (int):
    ///     Returns 0 if successful; otherwise, returns an error code (errno).
    ///     Returns ETIMEDOUT if the signal was not received within the timeout
    ///     period.
    ///     Returns EINVAL if @a threadEventHandle is NULL.
    ///     Returns EINVAL_MUTEX if @a mutexHandle is NULL.
    //////////////////////////////////////////////////////////////////////////
    int waitthreadevent(ThreadEventHandle threadEventHandle, MutexHandle mutexHandle, TimeoutDelay milliseconds);


    //////////////////////////////////////////////////////////////////////////
    //  Function: signalthreadevent
    //
    /// @brief Mark the specified thread signaling object as being signaled.
    ///
    /// This helper function signals a thread signaling object.  At least one
    /// thread that is waiting on the signal (with waitthreadevent()) will
    /// wake up and proceed to execute.  The thread signaling object is always
    /// automatically reset after the waiting thread is resumed.
    ///
    /// If no thread is waiting on the signal, nothing happens and the
    /// thread signaling object remains in an unsignaled state.
    ///
    /// @note
    /// The thread calling signalthreadevent() must lock the thread
    /// synchronization object used in the call to waitthreadevent().
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   The thread signaling object is signaled.
    ///
    /// @param threadEventHandle
    ///     The handle to the signaling object to signal.
    ///
    /// @return (int):
    ///     Returns 0 if successful; otherwise, returns an error code (errno).
    //////////////////////////////////////////////////////////////////////////
    int signalthreadevent(ThreadEventHandle threadEventHandle);
    
} // end namespace InternalUtils

#endif // __THREADEVENT_H__
