/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
/// @internal
///
/// @brief Declaration of the mutex functions that handle synchronization
///        across threads.
///
//////////////////////////////////////////////////////////////////////////////

#ifndef __THREADMUTEX_H__
#define __THREADMUTEX_H__

#include "threadtypes.h"

namespace InternalUtils
{

    //////////////////////////////////////////////////////////////////////////
    //  Function: createmutex
    //
    /// @brief Creates a thread synchronization object.
    ///
    /// This helper function creates a thread synchronization object in such
    /// a way that is operating system-agnostic (that is, controlled by
    /// conditional macros during compilation).
    ///
    /// Although the term "mutex" is used for "mutually exclusive", each
    /// operating system will use the most appropriate synchronization object.
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   A synchronization object is created.
    ///
    /// @param pMutexHandle
    ///     Returns the handle to the synchronization object.
    ///
    /// @return (int):
    ///     Returns 0 if successful; otherwise, returns an error code (errno).
    //////////////////////////////////////////////////////////////////////////
    int createmutex(MutexHandle* pMutexHandle);


    //////////////////////////////////////////////////////////////////////////
    //  Function: destroymutex
    //
    /// @brief Destroys a thread synchronization object.
    ///
    /// This helper function destroys a thread synchronization object that
    /// was created by the createmutex() function.  After the object is
    /// destroyed, it can no longer be used.
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   A synchronization object is destroyed.
    ///
    /// @param mutexHandle
    ///     The handle to the synchronization object to be destroyed.
    ///
    /// @return (int):
    ///     Returns 0 if successful; otherwise, returns an error code (errno).
    //////////////////////////////////////////////////////////////////////////
    int destroymutex(MutexHandle mutexHandle);


    //////////////////////////////////////////////////////////////////////////
    //  Function: lockmutex
    //
    /// @brief Acquire a lock on the specified thread synchronization object,
    ///        with an optional timeout.
    ///
    /// This helper function waits the specified number of milliseconds to
    /// acquire a lock on the specified thread synchronization object.  If the
    /// timeout is TIMEOUT_INFINITE, then this function will block
    /// indefinitely until the lock is acquired.
    ///
    /// If the lock is successfully acquired, call the unlockmutex() function
    /// to release the lock.  Calling lockmutex() increments an internal
    /// counter on the lock so a thread will not deadlock itself acquiring a
    /// lock it already has.  There must be a corresponding call to
    /// unlockmutex() for each call to lockmutex().
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   If the timeout period has not expired, the lock is acquired.
    ///
    /// @param mutexHandle
    ///     The handle to the synchronization object to lock.
    /// @param milliseconds
    ///     The number of milliseconds to wait to acquire the lock.  Pass
    ///     TIMEOUT_INFINITE to wait forever.
    ///
    /// @return (int):
    ///     Returns 0 if successful; otherwise, returns an error code (errno).
    ///     Returns ETIMEDOUT if the lock was not acquired within the timeout
    ///     period.
    //////////////////////////////////////////////////////////////////////////
    int lockmutex(MutexHandle mutexHandle, TimeoutDelay milliseconds);


    //////////////////////////////////////////////////////////////////////////
    //  Function: unlockmutex
    //
    /// @brief Release a previously acquired lock on a thread synchronization
    ///        object.
    ///
    /// This helper function releases a lock on a thread synchronization
    /// object that was previously acquired with the lockmutex() function.
    /// For each call to lockmutex(), there must be a call to unlockmutex().
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   The lock is released.
    ///
    /// @param mutexHandle
    ///     The handle to the synchronization object to release.
    ///
    /// @return (int):
    ///     Returns 0 if successful; otherwise, returns an error code (errno).
    //////////////////////////////////////////////////////////////////////////
    int unlockmutex(MutexHandle mutexHandle);
    

    //////////////////////////////////////////////////////////////////////////
    //  Function: trylockmutex
    //
    /// @brief Try to acquire a lock on the specified thread synchronization
    ///        object.
    ///
    /// This helper function tries to acquire a lock on a thread
    /// synchronization object.  If successful, the object is locked as though
    /// lockmutex() was called.  If unsuccessful, an error code is returned
    /// and the lock is not acquired.
    ///
    /// If the lock is successfully acquired, call the unlockmutex() function
    /// to release the lock.
    ///
    /// If a thread calls trylockmutex() on a thread synchronization object
    /// it has already acquired, the call is successful and the internal
    /// counter on the object is incremented (just like lockmutex()).  The
    /// thread must call unlockmutex() once for each call to trylockmutex().
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   If the lock is available, the lock is acquired.
    ///
    /// @param mutexHandle
    ///     The handle to the synchronization object to try to lock.
    ///
    /// @return (int):
    ///     Returns 0 if successful; otherwise, returns an error code (errno).
    ///     Returns EBUSY if the lock was not acquired (the lock was taken by
    ///     another thread).
    //////////////////////////////////////////////////////////////////////////
    int trylockmutex(MutexHandle mutexHandle);


} // end namespace InternalUtils

#endif // __THREADMUTEX_H__
