/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#include "stdafx.h"

#include <regex>

#include "CommonImpl.h"

#include <Foundation/Logging/OpenIPC_Logger.h>
#include <Foundation/Error/OpenIPC_ErrorContext.h>
#include <Foundation/Telemetry/Telemetry.h>

#include <Components/Configuration/ConfigurationErrors.h>

#include <StructuredData.h>
#include <ModuleAccess.h>
#include <ExecutionEnvironment.h>
#include <SafeString.h>
#include <ManipulatePath.h>

#include <PushNewOverride.h>
#include <boost/filesystem.hpp>
#include <boost/thread/tss.hpp>
#include <boost/algorithm/string/predicate.hpp>
#include <PopNewOverride.h>

#include <MemoryCopy.h>

struct CommonThreadContext
{
    int32_t _clientId;

    CommonThreadContext()
        : _clientId(-1)
    {
    }

};

static boost::thread_specific_ptr<CommonThreadContext> CommonThreadContextInstances;

CommonImpl CommonImpl::_instance;

const std::string CommonImpl::_crumbFileSuffix  = "_crumb.txt";
const std::string CommonImpl::_stacktraceSuffix = "_stacktrace.log";
const std::string CommonImpl::_crashdumpSuffix  = "_crashdump.dmp";

#if defined(HOST_WINDOWS)
#include <codecvt>
#else
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#endif

#if defined(HOST_WINDOWS)
    std::string GetErrorString(DWORD error)
    {
        LPWSTR buff = nullptr;
        DWORD  size = FormatMessageW(
            FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS,
            NULL,
            error,
            MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
            (LPWSTR)&buff,
            0,
            NULL);

        //FormatMessage returns the number of characters in the buffer _excluding_ the terminating null character

        if ((size != 0) && (buff[size - 1] == L'\n'))
        {
            //The error message that's returned ends in a newline character, we need to strip that out
            buff[size - 1] = L'\0';

            if ((size > 1) && (buff[size - 2] == L'\r'))
            {
                buff[size - 2] = L'\0';
            }
        }

        std::string message;
        if (size == 0)
        {
            message = "FormatMessage() failed to get the error description string";
        }
        else
        {
            std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>, wchar_t> converter;
            message = converter.to_bytes(buff);
        }
        LocalFree(buff);
        buff = nullptr;

        return message;
    }

#endif

void _OnLoggingPresetChangedCallback(const char*)
{
    CommonImpl::GetInstance().SaveLoggingConfiguration();
}

OpenIPC_Error OpenIPC_GetCommon(OpenIPC_CommonComponent** component)
{
    OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

    if (component == nullptr)
    {
        openIPCError = OpenIPC_Error_Null_Pointer;
        POST_ERROR_MESSAGE(openIPCError, "OpenIPC_GetCommon() argument error (component) : Null pointer");
    }
    else
    {
        CommonImpl& i = CommonImpl::GetInstance();
        *component = &(OpenIPC_CommonComponent&)i;
    }

    return openIPCError;
}

CommonImpl& CommonImpl::GetInstance()
{
    return _instance;
}

CommonImpl::CommonImpl() :
    _clientDescriptorMap(),
    _clientCallbacks(),
    _logfileSuffix("Client"),
#if defined(HOST_WINDOWS)
        _crumbFileHandle(INVALID_HANDLE_VALUE)
#else
        _crumbFileDescriptor(-1)
#endif
{
    OpenIPC_CommonComponent::GetPath     = &CommonImpl::GetPath_Mediator;
    OpenIPC_CommonComponent::GetClientId = &CommonImpl::GetClientId_Mediator;
    OpenIPC_CommonComponent::SetClientId = &CommonImpl::SetClientId_Mediator;
    OpenIPC_CommonComponent::RegisterNewClient = &CommonImpl::RegisterNewClient_Mediator;
    OpenIPC_CommonComponent::CloseClient = &CommonImpl::CloseClient_Mediator;
    OpenIPC_CommonComponent::GetNumRegisteredClients = &CommonImpl::GetNumRegisteredClients_Mediator;
    OpenIPC_CommonComponent::GetRegisteredClients    = &CommonImpl::GetRegisteredClients_Mediator;
    OpenIPC_CommonComponent::IsClientLocal = &CommonImpl::IsClientLocal_Mediator;
    OpenIPC_CommonComponent::RegisterClientCallback   = &CommonImpl::RegisterClientCallback_Mediator;
    OpenIPC_CommonComponent::UnregisterClientCallback = &CommonImpl::UnregisterClientCallback_Mediator;
    OpenIPC_CommonComponent::InitializeLogging       = &CommonImpl::InitializeLogging_Mediator;
    OpenIPC_CommonComponent::UninitializeLogging     = &CommonImpl::UninitializeLogging_Mediator;
    OpenIPC_CommonComponent::GetOpenIpcHomeDirectory = &CommonImpl::GetOpenIpcHomeDirectory_Mediator;
    OpenIPC_CommonComponent::GetServerConfiguration  = &CommonImpl::GetServerConfiguration_Mediator;
    OpenIPC_CommonComponent::FlushLogs   = &CommonImpl::FlushLogs_Mediator;
    OpenIPC_CommonComponent::ClearLogs   = &CommonImpl::ClearLogs_Mediator;
    OpenIPC_CommonComponent::ArchiveLogs = &CommonImpl::ArchiveLogs_Mediator;
    OpenIPC_CommonComponent::SaveLoggingConfiguration = &CommonImpl::SaveLoggingConfiguration_Mediator;
    OpenIPC_CommonComponent::InitializeProcessKilledDetector   = &CommonImpl::InitializeProcessKilledDetector_Mediator;
    OpenIPC_CommonComponent::UninitializeProcessKilledDetector = &CommonImpl::UninitializeProcessKilledDetector_Mediator;

    boost::filesystem::path::imbue(std::locale());
}

OpenIPC_Error CommonImpl::GetPath_Impl(
    IN OUT size_t* pathSize,
    OUT char*      path)
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;

    if (!path || !pathSize)
    {
        error = OpenIPC_Error_Null_Pointer;
    }

    std::string moduleFileName;
    if (OpenIPC_PASS(error))
    {
        error = CommonUtils::GetCurrentModulePath(moduleFileName);
    }

    if (OpenIPC_PASS(error))
    {
        if (boost::filesystem::exists(moduleFileName))
        {
            const std::string pathString = boost::filesystem::canonical(moduleFileName).parent_path().string();

            if (pathString.size() >= *pathSize)
            {
                *pathSize = pathString.size() + 1; // '+1' to account for the terminating null character
                error     = OpenIPC_Error_Bad_Argument;
            }
            else
            {
                // Copy the resulting path to the argument
                CommonUtils::SafeStringCopy(path, pathString.size() + 1, pathString);
                *pathSize = pathString.size() + 1; // '+1' to account for the terminating null character
            }
        }
        else
        {
            error = OpenIPC_Error_Internal_Error;
        }
    }

    return error;
}

OpenIPC_Error CommonImpl::GetClientId_Impl(
    OUT int32_t* clientId)
{
    OpenIPC_Error errorCode = OpenIPC_Error_No_Error;

    CommonThreadContext* context = CommonThreadContextInstances.get();
    if (context == nullptr)
    {
        context = new CommonThreadContext();

        CommonThreadContextInstances.reset(context);
    }

    if (clientId != nullptr)
    {
        *clientId = context->_clientId;
    }
    else
    {
        errorCode = OpenIPC_Error_Null_Pointer;
    }

    return errorCode;
}

OpenIPC_Error CommonImpl::SetClientId_Impl(
    IN int32_t clientId)
{
    CommonThreadContext* context = CommonThreadContextInstances.get();
    if (context == nullptr)
    {
        context = new CommonThreadContext();

        CommonThreadContextInstances.reset(context);
    }

    context->_clientId = clientId;

    return OpenIPC_Error_No_Error;
}

OpenIPC_Error CommonImpl::RegisterNewClient_Impl(
    IN int32_t      clientId,
    IN const char*  clientIdentifier,
    IN OpenIPC_Bool isClientLocal)
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;

    if (!clientIdentifier)
    {
        error = OpenIPC_Error_Null_Pointer;
    }

    OpenIPC_CommonClientDescriptor clientDescriptor = {};
    if (OpenIPC_PASS(error))
    {
        std::lock_guard<std::mutex> lock(_clientDescriptorMutex);

        if (_clientDescriptorMap.find(clientId) != _clientDescriptorMap.end())
        {
            error = OpenIPC_Error_Internal_Error;
        }

        if (OpenIPC_PASS(error))
        {
            clientDescriptor.ClientId = clientId;
            clientDescriptor.IsLocal  = (isClientLocal == OpenIPC_TRUE);
            CommonUtils::SafeStringCopy(clientDescriptor.ClientIdentifier, sizeof(clientDescriptor.ClientIdentifier), clientIdentifier);
            _clientDescriptorMap[clientId] = clientDescriptor;

            error = SetClientId_Impl(clientId);
        }
    }

    if (OpenIPC_PASS(error))
    {
        std::lock_guard<std::mutex> lock(_clientCallbackMutex);

        for (auto it = _clientCallbacks.begin(); it != _clientCallbacks.end(); ++it)
        {
            (std::get<0>(*it))(&clientDescriptor, OpenIPC_Common_ClientEvent_Connected, std::get<1>(*it));
        }
    }

    return error;
}

OpenIPC_Error CommonImpl::CloseClient_Impl(
    IN int32_t clientId)
{
    std::unique_lock<std::mutex>   clientDescriptoLock(_clientDescriptorMutex);
    OpenIPC_CommonClientDescriptor descriptor = _clientDescriptorMap[clientId];

    clientDescriptoLock.unlock();

    std::lock_guard<std::mutex> lock(_clientCallbackMutex);

    for (auto it = _clientCallbacks.begin(); it != _clientCallbacks.end(); ++it)
    {
        (std::get<0>(*it))(&descriptor, OpenIPC_Common_ClientEvent_Disconnected, std::get<1>(*it));
    }

    //Remove it from the descriptor map
    clientDescriptoLock.lock();
    _clientDescriptorMap.erase(clientId);

    return OpenIPC_Error_No_Error;
}

OpenIPC_Error CommonImpl::IsClientLocal_Impl(
    IN int32_t        clientId,
    OUT OpenIPC_Bool* isLocal)
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;

    if (isLocal == nullptr)
    {
        error = OpenIPC_Error_Null_Pointer;
        POST_ERROR_MESSAGE(error, "The 'isLocal' parameter that was passed to Common::IsClientLocal is null");
    }

    if (OpenIPC_PASS(error))
    {
        std::lock_guard<std::mutex> lock(_clientDescriptorMutex);
        const auto& it = _clientDescriptorMap.find(clientId);

        if (it == _clientDescriptorMap.end())
        {
            error = OpenIPC_Error_Internal_Error;
            POST_ERROR_MESSAGE(error, "The client with client ID '" << LOG_INT32_DEC(clientId) << "' is unknown");
        }
        else
        {
            *isLocal = it->second.IsLocal ? OpenIPC_TRUE : OpenIPC_FALSE;
        }
    }

    return error;
}

OpenIPC_Error CommonImpl::GetNumRegisteredClients_Impl(
    OUT uint32_t* numClients)
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;

    if (!numClients)
    {
        error = OpenIPC_Error_Null_Pointer;
        POST_ERROR_MESSAGE(error, "The 'numClients' parameter that was passed to Common::GetNumRegisteredClients is null");
    }

    if (OpenIPC_PASS(error))
    {
        std::lock_guard<std::mutex> lock(_clientDescriptorMutex);
        *numClients = static_cast<uint32_t>(_clientDescriptorMap.size());
    }

    return error;
}

OpenIPC_Error CommonImpl::GetRegisteredClients_Impl(
    IN uint32_t                         maxNumClients,
    OUT uint32_t*                       numClients,
    OUT OpenIPC_CommonClientDescriptor* clients)
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;

    if (!clients)
    {
        error = OpenIPC_Error_Null_Pointer;
        POST_ERROR_MESSAGE(error, "The 'clients' parameter that was passed to Common::GetRegisteredClients is null");
    }
    else if (!numClients)
    {
        error = OpenIPC_Error_Null_Pointer;
        POST_ERROR_MESSAGE(error, "The 'numClients' parameter that was passed to Common::GetRegisteredClients is null");
    }

    if (OpenIPC_PASS(error))
    {
        std::lock_guard<std::mutex> lock(_clientDescriptorMutex);
        *numClients = 0;
        for (auto it = _clientDescriptorMap.begin(); (it != _clientDescriptorMap.end()) && (*numClients < maxNumClients); ++it)
        {
            clients[*numClients] = it->second;
            *numClients += 1;
        }
    }

    return error;
}

OpenIPC_Error CommonImpl::RegisterClientCallback_Impl(
    IN OpenIPC_CommonClientCallback_Function callback,
    IN void*                                 data)
{
    std::lock_guard<std::mutex> lock(_clientCallbackMutex);

    _clientCallbacks.insert(std::tuple<OpenIPC_CommonClientCallback_Function, void*>(callback, data));

    return OpenIPC_Error_No_Error;
}

OpenIPC_Error CommonImpl::UnregisterClientCallback_Impl(
    IN OpenIPC_CommonClientCallback_Function callback,
    IN void*                                 data)
{
    std::lock_guard<std::mutex> lock(_clientCallbackMutex);

    _clientCallbacks.erase(std::tuple<OpenIPC_CommonClientCallback_Function, void*>(callback, data));

    return OpenIPC_Error_No_Error;
}

void ParseSinkBase(DataElement* element, std::vector<std::string>& sinkTargets)
{
    DataElement* sinkTargetElement = element->FindChildByName("SinkTarget");
    while (sinkTargetElement)
    {
        if (sinkTargetElement->Name() != "SinkTarget")
        {
            sinkTargetElement = sinkTargetElement->NextSibling();
            continue;
        }

        std::string sinkTargetValue = sinkTargetElement->FindAttribute("Name")->Value();

        sinkTargets.push_back(sinkTargetValue);

        sinkTargetElement = sinkTargetElement->NextSibling();
    }
}

OpenIPC_Error CommonImpl::InitializeLogging_Impl(
    IN const char* suffix,
    IN bool        persistLoggingPreset)
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;
    _persistLoggingPreset = persistLoggingPreset;

    size_t pathSize = 1024;
    std::vector<char> path(pathSize);
    error = GetPath_Impl(&pathSize, path.data());
    if (error == OpenIPC_Error_Bad_Argument)
    {
        path.resize(pathSize);
        error = GetPath_Impl(&pathSize, path.data());
    }

    if (OpenIPC_PASS(error))
    {
        bool initialized = OpenIPC_LoggingInitializeLogging(suffix);

        if (initialized)
        {
            if (suffix)
            {
                _logfileSuffix = suffix;
            }
            // Unregister the callback to save the logging preset to file
            // whenever it is changed so it won't be re-written while reading
            OpenIPC_LoggingRemoveOnPresetChangedCallback(_OnLoggingPresetChangedCallback);

            std::string openIpcHomeDirectory;
            if (OpenIPC_PASS(error))
            {
                std::vector<char> directory(1024);
                size_t size = directory.size();
                error = GetOpenIpcHomeDirectory_Impl(directory.data(), &size);

                if (error == OpenIPC_Error_Bad_Argument)
                {
                    directory.resize(size);
                    error = GetOpenIpcHomeDirectory_Impl(directory.data(), &size);
                }

                if (OpenIPC_PASS(error))
                {
                    openIpcHomeDirectory = directory.data();
                }
            }

            // Load the logging XML configuration file
            DataDocument document;
            if (OpenIPC_PASS(error))
            {
                try
                {
                    document.LoadXmlFile((boost::filesystem::path(path).parent_path() / "Config" / "Logging.xml").string(), true);
                }
                catch (XmlLoadException& exception)
                {
                    error = OpenIPC_Error_Invalid_Configuration;
                    POST_ERROR_MESSAGE(error, exception.Message());
                }
                catch (std::exception& exception)
                {
                    error = OpenIPC_Error_Invalid_Configuration;
                    POST_ERROR_MESSAGE(error, exception.what());
                }
            }

            if (OpenIPC_PASS(error))
            {
                DataElement* element = document.RootElement();
                if (element)
                {
                    element = element->FirstChild();
                }

                // For each element
                while (element && OpenIPC_PASS(error))
                {
                    const std::string& elementName = element->Name();

                    // <ConsoleSink> element
                    if (elementName == "ConsoleSink" && element->FindAttribute("Enabled") && element->FindAttribute("Enabled")->ToBool())
                    {
                        std::vector<std::string> sinkTargets;
                        ParseSinkBase(element, sinkTargets);

                        std::vector<const char*> cstrings;
                        for (auto& string : sinkTargets)
                        {
                            cstrings.push_back(string.c_str());
                        }

                        OpenIPC_LoggingAddConsoleSink(cstrings.data(), static_cast<uint32_t>(cstrings.size()));
                    }

                    // <FileSink> element
                    else if (elementName == "FileSink" && element->FindAttribute("Enabled") && element->FindAttribute("Enabled")->ToBool())
                    {
                        std::vector<std::string> sinkTargets;
                        ParseSinkBase(element, sinkTargets);

                        DataAttribute* nameValue    = element->FindChildByName("Name")->FindAttribute("Value");
                        DataElement*   rolloverSize = element->FindChildByName("RolloverSize");
                        DataElement*   maxSize = element->FindChildByName("MaxSize");
                        DataElement*   shortFormatterElement = element->FindChildByName("ShortFormatter");

                        const std::string& fileName = nameValue->Value();

                        uint32_t fileSize       = 0;
                        uint32_t maxFileSize    = 0;
                        bool     shortFormatter = false;

                        if (rolloverSize != nullptr)
                        {
                            DataAttribute* value = rolloverSize->FindAttribute("Value");
                            if (value)
                            {
                                fileSize = value->ToUInt32();
                            }
                        }

                        if (maxSize != nullptr)
                        {
                            DataAttribute* value = maxSize->FindAttribute("Value");
                            if (value)
                            {
                                maxFileSize = value->ToUInt32();
                            }
                        }

                        if (shortFormatterElement != nullptr)
                        {
                            DataAttribute* value = shortFormatterElement->FindAttribute("Value");
                            if (value)
                            {
                                shortFormatter = value->ToBool();
                            }
                        }

                        std::vector<const char*> cstrings;
                        for (auto& string : sinkTargets)
                        {
                            cstrings.push_back(string.c_str());
                        }

                        OpenIPC_LoggingAddFileSink(openIpcHomeDirectory.c_str(), fileName.c_str(), fileSize, maxFileSize, cstrings.data(), static_cast<uint32_t>(cstrings.size()), shortFormatter);
                    }

                    // <SyslogSink> element
                    else if (elementName == "SyslogSink" && element->FindAttribute("Enabled") && element->FindAttribute("Enabled")->ToBool())
                    {
                        std::vector<std::string> sinkTargets;
                        ParseSinkBase(element, sinkTargets);

                        DataElement* targetElement = element->FindChildByName("Target");
                        DataElement* shortFormatterElement = element->FindChildByName("ShortFormatter");

                        uint16_t    port    = 514;
                        std::string address = "127.0.0.1";
                        bool shortFormatter = false;

                        if (targetElement != nullptr)
                        {
                            DataAttribute* addressValue = targetElement->FindAttribute("Address");
                            DataAttribute* portValue    = targetElement->FindAttribute("Port");

                            if (addressValue != nullptr)
                            {
                                address = addressValue->Value();
                            }

                            if (portValue != nullptr)
                            {
                                std::stringstream ss(portValue->Value());
                                ss >> port;

                                if (ss.fail())
                                {
                                    error = OpenIPC_Error_Invalid_Configuration;
                                    POST_ERROR_MESSAGE(error, "The given SysLogSink port value is not a number in the range 0 - 65535");
                                }
                            }
                        }

                        if (shortFormatterElement)
                        {
                            DataAttribute* valueAttr = shortFormatterElement->FindAttribute("Value");
                            if (valueAttr)
                            {
                                shortFormatter = valueAttr->ToBool();
                            }
                        }

                        if (OpenIPC_PASS(error))
                        {
                            std::vector<const char*> cstrings;
                            for (auto& string : sinkTargets)
                            {
                                cstrings.push_back(string.c_str());
                            }

                            OpenIPC_LoggingAddSyslogSink(address.c_str(), port, cstrings.data(), static_cast<uint32_t>(cstrings.size()), shortFormatter);
                        }
                    }

                    // <Presets> element
                    else if (elementName == "Presets")
                    {
                        // <Preset> elements
                        DataElement* presetElement = element->FindChildByName("Preset");
                        while (presetElement)
                        {
                            if (presetElement->Name() != "Preset")
                            {
                                presetElement = presetElement->NextSibling();
                                continue;
                            }

                            // Create the preset
                            std::string presetName = presetElement->FindAttribute("Name")->Value();
                            OpenIPC_LoggingCreateLoggingPreset(presetName.c_str());

                            // <Logger> elements
                            DataElement* loggerElement = presetElement->FindChildByName("Logger");
                            while (loggerElement)
                            {
                                if (loggerElement->Name() != "Logger")
                                {
                                    loggerElement = loggerElement->NextSibling();
                                    continue;
                                }

                                std::string name  = loggerElement->FindAttribute("Name")->Value();
                                std::string level = loggerElement->FindAttribute("Level")->Value();
                                std::vector<std::string> sinkTargets = loggerElement->FindAttribute("SinkTarget")->ToCommaSeparatedList();
                                std::vector<const char*> c_sinkTargets;
                                for (auto const& sinkTarget : sinkTargets)
                                {
                                    c_sinkTargets.push_back(sinkTarget.c_str());
                                }

                                OpenIPC_LoggingSeverityLevel severity = OpenIPC_LoggingSeverityLevel_Off;
                                if (boost::iequals(level, "trace"))
                                {
                                    severity = OpenIPC_LoggingSeverityLevel_Trace;
                                }
                                else if (boost::iequals(level, "debug"))
                                {
                                    severity = OpenIPC_LoggingSeverityLevel_Debug;
                                }
                                else if (boost::iequals(level, "info"))
                                {
                                    severity = OpenIPC_LoggingSeverityLevel_Info;
                                }
                                else if (boost::iequals(level, "warning"))
                                {
                                    severity = OpenIPC_LoggingSeverityLevel_Warning;
                                }
                                else if (boost::iequals(level, "error"))
                                {
                                    severity = OpenIPC_LoggingSeverityLevel_Error;
                                }
                                else if (boost::iequals(level, "off"))
                                {
                                    severity = OpenIPC_LoggingSeverityLevel_Off;
                                }

                                // Set the logger level for thsi preset
                                OpenIPC_LoggingSetPresetLoggerLevel(presetName.c_str(), name.c_str(), severity, c_sinkTargets.data(), static_cast<uint32_t>(c_sinkTargets.size()));

                                loggerElement = loggerElement->NextSibling();
                            }

                            presetElement = presetElement->NextSibling();
                        }

                        // Load the preset specified to be loaded
                        DataAttribute* loadPresetAttribute = element->FindAttribute("Load");
                        if (loadPresetAttribute)
                        {
                            OpenIPC_LoggingLoadLoggingPreset(loadPresetAttribute->Value().c_str());
                        }
                    }

                    element = element->NextSibling();
                }
            }

            if (_persistLoggingPreset)
            {
                // Register the callback to save the logging preset to file
                // whenever it is changed
                OpenIPC_LoggingAddOnPresetChangedCallback(_OnLoggingPresetChangedCallback);
            }
        }
    }

    return error;
}

OpenIPC_Error CommonImpl::UninitializeLogging_Impl()
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;

    if (!OpenIPC_LoggingUninitializeLogging())
    {
        error = OpenIPC_Error_Internal_Error;
    }

    return error;
}

OpenIPC_Error CommonImpl::GetOpenIpcHomeDirectory_Impl(
    IN char*       directory,
    IN OUT size_t* size)
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;

    if (!directory || !size)
    {
        error = OpenIPC_Error_Null_Pointer;
        POST_ERROR_MESSAGE(error, "A parameter that was passed to Common::GetOpenIpcHomeDirectory was NULL.");
    }

    std::lock_guard<std::mutex> lock(_serverConfigurationMutex);
    if (OpenIPC_PASS(error))
    {
        error = _PopulateServerConfiguration();
    }

    std::string path;
    if (OpenIPC_PASS(error))
    {
        path = _serverConfiguration.OpenIpcHomeDirectory;
        if (path.empty())
        {
            //The user did not override the location. The default location is in the user's home directory
            error = CommonUtils::GetUserHomeDirectory(path);

            if (OpenIPC_PASS(error))
            {
                path = (boost::filesystem::path(path) / ".OpenIPC").string();
            }
        }
    }

    if (OpenIPC_PASS(error))
    {
        if ((*size == 0) || (path.size() > (*size - 1)))
        {
            *size = path.size() + 1;
            error = OpenIPC_Error_Bad_Argument;
        }
        else
        {
            *size = path.size() + 1;
            CommonUtils::SafeStringCopy(directory, *size, path);
        }
    }

    return error;
}

OpenIPC_Error CommonImpl::GetServerConfiguration_Impl(
    OUT OpenIPC_CommonServerConfiguration* serverConfiguration)
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;

    if (!serverConfiguration)
    {
        error = OpenIPC_Error_Null_Pointer;
        POST_ERROR_MESSAGE(error, "A parameter that was passed to Common::GetServerConfiguration was NULL.");
    }

    std::lock_guard<std::mutex> lock(_serverConfigurationMutex);
    if (OpenIPC_PASS(error))
    {
        error = _PopulateServerConfiguration();
    }

    if (OpenIPC_PASS(error))
    {
        CommonUtils::MemoryCopy(serverConfiguration, sizeof(OpenIPC_CommonServerConfiguration), &(_serverConfiguration.Configuration), sizeof(OpenIPC_CommonServerConfiguration));
    }

    return error;
}

class ScopedLoggingLock
{
public:
    ScopedLoggingLock()
    {
        OpenIPC_LoggingLock();
    }

    ~ScopedLoggingLock()
    {
        OpenIPC_LoggingUnlock();
    }

};

OpenIPC_Error CommonImpl::FlushLogs_Impl()
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;

    std::lock_guard<std::mutex> lock(_loggingOperationMutex);
    ScopedLoggingLock loggingLock;

    error = _DisableLogging();
    if (OpenIPC_PASS(error))
    {
        error = _ReenableLogging();
    }

    return error;
}

OpenIPC_Error CommonImpl::ClearLogs_Impl()
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;

    std::lock_guard<std::mutex> lock(_loggingOperationMutex);
    ScopedLoggingLock loggingLock;

    error = _DisableLogging();
    if (OpenIPC_PASS(error))
    {
        if (!OpenIPC_LoggingClearLogs())
        {
            error = OpenIPC_Error_Internal_Error;
            POST_ERROR_MESSAGE(error, OpenIPC_LoggingGetLastErrorMessage());
        }

        OpenIPC_Error reenableError = _ReenableLogging();
        if (OpenIPC_PASS(error) && !OpenIPC_PASS(reenableError))
        {
            error = reenableError;
        }
    }

    return error;
}

OpenIPC_Error CommonImpl::ArchiveLogs_Impl()
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;

    std::lock_guard<std::mutex> lock(_loggingOperationMutex);
    ScopedLoggingLock loggingLock;

    error = _DisableLogging();
    if (OpenIPC_PASS(error))
    {
        if (!OpenIPC_LoggingArchiveLogs())
        {
            error = OpenIPC_Error_Internal_Error;
            POST_ERROR_MESSAGE(error, OpenIPC_LoggingGetLastErrorMessage());
        }

        OpenIPC_Error reenableError = _ReenableLogging();
        if (OpenIPC_PASS(error) && !OpenIPC_PASS(reenableError))
        {
            error = reenableError;
        }
    }

    return error;
}

OpenIPC_Error CommonImpl::SaveLoggingConfiguration_Impl()
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;

    std::lock_guard<std::mutex> lock(_loggingOperationMutex);
    ScopedLoggingLock loggingLock;

    // Get the logging preset that is loaded
    char loggingPresetName[OpenIPC_LOGGING_MAX_PRESET_NAME_LENGTH] = { 0 };
    OpenIPC_LoggingGetLoadedLoggingPreset(loggingPresetName);
    if (strlen(loggingPresetName) > 0)
    {
        // Get the path to the OpenIPC config directory
        std::string configPathString;
        error = _GetOpenIpcConfigPath(configPathString);
        if (OpenIPC_PASS(error))
        {
            const boost::filesystem::path configPath(configPathString);
            const boost::filesystem::path loggingConfigFilePath = configPath / "Logging.xml";

            // Open both the logging file and the temporary logging file
            std::ifstream inputFile(loggingConfigFilePath.string());
            if (!inputFile.good())
            {
                error = OpenIPC_Error_Internal_Error;
                POST_ERROR_MESSAGE(error, "Failed to open logging configuration file '" << loggingConfigFilePath.string() << "'");
            }

            if (OpenIPC_PASS(error))
            {
                bool replaceFile = true;

                // Regular expression matching the line containing which preset is
                // loaded
                std::regex presetsRegex("(\\<\\s*Presets\\s+Load\\s*=\\s*\")(\\w+)(\"\\s*\\>)");

                // For each line in the file
                std::vector<std::string> lines;
                std::string line;
                while (std::getline(inputFile, line))
                {
                    // If this is the line containing the loaded preset
                    std::smatch match;
                    if (std::regex_search(line, match, presetsRegex))
                    {
                        // If the preset in the file is different from the current
                        // preset
                        if (!boost::iequals(match[2].str(), loggingPresetName))
                        {
                            // Replace the preset name with the loaded preset
                            line = std::regex_replace(line, presetsRegex, std::string("$1") + loggingPresetName + "$3");
                        }
                        else
                        {
                            // No need to update the logging configuration, the
                            // preset is the same
                            replaceFile = false;
                            break;
                        }
                    }

                    lines.push_back(line);
                }

                inputFile.close();

                // Re-write the logging configuration file with the modified
                // content if needed
                if (replaceFile)
                {
                    std::ofstream outputFile(loggingConfigFilePath.string());
                    if (!outputFile.good())
                    {
                        error = OpenIPC_Error_Internal_Error;
                        POST_ERROR_MESSAGE(error, "Failed to open logging configuration file '" << loggingConfigFilePath.string() << "'");
                    }

                    if (OpenIPC_PASS(error))
                    {
                        for (const std::string& modifiedLine : lines)
                        {
                            outputFile << modifiedLine << std::endl;
                        }
                    }
                }
            }
        }
    }

    return error;
}

OpenIPC_Error CommonImpl::InitializeProcessKilledDetector_Impl(
    OUT OpenIPC_Bool* found)
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;
    boost::filesystem::path crumbDirectory;
    std::string openIpcHomeDirectory;
    if (OpenIPC_PASS(error) && !found)
    {
        error = OpenIPC_Error_Null_Pointer;
        POST_ERROR_MESSAGE(error, "A parameter that was passed to Common::InitializeProcessKilledDetector was NULL.");
    }

    if (OpenIPC_PASS(error))
    {
        size_t sizeHomeDirectory = 0;
        error = GetOpenIpcHomeDirectory_Impl(&(openIpcHomeDirectory[0]), &sizeHomeDirectory);
        if (error == OpenIPC_Error_Bad_Argument && sizeHomeDirectory != 0)
        {
            openIpcHomeDirectory.resize(sizeHomeDirectory - 1); //The null character isn't included in std::string's size.
            error = GetOpenIpcHomeDirectory_Impl(&(openIpcHomeDirectory[0]), &sizeHomeDirectory);
        }
    }

    if (OpenIPC_PASS(error))
    {
        crumbDirectory = openIpcHomeDirectory;
        error = CommonUtils::SanitizePath(crumbDirectory);
    }

    if (OpenIPC_PASS(error))
    {
        _crumbFileName = (crumbDirectory / (_logfileSuffix + _crumbFileSuffix)).string();
    }

    if (OpenIPC_PASS(error))
    {
        std::unique_ptr<OpenIPC_Logger> logger(OpenIPC_Logger::GetLogger(OpenIPC_LoggingLoggerId::OpenIPC_LoggingLoggerId_IpcInterface));
        *found = OpenIPC_FALSE;
        for (auto& file : boost::filesystem::directory_iterator(crumbDirectory))
        {
            std::string filePath = file.path().string();
            if (filePath.compare(filePath.size() - _crumbFileSuffix.size(), _crumbFileSuffix.size(), _crumbFileSuffix) == 0)
            {
                if (OpenIPC_PASS(_LockAndDeleteFile(filePath))) //If we couldn't lock it (or access it), we should just move on, not fail the flow.
                {
                    *found = OpenIPC_TRUE;
                    std::string cleanFilePath;
                    CommonUtils::CleanUserDirectoryFromString(cleanFilePath, filePath);
                    OpenIPC_LOG(logger, OpenIPC_LoggingSeverityLevel_Warning, "Found crumb file (a previous instance of OpenIPC didn't exit cleanly): " << filePath);
                    std::string session = file.path().filename().string();
                    session.resize(session.size() - _crumbFileSuffix.size());
                    TELEMETRY_BUILD_DATA(Telemetry_Event_Server_Previous_Process_Killed, "session",    session);
                    TELEMETRY_BUILD_DATA(Telemetry_Event_Server_Previous_Process_Killed, "crumb_file", cleanFilePath);
                    filePath.resize(filePath.size() - _crumbFileSuffix.size());
                    cleanFilePath.resize(cleanFilePath.size() - _crumbFileSuffix.size());
                    if (boost::filesystem::exists(filePath + _stacktraceSuffix))
                    {
                        OpenIPC_LOG(logger, OpenIPC_LoggingSeverityLevel_Warning, "Found stack trace: " << filePath << _stacktraceSuffix);
                        TELEMETRY_BUILD_DATA(Telemetry_Event_Server_Previous_Process_Killed, "stacktrace_filename", cleanFilePath + _stacktraceSuffix);
                        Telemetry_UploadFile(Telemetry_Session_Server_Lifespan, Telemetry_Event_Server_Previous_Process_Killed, "stacktrace_file", (filePath + _stacktraceSuffix).c_str());
                    }
                    if (boost::filesystem::exists(filePath + _crashdumpSuffix))
                    {
                        OpenIPC_LOG(logger, OpenIPC_LoggingSeverityLevel_Warning, "Found crash dump: " << filePath << _crashdumpSuffix);
                        TELEMETRY_BUILD_DATA(Telemetry_Event_Server_Previous_Process_Killed, "crashdump_filename", cleanFilePath + _crashdumpSuffix);
                        Telemetry_UploadFile(Telemetry_Session_Server_Lifespan, Telemetry_Event_Server_Previous_Process_Killed, "crashdump_file", (filePath + _crashdumpSuffix).c_str());
                    }
                    Telemetry_SendEvent(Telemetry_Session_Server_Lifespan, Telemetry_Event_Server_Previous_Process_Killed);
                }
            }
        }
    }

    if (OpenIPC_PASS(error))
    {
        error = _SaveCrumbFile();
    }

    return error;
}

OpenIPC_Error CommonImpl::UninitializeProcessKilledDetector_Impl()
{
    _DeleteCrumbFile();
    return OpenIPC_Error_No_Error;
}

OpenIPC_Error CommonImpl::_DisableLogging()
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;

    // Save the currently set severity level for all logs to restore after
    // the flush
    OpenIPC_LoggingPushLogSeverityLevels();

    // Uninitializing and re-initialing logging causes all active log files to
    // get moved over to the "Log" directory (as if a file roll-over occurred)
    // and for new files
    error = UninitializeLogging_Impl();

    return error;
}

OpenIPC_Error CommonImpl::_ReenableLogging()
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;

    error = InitializeLogging_Impl(nullptr, _persistLoggingPreset);
    if (OpenIPC_PASS(error))
    {
        // Restore the severity level for all logs from before the flush
        OpenIPC_LoggingPopLogSeverityLevels();
    }

    return error;
}

OpenIPC_Error CommonImpl::_PopulateServerConfiguration()
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;

    if (_serverConfiguration.Loaded == false)
    {
        OpenIPC_CommonServerConfiguration& config = _serverConfiguration.Configuration;

        // Set the defaults
        config.GuiEnabled = false;
        config.GuiMaxLogMessagesCount = 10000;
        config.ShutdownWhenNoClientsAreConnected = true;
        config.AllowExternalConnections = false;
        config.StartPort = 0; //The defaults for the ports don't matter much because the XSD requires specifying them
        config.EndPort   = 0;

        // Get the path where the server executable is
        size_t pathSize = 1024;
        std::vector<char> path(pathSize);

        if (OpenIPC_PASS(error))
        {
            error = GetPath_Impl(&pathSize, path.data());

            if (error == OpenIPC_Error_Bad_Argument)
            {
                path.resize(pathSize);
                error = GetPath_Impl(&pathSize, path.data());
            }
        }

        DataDocument document;
        if (OpenIPC_PASS(error))
        {
            // Load the server process XML configuration file
            try
            {
                document.LoadXmlFile((boost::filesystem::path(path).parent_path() / "Config" / "ServerProcess.xml").string(), true);
            }
            catch (const DataStoreException& ex)
            {
                error = OpenIPC_Error_Invalid_Configuration;
                POST_ERROR_MESSAGE(error, "Failed to parse ServerProcess.xml: " << ex.Message());
            }
            catch (const std::exception& ex)
            {
                error = OpenIPC_Error_Invalid_Configuration;
                POST_ERROR_MESSAGE(error, "Failed to parse ServerProcess.xml: " << ex.what());
            }
        }

        if (OpenIPC_PASS(error))
        {
            // If the server process XML was valid
            DataElement* element = document.RootElement()->FirstChild();
            while (element)
            {
                if (element->Name() == "Gui")
                {
                    config.GuiEnabled = element->FindAttribute("Enable")->ToBool();

                    DataElement* maxLogMessagesElement = element->FindChildByName("MaxLogMessageCount");
                    if (maxLogMessagesElement)
                    {
                        config.GuiMaxLogMessagesCount = maxLogMessagesElement->FindAttribute("Value")->ToUInt32();
                    }
                }
                else if (element->Name() == "AllowExternalConnections")
                {
                    config.AllowExternalConnections = element->FindAttribute("Value")->ToBool();
                }
                else if (element->Name() == "PortRange")
                {
                    config.StartPort = element->FindAttribute("Start")->ToUInt16();
                    config.EndPort   = element->FindAttribute("End")->ToUInt16();
                }
                else if (element->Name() == "ShutdownWhenNoClientsAreConnected")
                {
                    config.ShutdownWhenNoClientsAreConnected = element->FindAttribute("Enable")->ToBool();
                }
                else if (element->Name() == "OpenIPCHomeDirectory")
                {
                    _serverConfiguration.OpenIpcHomeDirectory = element->FindAttribute("Value")->Value();
                }

                element = element->NextSibling();
            }
        }

        //Perform some sanity checks on the port range
        if (OpenIPC_PASS(error) && (config.StartPort > config.EndPort))
        {
            error = OpenIPC_Error_Invalid_Configuration;
            POST_ERROR_MESSAGE(error, "The PortRange element in ServerProcess.xml has a start port that is greater than the end port");
        }

        if (OpenIPC_PASS(error) && (config.StartPort == 0))
        {
            error = OpenIPC_Error_Invalid_Configuration;
            POST_ERROR_MESSAGE(error, "The start port in the PortRange element in ServerProcess.xml cannot be 0");
        }

        if (OpenIPC_PASS(error) && (config.EndPort == 0))
        {
            error = OpenIPC_Error_Invalid_Configuration;
            POST_ERROR_MESSAGE(error, "The end port in the PortRange element in ServerProcess.xml cannot be 0");
        }

        if (OpenIPC_PASS(error))
        {
            _serverConfiguration.Loaded = true;
        }
    }

    return error;
}

OpenIPC_Error CommonImpl::_GetOpenIpcBinPath(std::string& path)
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;

    // Get the install directory of the OpenIPC
    size_t pathSize = 1024;
    std::vector<char> pathVector(pathSize);
    error = GetPath_Impl(&pathSize, pathVector.data());

    if (error == OpenIPC_Error_Bad_Argument)
    {
        pathVector.resize(pathSize);
        error = GetPath_Impl(&pathSize, pathVector.data());
    }

    if (OpenIPC_PASS(error))
    {
        path = std::string(pathVector.data());
    }

    return error;
}

OpenIPC_Error CommonImpl::_GetOpenIpcConfigPath(std::string& path)
{
    OpenIPC_Error error = _GetOpenIpcBinPath(path);
    if (OpenIPC_PASS(error))
    {
        const boost::filesystem::path binPath(path);
        const boost::filesystem::path configPath = binPath.parent_path() / "Config";

        path = configPath.string();
    }

    return error;
}

OpenIPC_Error CommonImpl::_SaveCrumbFile()
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;
    #if defined(HOST_WINDOWS)
        std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>, wchar_t> converter;
        std::wstring ws_crumbFileName = converter.from_bytes(_crumbFileName.c_str());

        _crumbFileHandle = CreateFileW(
            ws_crumbFileName.c_str(),
            GENERIC_WRITE | DELETE, //Open the file for writing and deleting
            NULL,                   //Don't allow any sharing (no one else can open a handle to this file)
            NULL,                   //No security attributes
            CREATE_NEW,             //Only create a new file (will fail if file already exists)
            FILE_ATTRIBUTE_NORMAL,  //Normal file (I thought about setting delete on close, but then this file wouldn't stick around like we want)
            NULL                    //No template
            );
        if (_crumbFileHandle == INVALID_HANDLE_VALUE)
        {
            DWORD winError = GetLastError();
            error = OpenIPC_Error_Cannot_Open_File;
            POST_ERROR_MESSAGE(error, "Failed to create file with exclusive access (file = \"" << _crumbFileName << "\", error = " << LOG_UINT16_HEX(winError) << ", message = \"" << GetErrorString(winError) << "\")");
        }
    #else
        _crumbFileDescriptor = open(_crumbFileName.c_str(), O_RDWR | O_CLOEXEC | O_CREAT | O_EXCL, S_IRWXU);
        if (_crumbFileDescriptor < 0)
        {
            //TODO: get specific error and make a message out of it.
            error = OpenIPC_Error_Cannot_Open_File;
            POST_ERROR_MESSAGE(error, "Failed to create file with exclusive access (file = \"" << _crumbFileName << "\")");
        }
    #endif
    return error;
}

void CommonImpl::_DeleteCrumbFile()
{
    #if defined(HOST_WINDOWS)
        if (_crumbFileHandle != INVALID_HANDLE_VALUE)
        {
            FILE_DISPOSITION_INFO fdi;
            fdi.DeleteFileW = TRUE; //This will cause the file to be deleted when we close the handle
            SetFileInformationByHandle(
                _crumbFileHandle,
                FileDispositionInfo,
                &fdi,
                sizeof(FILE_DISPOSITION_INFO)
                );
            CloseHandle(_crumbFileHandle);
            _crumbFileHandle = INVALID_HANDLE_VALUE;
        }
    #else
        if (_crumbFileDescriptor >= 0)
        {
            unlink(_crumbFileName.c_str());
            close(_crumbFileDescriptor);
            _crumbFileDescriptor = -1;
        }
    #endif
}

OpenIPC_Error CommonImpl::_LockAndDeleteFile(std::string const& file)
{
    OpenIPC_Error error = OpenIPC_Error_No_Error;
    #if defined(HOST_WINDOWS)
        HANDLE handle = INVALID_HANDLE_VALUE;
        std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>, wchar_t> converter;
        std::wstring ws_fileName = converter.from_bytes(file.c_str());
        handle = CreateFileW(
            ws_fileName.c_str(),
            DELETE,                    //We're just going to delete it.
            NULL,                      //No sharing, we want exclusive access
            NULL,                      //No security attributes
            OPEN_EXISTING,             //Make sure the file exists
            FILE_FLAG_DELETE_ON_CLOSE, //Delete it when we close the handle
            NULL                       //No template
            );
        if (handle == INVALID_HANDLE_VALUE)
        {
            error = OpenIPC_Error_Cannot_Open_File;
        }
        else
        {
            CloseHandle(handle);
        }
    #else
        int fd = open(file.c_str(), O_RDWR | O_CLOEXEC, S_IRWXU);
        if (fd < 0)
        {
            error = OpenIPC_Error_Cannot_Open_File;
        }
        else
        {
            unlink(file.c_str());
            close(fd);
        }
    #endif
    return error;
}
