/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
///
/// @see
/// @link Overview_Commone Common Overview@endlink
///
/// @page Overview_Common Overview: Common
/// The common subsystem provides functionality common between each subsystem.
///
/// See OpenIPC_CommonComponent for details of the Common functions.
///
//////////////////////////////////////////////////////////////////////////////
#pragma once

#include <Components/Common/Common.h>

#include <map>
#include <set>
#include <tuple>
#include <mutex>

///
/// @brief Contains the methods of the Common component.
class CommonImpl : public OpenIPC_CommonComponent
{
public:
    static CommonImpl& GetInstance();

    ///
    /// @brief Gets the path that the OpenIPC is installed in.
    ///
    /// @param[in, out] pathSize
    ///     Going in, the maximum number of characters that can be written to the buffer.
    ///     Going out, the number of characters that are required for the full path.
    /// @param[out] path
    ///     The buffer to write the path to.
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure.
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    /// @retval OpenIPC_Error_Null_Pointer
    ///     The @a path or @a pathSize parameter is NULL.
    /// @retval OpenIPC_Error_Bad_Argument
    ///     The given buffer is tool small. The @a pathSize parameter will contain the needed size.
    OpenIPC_Error GetPath_Impl(
        IN OUT size_t* pathSize,
        OUT char*      path);

    ///
    /// @brief Gets the client ID that the current thread belongs to.
    /// If the thread was not created by the server it gets a client ID
    /// of -1. -1 is also the client ID used by in-process clients.
    ///
    /// @param[out] clientId
    ///     The client ID.
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure.
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    /// @retval OpenIPC_Error_Null_Pointer
    ///     The @a clientId parameter is NULL.
    OpenIPC_Error GetClientId_Impl(
        OUT int32_t* clientId);

    ///
    /// @brief Sets the client ID that the current thread belongs to.
    /// The Server assigns the current thread to a client ID by calling
    /// this method.
    ///
    /// @param[in] clientId
    ///     The client ID.
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure.
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    OpenIPC_Error SetClientId_Impl(
        IN int32_t clientId);

    ///
    /// @brief Registers a new client.
    ///
    /// @param[in] clientId
    ///     The client's unique ID.
    /// @param[in] clientIdentifier
    ///     The identifier string for this client.
    /// @param[in] isClientLocal
    ///     Whether or not the client is local.
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure.
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    /// @retval OpenIPC_Error_Null_Pointer
    ///     The @a 'clientIdentifier' parameter is NULL.
    OpenIPC_Error RegisterNewClient_Impl(
        IN int32_t      clientId,
        IN const char*  clientIdentifier,
        IN OpenIPC_Bool isClientLocal);

    ///
    /// @brief Notifies any subscribers that the client is closed.
    ///
    /// @param[in] clientId
    ///     The client ID.
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure.
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    OpenIPC_Error CloseClient_Impl(
        IN int32_t clientId);

    ///
    /// @brief Gets the number of registered clients.
    ///
    /// @param[out] numClients
    ///     The number of registered clients.
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure.
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    /// @retval OpenIPC_Error_Null_Pointer
    ///     The @a numClients parameter is NULL.
    OpenIPC_Error GetNumRegisteredClients_Impl(
        OUT uint32_t* numClients);

    ///
    /// @brief Gets the list of registered clients.
    ///
    /// @param[in] maxNumClients
    ///     The maximum number of registered clients to return.
    /// @param[out] numClients
    ///     The number of clients returned.
    /// @param[out] clients
    ///     The list of clients.
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure.
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    /// @retval OpenIPC_Error_Null_Pointer
    ///     The @a clients or @a numClients parameter is NULL.
    OpenIPC_Error GetRegisteredClients_Impl(
        IN uint32_t                         maxNumClients,
        OUT uint32_t*                       numClients,
        OUT OpenIPC_CommonClientDescriptor* clients);

    ///
    /// @brief Gets whether the client is on the same machine as the server.
    ///
    /// @param[in] clientId
    ///     The client ID.
    ///
    /// @param[out] isLocal
    ///     True if the client is on the same machine as the server
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure.
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    OpenIPC_Error IsClientLocal_Impl(
        IN int32_t        clientId,
        OUT OpenIPC_Bool* isLocal);

    ///
    /// @brief Adds a callback function to be called whenever a client
    /// is connected/disconnected.
    ///
    /// @param[in] callback
    ///     The function to call.
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure.
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    OpenIPC_Error RegisterClientCallback_Impl(
        IN OpenIPC_CommonClientCallback_Function callback,
        IN void*                                 data);

    ///
    /// @brief Removes a callback function.
    ///
    /// @param[in] callback
    ///     The callback function.
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure.
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    OpenIPC_Error UnregisterClientCallback_Impl(
        IN OpenIPC_CommonClientCallback_Function callback,
        IN void*                                 data);

    ///
    /// @brief Initializes logging based on the logging XML configuration file.
    ///
    /// @param[in] suffix
    ///     The log file suffix (null for default)
    /// @param[in] persistLoggingPreset
    ///     Specifies if logging preset changes should be persisted to the logging xml file
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure.
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    OpenIPC_Error InitializeLogging_Impl(
        IN const char* suffix,
        IN bool        persistLoggingPreset);

    ///
    /// @brief Uninitializes logging
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure.
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    OpenIPC_Error UninitializeLogging_Impl(
        void);

    ///
    /// @brief Gets the location of the OpenIPC home directory.
    ///
    /// Examples:
    ///     - C:\Users\bob\.OpenIPC
    ///     - /home/bob/.OpenIPC
    ///
    /// @param[out]    directory A pointer to a buffer where the directory path can be stored
    /// @param[in,out] size      Going in: the size of the buffer. Going out: The size of the directory path
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    /// @retval OpenIPC_Error_Bad_Argument
    ///     Indicates that the given buffer size is too small. @a size will contain the required size.
    /// @retval OpenIPC_Error_Null_Pointer
    ///     Indicates that one of the parameters is NULL.
    OpenIPC_Error GetOpenIpcHomeDirectory_Impl(
        IN char*       directory,
        IN OUT size_t* size);

    ///
    /// @brief Gets the server configuration.
    ///
    /// This is mostly the configuration parameters specified in the ServerProcess.xml file
    ///
    /// @param[out] serverConfiguration A pointer to a struct that will be filled with the server configuration
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    /// @retval OpenIPC_Error_Null_Pointer
    ///     The give @a serverConfiguration parameter is NULL.
    OpenIPC_Error GetServerConfiguration_Impl(
        OUT OpenIPC_CommonServerConfiguration* serverConfiguration
        );

    ///
    /// @brief Flush all pending log messages and begin writing to new log
    ///        files.
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    OpenIPC_Error FlushLogs_Impl();

    ///
    /// @brief Flush all pending log messages, delete all non-archived log
    ///        files, and begin writing to new log files.
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    OpenIPC_Error ClearLogs_Impl();

    ///
    /// @brief Flush all pending log messages, archive all non-archived log
    ///        files, and begin writing to new log files.
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    OpenIPC_Error ArchiveLogs_Impl();

    ///
    /// @brief If the current logging configuration is different than the
    ///        configuration on disk, then update the configuration.
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    OpenIPC_Error SaveLoggingConfiguration_Impl();

    ///
    /// @brief Creates a file that will be left behind if we don't shut down cleanly,
    ///        and checks if any such files are left from previous runs
    ///
    /// @param[out]    found Will be set to true if we found a file left behind
    ///                      from a previous instance that wasn't shut down.
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    /// @retval OpenIPC_Error_Null_Pointer
    ///     Indicates that one of the parameters is NULL.
    OpenIPC_Error InitializeProcessKilledDetector_Impl(
        OUT OpenIPC_Bool* found);

    ///
    /// @brief Deletes the file created by OpenIPC_CommonInitializeProcessKilledDetector,
    ///        which means that this run shut down cleanly.
    ///
    /// @return (OpenIPC_Error): A code indicating success or failure
    /// @retval OpenIPC_Error_No_Error
    ///     Indicates success.
    OpenIPC_Error UninitializeProcessKilledDetector_Impl();

    static OpenIPC_Error GetPath_Mediator(
        IN OUT size_t* pathSize,
        OUT char*      path)
    {
        return GetInstance().GetPath_Impl(pathSize, path);
    }

    static OpenIPC_Error GetClientId_Mediator(
        OUT int32_t* clientId)
    {
        return GetInstance().GetClientId_Impl(clientId);
    }

    static OpenIPC_Error SetClientId_Mediator(
        IN int32_t clientId)
    {
        return GetInstance().SetClientId_Impl(clientId);
    }

    static OpenIPC_Error RegisterNewClient_Mediator(
        IN int32_t      clientId,
        IN const char*  clientIdentifier,
        IN OpenIPC_Bool isClientLocal)
    {
        return GetInstance().RegisterNewClient_Impl(clientId, clientIdentifier, isClientLocal);
    }

    static OpenIPC_Error CloseClient_Mediator(
        IN int32_t clientId)
    {
        return GetInstance().CloseClient_Impl(clientId);
    }

    static OpenIPC_Error GetNumRegisteredClients_Mediator(
        OUT uint32_t* numClients)
    {
        return GetInstance().GetNumRegisteredClients_Impl(numClients);
    }

    static OpenIPC_Error GetRegisteredClients_Mediator(
        IN uint32_t                         maxNumClients,
        OUT uint32_t*                       numClients,
        OUT OpenIPC_CommonClientDescriptor* clients)
    {
        return GetInstance().GetRegisteredClients_Impl(maxNumClients, numClients, clients);
    }

    static OpenIPC_Error IsClientLocal_Mediator(
        IN int32_t        clientId,
        OUT OpenIPC_Bool* isLocal)
    {
        return GetInstance().IsClientLocal_Impl(clientId, isLocal);
    }

    static OpenIPC_Error RegisterClientCallback_Mediator(
        IN OpenIPC_CommonClientCallback_Function callback,
        IN void*                                 data)
    {
        return GetInstance().RegisterClientCallback_Impl(callback, data);
    }

    static OpenIPC_Error UnregisterClientCallback_Mediator(
        IN OpenIPC_CommonClientCallback_Function callback,
        IN void*                                 data)
    {
        return GetInstance().UnregisterClientCallback_Impl(callback, data);
    }

    static OpenIPC_Error InitializeLogging_Mediator(
        IN const char* suffix,
        IN bool        persistLoggingPreset)
    {
        return GetInstance().InitializeLogging_Impl(suffix, persistLoggingPreset);
    }

    static OpenIPC_Error UninitializeLogging_Mediator(
        void)
    {
        return GetInstance().UninitializeLogging_Impl();
    }

    static OpenIPC_Error GetOpenIpcHomeDirectory_Mediator(
        IN char*       directory,
        IN OUT size_t* size)
    {
        return GetInstance().GetOpenIpcHomeDirectory_Impl(directory, size);
    }

    static OpenIPC_Error GetServerConfiguration_Mediator(
        OUT OpenIPC_CommonServerConfiguration* serverConfiguration)
    {
        return GetInstance().GetServerConfiguration_Impl(serverConfiguration);
    }

    static OpenIPC_Error FlushLogs_Mediator()
    {
        return GetInstance().FlushLogs_Impl();
    }

    static OpenIPC_Error ClearLogs_Mediator()
    {
        return GetInstance().ClearLogs_Impl();
    }

    static OpenIPC_Error ArchiveLogs_Mediator()
    {
        return GetInstance().ArchiveLogs_Impl();
    }

    static OpenIPC_Error SaveLoggingConfiguration_Mediator()
    {
        return GetInstance().SaveLoggingConfiguration_Impl();
    }

    static OpenIPC_Error InitializeProcessKilledDetector_Mediator(OpenIPC_Bool* found)
    {
        return GetInstance().InitializeProcessKilledDetector_Impl(found);
    }

    static OpenIPC_Error UninitializeProcessKilledDetector_Mediator()
    {
        return GetInstance().UninitializeProcessKilledDetector_Impl();
    }

private:
    CommonImpl();
    OpenIPC_Error _DisableLogging();
    OpenIPC_Error _ReenableLogging();
    OpenIPC_Error _PopulateServerConfiguration();
    OpenIPC_Error _GetOpenIpcBinPath(std::string& path);
    OpenIPC_Error _GetOpenIpcConfigPath(std::string& path);
    OpenIPC_Error _SaveCrumbFile();
    void _DeleteCrumbFile();
    OpenIPC_Error _LockAndDeleteFile(std::string const& file);

    static CommonImpl _instance;

    std::map<int32_t, OpenIPC_CommonClientDescriptor> _clientDescriptorMap;
    std::mutex _clientDescriptorMutex;

    std::mutex _clientCallbackMutex;
    std::set<std::tuple<OpenIPC_CommonClientCallback_Function, void*>> _clientCallbacks;

    std::mutex _serverConfigurationMutex;
    struct ServerConfigurationContainer
    {
        bool Loaded{ false };
        OpenIPC_CommonServerConfiguration Configuration{ false, 0, false, 0, 0, false };
        std::string OpenIpcHomeDirectory;
    };
    ServerConfigurationContainer _serverConfiguration;

    std::mutex _loggingOperationMutex;

    std::string _logfileSuffix;
    std::string _crumbFileName;
    static const std::string _crumbFileSuffix;
    static const std::string _stacktraceSuffix;
    static const std::string _crashdumpSuffix;
    #if defined(HOST_WINDOWS)
        HANDLE _crumbFileHandle;
    #else
        int _crumbFileDescriptor;
    #endif
    bool _persistLoggingPreset{ true };
};
