/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#pragma once

#include <StructuredData.h>

#include <Foundation/Error/Error.h>
#include <Components/Common/Common.h>

#include <string>
#include <map>
#include <memory>

class ConfigurationManager
{
public:

    // Sets the name of the configuration to be selected
    void SetConfigName(const std::string& configName);

    // Gets the name of the configuration to be selected
    const std::string& GetConfigName() const;

    // Selects the configuration that is set if needed
    void Finalize();

    // Selects a configuration by name
    void SelectConfig(const std::string& configName, const std::string& configFilePath = "");

    // Selects a configuration by file path
    void SelectConfigFile(const std::string& configFilePath);

    // Loads a probe configuration by name
    void LoadReferencedProbeConfig(const std::string& probeConfigName);

    // Sets automatic configuration is enabled
    void SetAutomaticConfigurationEnabled(bool enabled);

    // Returns whether automatic configuration is enabled
    bool IsAutomaticConfigurationEnabled() const;

    // Sets the parameter values for the selected configuration
    void SetParameterValues(const DataAttribute::ParameterValues& parameterValues);

    // Gets the root element of the selected configuration
    const DataElement* GetPrimaryElement();

    // Gets the root element of the selected probe configuration
    const DataElement* GetProbeElement();

    // Gets the root element of the selected probe configuration for the
    // probe plugin of the specified name
    const DataElement* GetProbePluginElement(const std::string& pluginName);

    // Gets the root element of the selected run control configuration
    const DataElement* GetRunControlElement();

    // Gets the name of all enabled probe plugins in the selected configuration
    std::vector<std::string> GetEnabledProbePluginNames() const;

    // Gets the name of all enabled run control plugins in the selected configuration
    std::vector<std::string> GetEnabledRunControlPluginNames() const;

private:
    struct Parameter
    {
        std::string Name;
        bool HasDefaultValue;
        std::string DefaultValue;
        std::string ValidValues;
        const DataElement* Element;
    };

    void _ParseParameters(const DataElement& element);
    void _ParseParameterValues(const DataElement& element);
    void _ValidateParameterValue(const Parameter& parameter, const std::string& value);
    void _LoadOpenIpcConfig(const std::string& configFilePath, const std::string& configName);
    void _LoadProbeConfig();
    void _ResolveProbeConfigIncludes(DataElement& probePluginElement);
    void _LoadRunControlConfig();
    void _LoadReferencedRunControlConfig(const std::string& runControlConfigName);
    void _EnsureParametersResolved();
    void _EnsureConfigurationLoadedAndParametersResolved();
    void _ProcessDefaultConfigFile(const std::string& configFilePath, const std::string& schemaFilePath, std::string& configName, std::string& referencedConfigFilePath, bool& automaticConfigurationEnabled);
    void _IncludeConfigDirectory();

    static void _ResolveConfigFilePath(const std::string& configFilePath, std::string& resolvedConfigFilePath, std::string& schemaFilePath);
    static std::string _GetOpenIpcPath();
    static std::string _GetConfigDirectoryPath();
    static void _ReportDataStoreExceptions(const std::vector<DataStoreException>& dataStoreExceptions);

    std::string _configName;
    std::string _configFilePath;

    bool _configurationLoaded{ false };
    bool _automaticConfigurationEnabled{ false };

    DataDocument _primaryConfigDocument;
    DataElement* _primaryElement{ nullptr };

    std::vector<std::shared_ptr<DataDocument>> _probeConfigDocuments;
    DataElement* _probeElement{ nullptr };
    std::vector<DataElement*> _probeElements;
    std::vector<DataElement*> _probePluginElements;

    DataDocument _runControlConfigDocument;
    DataElement* _runControlElement{ nullptr };

    std::map<std::string, Parameter> _parameters;
    std::map<std::string, std::string> _parameterValues;
    bool _parametersResolved{ false };
};
