#include <math.h>
#include <emmintrin.h>
#include "ASDProtocol.h"
#include "InterfaceInstanceI2cASD.hpp"

ASDProtocol::ASDProtocol(std::shared_ptr<Transport> transport, std::shared_ptr<ProbeInstanceASD> probe, std::shared_ptr<ConnectionParameters> connectionParameters)
{
	UseTransport = transport;
	probeInstance = probe;
	this->init();
	//TODO: Need to refactor this code
	this->Timeout = connectionParameters->protocolParameters->timeout;
	this->payloadSize = connectionParameters->protocolParameters->payloadSize;
	this->maxNumBuffers = connectionParameters->protocolParameters->maxNumBuffers;
	this->supportedJtagChains = connectionParameters->protocolParameters->supportedJtagChains;
	this->supportedI2cBusesCount = connectionParameters->protocolParameters->supportedI2cBusesCount;
	for(unsigned int i=0; i<connectionParameters->protocolParameters->supportedI2cBusesCount; i++) {
		this->supportedI2cBusAddresses[i].busSelect = connectionParameters->protocolParameters->supportedI2cBusAddresses[i].busSelect;
		this->supportedI2cBusAddresses[i].busSpeed = connectionParameters->protocolParameters->supportedI2cBusAddresses[i].busSpeed;
	}
	this->connect();
	connectionParameters->protocolParameters->timeout = this->Timeout;
	connectionParameters->protocolParameters->payloadSize = this->payloadSize;
	connectionParameters->protocolParameters->maxNumBuffers = this->maxNumBuffers;
	connectionParameters->protocolParameters->supportedJtagChains = this->supportedJtagChains;
	connectionParameters->protocolParameters->supportedI2cBusesCount = this->supportedI2cBusesCount;
	for(unsigned int i=0; i<this->supportedI2cBusesCount; i++) {
		connectionParameters->protocolParameters->supportedI2cBusAddresses[i].busSelect = this->supportedI2cBusAddresses[i].busSelect;
		connectionParameters->protocolParameters->supportedI2cBusAddresses[i].busSpeed = this->supportedI2cBusAddresses[i].busSpeed;
	}
}

ASDProtocol::ASDProtocol()
{
	this->init();
}

void ASDProtocol::init()
{
	cmd_stat = 0;
	setWaitPRDYtimeoutSent = false;
	lastjtagstate = JtagStateEncode::JtagTLR;
	// Setup padding structure
	padding.drPaddingNearTDI = 0;
	padding.drPaddingNearTDO = 0;
	padding.irPaddingNearTDI = 0;
	padding.irPaddingNearTDO = 0;
	jtagSettings.JtagChainProtocolId = 255;
	jtagSettings.JtagDivisor = 0;
	jtagSettings.JtagPrescaler = 0;
	jtagSettings.SendWaitSync = false;
	params.InterfacePaddingSettings = &padding;
	params.JtagChainSettings = &jtagSettings;
	params.JtagChainSelectMode = JTAG_CHAIN_SELECT_MODE_SINGLE;
	params.JtagChainSettings->MultiBundleExecute = false;
	params.JtagChainSettings->MultiBundleFinalInterface = false;
	params.JtagDriverMode = JTAG_DRIVER_MODE_SOFTWARE;
	params.JtagSyncDelayValue = DEFAULT_JTAG_SYNC_DELAY_VALUE;
	params.JtagSyncTimeoutValue = DEFAULT_JTAG_SYNC_TIMEOUT_VALUE;
	multiChainBitVec.clear();
	multiChainDataActive = false;
	largestId = 0;
}

Connection_Error ASDProtocol::ClearPRDYTimeout(void)
{
	write_message.buffer[write_message_offset++] = CLEAR_WAITPRDY_TIMEOUT;
	write_message.buffer[write_message_offset++] = STATE_CHANGE_BYTE + (char)JtagTLR;
	write_message.buffer[write_message_offset++] = STATE_CHANGE_BYTE + (char)JtagRTI;
	return No_Error;
}

Connection_Error ASDProtocol::SetHWStateToTheOutMsg(JtagChainParameters *_params)
{
	char tmp_buffer[35];
	uint16_t cnt = 0;
	Connection_Error ret = No_Error;

	if (!_params)
		return ret;

	if (_params->InterfacePaddingSettings->drPaddingNearTDI != padding.drPaddingNearTDI) {
		padding.drPaddingNearTDI = _params->InterfacePaddingSettings->drPaddingNearTDI;
		tmp_buffer[cnt++] = 0x02;
		tmp_buffer[cnt++] = _params->InterfacePaddingSettings->drPaddingNearTDI & 0xFF;
	}

	if (_params->InterfacePaddingSettings->drPaddingNearTDO != padding.drPaddingNearTDO) {
		padding.drPaddingNearTDO = _params->InterfacePaddingSettings->drPaddingNearTDO;
		tmp_buffer[cnt++] = 0x03;
		tmp_buffer[cnt++] = _params->InterfacePaddingSettings->drPaddingNearTDO & 0xFF;
	}

	if (_params->InterfacePaddingSettings->irPaddingNearTDI != padding.irPaddingNearTDI) {
		padding.irPaddingNearTDI = _params->InterfacePaddingSettings->irPaddingNearTDI;
		tmp_buffer[cnt++] = 0x04;
		tmp_buffer[cnt++] = _params->InterfacePaddingSettings->irPaddingNearTDI & 0xFF;
		tmp_buffer[cnt++] = (_params->InterfacePaddingSettings->irPaddingNearTDI >> 8) & 0xFF;
	}

	if (_params->InterfacePaddingSettings->irPaddingNearTDO != padding.irPaddingNearTDO) {
		padding.irPaddingNearTDO = _params->InterfacePaddingSettings->irPaddingNearTDO;
		tmp_buffer[cnt++] = 0x05;
		tmp_buffer[cnt++] = _params->InterfacePaddingSettings->irPaddingNearTDO & 0xFF;
		tmp_buffer[cnt++] = (_params->InterfacePaddingSettings->irPaddingNearTDO >> 8) & 0xFF;
	}

	if (_params->JtagChainSettings->SendWaitSync) {
		tmp_buffer[cnt++] = WAIT_SYNC;
		tmp_buffer[cnt++] = params.JtagSyncTimeoutValue & 0xFF; // LSB FIRST
		tmp_buffer[cnt++] = (params.JtagSyncTimeoutValue & 0xFF00) >> 8; // MSB
		tmp_buffer[cnt++] = params.JtagSyncDelayValue & 0xFF; // LSB FIRST
		tmp_buffer[cnt++] = (params.JtagSyncDelayValue & 0xFF00) >> 8; // MSB
		_params->JtagChainSettings->SendWaitSync = false;
	}

	//FPGA configuration parameter:
	//command 0x02 set dr_prefix == drPaddingNearTDI 1 byte of data
	//command 0x03 set dr_postfix == drPaddingNearTDO 1 byte of data
	//command 0x04 set ir_prefix == irPaddingNearTDI 2 bytes of data
	//command 0x05 set ir_postfix == irPaddingNearTDO 2 bytes of data
	//command 0x18 set padding bit == drValueConstantOne 1 byte of data


	if (_params->JtagChainSettings->JtagChainProtocolId != jtagSettings.JtagChainProtocolId) {
		jtagSettings.JtagChainProtocolId = _params->JtagChainSettings->JtagChainProtocolId;
		jtagSettings.JtagDivisor = _params->JtagChainSettings->JtagDivisor;
		jtagSettings.JtagPrescaler = _params->JtagChainSettings->JtagPrescaler;
		PPI_LOG(probeInstance->deviceID, PPI_debugNotification,
			"Detected scan chain ID request for: " << jtagSettings.JtagChainProtocolId);

		if (this->params.JtagChainSelectMode == JTAG_CHAIN_SELECT_MODE_SINGLE) {
			tmp_buffer[cnt++] = WRITE_PINS;
			tmp_buffer[cnt++] = SCAN_CHAIN_SELECT + (jtagSettings.JtagChainProtocolId & MAX_SCAN_CHAINS);
		}
		else {
			uint8_t toSend = 0;
			if (_params->JtagChainSettings->MultiBundleExecute) {
				// build multiChainBitVec for multi bundle execution case
				multiChainDataActive = true;
				if (jtagSettings.JtagChainProtocolId > largestId)
					largestId = jtagSettings.JtagChainProtocolId;
				toSend = largestId / BITS_IN_VECTOR_BYTE;
				ret = this->BuildBitVecBuffer(toSend);
				if (!CONNECTION_PASS(ret))
					return ret;
				if (!_params->JtagChainSettings->MultiBundleFinalInterface) {
					// return to avoid sending paddings
					if (cnt != 0) {
						PPI_LOG(probeInstance->deviceID, PPI_debugNotification,
							"Detected Multi_Bundle_Error condition ");
						return Multi_Bundle_Error;
					}
					else
						return ret;
				}
			}
			else {
				// build multiChainBitVec for single bundle execution case
				multiChainBitVec.clear();
				toSend = jtagSettings.JtagChainProtocolId / BITS_IN_VECTOR_BYTE;
				ret = this->BuildBitVecBuffer(toSend);
				if (!CONNECTION_PASS(ret))
					return ret;
			}
			tmp_buffer[cnt++] = WRITE_PINS;
			tmp_buffer[cnt++] = SCAN_CHAIN_SELECT + (toSend & MAX_SCAN_CHAINS);
			for (auto it = multiChainBitVec.begin(); it != multiChainBitVec.end(); it++) {
				tmp_buffer[cnt++] = *it;
			}
		}
		tmp_buffer[cnt++] = CMD_WRITECFG_JTAG_FREQ;
		tmp_buffer[cnt++] = ((jtagSettings.JtagPrescaler & 0x3) << 6) | (jtagSettings.JtagDivisor & 0x3F);
		// tmp_buffer is ready, clear all flags
		if (_params->JtagChainSettings->MultiBundleFinalInterface) {
			multiChainDataActive = false;
			multiChainBitVec.clear();
			largestId = 0;
		}
	}

	if (cnt > 0) {
		ret = checkPayloadSize(cnt);
		if (!CONNECTION_PASS(ret))
			return ret;
		memcpy_s(write_message.buffer + write_message_offset,
			payloadSize - write_message_offset,
			tmp_buffer, cnt);

		write_message_offset += cnt;
	}

	return ret;
}

Connection_Error ASDProtocol::DecryptData(struct message *message)
{
	return No_Error;
}

Connection_Error ASDProtocol::EncryptData(struct message *message)
{
	return No_Error;
}

Connection_Error ASDProtocol::ProcessReadBuffers(struct message *message, std::shared_ptr<std::list<std::shared_ptr<OutputBuffers>>> tmp_input_list)
{
	unsigned int currentBufferLocation = 0;
	Connection_Error error = No_Error;

	if (!tmp_input_list || !message)
		return error;

	while (!tmp_input_list->empty() && currentBufferLocation < message->buffer_size)
	{
		// recommended fix from kw for spectre.variant1 warning
		_mm_lfence();
		char command;
		command = message->buffer[currentBufferLocation];
		uint16_t size = command & 0x3F;
		uint16_t copied_size = 0;

		if (command != tmp_input_list->front()->cmd && !tmp_input_list->front()->buffer_incomplete) {
			error = Error_Processing_Received_Data;
			goto end;
		}
		if (msgType != JTAG_MESSAGE_TYPE)
			size = tmp_input_list->front()->pm_message_respond_size;

		if (!tmp_input_list->front()->buffer_incomplete && msgType != DMA_MESSAGE_TYPE_READ && msgType != DMA_MESSAGE_TYPE_WRITE)
			currentBufferLocation++;

		if (tmp_input_list->front()->buffer_incomplete || (msgType != JTAG_MESSAGE_TYPE) ||
			(probeInstance != nullptr &&
				!probeInstance->ProcessSpecialCommands(command, tmp_input_list->front(), message->buffer[currentBufferLocation], &size))
			|| probeInstance == nullptr) {

			if (tmp_input_list->front()->buffer_incomplete) {
				uint16_t tmp_size = tmp_input_list->front()->size;
				if (tmp_size == 0)
					tmp_size = 64;
				size = tmp_size - (tmp_input_list->front()->current_position * 8);
			}
			else
				tmp_input_list->front()->size = size;

			if (size == 0)
				size = 64;
			size = (size + 7) / 8;
			copied_size = size;

			if ((size + currentBufferLocation) >= message->buffer_size) {
				copied_size = message->buffer_size - (uint16_t)currentBufferLocation;
				tmp_input_list->front()->current_position = copied_size;
				tmp_input_list->front()->buffer_incomplete = true;
			}
			if (copied_size > 0) {
				memcpy_s(tmp_input_list->front()->buffer,
					tmp_input_list->front()->buffer_size,
					message->buffer + currentBufferLocation, copied_size);

				if (copied_size == size) {
					tmp_input_list->pop_front();
				}
			}
		}
		else {
			tmp_input_list->pop_front();
		}
		currentBufferLocation += copied_size;
	}
end:
	return error;
}

Connection_Error ASDProtocol::ProcessReadBuffersI2c(struct message *message, std::shared_ptr<std::list<std::shared_ptr<OutputBuffers>>> tmp_input_list)
{
	Connection_Error error = No_Error;
	std::list<std::shared_ptr<OutputBuffers>>::iterator it;
	unsigned int currentBufferLocation = 0;

	if (!tmp_input_list || !message) {
		error = I2C_Response_Error;
		goto end;
	}

	while (!tmp_input_list->empty()) {
		it = tmp_input_list->begin();
		if (((*it)->cmd & I2C_CMD_MASK) == I2C_READ_CMD) {
			error = this->ProcessI2cReadResponse(message, currentBufferLocation, (*it));
			currentBufferLocation = currentBufferLocation + I2C_CMD_SIZE + (unsigned int)(*it)->buffer_size;
		}
		else if (((*it)->cmd & I2C_CMD_MASK) == I2C_WRITE_CMD) {
			error = this->ProcessI2cWriteResponse(message, currentBufferLocation, (*it));
			currentBufferLocation = currentBufferLocation + I2C_CMD_SIZE;
		}
		else {
			error = I2C_Unknown_Response;
			break;
		}
		if (error != No_Error)
			break;
		tmp_input_list->pop_front();
		if (currentBufferLocation >= message->buffer_size)
			break;
	}

end:
	return error;
}

Connection_Error ASDProtocol::ProcessI2cReadResponse(struct message *message, size_t pos, std::shared_ptr<OutputBuffers> tmp_input)
{
	Connection_Error error = No_Error;

	bool ack = (message->buffer[pos+1] & I2C_WRITE_ACK_MASK) ? true: false;  // ack (1-bit)
	uint8_t n_size = message->buffer[pos+1] & I2C_SIZE_MASK;  // n-size (4-bit)
	std::vector<uint8_t> buff;

	if (tmp_input->addressAck != nullptr) {
		ack ? *tmp_input->addressAck = 0x01 : *tmp_input->addressAck = 0x00;
	}

	for (uint16_t i = 2; i < n_size + 2; i++) {
		buff.push_back(message->buffer[pos + i]);
	}

	memcpy_s(tmp_input->buffer,
		buff.size(),
		reinterpret_cast<uint8_t*>(buff.data()), buff.size());

	return error;
}

Connection_Error ASDProtocol::ProcessI2cWriteResponse(struct message *message, size_t pos, std::shared_ptr<OutputBuffers> tmp_input)
{
	Connection_Error error = No_Error;
	bool ack = (message->buffer[pos + 1] & I2C_WRITE_ACK_MASK) ? true : false;  // ack (1-bit)
	uint8_t n_size = message->buffer[pos+1] & I2C_SIZE_MASK;  // n-size (4-bit)
	std::vector<uint8_t> acks;

	if (tmp_input->addressAck != nullptr) {
		ack ? *tmp_input->addressAck = 0x01 : *tmp_input->addressAck = 0x00;
	}

	// Notes: lastDataAck=nullptr when python caller is lastDataAck=False
	if (tmp_input->lastDataAck == nullptr) {
		goto end;
	}

	// Notes: n_size + 1 because (address ack + data ack(s)) requires by PPI API
	for (uint8_t i = 0; i < n_size + 1; i++) {
		ack ? acks.push_back(0x01) : acks.push_back(0x00);
	}

	memcpy_s(tmp_input->lastDataAck, acks.size(), reinterpret_cast<uint8_t*>(acks.data()), acks.size());

end:
	return error;
}

Connection_Error ASDProtocol::doubleShiftCheck(JtagStateEncode state)
{
	Connection_Error error = No_Error;
	if ((lastjtagstate == JtagShfDR && lastjtagstate == state) || (lastjtagstate == JtagShfIR && lastjtagstate == state)) {
		error = checkPayloadSize(sizeof(state));
		if (CONNECTION_PASS(error))
			write_message.buffer[write_message_offset++] = STATE_CHANGE_BYTE + (char)JtagSelDR;
	}

	return error;
}
Connection_Error ASDProtocol::checkPayloadSize(size_t size)
{
	Connection_Error error = No_Error;
	if ((write_message_offset + size) > write_message.max_payload) {
		error = SendData(false, No_Error);
	}
	return error;
}

Connection_Error ASDProtocol::AddOutputBuffer(char* buffer, size_t buffer_size, char cmd, uint16_t pm_message_size, GotoStateOptions params)
{
	std::shared_ptr<OutputBuffers> out_buffer = std::make_shared<OutputBuffers>();
	if (!out_buffer)
		return Bad_Argument;
	out_buffer->buffer = buffer;
	out_buffer->buffer_size = buffer_size;
	out_buffer->buffer_incomplete = false;
	out_buffer->cmd = cmd;
	out_buffer->current_position = 0;
	out_buffer->pm_message_respond_size = pm_message_size;
	out_buffer->size = 0;
	out_buffer->lastDataAck = params.writeParams.lastDataAck;
	if (params.isI2cRead) {
		out_buffer->addressAck = params.readParams.addressAck;
	}
	else {
		out_buffer->addressAck = params.writeParams.addressAck;
	}

	if (!mutex_lock(list_mutex)) {
		return Could_Not_Handle_Mutex;
	}
	if (out_buffers_list == NULL) {
		out_buffers_list = std::make_shared<std::list<std::shared_ptr<OutputBuffers>>>();
	}
	out_buffers_list->push_back(out_buffer);
	if (!mutex_unlock(list_mutex)) {
		return Could_Not_Handle_Mutex;
	}
	return No_Error;
}

Connection_Error ASDProtocol::BuildBitVecBuffer(uint8_t toSend)
{
	Connection_Error ret = No_Error;
	if (toSend > MAX_BIT_VECTOR_BYTES - 1) {
		ret = Multi_Bundle_Error;
	}
	else {
		if (multiChainBitVec.size() <= toSend) {
			multiChainBitVec.resize(toSend + 1);
		}
		multiChainBitVec.at(toSend) |= 1 << (jtagSettings.JtagChainProtocolId % BITS_IN_VECTOR_BYTE);
	}
	return ret;
}

Connection_Error ASDProtocol::ShiftJTAGProtocol(OpenIPC_DeviceId deviceId, JtagStateEncode state, char *writeBuffer, size_t writeSize, char* readBuffer, size_t readSize, GotoStateOptions *goto_state_opt)
{
	Connection_Error error = No_Error;
	char scan_cmd = 0;
	char *writeBufferTmp = writeBuffer;
	char *readBufferTmp = readBuffer;
	size_t readBufferTmpSize = readSize;
	size_t writeNumberBytes = (writeSize + 7) / 8;
	size_t readNumberBytes = (readSize + 7) / 8;
	size_t writeNumBits = writeSize;
	size_t readNumBits = readSize;
	bool lastPacket = false;
	//if we are sending the data to multiChain, no need to build the data for every chain.
	//send it only with the last one
	if (multiChainDataActive)
		return error;

	if ((JtagStateEncodeASD)state != JtagNoStateChange) {
		error = checkPayloadSize(sizeof(state));
		if (!CONNECTION_PASS(error)) {
			return error;
		}
		write_message.buffer[write_message_offset++] = STATE_CHANGE_BYTE + (char)state;
		lastjtagstate = state;
		if (goto_state_opt != NULL && goto_state_opt->options != NULL && goto_state_opt->options->waitForTrigger) {
			error = checkPayloadSize(3);
			if (!CONNECTION_PASS(error)) {
				return error;
			}
			if (goto_state_opt->number_of_clocks_in_state == 0) goto_state_opt->number_of_clocks_in_state = 1;
			if (!setWaitPRDYtimeoutSent) {
				write_message.buffer[write_message_offset++] = 0x6; //set waitprdy timeout
				write_message.buffer[write_message_offset++] = (char)((log(goto_state_opt->number_of_clocks_in_state) / log(2)) + 1);
				setWaitPRDYtimeoutSent = true;
			}
			write_message.buffer[write_message_offset++] = WAIT_PRDY;
		}
		else if (goto_state_opt != NULL && goto_state_opt->number_of_clocks_in_state > 1) {
			uint32_t  count = goto_state_opt->number_of_clocks_in_state;
			char waitCyclesComand = CMD_WAITCYCLES;
			if (state == JtagTLR || state == JtagRTI || state == JtagPauDR || state == JtagPauIR)  waitCyclesComand |= 0x1;
			PPI_LOG(probeInstance->deviceID, PPI_traceNotification, "Sending Wait Cycles Cmd, TCKenable = " << (waitCyclesComand & 1) << ", Cycles = " << count);

			while (count > 0)
			{
				error = checkPayloadSize(2);
				if (!CONNECTION_PASS(error)) {
					return error;
				}
				uint32_t tmp_count = count;
				if (tmp_count > 256) tmp_count = 256;
				count -= tmp_count;
				if (tmp_count == 256) tmp_count = 0;
				write_message.buffer[write_message_offset++] = waitCyclesComand;
				write_message.buffer[write_message_offset++] = char(tmp_count);

			}
		}
	}

	while (writeNumberBytes > 0 || readNumberBytes > 0)
	{
		char writeSizeTmp = (char)std::min((size_t)MAX_RW_SCAN_SIZE, writeNumberBytes);
		char readSizeTmp = (char)std::min((size_t)MAX_RW_SCAN_SIZE, readNumberBytes);
		char writeNumBitsTmp;
		char readNumBitsTmp;

		if (writeSizeTmp > 0) {
			error = checkPayloadSize(writeSizeTmp);
			if (!CONNECTION_PASS(error)) {
				return error;
			}
		}

		if (readSizeTmp > 0) {
			error = checkPayloadSize(sizeof(scan_cmd));
			if (!CONNECTION_PASS(error)) {
				return error;
			}
		}

		if (writeSizeTmp == writeNumberBytes)
			writeNumBitsTmp = (char)writeNumBits;
		else
			writeNumBitsTmp = writeSizeTmp * 8;

		if (readSizeTmp == readNumberBytes)
			readNumBitsTmp = (char)readNumBits;
		else
			readNumBitsTmp = readSizeTmp * 8;
		readNumberBytes -= readSizeTmp;
		writeNumberBytes -= writeSizeTmp;
		writeNumBits -= writeNumBitsTmp;
		readNumBits -= readNumBitsTmp;
		if (state == JtagShfDR || state == JtagShfIR) {
			if (writeNumberBytes == 0 && readNumberBytes == 0) {
				lastPacket = true;
				error = checkPayloadSize(writeSizeTmp + 2 * sizeof(scan_cmd));
				if (!CONNECTION_PASS(error)) {
					return error;
				}
			}
			if (writeBuffer && readBuffer) {
				if (writeNumberBytes != readNumberBytes)
					return Invalid_Data_Size;
				scan_cmd = ((uint8_t)RW_SCAN) | (writeNumBitsTmp & 0x3f);
				error = checkPayloadSize(writeSizeTmp + sizeof(scan_cmd));
				if (!CONNECTION_PASS(error)) {
					return error;
				}
				error = AddOutputBuffer(readBufferTmp, readBufferTmpSize, scan_cmd, 0, {});
				if (!CONNECTION_PASS(error)) {
					return error;
				}
				readBufferTmp += writeSizeTmp;
				readBufferTmpSize -= writeSizeTmp;
			}
			else if (writeBuffer) {
				scan_cmd = ((char)WRITE_SCAN) | (writeNumBitsTmp & 0x3f);
				error = checkPayloadSize(writeSizeTmp + sizeof(scan_cmd));
				if (!CONNECTION_PASS(error)) {
					return error;
				}
			}
			else if (readBuffer) {
				scan_cmd = ((uint8_t)READ_SCAN) | (readNumBitsTmp & 0x3f);
				error = checkPayloadSize(sizeof(scan_cmd));
				if (!CONNECTION_PASS(error)) {
					return error;
				}
				error = AddOutputBuffer(readBufferTmp, readBufferTmpSize, scan_cmd, 0, {});
				if (!CONNECTION_PASS(error)) {
					return error;
				}
				readBufferTmp += readSizeTmp;
				readBufferTmpSize -= readSizeTmp;
			}
			else
				return Invalid_Data_Size;
			if (scan_cmd != 0) {
				write_message.buffer[write_message_offset++] = scan_cmd;
			}
			else
				return Invalid_Data_Size;
		}
		else if ((JtagStateEncodeASD)state == JtagNoStateChange && readBuffer && writeBuffer && readNumBitsTmp != 0) { //we are reading configuration
			error = checkPayloadSize((readNumBitsTmp + 7) / 8);
			if (!CONNECTION_PASS(error)) {
				return error;
			}
			error = AddOutputBuffer(readBuffer, readSize, writeBuffer[0], readNumBitsTmp, {});
			if (!CONNECTION_PASS(error)) {
				return error;
			}
		}

		if (writeBuffer) {
			for (int i = 0; i < writeSizeTmp; i++)
			{
				write_message.buffer[write_message_offset++] = writeBufferTmp[i];
			}
			writeBufferTmp += writeSizeTmp;
		}
		if (lastPacket) {
			if (state == JtagShfDR) {
				write_message.buffer[write_message_offset++] = STATE_CHANGE_BYTE + (char)JtagEx1DR;
			}
			else if (state == JtagShfIR) {
				write_message.buffer[write_message_offset++] = STATE_CHANGE_BYTE + (char)JtagEx1IR;
			}
		}
	}
	return error;
}

Connection_Error ASDProtocol::ShiftI2cProtocol(OpenIPC_DeviceId deviceId, char *writeBuffer, size_t writeSize, char* readBuffer, size_t readSize, GotoStateOptions *i2c)
{
	std::vector<uint8_t> buff;
	Connection_Error error = No_Error;
	PPI_bool continueBit = false; // TBD

	if (!i2c) {
		error = I2C_Request_Error;
		goto end;
	}

	if (i2c->isI2cRead) {
		buff.push_back(I2C_READ_CMD | (continueBit) | (uint8_t)(i2c->readParams.bufferLength));
		buff.push_back((uint8_t)(i2c->readParams.deviceId << 1) | (uint8_t)i2c->readParams.forceStop);
	}
	else {
		buff.push_back(I2C_WRITE_CMD | (continueBit) | (uint8_t)(i2c->writeParams.bufferLength));
		buff.push_back((uint8_t)(i2c->writeParams.deviceId << 1) | (uint8_t)i2c->writeParams.forceStop);
		for (uint32_t i = 0; i < i2c->writeParams.bufferLength; i++) {
			buff.push_back(i2c->writeParams.writeBuffer[i]);
		}
	}

	if (buff.size() > 0) {
		error = checkPayloadSize(buff.size());
		if (!CONNECTION_PASS(error)) {
			error = I2C_Request_Error;
			goto end;
		}

		if (i2c->isI2cRead) {
			AddOutputBuffer((char*)i2c->readParams.readBuffer, i2c->readParams.bufferLength, buff.front(), 0, *i2c);
		}
		else {
			AddOutputBuffer((char*)i2c->writeParams.writeBuffer, i2c->writeParams.bufferLength, buff.front(), 0, *i2c);
		}

		memcpy_s(write_message.buffer + write_message_offset,
			payloadSize - write_message_offset,
			(char*)reinterpret_cast<uint8_t*>(buff.data()), buff.size());
		write_message_offset += (uint16_t)buff.size();
	}

end:
	return error;
}

Connection_Error ASDProtocol::SetHardwareLogLevel(LoggingConfiguration log_config)
{
	setMsgType(AGENT_CONTROL_MESSAGE_TYPE);
	cmd_stat = HARDWARE_CONFIGURATION;
	write_message.buffer[write_message_offset++] = AGENT_CONFIG_TYPE_LOGGING;
	write_message.buffer[write_message_offset++] = log_config.value;
	return SendData(true, No_Error);
}

Connection_Error ASDProtocol::SetHardwareGpioConfig(const uint8_t gpio_config)
{
	setMsgType(AGENT_CONTROL_MESSAGE_TYPE);
	cmd_stat = HARDWARE_CONFIGURATION;
	write_message.buffer[write_message_offset++] = AGENT_CONFIG_TYPE_GPIO;
	write_message.buffer[write_message_offset++] = gpio_config;
	return SendData(true, No_Error);
}

Connection_Error ASDProtocol::SetHardwareJTAGDriverMode(JtagChainParameters params)
{
	setMsgType(AGENT_CONTROL_MESSAGE_TYPE);
	cmd_stat = HARDWARE_CONFIGURATION;
	write_message.buffer[write_message_offset++] = AGENT_CONFIG_TYPE_JTAG_DRIVER_MODE;
	write_message.buffer[write_message_offset++] = (uint8_t)(params.JtagDriverMode | params.JtagChainSelectMode << 1);
	this->params.JtagChainSelectMode = params.JtagChainSelectMode;
	this->params.JtagSyncDelayValue = params.JtagSyncDelayValue;
	this->params.JtagSyncTimeoutValue = params.JtagSyncTimeoutValue;
	return SendData(true, No_Error);
}

Connection_Error ASDProtocol::StartDataTransfer(unsigned int messageType, JtagChainParameters *params)
{
	return ASDBaseProtocol::StartDataTransfer(messageType, params);
}

Connection_Error ASDProtocol::EndDataTransfer(JtagChainParameters *params)
{
	return ASDBaseProtocol::EndDataTransfer(params);
}

unsigned int ASDProtocol::readPacket(void)
{
	return ASDBaseProtocol::readPacket();
}

Connection_Error ASDProtocol::UpdateInterfacePaddingSettings(InterfacePadding *interfacePaddingSettings)
{
	JtagChainParameters chain_params;
	chain_params.JtagChainSettings = &jtagSettings;
	chain_params.InterfacePaddingSettings = interfacePaddingSettings;
	return SetHWStateToTheOutMsg(&chain_params);
}
