
#include "OSDependenciesImpl.h"

#if defined(HOST_LINUX) || defined(HOST_DARWIN)
#include <sys/socket.h>
#include <poll.h>
#include <unistd.h>
#include <netinet/in.h>
#include <netdb.h>
#include <netinet/tcp.h>
#else
#include <winsock2.h>
#include <ws2tcpip.h>
#endif

OSDepSocket OSDependenciesImpl::InitSocket()
{
	OSDepSocket socket;
#if defined(HOST_LINUX) || defined(HOST_DARWIN)
	socket = -1;
#else
	socket = INVALID_SOCKET;
	WSADATA wsaData;
	// Initialize Winsock
	WSAStartup(MAKEWORD(2, 2), &wsaData);
#endif // DARWIN
	return socket;
}

bool OSDependenciesImpl::IsSocketValid(OSDepSocket socket)
{
#if defined(HOST_LINUX) || defined(HOST_DARWIN)
	return socket != -1;
#else
	return socket != INVALID_SOCKET;
#endif // DARWIN
}

OSDepSocket OSDependenciesImpl::Socket(int socket_family, int socket_type, int protocol)
{
	return socket(socket_family, socket_type, protocol);
}

int OSDependenciesImpl::GetAddressInfo(const char* __name, const char* __service,
	const struct addrinfo* _req, struct addrinfo** __pai)
{
	return getaddrinfo(__name, __service, _req, __pai);
}

void OSDependenciesImpl::FreeAddressInfo(struct addrinfo *__ai)
{
	freeaddrinfo(__ai);
}

bool OSDependenciesImpl::SetSockOptKeepAlive(OSDepSocket s)
{
	bool result = true;
#if defined(HOST_LINUX) || defined(HOST_DARWIN)
	int optval = 1;
#else
	char optval = 1;
#endif //HOST_DARWIN
	if (setsockopt(s, SOL_SOCKET, SO_KEEPALIVE, &optval, sizeof(optval)) < 0) {
		result = false;
	}
	return result;
}

bool OSDependenciesImpl::SetSockOptNoDelay(OSDepSocket s)
{
	bool result = true;
#if defined(HOST_LINUX) || defined(HOST_DARWIN)
	int optval = 1;
#else
	char optval = 1;
#endif //HOST_DARWIN
	if (setsockopt(s, IPPROTO_TCP, TCP_NODELAY, &optval, sizeof(optval)) < 0) {
		result = false;
	}
	return result;
}

int OSDependenciesImpl::Connect(OSDepSocket sockfd, const void* addr, unsigned int addrlen)
{
	return connect(sockfd, static_cast<const sockaddr*>(addr), addrlen);
}

int OSDependenciesImpl::Close(OSDepSocket sockfd)
{
#if defined(HOST_LINUX) || defined(HOST_DARWIN)
	return close(sockfd);
#else
	shutdown(sockfd, SD_BOTH);
	closesocket(sockfd);
	WSACleanup();
	return 0;
#endif // DARWIN
}

int OSDependenciesImpl::Send(OSDepSocket sockfd, const void* buf, int len, int flags)
{
#if defined(HOST_LINUX) || defined(HOST_DARWIN)
	(void) flags;
	int cnt = 0;
	cnt = static_cast<int>(write(sockfd, buf, static_cast<size_t>(len)));
	if (cnt != len)
		cnt = -1;
	return cnt;
#else
	return send(sockfd, (const char*)buf, len, flags);
#endif //HOST_DARWIN
}

int OSDependenciesImpl::Receive(OSDepSocket sockfd, void* buf, int len, int flags)
{
	return static_cast<int>(recv(sockfd, (char*)buf, static_cast<size_t>(len), flags));
}

bool OSDependenciesImpl::AwaitData(OSDepSocket sockfd)
{
	bool result = false;
#if defined(HOST_LINUX) || defined(HOST_DARWIN)
	struct pollfd poll_fd;
	poll_fd.fd = sockfd;
	poll_fd.events = POLLIN;

	poll(&poll_fd, 1, -1);
	if (poll_fd.revents & POLLIN) {
		result = true;
	}
#else
	u_long iMode = 0;
	WSAPOLLFD socketFd;
	int ret = 0;
	socketFd.fd = sockfd;
	ret = ioctlsocket(sockfd, FIONBIO, &iMode);
	if (ret != NO_ERROR) {
		result = false;
	}
	socketFd.events = POLLRDNORM;

	ret = WSAPoll(&socketFd, 1, -1);
	if (ret > 0 && ret != SOCKET_ERROR) {
		if (socketFd.revents & POLLRDNORM) {
			result = true;
		}
	}
#endif //HOST_DARWIN
	return result;
}