//////////////////////////////////////////////////////////////////////////////
//
//                      INTEL CONFIDENTIAL
//       Copyright 2016-2017 Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its
// suppliers. Title to the Material remains with Intel Corporation, its
// suppliers, or licensors. The Material contains trade secrets and
// proprietary and confidential information of Intel Corporation, its
// suppliers, and licensors, and is protected by worldwide copyright and trade
// secret laws and treaty provisions. No part of the Material may be used,
// copied, reproduced, modified, published, uploaded, posted, transmitted,
// distributed, or disclosed in any way without Intel's prior express written
// permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or
// delivery of the Materials, either expressly, by implication, inducement,
// estoppel or otherwise. Any license under such intellectual property rights
// must be express and approved by Intel in writing.
//
// Unless otherwise agreed by Intel in writing, you may not remove or alter
// this notice or any other notice embedded in Materials by Intel or Intel's
// suppliers or licensors in any way.
//
//////////////////////////////////////////////////////////////////////////////
///  @file
///
///  @brief Contains methods for obtaining instances of the Probe interfaces
///
///  For additional information on obtaining and using instances, see @ref probeusage.
///
//////////////////////////////////////////////////////////////////////////////

#include "RemoteConnection.h"

#include <vector>
#include <string>
#include <map>
#include <memory>
#include <mutex>
#include <Components/Common/CommonErrors.h>
#include "Foundation/Error/ErrorTypes.h"
#include "Foundation/Types.h"
#include <ProbeTypes.h>
#include <TapStateMachineEncode.h>
#include "Connection.h"

std::map<std::string, std::shared_ptr<Connection>> ip2Connection;
std::map<OpenIPC_DeviceId, std::shared_ptr<Connection>> devid2Connection;
std::mutex remoteConnectionAcccessControl;

OpenIPC_Error RegisterConnection(std::shared_ptr<Connection> connection)
{
	if (connection != nullptr) {
		ip2Connection[connection->Key] = connection;
		return OpenIPC_Error_No_Error;
	}
	return OpenIPC_Error_Bad_Argument;
}

OpenIPC_Error RegisterProbeConnection(OpenIPC_DeviceId deviceId, std::shared_ptr<Connection> connection)
{
	devid2Connection[deviceId] = connection;
	return OpenIPC_Error_No_Error;
}

EXPORT_PREFIX EXTERN_PREFIX Connection_Error RegisterProbePlugin(std::shared_ptr<ProbeInstanceASD> probe)
{
	std::string address = probe->connectionParameters->transportParameters->ipAddress;
	std::string port = probe->connectionParameters->transportParameters->ipPort;

	std::map<OpenIPC_DeviceId, std::shared_ptr<Connection>>::iterator iter = devid2Connection.find(probe->deviceID);
	auto connectionKey = Connection::GenerateKey(address, port);
	std::map<std::string, std::shared_ptr<Connection>>::iterator iterIp = ip2Connection.find(connectionKey);
	PPI_LOG(probe->deviceID, PPI_traceNotification, "Registration");

	if (iter != devid2Connection.end())
		return User_Already_Registered;

	if (iterIp != ip2Connection.end() && iter != devid2Connection.end()) {
		std::shared_ptr<Connection> connect = iter->second;
		RegisterProbeConnection(probe->deviceID, connect);
		connect->SetProbeDeviceId(probe->deviceID);
		if (connect->ProbeRegistered())
			return User_Already_Registered;
		return connect->RegisterProbeInstance(probe);
	}

	PPI_LOG(probe->deviceID, PPI_debugNotification, "Connecting to BMC " << address << " port: " << port);

	Connection_Error conn_error = Failed_To_Register;

	std::shared_ptr<Connection> connect = std::make_shared<Connection>(probe);
	if (connect && connect->InitSuccess()) {
		PPI_LOG(probe->deviceID, PPI_traceNotification, "Connected to: " << address << " port: " << port);
		PluginNotifications::GetInstance().Notify(
			probe->deviceID,
			OpenIPC_Error_No_Error,
			PPI_infoNotification,
			[&](std::basic_ostream<char>& stream)
			{
				std::string protocolStr = "None";
				if (probe->connectionParameters->transportParameters->socketType== SSL_SOCKET)
					protocolStr = "TLS_AUTH";
				stream << "ASD plugin connected to: " << address << " port: " << port <<
					" payloadSize: "<< probe->connectionParameters->protocolParameters->payloadSize <<
					" maxNumBuffers: " << probe->connectionParameters->protocolParameters->maxNumBuffers <<
					" securityProtocol: " << protocolStr;
			});
		RegisterProbeConnection(probe->deviceID, connect);
		RegisterConnection(connect);
		return No_Error;
	}
	else if (!connect) {
		PPI_LOG(probe->deviceID, PPI_errorNotification, "Unable to create shared connection object");
	}
	else {
		conn_error = connect->GetConnectionError();
		if (conn_error == Could_Not_Contact_BMC) {
			PPI_LOG(probe->deviceID, PPI_errorNotification, "Failed to contact the BMC");
			PluginNotifications::GetInstance().Notify(
				probe->deviceID,
				conn_error,
				PPI_errorNotification,
				[&](std::basic_ostream<char>& stream)
				{
					stream << "ASD plugin failed to connect the host address '" << address << "' using port '" << port << "'.";
				});
			PluginNotifications::GetInstance().Notify(
				probe->deviceID,
				conn_error,
				PPI_errorNotification,
				[&](std::basic_ostream<char>& stream)
				{
					stream << "Ensure that the BMC is powered on and that the use of port '" << port << "' is allowed on your network.";
				});
		}
		else if (conn_error == Could_Not_Connect_To_BMC) {
			PPI_LOG(probe->deviceID, PPI_errorNotification, "Failed to connect to the BMC");
			PluginNotifications::GetInstance().Notify(
				probe->deviceID,
				OpenIPC_Error_Initialization_Failed,
				PPI_errorNotification,
				[&](std::basic_ostream<char>& stream)
				{
					stream << "ASD failed to connect to " << address << " on port " << port << ".";
				});
			PluginNotifications::GetInstance().Notify(
				probe->deviceID,
				OpenIPC_Error_Initialization_Failed,
				PPI_errorNotification,
				[&](std::basic_ostream<char>& stream)
				{
					stream << "Please ensure use of the correct port number and ensure that the JTAG socket server process is started.";
				});
		}
		else if (conn_error == Connection_To_BMC_Rejected) {
			PPI_LOG(probe->deviceID, PPI_errorNotification, "Connection to BMC was rejected");
			PluginNotifications::GetInstance().Notify(
				probe->deviceID,
				conn_error,
				PPI_errorNotification,
				[&](std::basic_ostream<char>& stream)
				{
					stream << "ASD connected to " << address << " but was rejected. Is another debug client connected?";
				});
		}
		else {
			PPI_LOG(probe->deviceID, PPI_errorNotification, "Unable to connect to BMC");
			PluginNotifications::GetInstance().Notify(
				probe->deviceID,
				conn_error,
				PPI_errorNotification,
				[&](std::basic_ostream<char>& stream)
			{
				if (probe->connectionParameters->transportParameters->socketType == SSL_SOCKET) {
					stream << "ASD plugin was unable to connect to the BMC in TLS mode";
				}
				else {
					stream << "ASD plugin was unable to connect to the BMC in non-TLS mode";
				}
			});
		}
	}

	return conn_error;
}

EXPORT_PREFIX Connection_Error UnRegisterPlugin(OpenIPC_DeviceId deviceId)
{
	std::map<OpenIPC_DeviceId, std::shared_ptr<Connection>>::iterator iter = devid2Connection.find(deviceId);
	std::map<std::string, std::shared_ptr<Connection>>::iterator iterIp;

	if (iter == devid2Connection.end())
		return No_Error;
	std::shared_ptr<Connection> ptr = iter->second;
	devid2Connection.erase(iter);

	for (iter = devid2Connection.begin(); iter != devid2Connection.end(); iter++)
	{
		if (iter->second == ptr)
			return No_Error;
	}

	for (iterIp = ip2Connection.begin(); iterIp != ip2Connection.end(); iterIp++)
	{
		if (iterIp->second == ptr) {
			ip2Connection.erase(iterIp);
			return No_Error;
		}
	}
	return Failed_To_UnRegister;
}

EXPORT_PREFIX Connection_Error RegisterRCPlugin(OpenIPC_DeviceId deviceId, std::string &address, std::string &port)
{
	std::map<OpenIPC_DeviceId, std::shared_ptr<Connection>>::iterator iter = devid2Connection.find(deviceId);
	std::map<std::string, std::shared_ptr<Connection>>::iterator iterIp = ip2Connection.find(Connection::GenerateKey(address, port));

	if (iter != devid2Connection.end())
		return User_Already_Registered;
	if (iterIp == ip2Connection.end())
		return Probe_Plugin_Not_Registered;

	std::shared_ptr<Connection> connect = iterIp->second;
	if (connect->SetRCDeviceId(deviceId))
		RegisterProbeConnection(deviceId, connect);
	else
		return Error_Connection_Taken;
	return No_Error;
}

EXPORT_PREFIX Connection_Error StartDataTransfer(OpenIPC_DeviceId deviceId, unsigned int messageType, JtagChainParameters *params)
{
	std::map<OpenIPC_DeviceId, std::shared_ptr<Connection>>::iterator iter = devid2Connection.find(deviceId);

	if (iter == devid2Connection.end())
		return User_Not_Registered;
	std::shared_ptr<Connection> connect = iter->second;
	PPI_LOG(deviceId, PPI_traceNotification, "Starting data transfer");
	return connect->StartDataTransfer(deviceId, messageType, params);
}

EXPORT_PREFIX Connection_Error EndDataTransfer(OpenIPC_DeviceId deviceId, JtagChainParameters *params)
{
	std::map<OpenIPC_DeviceId, std::shared_ptr<Connection>>::iterator iter = devid2Connection.find(deviceId);
	if (iter == devid2Connection.end())
		return User_Not_Registered;
	std::shared_ptr<Connection> connect = iter->second;
	PPI_LOG(deviceId, PPI_traceNotification, "Ending data transfer");
	return connect->EndDataTransfer(deviceId, params);
}

EXPORT_PREFIX Connection_Error UpdateInterfacePaddingSettings(OpenIPC_DeviceId deviceId, InterfacePadding *interfacePaddingSettings)
{
	std::map<OpenIPC_DeviceId, std::shared_ptr<Connection>>::iterator iter = devid2Connection.find(deviceId);

	if (iter == devid2Connection.end())
		return User_Not_Registered;
	std::shared_ptr<Connection> connect = iter->second;
	return connect->UpdateInterfacePaddingSettings(interfacePaddingSettings);
}

EXPORT_PREFIX Connection_Error RemoteShift(OpenIPC_DeviceId deviceId, JtagStateEncodeASD state, char *writeBuffer, size_t writeSize, char* readBuffer, size_t readSize, GotoStateOptions *goto_state_opt)
{
	std::map<OpenIPC_DeviceId, std::shared_ptr<Connection>>::iterator iter = devid2Connection.find(deviceId);
	if (iter == devid2Connection.end())
		return User_Not_Registered;
	std::shared_ptr<Connection> connect = iter->second;
	return connect->Shift(deviceId, (JtagStateEncode)state, writeBuffer, writeSize, readBuffer, readSize, goto_state_opt);
}

EXPORT_PREFIX Connection_Error EnableEncryption(OpenIPC_DeviceId deviceId, bool enable)
{
	std::map<OpenIPC_DeviceId, std::shared_ptr<Connection>>::iterator iter = devid2Connection.find(deviceId);
	if (iter == devid2Connection.end())
		return User_Not_Registered;
	std::shared_ptr<Connection> connect = iter->second;
	connect->EnableEncryption(enable);
	return No_Error;
}

// Set remote logging configuration on all registered connections
EXPORT_PREFIX Connection_Error ChangeHardwareLogLevel(LoggingConfiguration log_config)
{
	std::map<OpenIPC_DeviceId, std::shared_ptr<Connection>>::iterator iter;
	for (iter = devid2Connection.begin(); iter != devid2Connection.end(); iter++) {
		std::shared_ptr<Connection> connection = iter->second;
		connection->SetHardwareLogLevel(log_config);
	}
	return No_Error;
}

EXPORT_PREFIX Connection_Error SetHardwareGpioConfig(OpenIPC_DeviceId deviceId, const uint8_t gpio_config)
{
	std::map<OpenIPC_DeviceId, std::shared_ptr<Connection>>::iterator iter = devid2Connection.find(deviceId);
	if (iter == devid2Connection.end())
		return User_Not_Registered;
	std::shared_ptr<Connection> connect = iter->second;
	connect->SetHardwareGpioConfig(gpio_config);
	return No_Error;
}

EXPORT_PREFIX Connection_Error SetHardwareJTAGDriverMode(OpenIPC_DeviceId deviceId, JtagChainParameters params)
{
	std::map<OpenIPC_DeviceId, std::shared_ptr<Connection>>::iterator iter = devid2Connection.find(deviceId);
	if (iter == devid2Connection.end())
		return User_Not_Registered;
	std::shared_ptr<Connection> connect = iter->second;
	connect->SetHardwareJTAGDriverMode(params);
	return No_Error;
}

EXPORT_PREFIX Connection_Error ChangeStreamLogLevel(PPI_Stream stream, PPI_NotificationLevel level)
{
	Connection_Error result = No_Error;
	PluginLogger::GetInstance().Level(level, stream);

	if (stream == PPI_BACKEND_HARDWARE_COMMUNICATION_STREAM) {
		// Set Hardware Log Configuration for each registered connection
		LoggingConfiguration log_config = { {0} };
		log_config.Parsed.logging_stream = stream;
		log_config.Parsed.logging_level = level;
		result = ChangeHardwareLogLevel(log_config);
	}
	return result;
}
