//////////////////////////////////////////////////////////////////////////////
//
//                      INTEL CONFIDENTIAL
//       Copyright 2017 Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its
// suppliers. Title to the Material remains with Intel Corporation, its
// suppliers, or licensors. The Material contains trade secrets and
// proprietary and confidential information of Intel Corporation, its
// suppliers, and licensors, and is protected by worldwide copyright and trade
// secret laws and treaty provisions. No part of the Material may be used,
// copied, reproduced, modified, published, uploaded, posted, transmitted,
// distributed, or disclosed in any way without Intel's prior express written
// permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or
// delivery of the Materials, either expressly, by implication, inducement,
// estoppel or otherwise. Any license under such intellectual property rights
// must be express and approved by Intel in writing.
//
// Unless otherwise agreed by Intel in writing, you may not remove or alter
// this notice or any other notice embedded in Materials by Intel or Intel's
// suppliers or licensors in any way.
//
//////////////////////////////////////////////////////////////////////////////
#include "TransportTLSPassPhrase.hpp"

#ifdef NEED_SAFE_CLIB
#include "safe_lib.h"
#endif

TransportTLSPassPhrase::TransportTLSPassPhrase(const char* address,
	const char* sslPort, const char* targetChainFile,
	std::shared_ptr<SSLMethod> sslMethod,
	std::string &passphrase, std::shared_ptr<OSDependencies> deps) :
	TransportTLS (address, sslPort, targetChainFile, sslMethod, deps) {
	SetPassphrase(&passphrase[0]);
}

void TransportTLSPassPhrase::SetPassphrase(char *passphrase) {
	if(passphrase == NULL) {
		PassPhraseLength = 0;
	} else {
		size_t safeLength = strlen(passphrase) & BG_PASSPHRASE_LEN_MASK; // force length range
		PassPhraseLength = (uint8_t)MAX_BG_PASSPHRASE_LEN;     // force Passphrase length to be always a fixed size
		std::fill(&ASDBG_PassPhrase[0], &ASDBG_PassPhrase[MAX_BG_PASSPHRASE_LEN], 0x0);
		ASDBG_PassPhrase[0] = CLIENT_AUTHENTICATION_VERSION;
		memcpy_s(&ASDBG_PassPhrase[1], MAX_BG_PASSPHRASE_LEN, passphrase, safeLength);
	}
}

// After successful TLS connection, a final passphrase step and check must also pass
Connection_Error TransportTLSPassPhrase::Connect() {
	Connection_Error result = TransportTLS::Connect();  // Already verifies that SslMethod is not null, so no need to check again
	if(CONNECTION_PASS(result)) {
		// Do passphrase authentication
		if (PassPhraseLength) {
			// Write passphrase to target
			SslMethod->Write(ASDBG_PassPhrase, PassPhraseLength + 1);
		} else {
			TransportMessage message = { (unsigned int)result, "Passphrase is empty" };
			RegisterMessage(message);
		}

		// Receive passphrase result
		char passphraseResultBytes[2] = {0};
		auto len = SslMethod->Read(passphraseResultBytes, 2);


		if ((len == 2) &&
			passphraseResultBytes[0] == CLIENT_AUTHENTICATION_VERSION &&
			passphraseResultBytes[1] == AUTH_HANDSHAKE_SUCCESS) {
			result = Connection_Error::No_Error;
		} else {
			if (passphraseResultBytes[0] != CLIENT_AUTHENTICATION_VERSION) {
				Message = "Authentication versions don't match.";
			} else {
				switch (passphraseResultBytes[1]) {
				case AUTH_HANDSHAKE_SYSERR:
					Message = "Authentication system error";
					break;
				case AUTH_HANDSHAKE_BUSY:
					Message = "Authentication already in progress for another client.";
					break;
				case AUTH_HANDSHAKE_FAILURE:
					Message = "Authentication handshake failure";
					break;
				}
			}
			result = Connection_Error::Could_Not_Connect_To_BMC;
		}
	}
	if (!CONNECTION_PASS(result)) {
		if (!Message.empty()) {
			TransportMessage message = { (unsigned int)result, Message };
			RegisterMessage(message);
		}
	}

	return result;
}
