//////////////////////////////////////////////////////////////////////////////
//
//                      INTEL CONFIDENTIAL
//       Copyright 2017 Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its
// suppliers. Title to the Material remains with Intel Corporation, its
// suppliers, or licensors. The Material contains trade secrets and
// proprietary and confidential information of Intel Corporation, its
// suppliers, and licensors, and is protected by worldwide copyright and trade
// secret laws and treaty provisions. No part of the Material may be used,
// copied, reproduced, modified, published, uploaded, posted, transmitted,
// distributed, or disclosed in any way without Intel's prior express written
// permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or
// delivery of the Materials, either expressly, by implication, inducement,
// estoppel or otherwise. Any license under such intellectual property rights
// must be express and approved by Intel in writing.
//
// Unless otherwise agreed by Intel in writing, you may not remove or alter
// this notice or any other notice embedded in Materials by Intel or Intel's
// suppliers or licensors in any way.
//
//////////////////////////////////////////////////////////////////////////////
#include "PPI_InterfaceTriggerResponse.h"
#include "RemoteConnection.h"

#include "BundleHelpersASD.h"
#include "I2cDefinitions.h"

OpenIPC_Error PPI_InterfaceTriggerResponseBreakAll::SetIsEnabled(bool enabled)
{
	unsigned char buf[2];
	PPI_InterfaceTriggerResponse::SetIsEnabled(enabled);
	Connection_Error ret;
	buf[0] = 0x0; //write config cmd Probe Config reg
	if (enabled) {
		buf[1] = 0x80;
		PPI_LOG(probe->deviceID, PPI_debugNotification, "Enabling BreakAll Reporting");
	}
	else {
		buf[1] = 0x0; // breakall report setting default off
		PPI_LOG(probe->deviceID, PPI_debugNotification, "Disabling BreakAll Reporting");
	}
	StartDataTransfer(probe->deviceID, JTAG_MESSAGE_TYPE, NULL);
	ret = RemoteShift(probe->deviceID, JtagNoStateChange, (char*)buf, 16, NULL, 0);
	EndDataTransfer(probe->deviceID);
	return To_OpenIPC_Error(ret);
}

OpenIPC_Error PPI_InterfaceTriggerResponseResetBreak::SetIsEnabled(bool enabled)
{
	unsigned char buf[2];
	OpenIPC_Error ret;
	PPI_InterfaceTriggerResponse::SetIsEnabled(enabled);
	buf[0] = 0x0; //write config cmd Probe Config reg
	if (enabled) {
		buf[1] = 0x81;
		PPI_LOG(probe->deviceID, PPI_debugNotification, "Enabling ResetBreak Reporting");
	}
	else {
		buf[1] = 0x1; // resetBreak report setting default off
		PPI_LOG(probe->deviceID, PPI_debugNotification, "Disabling ResetBreak Reporting");
	}
	StartDataTransfer(probe->deviceID, JTAG_MESSAGE_TYPE, NULL);
	ret = RemoteShift(probe->deviceID, JtagNoStateChange, (char*)buf, 16, NULL, 0);
	EndDataTransfer(probe->deviceID);
	return ret;
}

static uint64_t getSlotPosition(PPI_SlotHandle handle) {
	return ((uint64_t)handle - 1);
}

PROBEPLUGIN_API OpenIPC_Error PPI_Pins_SetPin(
	PPI_ProbeBundleHandle handle,
	PPI_Pins_TypeEncode pin,
	PPI_bool value,
	const PPI_Pins_PinReadSetOptions* const options) {
	SetPinCommand *cmd = new SetPinCommand();
	if (options != NULL && options->options == PINS_Save_Restore_From_Slot) {
		cmd->savedSlot = (uint8_t)getSlotPosition(options->saved_slot);
		cmd->options = options->options;
	}
	else {
		cmd->savedSlot = 0;
		cmd->options = PINS_Default;
	}
	cmd->pin = pin;
	cmd->plugin = GetProbePluginInstanceASD();
	cmd->pinValue = value;

	return handle_command(cmd, handle);
}

PROBEPLUGIN_API OpenIPC_Error PPI_Pins_ReadPin(
	PPI_ProbeBundleHandle handle,
	PPI_Pins_TypeEncode pin,
	uint32_t* pinValue,
	const PPI_Pins_PinReadSetOptions* const options) {
	ReadPinCommand *cmd = new ReadPinCommand();
	if (options != NULL && options->options == PINS_Save_Restore_From_Slot) {
		cmd->savedSlot = (uint8_t)getSlotPosition(options->saved_slot);
		cmd->options = options->options;
	}
	else {
		cmd->savedSlot = 0;
		cmd->options = PINS_Default;
	}
	cmd->pin = pin;
	cmd->plugin = GetProbePluginInstanceASD();
	cmd->pinValue = pinValue;
	return handle_command(cmd, handle);
}

PROBEPLUGIN_API OpenIPC_Error PPI_Pins_PulsePin(
	PPI_ProbeBundleHandle handle,
	PPI_Pins_TypeEncode pin,
	PPI_bool startingPinState,
	uint32_t widthOfPulseInMicroSeconds,
	const PPI_Pins_PinPulseOptions* const options) {
	OpenIPC_Error status;
	if (options != NULL && options->options == PINS_Save_Restore_From_Slot)
		return OpenIPC_Error_Not_Supported; // Not implemented yet
	SetPinCommand *setCmd = new SetPinCommand();
	setCmd->pin = pin;
	setCmd->plugin = GetProbePluginInstanceASD();
	setCmd->pinValue = startingPinState;
	status = handle_command(setCmd, handle);
	if (status != OpenIPC_Error_No_Error)
		return status;
	SetPinCommand *setCmd2 = new SetPinCommand();
	setCmd2->pin = pin;
	setCmd2->plugin = GetProbePluginInstanceASD();
	setCmd2->pinValue = !startingPinState;
	status = handle_command(setCmd2, handle);
	if (status != OpenIPC_Error_No_Error)
		return status;
	PinsDelayCommand *delayCmd = new PinsDelayCommand();
	delayCmd->timeInMicroSeconds = widthOfPulseInMicroSeconds;
	delayCmd->plugin = GetProbePluginInstanceASD();
	status = handle_command(delayCmd, handle);
	if (status != OpenIPC_Error_No_Error) {
		return status;
	}
	SetPinCommand *setCmd3 = new SetPinCommand();
	setCmd3->pin = pin;
	setCmd3->plugin = GetProbePluginInstanceASD();
	setCmd3->pinValue = startingPinState;
	status = handle_command(setCmd3, handle);
	return status;
}

PROBEPLUGIN_API OpenIPC_Error PPI_Slot_Size(PPI_SlotHandle handle, uint32_t* bitSize) {
	if (bitSize == NULL)
		return OpenIPC_Error_Null_Pointer;
	uint64_t slotPosition = getSlotPosition(handle);
	Local_Handle* current = (Local_Handle*)handle;
	assert(current->slots.size() >= slotPosition);
	*bitSize = static_cast<uint32_t>(current->slots[slotPosition]->bitsize);
	return OpenIPC_Error_No_Error;
}

PROBEPLUGIN_API OpenIPC_Error PPI_Slot_Modification(
	PPI_ProbeBundleHandle handle,
	PPI_SlotHandle savedSlot,
	uint32_t numberOfBits,
	const uint8_t* const mask,
	const uint8_t* const valueToOrIn,
	PPI_Slot_ModificationOptions* options
) {
	uint64_t slotLocation = getSlotPosition(savedSlot);
	uint32_t number_of_bytes = (numberOfBits + 7) / 8;

	SlotModificationCommand* cmd = new SlotModificationCommand();
	if (mask == nullptr) {
		cmd->mask.resize(0);
		cmd->mask.resize(number_of_bytes);
	} else {
		cmd->mask.assign(mask, mask + number_of_bytes);
	}
	if (valueToOrIn == nullptr) {
		cmd->valueToOrIn.resize(0);
		cmd->valueToOrIn.resize(number_of_bytes);
	} else {
		cmd->valueToOrIn.assign(valueToOrIn, valueToOrIn + number_of_bytes);
	}
	cmd->savedSlot = (uint32_t)slotLocation;
	return handle_command(cmd, handle);
}

PROBEPLUGIN_API OpenIPC_Error PPI_Slot_Info(
	PPI_ProbeBundleHandle handle,
	PPI_SlotHandle savedSlot,
	uint32_t* lengthOfDataInSlot
) {
	if (lengthOfDataInSlot == nullptr)
		return OpenIPC_Error_Null_Pointer;
	uint64_t slotLocation = getSlotPosition(savedSlot);

	SlotInfoCommand* cmd = new SlotInfoCommand();
	cmd->lengthOfDataInSlot = lengthOfDataInSlot;
	cmd->savedSlot = (uint8_t)slotLocation;
	return handle_command(cmd, handle);
}

PROBEPLUGIN_API OpenIPC_Error PPI_Slot_ComparisonToConstant(
	PPI_ProbeBundleHandle handle,
	PPI_SlotHandle savedSlot,
	uint32_t numberOfBits,
	const uint8_t* const mask,
	const uint8_t* const match,
	bool* comparisonResult,
	PPI_Slot_ComparisonOptions* options
) {
	if (comparisonResult == nullptr) {
		return OpenIPC_Error_Null_Pointer;
	}
	uint64_t slotLocation = getSlotPosition(savedSlot);
	uint32_t number_of_bytes = (numberOfBits + 7) / 8;

	SlotComparisonCommand* cmd = new SlotComparisonCommand();
	if(mask == nullptr) {
		cmd->mask.resize(0);
		cmd->mask.resize(number_of_bytes);
	} else {
		cmd->mask.assign(mask, mask + number_of_bytes);
	}
	if(match == nullptr) {
		cmd->match.resize(0);
		cmd->match.resize(number_of_bytes);
	} else {
		cmd->match.assign(match, match + number_of_bytes);
	}
	cmd->comparisonResult = comparisonResult;
	cmd->numberOfBits = numberOfBits;
	cmd->savedSlot = (uint8_t)slotLocation;
	return handle_command(cmd, handle);
}

PROBEPLUGIN_API OpenIPC_Error PPI_Pins_AssertPin(
	PPI_ProbeBundleHandle handle,
	PPI_Pins_TypeEncode pin,
	const PPI_Pins_PinReadSetOptions* const options) {
	AssertPinCommand *cmd = new AssertPinCommand();
	if (options != NULL && options->options == PINS_Save_Restore_From_Slot) {
		cmd->savedSlot = (uint8_t)getSlotPosition(options->saved_slot);
		cmd->options = options->options;
	}
	else {
		cmd->savedSlot = 0;
		cmd->options = PINS_Default;
	}
	cmd->pin = pin;
	cmd->plugin = GetProbePluginInstanceASD();
	return handle_command(cmd, handle);
}

PROBEPLUGIN_API OpenIPC_Error PPI_Pins_DeassertPin(
	PPI_ProbeBundleHandle handle,
	PPI_Pins_TypeEncode pin,
	const PPI_Pins_PinReadSetOptions* const options) {
	DeassertPinCommand *cmd = new DeassertPinCommand();
	if (options != NULL && options->options == PINS_Save_Restore_From_Slot) {
		cmd->savedSlot = (uint8_t)getSlotPosition(options->saved_slot);
		cmd->options = options->options;
	}
	else {
		cmd->savedSlot = 0;
		cmd->options = PINS_Default;
	}
	cmd->pin = pin;
	cmd->plugin = GetProbePluginInstanceASD();
	return handle_command(cmd, handle);
}

PROBEPLUGIN_API OpenIPC_Error PPI_I2C_DirectRead(
	PPI_ProbeBundleHandle handle,
	uint8_t deviceId,
	uint32_t bufferLength,
	uint8_t* readBuffer,
	PPI_bool forceStop,
	PPI_bool* addressAck,
	const PPI_I2C_Options* const options) {
	I2cReadCommandASD *cmd = new I2cReadCommandASD();
	cmd->deviceId = deviceId >> 1; // 7-bit addressing
	cmd->bufferLength = bufferLength;
	cmd->readBuffer = readBuffer;
	cmd->forceStop = forceStop;
	cmd->addressAck = addressAck;
	cmd->plugin = GetProbePluginInstanceASD();
	return handle_command(cmd, handle);
}

PROBEPLUGIN_API OpenIPC_Error PPI_I2C_DirectWrite(
	PPI_ProbeBundleHandle handle,
	uint8_t deviceId,
	uint32_t bufferLength,
	const uint8_t* const writeBuffer,
	PPI_bool forceStop,
	PPI_bool* addressAck,
	uint32_t* lastDataAck,
	const PPI_I2C_Options* const options) {
	I2cWriteCommandASD *cmd = new I2cWriteCommandASD();
	cmd->deviceId = deviceId >> 1; // 7-bit addressing
	cmd->bufferLength = bufferLength;
	cmd->writeBuffer = writeBuffer;
	cmd->forceStop = forceStop;
	cmd->addressAck = addressAck;
	cmd->lastDataAck = lastDataAck;
	cmd->plugin = GetProbePluginInstanceASD();
	return handle_command(cmd, handle);
}
