//////////////////////////////////////////////////////////////////////////////
//
//                      INTEL CONFIDENTIAL
//       Copyright 2016-2017 Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its
// suppliers. Title to the Material remains with Intel Corporation, its
// suppliers, or licensors. The Material contains trade secrets and
// proprietary and confidential information of Intel Corporation, its
// suppliers, and licensors, and is protected by worldwide copyright and trade
// secret laws and treaty provisions. No part of the Material may be used,
// copied, reproduced, modified, published, uploaded, posted, transmitted,
// distributed, or disclosed in any way without Intel's prior express written
// permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or
// delivery of the Materials, either expressly, by implication, inducement,
// estoppel or otherwise. Any license under such intellectual property rights
// must be express and approved by Intel in writing.
//
// Unless otherwise agreed by Intel in writing, you may not remove or alter
// this notice or any other notice embedded in Materials by Intel or Intel's
// suppliers or licensors in any way.
//
//////////////////////////////////////////////////////////////////////////////
#include <iostream>
#include <vector>
#include <future>
#include <thread>


#include "Public/StructuredData.h"					// For XML support

#include "ProbePluginASD_JTAG.hpp"
#include "Command.h"
#include "RemoteConnection.h"

#ifdef USE_INTERFACE_SCAN
#include "MetaEncoding.h"
#include "I2cEncoding.h"
#include <deque>
#else
#include <cstdint>
#include <cstdlib>
#include <cstddef>
#include <limits>
#include <type_traits>
#include <stdexcept>
#endif // USE_INTERFACE_SCAN
#include <memory>

#include <Foundation/Types.h>
#include <Foundation/Error/ErrorTypes.h>
#include <Foundation/BitData/BitData.h>
#include "ProbePluginErrors.h"
#ifdef USE_INTERFACE_SCAN
#include "PinsEncoding.h"
#endif

#include <stringcopy.h>
#include "JTAGRegisterBasedOperations.h"
#include "JtagStateBasedOperations.h"
#include "SlotOperations.h"
#include "PinOperations.h"
#include "I2CDirectOperations.h"

#include "Handle.h"
#include "BundleHelpersASD.h"

#include <algorithm>
#include <cmath>
#include <cstring>

#ifdef NEED_SAFE_CLIB
#include "safe_lib.h"
#endif

const char* _pluginname = "ASD_JTAG";
const char* _probename = "ASD_JTAGProbe";

//////////////////////////////////////////////////////////////////////////////
// GetInstance for enforcing a singleton
//////////////////////////////////////////////////////////////////////////////
ProbePluginASD_JTAG* ProbePluginASD_JTAG::GetInstance(ProbePluginASD_JTAG *instance_override)
{
	// Pass in _instance override when BundleHelpers are needed outside of the normal context
	// BundleHelpers should be refactored to allow passing in the plugin and only use the
	//  global singleton when not otherwise defined
	static ProbePluginASD_JTAG* _instance = nullptr;

	if (_instance == nullptr) {
		if (instance_override == nullptr) {
			_instance = new ProbePluginASD_JTAG();
		} else {
			_instance = instance_override;
		}
	}

	return _instance;
}

//////////////////////////////////////////////////////////////////////////////
// GetProbePluginInstance (needed for C interfaces)
//////////////////////////////////////////////////////////////////////////////
CProbePlugin* GetProbePluginInstance()
{
	return (CProbePlugin*)ProbePluginASD_JTAG::GetInstance();
}

//////////////////////////////////////////////////////////////////////////////
// ProbePluginASD_JTAG constructor
//////////////////////////////////////////////////////////////////////////////
ProbePluginASD_JTAG::ProbePluginASD_JTAG()
    : CProbePlugin(::_pluginname),
      keepLocks({}), protocolFilters({})
{
	_probes.clear();
}

int InterfaceInstanceJtagASD::GetDeviceId()
{
	return deviceID;
}

std::shared_ptr<InterfaceInstance> ProbePluginASD_JTAG::InterfaceFromId(OpenIPC_DeviceId interfaceID)
{
	std::map<OpenIPC_DeviceId, std::weak_ptr<InterfaceInstance>>::iterator iter;
	std::shared_ptr<InterfaceInstance> empty;
	for (iter = _deviceid2probeinterface.begin(); iter != _deviceid2probeinterface.end(); iter++) {
		if (iter->first == interfaceID)
			return iter->second.lock();
	}
	return empty;
}

// OpenIPC calls this method to register the log handler function
OpenIPC_Error ProbePluginASD_JTAG::PluginSetLogEventHandler(PluginLogCallbackHandler logHandlerFunction)
{
	std::lock_guard<std::recursive_mutex> _(_pluginMutex);
	OpenIPC_Error result = To_OpenIPC_Error(RegisterLogHandler(logHandlerFunction));
	if (!OpenIPC_PASS(result)) {
		return OpenIPC_Error_Callback_Already_Initialized;
	}
	return PluginLogger::GetInstance().Handler(logHandlerFunction);
}

// OpenIPC calls this method to register the stream log handler function
OpenIPC_Error ProbePluginASD_JTAG::PPI_PluginSetStreamLogEventHandler(PluginStreamLogCallbackHandler logCallBackFunction)
{
	std::lock_guard<std::recursive_mutex> _(_pluginMutex);
	OpenIPC_Error result = To_OpenIPC_Error(RegisterStreamLogHandler(logCallBackFunction));
	if (!OpenIPC_PASS(result)) {
		return OpenIPC_Error_Callback_Already_Initialized;
	}
	return PluginLogger::GetInstance().Handler(logCallBackFunction);
}

// OpenIPC calls this method to set log level
OpenIPC_Error ProbePluginASD_JTAG::PPI_PluginSetNotificationRequiredForStreamLogCallBack( PPI_Stream stream, PPI_NotificationLevel level)
{
	std::lock_guard<std::recursive_mutex> _(_pluginMutex);
	PluginLogger::GetInstance().Level(level, stream); // ProbePluginASD_JTAG logger
	return To_OpenIPC_Error(ChangeStreamLogLevel(stream, level)); // ASD_Connection logger
}

OpenIPC_Error ProbePluginASD_JTAG::InterfaceOperationCancel(OpenIPC_DeviceId interfaceID)
{
	// Right now, this is not multi-threaded, so this operation will always work
	return OpenIPC_Error_No_Error;
}

// OpenIPC calls this method to register the notification handler function
OpenIPC_Error ProbePluginASD_JTAG::PluginRegisterNotificationHandler(PluginNotificationCallbackHandler notificationHandlerFunction)
{
	std::lock_guard<std::recursive_mutex> _(_pluginMutex);
	OpenIPC_Error result = To_OpenIPC_Error(RegisterNotificationHandler(notificationHandlerFunction));
	if (!OpenIPC_PASS(result)) {
		return result;
	}
	return PluginNotifications::GetInstance().Handler(notificationHandlerFunction);
}

OpenIPC_Error ProbePluginASD_JTAG::PluginSetNotificationRequiredForNotificationCallBack(PPI_NotificationLevel severity)
{
	std::lock_guard<std::recursive_mutex> _(_pluginMutex);
	PluginNotifications::GetInstance().Level(severity);
	return To_OpenIPC_Error(ChangeNotificationLevel(severity));
}

OpenIPC_Error ProbePluginASD_JTAG::PluginRegisterEventHandler(
	PluginEventCallbackHandler eventHandlerFunction
	)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

	if (_eventHandlerFunction == nullptr)
	{
		std::map<OpenIPC_DeviceId, std::weak_ptr<ProbeInstance>>::iterator iter;
		for (iter = _refid2ProbeInstance.begin(); iter != _refid2ProbeInstance.end(); iter++){
			std::shared_ptr<ProbeInstanceASD> probeShared = std::dynamic_pointer_cast<ProbeInstanceASD>(iter->second.lock());
			if (probeShared == nullptr)
				return OpenIPC_Error_Null_Pointer;
			probeShared->ProbeRegisterEventHandler(eventHandlerFunction);
		}
		_eventHandlerFunction = eventHandlerFunction;
	}
	else
	{
		openIPCError = OpenIPC_Error_Callback_Already_Initialized;
		PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
	}

	return openIPCError;
}

OpenIPC_Error ProbePluginASD_JTAG::InterfaceGetInfoJTAG(OpenIPC_DeviceId probeID, PPI_RefId interface_refid, PPI_InterfaceJTAGCapabilities* capabilities)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

	std::lock_guard<std::recursive_mutex> lock(_pluginMutex);
	std::shared_ptr<ProbeInstance> probe;
	openIPCError = _LookupProbe(probeID, probe);
	if (OpenIPC_PASS(openIPCError))
	{
		std::weak_ptr<InterfaceInstance> probeinterface_ref;
		openIPCError = probe->GetInterfaceFromRefid(interface_refid, probeinterface_ref);
		// probeinterface will be an empty shared_ptr if either
		// 1. the probeinterface_ref weak_ptr points to nothing
		// 2. the dynamic_pointer_cast downcast fails.
		auto probeinterface = std::dynamic_pointer_cast<InterfaceInstanceJtagASD>(probeinterface_ref.lock());
		if (OpenIPC_PASS(openIPCError) && (probeinterface != nullptr) && (probeinterface.get() != nullptr))
		{
			capabilities->supportPauDRInRegisterInterface = probeinterface->capabilities.supportPauDRInRegisterInterface;
			capabilities->supportTLR = probeinterface->capabilities.supportTLR;
			capabilities->supportTRST = probeinterface->capabilities.supportTRST;
		}
		else if (probeinterface == nullptr)
		{
			openIPCError = OpenIPC_Error_Invalid_Device_ID;
			PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
		}
	}
	else
	{
		openIPCError = OpenIPC_Error_Invalid_Device_ID;
		PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
	}

	return openIPCError;
}

OpenIPC_Error ProbePluginASD_JTAG::GetInterfaceJTAGPadding(OpenIPC_DeviceId device,
														 uint32_t* irPaddingNearTDI,
														 uint32_t* irPaddingNearTDO,
														 uint32_t* drPaddingNearTDI,
														 uint32_t* drPaddingNearTDO,
														 PPI_bool* drValueConstantOne
														 )
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

	std::shared_ptr<InterfaceInstanceJtagASD> probeinterface;
	if (OpenIPC_PASS(openIPCError))
	{
		openIPCError = _LookupInterface(device, probeinterface);
	}

	if (OpenIPC_PASS(openIPCError))
	{
		std::lock_guard<std::recursive_mutex> _(probeinterface->_instanceMutex);
		openIPCError = probeinterface->GetInterfacePadding(irPaddingNearTDI, irPaddingNearTDO,
			                                drPaddingNearTDI, drPaddingNearTDO,
											drValueConstantOne);
	}
	return openIPCError;
}

OpenIPC_Error ProbePluginASD_JTAG::SetInterfaceJTAGPadding( OpenIPC_DeviceId device,
														 uint32_t irPaddingNearTDI,
														 uint32_t irPaddingNearTDO,
														 uint32_t drPaddingNearTDI,
														 uint32_t drPaddingNearTDO,
														 PPI_bool drValueConstantOne
														 )
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
	std::shared_ptr<InterfaceInstanceJtagASD> probeinterface;
	openIPCError = _LookupInterface(device, probeinterface);
	if (OpenIPC_PASS(openIPCError))
	{
		 std::lock_guard<std::recursive_mutex> _(probeinterface->_instanceMutex);
		 openIPCError = probeinterface->SetInterfacePadding(irPaddingNearTDI, irPaddingNearTDO,
															drPaddingNearTDI, drPaddingNearTDO,
															drValueConstantOne);
	}
	return openIPCError;
}

OpenIPC_Error ProbePluginASD_JTAG::PPI_JTAG_GetCurrentBundlePadding(PPI_ProbeBundleHandle bundle, int32_t* irPaddingNearTDI, int32_t* irPaddingNearTDO, int32_t* drPaddingNearTDI,
	int32_t* drPaddingNearTDO, PPI_bool* drValueConstantOne) {
	auto result = OpenIPC_Error_No_Error;
	if (irPaddingNearTDI == nullptr ||
		irPaddingNearTDO == nullptr ||
		drPaddingNearTDI == nullptr ||
		drPaddingNearTDO == nullptr ||
		drValueConstantOne == nullptr)
	{
		PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, result = OpenIPC_Error_Null_Pointer, "PPI_JTAG_GetCurrentBundlePadding has at least one parameter that is null.");
	}
	else
	{
		Local_Handle* current;
		if (bundle == PPI_PROBE_LOCK_HOLD || bundle == PPI_PROBE_LOCK_RELEASE) {
			current = getLockedHandle();
		}
		else {
			current = (Local_Handle*)bundle;
		}
		*irPaddingNearTDI = current->ir_pre;
		*irPaddingNearTDO = current->ir_post;
		*drPaddingNearTDI = current->dr_pre;
		*drPaddingNearTDO = current->dr_post;
	}
	return result;
}

OpenIPC_Error ProbePluginASD_JTAG::PPI_JTAG_UpdateBundlePadding(PPI_ProbeBundleHandle bundle, int32_t irPaddingNearTDI, int32_t irPaddingNearTDO, int32_t drPaddingNearTDI,
		int32_t drPaddingNearTDO, PPI_bool drValueConstantOne) {
	JTAGUpdateBundlePadding *cmd = new JTAGUpdateBundlePadding();

	cmd->irPaddingNearTDI = irPaddingNearTDI;
	cmd->irPaddingNearTDO = irPaddingNearTDO;
	cmd->drPaddingNearTDI = drPaddingNearTDI;
	cmd->drPaddingNearTDO = drPaddingNearTDO;
	cmd->drValueConstantOne = drValueConstantOne;

	return handle_command(cmd, bundle);
}

ProbePluginASD_JTAG* GetProbePluginInstanceASD()
{
	return ProbePluginASD_JTAG::GetInstance();
}

#ifdef USE_INTERFACE_SCAN
inline PinTypeEncode LookupInterfaceScanPin(PPI_Pins_TypeEncode pin, bool& isValid)
{
	isValid = true;
	switch (pin)
	{
	case PPI_PINS_hook0:
		return Pin_Hook0;
	case PPI_PINS_hook1:
		return Pin_Hook1;
	case PPI_PINS_hook2:
		return Pin_Hook2;
	case PPI_PINS_hook3:
		return Pin_Hook3;
	case PPI_PINS_hook4:
		return Pin_Hook4;
	case PPI_PINS_hook5:
		return Pin_Hook5;
	case PPI_PINS_hook6:
		return Pin_Hook6;
	case PPI_PINS_hook7:
		return Pin_Hook7;
	case PPI_PINS_hook8:
		return Pin_Hook7;
	case PPI_PINS_hook9:
		return Pin_Hook7;
	case PPI_PINS_Pod_Prsnt1_N:
		isValid = false;
		return Pin_DBR;
	case PPI_PINS_Pod_Prsnt2_N:
		isValid = false;
		return Pin_DBR;
	case PPI_PINS_Preq_N:
		return Pin_PREQ;
	case PPI_PINS_Prdy_N:
		return Pin_PRDY;
	case PPI_PINS_TRST:
		return Pin_TRSTn;
	case PPI_PINS_TMS_bit:
	case PPI_PINS_TDI_bit:
	case PPI_PINS_TDO_bit:
	case PPI_PINS_TCK_bit:
		isValid = false;
		return Pin_TRSTn;
	case PPI_PINS_PowerButton:
		return Pin_Hook1;
	case PPI_PINS_ResetButton:
		return Pin_Hook7;
	case PPI_PINS_SystemTapPowerGood:
		return Pin_PWRGOOD;
	case PPI_PINS_SystemReset:
	case PPI_PINS_CoreReset:
		return Pin_ResetOccurred;
	case PPI_PINS_TapReady:
	case PPI_PINS_CorePowerGood:
		return Pin_PWRGOOD;
	case PPI_PINS_PowerBreak1:
		return Pin_Hook3;
	case PPI_PINS_PowerBreak2:
		return Pin_Hook2;
	case PPI_PINS_PowerBreak3:
		return Pin_Hook2;
	case PPI_PINS_JtagDisable:
	case PPI_PINS_I2CDisable:
		isValid = false;
		return Pin_Hook0;
	case PPI_PINS_SystemPowerGood:
		return Pin_PWRGOOD;
	case PPI_PINS_SystemBootStall:
		return Pin_Hook3;
	case PPI_PINS_obsA_func0:
		return Pin_obsA_func0;
	case PPI_PINS_obsA_func1:
		return Pin_obsA_func1;
	case PPI_PINS_obsB_func0:
		return Pin_obsB_func0;
	case PPI_PINS_obsB_func1:
		return Pin_obsB_func1;
	case PPI_PINS_obsC_func0:
		return Pin_obsC_func0;
	case PPI_PINS_obsC_func1:
		return Pin_obsC_func1;
	case PPI_PINS_obsD_func0:
		return Pin_obsD_func0;
	case PPI_PINS_obsD_func1:
		return Pin_obsD_func1;
	case PPI_PINS_obsA_data0:
		return Pin_obsA_data0;
	case PPI_PINS_obsA_data1:
		return Pin_obsA_data1;
	case PPI_PINS_obsA_data2:
		return Pin_obsA_data2;
	case PPI_PINS_obsA_data3:
		return Pin_obsA_data3;
	case PPI_PINS_obsB_data0:
		return Pin_obsB_data0;
	case PPI_PINS_obsB_data1:
		return Pin_obsB_data1;
	case PPI_PINS_obsB_data2:
		return Pin_obsB_data2;
	case PPI_PINS_obsB_data3:
		return Pin_obsB_data3;
	case PPI_PINS_obsC_data0:
		return Pin_obsC_data0;
	case PPI_PINS_obsC_data1:
		return Pin_obsC_data1;
	case PPI_PINS_obsC_data2:
		return Pin_obsC_data2;
	case PPI_PINS_obsC_data3:
		return Pin_obsC_data3;
	case PPI_PINS_obsD_data0:
		return Pin_obsD_data0;
	case PPI_PINS_obsD_data1:
		return Pin_obsD_data1;
	case PPI_PINS_obsD_data2:
		return Pin_obsD_data2;
	case PPI_PINS_obsD_data3:
		return Pin_obsD_data3;
	case PPI_PINS_NO_ACTION:
		isValid = false;
		return Pin_Hook0;
	default:
		isValid = false;
		return Pin_Hook0;
	}
}

inline PPI_Pins_TypeEncode LookupPPIPin(PinTypeEncode pin, bool& isValid)
{
	isValid = true;
	switch (pin)
	{
	case Pin_Hook0:
		return PPI_PINS_hook0;
	case Pin_Hook1:
		return PPI_PINS_hook1;
	case Pin_Hook2:
		return PPI_PINS_hook2;
	case Pin_Hook3:
		return PPI_PINS_hook3;
	case Pin_Hook4:
		return PPI_PINS_hook4;
	case Pin_Hook5:
		return PPI_PINS_hook5;
	case Pin_Hook6:
		return PPI_PINS_hook6;
	case Pin_Hook7:
		return PPI_PINS_hook7;
	case Pin_ResetOccurred:
		return PPI_PINS_SystemReset;
	case Pin_DBR:
		return PPI_PINS_Pod_Prsnt1_N; // I think
	case Pin_PREQ:
		return PPI_PINS_Preq_N;
	case Pin_PRDY:
		return PPI_PINS_Prdy_N;
	case Pin_PWRGOOD:
		return PPI_PINS_SystemPowerGood;
	case Pin_TRSTn:
		return PPI_PINS_TRST;
	case Pin_obsA_func0:
		return PPI_PINS_obsA_func0;
	case Pin_obsA_func1:
		return PPI_PINS_obsA_func1;
	case Pin_obsB_func0:
		return PPI_PINS_obsB_func0;
	case Pin_obsB_func1:
		return PPI_PINS_obsB_func1;
	case Pin_obsC_func0:
		return PPI_PINS_obsC_func0;
	case Pin_obsC_func1:
		return PPI_PINS_obsC_func1;
	case Pin_obsD_func0:
		return PPI_PINS_obsD_func0;
	case Pin_obsD_func1:
		return PPI_PINS_obsD_func1;
	case Pin_obsA_data0:
		return PPI_PINS_obsA_data0;
	case Pin_obsA_data1:
		return PPI_PINS_obsA_data1;
	case Pin_obsA_data2:
		return PPI_PINS_obsA_data2;
	case Pin_obsA_data3:
		return PPI_PINS_obsA_data3;
	case Pin_obsB_data0:
		return PPI_PINS_obsB_data0;
	case Pin_obsB_data1:
		return PPI_PINS_obsB_data1;
	case Pin_obsB_data2:
		return PPI_PINS_obsB_data2;
	case Pin_obsB_data3:
		return PPI_PINS_obsB_data3;
	case Pin_obsC_data0:
		return PPI_PINS_obsC_data0;
	case Pin_obsC_data1:
		return PPI_PINS_obsC_data1;
	case Pin_obsC_data2:
		return PPI_PINS_obsC_data2;
	case Pin_obsC_data3:
		return PPI_PINS_obsC_data3;
	case Pin_obsD_data0:
		return PPI_PINS_obsD_data0;
	case Pin_obsD_data1:
		return PPI_PINS_obsD_data1;
	case Pin_obsD_data2:
		return PPI_PINS_obsD_data2;
	case Pin_obsD_data3:
		return PPI_PINS_obsD_data3;
	default:
		isValid = false;
		return PPI_PINS_hook0;
	}
}
#endif//USE_INTERFACE_SCAN


OpenIPC_Error OpenIPC_No_Vendor_Id = 1;
OpenIPC_Error OpenIPC_No_Ip_Info = 2;

static uint64_t getSlotPosition(PPI_SlotHandle handle){
	return ((uint64_t) handle - 1);
}

void ProbePluginASD_JTAG::SetProbeId(std::shared_ptr<ProbeInstanceASD> probe, OpenIPC_DeviceId probeID) {
	_deviceid2ProbeInstance[probeID] = probe;
	probe->deviceID = probeID;
	probe->SetVersion();
}

OpenIPC_Error ProbePluginASD_JTAG::ListProbes(uint32_t maxNumberOfProbes, OpenIPC_DeviceId *probes, uint32_t *numberOfProbes) {

	if (probes == nullptr)
		return maxNumberOfProbes == 0
			? OpenIPC_Error_No_Error : OpenIPC_Error_Null_Pointer;

	if (numberOfProbes == nullptr)
		return OpenIPC_Error_Null_Pointer;

	*numberOfProbes = (int32_t)_deviceid2ProbeInstance.size();
	uint32_t count = 0;
	for (auto &probe : _deviceid2ProbeInstance) {
		if (count < maxNumberOfProbes) {
			probes[count++] = probe.first;
		}
	}

	return OpenIPC_Error_No_Error;
}

OpenIPC_Error ProbePluginASD_JTAG::ListInterfaces(uint32_t maxNumberOfInterfaces, OpenIPC_DeviceId * interfaces, uint32_t * numberOfInterfaces)
{

	if (interfaces == nullptr)
		return maxNumberOfInterfaces == 0
			? OpenIPC_Error_No_Error : OpenIPC_Error_Null_Pointer;

	if (numberOfInterfaces == nullptr)
		return OpenIPC_Error_Null_Pointer;


	*numberOfInterfaces = (int32_t)_deviceid2probeinterface.size();
	uint32_t count = 0;
	for (auto &interface : _deviceid2probeinterface) {
		if (count < maxNumberOfInterfaces) {
			interfaces[count++] = interface.first;
		}
	}

	return OpenIPC_Error_No_Error;
}

OpenIPC_Error ProbePluginASD_JTAG::RegisterBundleFilter(ASD_JTAG_PROTOCOL protocol, std::shared_ptr<BundleFilter> filter)
{
	if (filter != nullptr) {
		protocolFilters[protocol] = filter;
		return OpenIPC_Error_No_Error;
	}
	return OpenIPC_Error_Null_Pointer;
}

OpenIPC_Error ProbePluginASD_JTAG::DeRegisterAllBundleFilters(void) {
	protocolFilters.clear();
	return OpenIPC_Error_No_Error;
}

std::shared_ptr<BundleFilter> ProbePluginASD_JTAG::GetBundleFilter(ASD_JTAG_PROTOCOL protocol)
{
	auto tentative = protocolFilters.find(protocol);
	if (tentative != protocolFilters.end()) {
		return tentative->second;
	}
	std::shared_ptr<BundleFilter> nullResult;
	return nullResult;
}

std::map<ASD_JTAG_PROTOCOL, std::shared_ptr<BundleFilter>> &ProbePluginASD_JTAG::BundleFilters() {
	return protocolFilters;
}


OpenIPC_Error ProbePluginASD_JTAG::LookupProbe(OpenIPC_DeviceId probeID, std::shared_ptr<ProbeInstanceASD>& probeInstance) {
	return _LookupProbe(probeID, probeInstance);
}

//////////////////////////////////////////////////////////////////////////////
// _GetProbes
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error ProbePluginASD_JTAG::_GetProbes(std::vector<std::weak_ptr<ProbeInstance>>& probes, const char* vendorinit)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
	uint32_t probe_idx = 1;

	probes.clear();

	if ((vendorinit != nullptr) && (vendorinit[0] != 0))
	{
		DataDocument doc("GetProbes");
		doc.LoadXmlString(std::string(vendorinit));
		DataElement* probeChild = doc.RootElement()->FirstChild();
		while (probeChild != nullptr) {
			auto probe = XmlToProbeInstance(probeChild, probe_idx);
			if (probe) {
				probes.push_back(probe);
				this->_probes.push_back(probe);
				probe_idx++;
			}

			probeChild = probeChild->NextSibling();
		}
	}

	return openIPCError;
}

std::shared_ptr<ProbeInstanceASD> ProbePluginASD_JTAG::XmlToProbeInstance(DataElement* element, uint32_t probe_idx) {
	std::shared_ptr<ProbeInstanceASD> result = nullptr;
	const std::string& elementName = element->Name();
	if (elementName == "Probe") {
		DataAttribute* attr = nullptr;
		if ((attr = element->FindAttribute("UniqueId")) != nullptr) {
			result = std::make_shared<ProbeInstanceASD>(_plugin_refid | (probe_idx << 16), _probename);
			result->UniqueId = attr->Value().c_str();
		}
		else if (probe_idx == 1) {
			result = std::make_shared<ProbeInstanceASD>(_plugin_refid | (probe_idx << 16), _probename);
		}
	}
	return result;
}

//////////////////////////////////////////////////////////////////////////////
// _GetProbeInterfaces
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error ProbePluginASD_JTAG::_GetProbeInterfaces(
	std::shared_ptr<ProbeInstance> probe,
	std::vector<std::weak_ptr<InterfaceInstance>>& probeinterfaces
	)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
	std::vector<uint32_t> refids;

	probeinterfaces.clear();
	openIPCError = probe->GetInterfaceRefids(refids);
	if (openIPCError == OpenIPC_Error_No_Error)
	for (auto iter = refids.cbegin(); iter < refids.cend(); iter++) {
		std::weak_ptr<InterfaceInstance> probeinterface;
		openIPCError = probe->GetInterfaceFromRefid(*iter, probeinterface);
		if (openIPCError != OpenIPC_Error_No_Error)
			break;
		probeinterfaces.push_back(probeinterface);
	}

	if (openIPCError != OpenIPC_Error_No_Error)
		POST_ERROR(openIPCError);
	return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
// PluginInitialize
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error ProbePluginASD_JTAG::PluginInitialize(PPI_RefId pluginid, const char* vendorinit)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
	std::string errorMessage = "PluginInitialize Error";

	openIPCError = CProbePlugin::PluginInitialize(pluginid, vendorinit);

	if (openIPCError != OpenIPC_Error_No_Error)
	{
		POST_ERROR_MESSAGE(openIPCError, errorMessage);
	}
	else
	{
		POST_ERROR(openIPCError);
	}

	return openIPCError;
}

OpenIPC_Error ProbePluginASD_JTAG::_GetDrivablePins(std::vector<PPI_Pins_TypeEncode>& pins) const
{
	// This is really a plugin level thing, not probe. If it becomes an issue, we will revist
	for (auto const& probe : this->_probes) {
		probe->GetDrivablePins(pins);
		if (pins.size() > 0)
			break;
	}
	return OpenIPC_Error_No_Error;
}

OpenIPC_Error ProbePluginASD_JTAG::_GetReadablePins(std::vector<PPI_Pins_TypeEncode>& pins) const
{
	// This is really a plugin level thing, not probe. If it becomes an issue, we will revist
	for (auto const& probe : this->_probes) {
		probe->GetReadablePins(pins);
		if (pins.size() > 0)
			break;
	}
	return OpenIPC_Error_No_Error;
}


//////////////////////////////////////////////////////////////////////////////
// ProbeGetInfo
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error ProbePluginASD_JTAG::ProbeGetInfo(PPI_RefId refId, PPI_ProbeInfo* info)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

	std::lock_guard<std::recursive_mutex> lock(_pluginMutex);

	std::shared_ptr<ProbeInstanceASD> probe;
	auto iter = _refid2ProbeInstance.find(refId);
	if (iter != _refid2ProbeInstance.end()
		&& (probe = std::dynamic_pointer_cast<ProbeInstanceASD>(iter->second.lock())))
	{
		InternalUtils::stringcopy(info->type, PPI_MAX_INFO_LEN, probe->typestring.c_str());
		InternalUtils::stringcopy(info->uniqueIdentifier, PPI_MAX_INFO_LEN, probe->UniqueId.c_str());
		InternalUtils::stringcopy(info->probeInfo, PPI_MAX_INFO_LEN, probe->GetVersion().c_str());
	}
	else
	{
		openIPCError = OpenIPC_Error_Invalid_Device_ID;
		PPI_ERROR(OpenIPC_INVALID_DEVICE_ID, openIPCError);
	}

	return openIPCError;
}

//////////////////////////////////////////////////////////////////////////////
// ProbeInitialize
//////////////////////////////////////////////////////////////////////////////
OpenIPC_Error ProbePluginASD_JTAG::ProbeInitialize(uint32_t probe_refid, OpenIPC_DeviceId probeID, const char* vendorinit)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
	std::shared_ptr<ProbeInstanceASD> probe = nullptr;
	std::shared_ptr<InterfaceInstanceRemotePins> pinInterface = nullptr;
	int cnt = 0;
	uint8_t hardware_gpio_config = 0;
	JtagChainParameters params;
	InterfacePadding padding = { 0 };
	JtagChainInfo jtagSettings = { 0 };
	params.InterfacePaddingSettings = &padding;
	params.JtagChainSettings = &jtagSettings;

	if ((vendorinit != nullptr) && (vendorinit[0] != 0))
	{
		DataDocument doc("ProbeInitialize");
		doc.LoadXmlString(std::string(vendorinit));

		DataAttribute* attr = nullptr;
		if ((attr = doc.RootElement()->FindAttribute("UniqueId")) != nullptr) {
			for (auto const& p : this->_probes) {
				if (p->UniqueId == attr->Value().c_str()) {
					probe = p;
					break;
				}
			}
		}
		else if (this->_probes.size() == 1) {
			// if only 1 probe is defined, then UniqueId is not required.
			probe = this->_probes[0];
		}
		else {
			openIPCError = OpenIPC_Error_Probe_Config_Undefined;
			goto end;
		}

		SetProbeId(probe, probeID);
		probe->this_shared = probe;

		DataElement* probeChild = doc.RootElement()->FirstChild();
		while (probeChild != nullptr)
		{
			const std::string& elementName = probeChild->Name();
			if (elementName == "Connection.Parameters")
			{
				openIPCError = probe->ProbeInitializeConnectionParams(probeChild, cnt);
				if (!OpenIPC_PASS(openIPCError))
					goto end;
			}
			else if(elementName == "Interface.Pins")
			{
				openIPCError = probe->ProbeInitializePins(pinInterface, probeChild, cnt);
				if (!OpenIPC_PASS(openIPCError))
					goto end;
			}
			else if (elementName == "Interface.I2C")
			{
				openIPCError = probe->ProbeInitializeI2C(probeChild, cnt);
				if (!OpenIPC_PASS(openIPCError))
					goto end;
			}
			else if(elementName == "Interface.Jtag")
			{
				openIPCError = probe->ProbeInitializeJtag(probeChild, cnt);
				if (!OpenIPC_PASS(openIPCError))
					goto end;
			}
			else if (elementName == "Hardware.Config")
			{
				openIPCError = probe->ProbeInitializeHwConfig(probeChild, hardware_gpio_config, &params);
			}
			probeChild = probeChild->NextSibling();
		}
	}
	else {
		return OpenIPC_No_Vendor_Id;
	}

	if (probe && probe->connectionParameters->connectionInitialized)
	{
		Connection_Error conn_error = RegisterProbePlugin(probe);
		openIPCError = To_OpenIPC_Error(conn_error);

		if (!OpenIPC_PASS(openIPCError)) {
			std::string text = Connection_Error_text(conn_error);
			POST_ERROR_MESSAGE(openIPCError, text);
			goto end;
		}

		if (probe->GetNextJtagInterfaceId() == 0) { // no jtag interface defined in XML, so use BMC value
			PPI_InterfaceJTAGCapabilities jtagCapabilities = { 1, 1, 0 };
			unsigned int jtagChains = probe->connectionParameters->protocolParameters->supportedJtagChains;
			if (jtagChains == 0) { // no BMC value then use a default value
				jtagChains = 1;
			}
			std::string finalJtagSpeed("1000000"); //default is 1MHz
			jtagSettings.JtagChainProtocolId = ASD_JTAG_PROTOCOL::DEFAULT;
			jtagSettings.JtagPrescaler = DEFAULT_JTAG_PRESCALER;
			jtagSettings.JtagDivisor = DEFAULT_JTAG_DIVISOR;

			for(unsigned int i = 0; i < jtagChains; i++) {
				auto jtagInterface = std::make_shared<InterfaceInstanceJtagASD>(probe_refid | cnt++, PPI_interfaceTypeJtag, jtagCapabilities, jtagSettings);
				jtagInterface->settings.Set(std::string("Jtag.TclkRate"), finalJtagSpeed);
				probe->clockCycleTimeinUs = 1000000.0 / (std::stol(finalJtagSpeed));
				probe->AddJtagInterface(jtagInterface);
				jtagSettings.JtagChainProtocolId++;
			}
		}

		if (probe->GetI2cInterfaceBusCount() == 0) { // no i2c interfaces defined in XML, so use BMC value
			PPI_InterfaceI2CCapabilities caps;
			I2cConfiguration config = {};
			unsigned int i2cBusCount = probe->connectionParameters->protocolParameters->supportedI2cBusesCount;
			I2cConfiguration * p_config = probe->connectionParameters->protocolParameters->supportedI2cBusAddresses;
			std::shared_ptr<InterfaceInstanceI2cASD> i2cInterface = nullptr;
			for(unsigned int i = 0; i < i2cBusCount; i++) {
				caps.minimumtimeGraularityInMicroSeconds = p_config[i].busSpeed;
				i2cInterface = std::make_shared<InterfaceInstanceI2cASD>(probe_refid | cnt++, PPI_interfaceTypeI2c, caps);
				config.busSpeed = p_config[i].busSpeed;
				config.busSelect = p_config[i].busSelect;
				i2cInterface->SetConfig(config);
				probe->AddI2CInterface(i2cInterface);
			}
		}


		if (pinInterface != nullptr)
		{
			probe->SetPinsInterface(pinInterface);
		}

		PluginNotifications::GetInstance().Notify(
			probe->deviceID,
			OpenIPC_Error_No_Error,
			PPI_infoNotification,
			[&](std::basic_ostream<char>& stream)
		{
			stream << "ASD Probe plugin version: " << probe->GetVersion();
		});
		// Warn user if they may have problems with Probe and BMC version combination
		if (probe->ProbeMajorMinor != probe->SocketServerMajorMinor) {
			PluginNotifications::GetInstance().Notify(
				probe->deviceID,
				OpenIPC_Error_No_Error,
				PPI_warningNotification,
				[&](std::basic_ostream<char>& stream) {
				std::string warning;
				if (probe->ProbeMajorMinor > probe->SocketServerMajorMinor)
					warning.assign("BMC Firmware does not match ASD Probe Plugin, please upgrade Firmware to " + probe->ProbeMajorMinor + ".x");
				else
					warning.assign("ASD Probe Plugin does not match BMC Firmware, please upgrade Plugin to " + probe->SocketServerMajorMinor + ".x");

				stream << warning;
			});
		}
		openIPCError = To_OpenIPC_Error(SetHardwareGpioConfig(probe->deviceID, hardware_gpio_config));
		if (!OpenIPC_PASS(openIPCError)) {
			POST_ERROR(openIPCError);
			goto end;
		}

		openIPCError = To_OpenIPC_Error(SetHardwareJTAGDriverMode(probe->deviceID, params));
		if (!OpenIPC_PASS(openIPCError)) {
			POST_ERROR(openIPCError);
			goto end;
		}

		if ((pinInterface != nullptr) && (pinInterface.get() != NULL))
			pinInterface->SettingsUpdated();
	}
	else {
		openIPCError = OpenIPC_No_Vendor_Id;
	}
end:
	return openIPCError;
}

OpenIPC_Error ProbePluginASD_JTAG::ProbeDeInitialize(OpenIPC_DeviceId probeID)
{
	OpenIPC_Error err = OpenIPC_Error_No_Error;
	std::map<OpenIPC_DeviceId, std::weak_ptr<ProbeInstance>>::iterator iter = _deviceid2ProbeInstance.find(probeID);
	if (iter == _deviceid2ProbeInstance.end())
		return err;
	std::shared_ptr<ProbeInstanceASD> probe = std::dynamic_pointer_cast<ProbeInstanceASD>(iter->second.lock());

	if ((probe != nullptr) && (probe.get() != NULL)) {
		err = To_OpenIPC_Error(UnRegisterPlugin(probe->deviceID));

		probe->JtagDeinitialize();
		probe->I2CDeinitialize();
	}
	CProbePlugin::ProbeDeInitialize(probeID);
	return err;
}

OpenIPC_Error ProbePluginASD_JTAG::InterfacePortRead(OpenIPC_DeviceId interfaceID, uint8_t* output, uint32_t maxoutputbytes, uint32_t* outputbytes)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;

	std::fill(output, output + maxoutputbytes, 0xFF);

	*outputbytes = maxoutputbytes;

	return openIPCError;
}

OpenIPC_Error ProbePluginASD_JTAG::InterfacePortOpenWindow(OpenIPC_DeviceId interfaceID, PPI_Trace_PortAccessMode accessMode)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
	return openIPCError;
}

OpenIPC_Error ProbePluginASD_JTAG::InterfacePortCloseWindow(OpenIPC_DeviceId interfaceID)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
	return openIPCError;
}

OpenIPC_Error ProbePluginASD_JTAG::InterfacePortIsReadDataAvailable(OpenIPC_DeviceId interfaceID, bool* isDataAvailable)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
	return openIPCError;
}

OpenIPC_Error ProbePluginASD_JTAG::InterfacePortIsWindowOpen(OpenIPC_DeviceId interfaceID, bool* isWindowOpen)
{
	OpenIPC_Error openIPCError = OpenIPC_Error_No_Error;
	return openIPCError;
}

OpenIPC_Error ProbePluginASD_JTAG::LockTargetInterface (OpenIPC_DeviceId deviceInterface){
	Local_Handle* lockedHandle = getLockedHandle();
	lockedHandle->target_valid = true;
	lockedHandle->target = deviceInterface;
	if (lockedHandle->start == NULL){
		lockedHandle->start = new StartCommand();
		lockedHandle->last = lockedHandle->start;
		lockedHandle->status = RUNNING;
	}
	return CProbePlugin::LockTargetInterface(deviceInterface);
}

PPI_ProbeBundleHandle ProbePluginASD_JTAG::BundleAllocate()
{
	Local_Handle* result = new Local_Handle();
	//TODO: Should add this to a constructor
	result->start = new StartCommand();
	result->last = result->start;
	result->dr_post = 0;
	result->dr_pre = 0;
	result->ir_post = 0;
	result->ir_pre = 0;
	result->status = RUNNING;
	result->number_of_bits_in_last_data = 0;
	result->enclosingContext = NULL;
	result->last_data = NULL;
	result->target_valid = false;
	return (PPI_ProbeBundleHandle)result;
}

OpenIPC_Error ProbePluginASD_JTAG::BundleExecute(PPI_ProbeBundleHandle current_handle,  OpenIPC_DeviceId deviceInterface, PPI_bool keepLock){
	Local_Handle* current;
	if (current_handle == PPI_PROBE_LOCK_HOLD || current_handle == PPI_PROBE_LOCK_RELEASE) {
		current = getLockedHandle();
	} else {
		current = (Local_Handle*) current_handle;
	}

	OpenIPC_Error result = OpenIPC_Error_No_Error;

	// Check interface for registered filter[s] by its JTAG protocol
	std::weak_ptr<InterfaceInstance> interfaceInstance = GetInstance()->InterfaceFromId(deviceInterface);
	auto probeInterface = std::dynamic_pointer_cast<InterfaceInstanceJtagASD>(interfaceInstance.lock());
	std::shared_ptr<BundleFilter> protocolFilter;
	if (probeInterface != nullptr && probeInterface.get() != NULL)
		protocolFilter = GetBundleFilter(probeInterface->EncryptionProtocol());

	if (protocolFilter != nullptr && protocolFilter.get() != NULL) {
		result = protocolFilter->Execute(current, deviceInterface, keepLock == 0 ? false : true);
	} else {
		result = ExecuteHandle(current, deviceInterface, keepLock == 0 ? false : true);
	}

	if (OpenIPC_PASS(result)) {
		// Do base class interface lock management
		result = CProbePlugin::BundleExecute(PPI_PROBE_LOCK_HOLD, deviceInterface, keepLock);
	}

	return result;
}

OpenIPC_Error DoExecuteMultiChain(ProbePluginASD_JTAG* instance, std::vector<uint32_t> chains, PPI_ProbeBundleHandle current_handle, PPI_bool keepLock, bool sendWaitSync) {
	OpenIPC_Error result = OpenIPC_Error_No_Error;
	std::weak_ptr<InterfaceInstance> interfaceInstance;

	for (auto it = chains.begin(); it != chains.end(); ++it)
	{
		interfaceInstance = instance->GetInstance()->InterfaceFromId(*it);
		auto jtag_interface = std::dynamic_pointer_cast<InterfaceInstanceJtagASD>(interfaceInstance.lock());
		if (jtag_interface != nullptr && jtag_interface.get() != NULL) {
			jtag_interface->SetMultiBundleExecute(true);
			if (it == chains.begin())
				jtag_interface->SetSendWaitSync(sendWaitSync);

			if (std::next(it) == chains.end()) // last element
				jtag_interface->SetMultiBundleFinalInterface(true);
			else
				jtag_interface->SetMultiBundleFinalInterface(false);
			result = instance->BundleExecute(current_handle, *it, keepLock);
			jtag_interface->SetMultiBundleExecute(false);
			jtag_interface->SetMultiBundleFinalInterface(false);
			jtag_interface->SetSendWaitSync(false);  // disable
		}
	}

	// At this point, all of the inputs have been sanity checked and we call into the probe plugin specific code
	if (OpenIPC_PASS(result))
	{
		// Do base class interface lock management
		result = instance->CProbePlugin::BundleExecuteMultiChain(PPI_PROBE_LOCK_HOLD, &chains[0], (uint32_t)chains.size(), keepLock);
	}
	return result;
}

OpenIPC_Error ProbePluginASD_JTAG::BundleExecuteMultiChain(PPI_ProbeBundleHandle current_handle, OpenIPC_DeviceId* deviceInterfaces, uint32_t deviceInterfacesLength, PPI_bool keepLock)
{
	OpenIPC_Error result = OpenIPC_Error_No_Error;
	std::weak_ptr<InterfaceInstance> interfaceInstance;

	if (deviceInterfaces == nullptr)
		return OpenIPC_Error_Null_Pointer;

	std::map<OpenIPC_DeviceId, std::vector<OpenIPC_DeviceId>> chainsByProbe;

	// build up a map of probes (keys) and lists of chains (values)
	for (uint32_t i = 0; OpenIPC_PASS(result) && i < deviceInterfacesLength; ++i)
	{
		interfaceInstance = GetInstance()->InterfaceFromId(deviceInterfaces[i]);
		auto jtag_interface = std::dynamic_pointer_cast<InterfaceInstanceJtagASD>(interfaceInstance.lock());
		if ((jtag_interface != nullptr) && (jtag_interface.get() != NULL))
			chainsByProbe[jtag_interface->probe->deviceID].push_back(deviceInterfaces[i]);
		else
			return OpenIPC_Error_Null_Pointer;
	}

	std::vector<std::future<OpenIPC_Error>> futures;

	// iterate through each probe
	for (auto& element : chainsByProbe) {
		// launch an async stash and store it in the futures vector
		futures.push_back(std::async(std::launch::async, &DoExecuteMultiChain, this, element.second, current_handle, keepLock, (chainsByProbe.size() > 1)));
	}

	// wait for each async task to end
	for (int i = 0; i < futures.size(); i++) {
		OpenIPC_Error future_result = futures[i].get();
		if (OpenIPC_PASS(result) && !OpenIPC_PASS(future_result)) {
			// store the first error found as the main 'result'
			result = future_result;
			// Do not break here if error is found.
			// Lets play it safe and wait for all the futures
			// to finish
		}
	}

	return result;
}

OpenIPC_Error ProbePluginASD_JTAG::BundleClear(PPI_ProbeBundleHandle handle) {
	if (handle != nullptr) {
		Local_Handle* current = (Local_Handle*)handle;
		current->dr_post = 0;
		current->dr_pre = 0;
		current->ir_post = 0;
		current->ir_pre = 0;
		current->start->Clear();
		current->last = current->start;
	}
	return OpenIPC_Error_No_Error;
}

OpenIPC_Error ProbePluginASD_JTAG::BundleFree(PPI_ProbeBundleHandle * handle)
{
	OpenIPC_Error result = OpenIPC_Error_No_Error;
	if (*handle != PPI_PROBE_LOCK_RELEASE && *handle != PPI_PROBE_LOCK_HOLD) {
		result = PPI_Bundle_Clear(*handle);
		Local_Handle* current = (Local_Handle*)(*handle);
		delete current->start;
		delete current;
		*handle = PPI_PROBE_LOCK_RELEASE;
	}
	return result;
}

/********** JTAG ***********/

OpenIPC_Error ProbePluginASD_JTAG::PPI_JTAG_GoToState(
	PPI_ProbeBundleHandle handle,
	JtagStateEncode goto_state,
	uint32_t number_of_clocks_in_state,
	const PPI_JTAG_StateGotoOptions* const options
	){
		GotoStateJTAGCommandASD* cmd = new GotoStateJTAGCommandASD();
		cmd->goto_state = goto_state;
		cmd->number_of_clocks_in_state = number_of_clocks_in_state;
		cmd->plugin = GetProbePluginInstanceASD();
		// We need to make a copy of options here
		if (options == nullptr) {
			cmd->options = nullptr;
		} else {
			auto optionsNew = new PPI_JTAG_StateGotoOptions();
			memcpy_s(optionsNew, sizeof(PPI_JTAG_StateGotoOptions), options, sizeof(PPI_JTAG_StateGotoOptions));
			cmd->options = optionsNew;
		}

		return handle_command(cmd, handle);
}

OpenIPC_Error ProbePluginASD_JTAG::PPI_JTAG_StateIRShift(
	PPI_ProbeBundleHandle handle,
	uint32_t shiftLengthBits,
	const uint8_t* const inBits,
	uint8_t* outBits,
	const PPI_JTAG_StateShiftOptions* const options
	){
		IRShiftJTAGCommandASD* cmd = new IRShiftJTAGCommandASD();
		uint32_t number_of_bytes = (shiftLengthBits+7)/8;
		if (options != NULL){
			cmd->savedSlot = (uint8_t)getSlotPosition(options->savedSlot);
			cmd->TdiTdoOptions = options->TdiTdoOptions;
		}else{
			cmd->savedSlot = 0;
			cmd->TdiTdoOptions = JtagOption_TDI_Default;
		}
		if (inBits != NULL){
			cmd->inBits.assign(inBits,inBits+number_of_bytes);
		} else {
			cmd->inBits.clear();
			//assignTDI(number_of_bytes,cmd->inBits,cmd->TdiTdoOptions);
		}
		cmd->shiftLengthBits = shiftLengthBits;
		cmd->outBits = outBits;
		cmd->plugin = GetProbePluginInstanceASD();
		return handle_command(cmd, handle);
}

OpenIPC_Error ProbePluginASD_JTAG::PPI_JTAG_StateDRShift(
	PPI_ProbeBundleHandle handle,
	uint32_t shiftLengthBits,
	const uint8_t* const inBits,
	uint8_t* outBits,
	const PPI_JTAG_StateShiftOptions* const options
	){
		DRShiftJTAGCommandASD* cmd = new DRShiftJTAGCommandASD();
		uint32_t number_of_bytes = (shiftLengthBits+7)/8;
		if (options != NULL){
			cmd->savedSlot = (uint8_t)getSlotPosition(options->savedSlot);
			cmd->TdiTdoOptions = options->TdiTdoOptions;
		}else{
			cmd->savedSlot = 0;
			cmd->TdiTdoOptions = JtagOption_TDI_Default;
		}
		if (inBits != NULL){
			cmd->inBits.assign(inBits,inBits+number_of_bytes);
		} else {
			cmd->inBits.clear();
			//assignTDI(number_of_bytes,cmd->inBits,cmd->TdiTdoOptions);
		}
		cmd->shiftLengthBits = shiftLengthBits;
		cmd->outBits = outBits;
		cmd->plugin = GetProbePluginInstanceASD();
		return handle_command(cmd, handle);
}

#ifdef USE_INTERFACE_SCAN
typedef struct WritebackRecord
{
	uint32_t            start;
	uint32_t			sizeInBits;
	bool                isIR;
} WritebackRecord;


OpenIPC_Error ProbePluginASD_JTAG::_JtagScan(InterfaceInstance& interfaceInstance,
													  const uint32_t* input,
													  uint32_t inputdwords,
													  uint32_t* output,
													  uint32_t maxoutputdwords,
													  uint32_t* outputdwords)
{
	OpenIPC_Error result = OpenIPC_Error_No_Error;
	OpenIPC_DeviceId interfaceID = interfaceInstance.deviceID;
	InterfaceInstanceJtagASD& jtagInterfaceInstance = *(InterfaceInstanceJtagASD*)&interfaceInstance;
	// Now, let's iterate over the input in order to figure out what bundle to make
	uint32_t currentDwordInOutput = 0;
	auto bundle = PPI_Bundle_Allocate();

	PPI_LOG_STREAM(interfaceID, PPI_HARDWARE_COMMUNICATION_STREAM, PPI_infoNotification, "Starting InterfaceScan translation on device: " << LOG_UINT32_HEX(interfaceID) << " payload is: " << LOG_UINT32_ARRAY(input, inputdwords));

	std::deque<std::unique_ptr<std::vector<uint32_t>>> localBuffers;
	std::vector<std::unique_ptr<PPI_JTAG_StateGotoOptions>> options;
	std::vector<WritebackRecord> writebacks;

	if (OpenIPC_PASS(result) && bundle == PPI_PROBE_LOCK_RELEASE)
	{
		// We *might* be able to handle this... but let's just assume that the adapter has done the right thing already and we have legit bundles
		result = OpenIPC_Error_Probe_Bundle_Invalid;
	}
	for (uint32_t currentDowrdInInput = 0; OpenIPC_PASS(result) && currentDowrdInInput < inputdwords; currentDowrdInInput++)
	{
		const MetaOpDwordEncode* metaopdwordptr = (const MetaOpDwordEncode*)&input[currentDowrdInInput];
		if (metaopdwordptr->metaop == 1)
		{
			if (metaopdwordptr->optype == MetaOpTypeEncode::Hold)
			{
				if (metaopdwordptr->start == 1)
				{
					jtagInterfaceInstance.lock = true;
					PPI_LOG_STREAM(interfaceID, PPI_HARDWARE_COMMUNICATION_STREAM, PPI_infoNotification, "Setting keep locks to true on device " << LOG_UINT32_HEX(interfaceID));
				}
				else
				{
					jtagInterfaceInstance.lock = false;
					PPI_LOG_STREAM(interfaceID, PPI_HARDWARE_COMMUNICATION_STREAM, PPI_infoNotification, "Setting keep locks to false on device " << LOG_UINT32_HEX(interfaceID));
				}
			}
		}
		else
		{
			if (!jtagInterfaceInstance.jtagActive)
			{
				auto leadingTCKs = interfaceInstance.settings.GetUint32("Jtag.LeadingTCKs");
				if (leadingTCKs > 0)
				{
					PPI_LOG_STREAM(interfaceID, PPI_HARDWARE_COMMUNICATION_STREAM, PPI_infoNotification, "Issuing JTAG GoTo State at the begining of a jtag scan: " << c_statestrASD[jtagInterfaceInstance.lastjtagstate] << " for count: 1");
					result = PPI_JTAG_GoToState(bundle, jtagInterfaceInstance.lastjtagstate, leadingTCKs, nullptr);
					if (!OpenIPC_PASS(result))
					{
						// bail from the loop so that we don't clobber the result error code
						break;
					}
				}

				jtagInterfaceInstance.jtagActive = true;
			}

			const JtagDwordEncode* jtagdwordptr = (const JtagDwordEncode*)(void*)metaopdwordptr;
			const uint32_t* inputbuffer = jtagdwordptr->usetdi ? &input[currentDowrdInInput + 1] : 0;
			JtagStateEncode deststate = (JtagStateEncode)jtagdwordptr->gotostate;

			assert(jtagdwordptr->count >= 1);
			assert((jtagdwordptr->count == 1) || (deststate == JtagRTI) || (deststate == JtagTLR) || (deststate == JtagPauIR) || (deststate == JtagPauDR) || (deststate == JtagShfDR) || (deststate == JtagShfIR));
			assert((uint8_t)jtagdwordptr->gotostate < 16);

			if ((deststate == JtagShfIR) || (deststate == JtagShfDR))
			{
				// Perform the shift and remember if we need to save TDO :)
				uint32_t* outBits = nullptr;
				if (jtagdwordptr->capturetdo)
				{
					WritebackRecord rec;
					rec.start = 0;
					rec.sizeInBits = jtagdwordptr->count;
					rec.isIR = (deststate == JtagShfIR);
					writebacks.push_back(rec);

					auto temp_1 = new std::vector<uint32_t>((jtagdwordptr->count + 31) / 32);
					std::unique_ptr<std::vector<uint32_t>> temp_2(temp_1);
					localBuffers.push_back(std::move(temp_2));

					outBits = localBuffers.back()->data();
					currentDwordInOutput += (jtagdwordptr->count + 31) / 32;
				}

				if (OpenIPC_PASS(result) && deststate == JtagShfIR)
				{
					PPI_LOG_STREAM(interfaceID, PPI_HARDWARE_COMMUNICATION_STREAM, PPI_infoNotification, "Issuing JTAG SHIR: " << c_statestrASD[deststate] << " for count: " << jtagdwordptr->count << " with data: " << LOG_UINT8_ARRAY(reinterpret_cast<const uint8_t*>(inputbuffer), (jtagdwordptr->count + 7) / 8));
					result = PPI_JTAG_StateIRShift(bundle, jtagdwordptr->count, reinterpret_cast<const uint8_t*>(inputbuffer), reinterpret_cast<uint8_t*>(outBits), nullptr);
				}
				else if (OpenIPC_PASS(result))
				{
					PPI_LOG_STREAM(interfaceID, PPI_HARDWARE_COMMUNICATION_STREAM, PPI_infoNotification, "Issuing JTAG SHDR: " << c_statestrASD[deststate] << " for count: " << jtagdwordptr->count << " with data: " << LOG_UINT8_ARRAY(reinterpret_cast<const uint8_t*>(inputbuffer), (jtagdwordptr->count + 7) / 8));
					result = PPI_JTAG_StateDRShift(bundle, jtagdwordptr->count, reinterpret_cast<const uint8_t*>(inputbuffer), reinterpret_cast<uint8_t*>(outBits), nullptr);
				}
			}
			else
			{
				assert(jtagdwordptr->usetdi == 0);
				assert(jtagdwordptr->capturetdo == 0);
				options.push_back(std::unique_ptr<PPI_JTAG_StateGotoOptions>(new PPI_JTAG_StateGotoOptions()));
				options.back()->waitForTrigger = jtagdwordptr->triggerstall;

				PPI_LOG_STREAM(interfaceID, PPI_HARDWARE_COMMUNICATION_STREAM, PPI_infoNotification, "Issuing JTAG GoTo State: " << c_statestrASD[deststate] << " for count: " << jtagdwordptr->count);
				result = PPI_JTAG_GoToState(bundle, deststate, jtagdwordptr->count, options.back().get());
			}
			jtagInterfaceInstance.lastjtagstate = deststate;

			if (jtagdwordptr->usetdi)
			{
				currentDowrdInInput += BitsToUint32s(jtagdwordptr->count);
			}
		}
	}

	if (jtagInterfaceInstance.jtagActive && OpenIPC_PASS(result))
	{
		// if there is not (or no longer) an active lock
		if (jtagInterfaceInstance.lock == false)
		{
			// always go back to the "park" state, i.e., RTI, if not already there, when the JTAG lock is released
			if (jtagInterfaceInstance.lastjtagstate != JtagRTI)
			{
				JtagStateEncode deststate = JtagRTI;
				PPI_LOG_STREAM(interfaceID, PPI_HARDWARE_COMMUNICATION_STREAM, PPI_infoNotification, "Issuing JTAG GoTo State at the end of a lock: " << c_statestrASD[deststate] << " for count: 1");
				result = PPI_JTAG_GoToState(bundle, deststate, 1, nullptr);
				jtagInterfaceInstance.lastjtagstate = JtagRTI;
			}

			jtagInterfaceInstance.jtagActive = false;
		}
	}


	// Ok, finally we do the work :)
	if (OpenIPC_PASS(result))
	{
		PPI_LOG_STREAM(interfaceID, PPI_HARDWARE_COMMUNICATION_STREAM, PPI_infoNotification, "Executing scan with keepLocks " << jtagInterfaceInstance.lock << " on device " << LOG_UINT32_HEX(interfaceID));
		result = PPI_Bundle_Execute(bundle, interfaceID, jtagInterfaceInstance.lock);
		PPI_Bundle_Free(&bundle);
	}

	// Update the last parameter...
	if (OpenIPC_PASS(result))
	{
		BitData outputBuffer = BitData_CreateLocalFromBuffer(maxoutputdwords * 32, maxoutputdwords * 32, output);
		uint32_t currentBit = 0;
		uint32_t place = 0;
		for (auto rec : writebacks)
		{
			if (currentBit + rec.sizeInBits > outputBuffer.bitsize)
			{
				result = OpenIPC_Error_Probe_Invalid_Parameter;
				PPI_ERROR_WITH_MESSAGE(interfaceID, result, "output buffer is not large enough for writeback!");
				break;
			}
			else
			{
				BitData currentChunk = BitData_CreateLocalFromBuffer(rec.sizeInBits, rec.sizeInBits, localBuffers[place]->data() + rec.start);
				BitData_Copy(&currentChunk, 0, &outputBuffer, currentBit, currentChunk.bitsize);
				currentBit += (uint32_t)currentChunk.bitsize;
#if defined(HOST_WINDOWS)
#pragma warning (disable : 4267)
#endif
				PPI_LOG_STREAM_FUNC(interfaceID, PPI_HARDWARE_COMMUNICATION_STREAM, PPI_infoNotification, [&](std::basic_ostream<char>& stream) \
				{
					std::vector<char> temp;
					temp.resize(2 + (3 + rec.sizeInBits) / 4 + 1);
					BitData_ToHex(&currentChunk, 0, rec.sizeInBits, temp.data(), temp.size());
					// Make sure last character is null..
					temp[temp.size() - 1] = '\0';
					if (rec.isIR)
					{
						stream << "JTAG IR writeback: " << rec.sizeInBits << " bits with data: " << temp.data();
					}
					else
					{
						stream << "JTAG DR writeback: " << rec.sizeInBits << " bits with data: " << temp.data();
					}
				});
#if defined(HOST_WINDOWS)
#pragma warning (default : 4267)
#endif
			}
			place++;

		}

		if(outputdwords != nullptr)
			*outputdwords = currentDwordInOutput;
	}

	return result;
}

OpenIPC_Error ProbePluginASD_JTAG::_PinsScan(InterfaceInstance& interfaceInstance,
													  const uint32_t* input,
													  uint32_t inputdwords,
													  uint32_t* output,
													  uint32_t maxoutputdwords,
													  uint32_t* outputdwords)
{
	OpenIPC_Error result = OpenIPC_Error_No_Error;
	uint32_t currentDwordInOutput = 0;
	OpenIPC_DeviceId interfaceID = interfaceInstance.deviceID;
	auto bundle = PPI_Bundle_Allocate();
	if (OpenIPC_PASS(result) && bundle == PPI_PROBE_LOCK_RELEASE)
	{
		// We *might* be able to handle this... but let's just assume that the adapter has done the right thing already and we have legit bundles
		result = OpenIPC_Error_Probe_Bundle_Invalid;
		PPI_ERROR_WITH_MESSAGE(interfaceID, result, "Unable to get a new bundle.");
	}
	std::vector<uint64_t> reads;
	for (uint32_t currentDowrdInInput = 0;  OpenIPC_PASS(result) && currentDowrdInInput < inputdwords; currentDowrdInInput++)
	{
		const MetaOpDwordEncode* metaopdwordptr = (const MetaOpDwordEncode*)&input[currentDowrdInInput];
		if (metaopdwordptr->metaop == 1)
		{
			if (metaopdwordptr->optype == MetaOpTypeEncode::Hold)
			{
				if (metaopdwordptr->start == 1)
				{
					this->keepLocks[interfaceID] = true;
					PPI_LOG_STREAM(interfaceID, PPI_infoNotification, PPI_infoNotification, "Setting keep locks to true.");
				}
				else
				{
					this->keepLocks[interfaceID] = false;
					PPI_LOG_STREAM(interfaceID, PPI_infoNotification, PPI_infoNotification, "Setting keep locks to false.");
				}
			}
		}
		else
		{
			const PinsDwordEncode* pinsdwordptr = (const PinsDwordEncode*)(void*)metaopdwordptr;
			BitData bdPinsVector = BitData_CreateLocalFromBuffer(64, 64, (void*)(&input[currentDowrdInInput + 1]));
			uint64_t pincount;
			BitData_PopCount(&bdPinsVector, 0, 64, &pincount);
			if (pincount > 1)
			{
				result = OpenIPC_Error_Interface_Pins_Pin_Not_Supported;
				PPI_ERROR_WITH_MESSAGE(interfaceID, result, "Multiple pin operations requested; not supported.");
				break;
			}
			else if (pincount == 0)
			{
				result = OpenIPC_Error_Interface_Pins_Pin_Not_Supported;
				PPI_ERROR_WITH_MESSAGE(interfaceID, result, "No pin operations requested; not supported.");
				break;
			}

			uint64_t pinselect;
			BitData_FindFirst(&bdPinsVector, 0, 64, &pinselect);
			PinTypeEncode temp = static_cast<PinTypeEncode>(pinselect);
			bool isValid;
			auto op = LookupPPIPin(temp, isValid);
			if (!isValid)
			{
				result = OpenIPC_Error_Interface_Pins_Pin_Not_Supported;
				PPI_ERROR_WITH_MESSAGE(interfaceID, result, "pin: " << temp << " is not a valid pin.");
				break;
			}

			uint32_t count = pinsdwordptr->largeCount ? pinsdwordptr->count << 12 : pinsdwordptr->count;
			count = count / 1000;

			if (pinsdwordptr->operation == PinsOperationEncode::pinsRead)
			{
				PPI_LOG_STREAM(interfaceID, PPI_infoNotification, PPI_infoNotification, "Reading pin: " << op);
				reads.push_back(pinselect);
				result = PPI_Pins_ReadPin(bundle, op, &output[currentDwordInOutput], nullptr);
				currentDwordInOutput += 2;
			}
			else if (pinsdwordptr->operation == PinsOperationEncode::pinsWriteAsserted)
				{
				PPI_LOG_STREAM(interfaceID, PPI_infoNotification, PPI_infoNotification, "Setting pin: " << op);
					result = PPI_Pins_SetPin(bundle, op, true, nullptr);
				}
				else if (pinsdwordptr->operation == PinsOperationEncode::pinsWriteDeAsserted)
				{
					PPI_LOG_STREAM(interfaceID, PPI_infoNotification, PPI_infoNotification, "Deasserting pin: " << op);
					result = PPI_Pins_SetPin(bundle, op, false, nullptr);
				}
				else if (pinsdwordptr->operation == PinsOperationEncode::pinsPulseAsserted)
				{
					PPI_LOG_STREAM(interfaceID, PPI_infoNotification, PPI_infoNotification, "Pulsing pin: " << op);
					result = PPI_Pins_PulsePin(bundle, op, false, count, nullptr);
				}
				else if (pinsdwordptr->operation == PinsOperationEncode::pinsPulseDeAsserted)
				{
					PPI_LOG_STREAM(interfaceID, PPI_infoNotification, PPI_infoNotification, "Pulsing pin deasserted: " << op);
					result = PPI_Pins_PulsePin(bundle, op, true, count, nullptr);
				}
				else
				{
					result = OpenIPC_Error_Interface_Pins_Operation_Not_Supported;
					PPI_ERROR_WITH_MESSAGE(interfaceID, result, "Pin operation type is not supported: " << pinsdwordptr->operation);
					break;
				}

			currentDowrdInInput += 2; // skip over QWORD payload of pin vector
			if (outputdwords != nullptr)
			{
				*outputdwords += 2;
			}
		}
	}
	if (OpenIPC_PASS(result))
	{
		PPI_LOG_STREAM(interfaceID, PPI_infoNotification, PPI_infoNotification, "Executing bundle");
		result = PPI_Bundle_Execute(bundle, interfaceID, this->keepLocks[interfaceID]);
	}
	if (outputdwords != nullptr && OpenIPC_PASS(result))
	{
		PPI_LOG_STREAM(interfaceID, PPI_infoNotification, PPI_infoNotification, "Writing number of dwords used in output.");
		uint32_t place = 0;
		for (auto read : reads)
		{
			if (read < 32)
			{
				output[place] = output[place] << read;
				output[place + 1] = 0;
			}
			else
			{
				output[place] = 0;
				output[place + 1] = output[place] << (read - 32);
			}
			place += 2;
		}
		*outputdwords = currentDwordInOutput;
	}
	PPI_Bundle_Free(&bundle);
	return result;
}

OpenIPC_Error ProbePluginASD_JTAG::_I2cScan(InterfaceInstance& interfaceInstance,
													 const uint32_t* input,
													 uint32_t inputdwords,
													 uint32_t* output,
													 uint32_t maxoutputdwords,
													 uint32_t* outputdwords)
{
	OpenIPC_Error result = OpenIPC_Error_No_Error;
	OpenIPC_DeviceId interfaceID = interfaceInstance.deviceID;
	uint32_t currentDwordInOutput = 0;

	auto bundle = PPI_Bundle_Allocate();
	if (OpenIPC_PASS(result) && bundle == PPI_PROBE_LOCK_RELEASE)
	{
		// We *might* be able to handle this... but let's just assume that the adapter has done the right thing already and we have legit bundles
		result = OpenIPC_Error_Probe_Bundle_Invalid;
	}
	for (uint32_t currentDowrdInInput = 0; OpenIPC_PASS(result) && currentDowrdInInput < inputdwords && currentDwordInOutput < maxoutputdwords; currentDowrdInInput++)
	{
		const MetaOpDwordEncode* metaopdwordptr = (const MetaOpDwordEncode*)&input[currentDowrdInInput];
		if (metaopdwordptr->metaop == 1)
		{
			if (metaopdwordptr->optype == MetaOpTypeEncode::Hold)
			{
				if (metaopdwordptr->start == 1)
				{
					this->keepLocks[interfaceID] = true;
				}
				else
				{
					this->keepLocks[interfaceID] = false;
				}
			}
		}
		else
		{
			const I2cDwordEncodeDescriptor* i2cdwordptr = (const I2cDwordEncodeDescriptor*)(void*)metaopdwordptr;

			uint32_t bytecnt = i2cdwordptr->bytecnt;
			if (i2cdwordptr->read != 0)
			{
				// add a read operation
				if (currentDwordInOutput + bytecnt + 1 <= maxoutputdwords)
				{
					result = PPI_I2C_DirectRead(bundle, interfaceID, bytecnt, reinterpret_cast<uint8_t*>(&output[currentDwordInOutput + 1]),
													  i2cdwordptr->forcestop, reinterpret_cast<uint8_t*>(&output[currentDwordInOutput]), nullptr);
					currentDwordInOutput += bytecnt + 1;
				}
				else
				{
					result = OpenIPC_Error_Probe_Invalid_Size;
				}
			}
			else
			{
				if (currentDwordInOutput + 2 <= maxoutputdwords)
				{
					result = PPI_I2C_DirectWrite(bundle, interfaceID, bytecnt, reinterpret_cast<const uint8_t*>(&input[currentDowrdInInput]),
													   i2cdwordptr->forcestop, reinterpret_cast<uint8_t*>(&output[currentDwordInOutput]),
													   &output[currentDwordInOutput + 1], nullptr);
					currentDowrdInInput += bytecnt;
				}
				else
				{
					result = OpenIPC_Error_Probe_Invalid_Size;
				}
			}
		}
	}
	if (OpenIPC_PASS(result))
	{
		result = PPI_Bundle_Execute(bundle, interfaceID, this->keepLocks[interfaceID]);
	}
	if (OpenIPC_PASS(result))
	{
		*outputdwords = currentDwordInOutput;
	}

	PPI_Bundle_Free(&bundle);
	return result;
}
#endif // USE_INTERFACE_SCAN

