/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
///  @file
///
///  @brief Implements functionality for accessing information about the current execution environment
//////////////////////////////////////////////////////////////////////////////

#include "ExecutionEnvironment.h"
#include <Components/Common/CommonErrors.h>
#include <Foundation/Error/ErrorMacros.h>

#include "PushNewOverride.h"
    #include <boost/asio/ip/host_name.hpp>
#include "PopNewOverride.h"

#include <vector>
#include <mutex>
#include <regex>

#include "PushNewOverride.h"
#include <boost/filesystem/operations.hpp>
#include "PopNewOverride.h"

#if defined(HOST_WINDOWS)
    #include <Shlobj.h> // For SHGetKnownFolderPath
    #include <codecvt>
#elif defined(HOST_LINUX) || defined(HOST_DARWIN)
    #include <unistd.h> // For sysconf(), getuid(), getpid(), gethostname()
    #include <sys/types.h>
    #include <pwd.h>   // For getpwuid_r()
    #include <fstream> // For std::ifstream
#endif

#if defined(HOST_DARWIN)
    #include <libproc.h> // For proc_pidpath
#endif

void InitializeBoostFilesystem()
{
    static bool s_initialized = false;
    static std::mutex s_mutex;
    std::lock_guard<std::mutex> lock(s_mutex);
    if (!s_initialized)
    {
        boost::filesystem::path::imbue(std::locale());
        s_initialized = true;
    }
}

namespace CommonUtils
{
    OpenIPC_Error GetUserHomeDirectory(OUT std::string& directory)
    {
        OpenIPC_Error error = OpenIPC_Error_No_Error;

        #if defined(HOST_WINDOWS)
            wchar_t* path   = nullptr;
            HRESULT  result = SHGetKnownFolderPath(FOLDERID_Profile, KF_FLAG_DEFAULT, NULL, &path);

            if (result != S_OK)
            {
                error = OpenIPC_Error_Internal_Error;
            }

            if (OpenIPC_PASS(error))
            {
                std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>, wchar_t> converter;
                directory = converter.to_bytes(path);
                CoTaskMemFree(path); path = nullptr;
            }
        #elif defined(HOST_LINUX) || defined(HOST_DARWIN)
            //First check the HOME environment variable
            char* homeDir = getenv("HOME");

            if (homeDir == nullptr)
            {
                //If the HOME environment variable is not set, get the password entry for the current user
                uid_t userID = getuid(); // Get the current user's ID
                long  maximumPwdSize = sysconf(_SC_GETPW_R_SIZE_MAX);

                if (maximumPwdSize < 0)
                {
                    //There is no limit. We should try and pick a sane value here
                    maximumPwdSize = 4096;
                }

                if (OpenIPC_PASS(error))
                {
                    std::vector<char> buffer(maximumPwdSize, 0);
                    struct passwd     pwd;
                    struct passwd*    resultPtr = nullptr;
                    int result = getpwuid_r(userID, &pwd, buffer.data(), buffer.size(), &resultPtr);

                    if ((result != 0) || (resultPtr == nullptr))
                    {
                        error = OpenIPC_Error_Internal_Error;
                    }

                    if (OpenIPC_PASS(error))
                    {
                        directory = pwd.pw_dir;
                    }
                }
            }
            else
            {
                directory = homeDir;
            }
        #else
            error = OpenIPC_Error_Not_Implemented;
        #endif

        return error;
    }

    OpenIPC_Error GetCurrentProcessNameAndId(OUT std::string& processName, OUT uint32_t& processID)
    {
        OpenIPC_Error error = OpenIPC_Error_No_Error;
        #if defined(HOST_WINDOWS)
            DWORD bufferSize     = 0;
            DWORD numCharsCopied = 0;
            std::vector<wchar_t> buffer;
            while (numCharsCopied == bufferSize)
            {
                bufferSize += 1024;
                buffer.resize(bufferSize, L'\0');
                numCharsCopied = ::GetModuleFileNameW(nullptr, buffer.data(), bufferSize);
                if (numCharsCopied == 0)
                {
                    //An error occured...
                    error = OpenIPC_Error_Internal_Error;
                    break;
                }
            }

            if (OpenIPC_PASS(error))
            {
                buffer[numCharsCopied] = L'\0';
                std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>> converter;
                InitializeBoostFilesystem();
                boost::filesystem::path fullPath = converter.to_bytes(buffer.data());
                processName = fullPath.stem().string();
            }

            if (OpenIPC_PASS(error))
            {
                processID = GetCurrentProcessId();
            }

        #elif defined(HOST_LINUX)
            pid_t pid = getpid();
            std::ifstream file("/proc/" + std::to_string(pid) + "/comm"); //It is ok to use std::ifstream here
                                                                          // because it can handle UTF-8 paths
                                                                          // on linux
            if (!file)
            {
                error = OpenIPC_Error_Internal_Error;
            }
            else
            {
                file >> processName;
                processID = static_cast<uint32_t>(pid);
            }
        #elif defined(HOST_DARWIN)
            pid_t pid = getpid();
            char  buffer[PROC_PIDPATHINFO_MAXSIZE] = { 0 };
            int   ret = proc_pidpath(pid, buffer, sizeof(buffer));
            if (ret <= 0)
            {
                error = OpenIPC_Error_Internal_Error;
            }
            else
            {
                processName = buffer;
                processID   = static_cast<uint32_t>(pid);
            }
        #endif
        return error;
    }

    OpenIPC_Error GetHostname(OUT std::string& hostname)
    {
        OpenIPC_Error error = OpenIPC_Error_No_Error;
        boost::system::error_code ec;
        hostname = boost::asio::ip::host_name(ec);
        if (ec)
        {
            error = OpenIPC_Error_Internal_Error;
        }
        return error;
    }

    OpenIPC_Error CleanUserDirectoryFromString(OUT std::string& out, IN std::string const& in)
    {
        OpenIPC_Error     error = OpenIPC_Error_No_Error;
        std::string       userDir;
        std::stringstream replacedStr;
        if (OpenIPC_PASS(error))
        {
            error = GetUserHomeDirectory(userDir);
        }
        if (OpenIPC_PASS(error))
        {
            std::stringstream userDirRegexStr;
            std::regex_replace(
                std::ostreambuf_iterator<decltype(userDirRegexStr)::char_type>(userDirRegexStr),
                userDir.begin(),
                userDir.end(),
                //std::regex("[\\/]"),
                std::regex("[\\\\/]"),
                "[\\\\/]");
            //"aa");

            std::regex_constants::syntax_option_type options = std::regex_constants::basic;
            #if defined(HOST_WINDOWS)
                options |= std::regex_constants::syntax_option_type::icase; //Case insensitive on windows.
            #endif
            std::regex userDirRegex(userDirRegexStr.str(), options);
            std::regex_replace(
                std::ostreambuf_iterator<decltype(replacedStr)::char_type>(replacedStr),
                in.begin(),
                in.end(),
                userDirRegex,
                #if defined(HOST_WINDOWS)
                    "%USERPROFILE%");
                #elif defined(HOST_LINUX)
                    "~");
                #elif defined(HOST_DARWIN)
                    "~");
            #endif
            out = replacedStr.str();
        }
        return error;
    }

    OpenIPC_Error CleanHostnameFromString(OUT std::string& out, IN std::string const& in)
    {
        OpenIPC_Error     error = OpenIPC_Error_No_Error;
        std::string       hostName;
        std::stringstream replacedStr;
        if (OpenIPC_PASS(error))
        {
            error = GetHostname(hostName);
        }
        if (OpenIPC_PASS(error))
        {
            std::regex_constants::syntax_option_type options = std::regex_constants::basic;
            options |= std::regex_constants::icase;

            std::regex userDirRegex(hostName, options);
            std::regex_replace(
                std::ostreambuf_iterator<decltype(replacedStr)::char_type>(replacedStr),
                in.begin(),
                in.end(),
                userDirRegex,
                #if defined(HOST_WINDOWS)
                    "%COMPUTERNAME%");
                #elif defined(HOST_LINUX)
                    "$hostname");
                #elif defined(HOST_DARWIN)
                    "$hostname");
            #endif
            out = replacedStr.str();
        }
        return error;
    }

    OpenIPC_Error GetCommonApplicationDataDirectory(OUT std::string& directory)
    {
        OpenIPC_Error error = OpenIPC_Error_No_Error;

        #if defined(HOST_WINDOWS)
            wchar_t* path   = nullptr;
            HRESULT  result = SHGetKnownFolderPath(FOLDERID_ProgramData, KF_FLAG_DEFAULT, NULL, &path);

            if (result != S_OK)
            {
                error = OpenIPC_Error_Internal_Error;
            }

            if (OpenIPC_PASS(error))
            {
                std::wstring_convert<std::codecvt_utf8_utf16<wchar_t>, wchar_t> converter;
                directory = converter.to_bytes(path);
                CoTaskMemFree(path); path = nullptr;
                directory += R"(\Intel\OpenIPC)";
            }
        #elif defined(HOST_LINUX) || defined(HOST_DARWIN)
            directory = R"(/etc/intel/openipc)";
        #else
            error = OpenIPC_Error_Not_Implemented;
        #endif

        return error;
    }

}
