//////////////////////////////////////////////////////////////////////////////
//
//                      INTEL CONFIDENTIAL
//       Copyright 2017 Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its
// suppliers. Title to the Material remains with Intel Corporation, its
// suppliers, or licensors. The Material contains trade secrets and
// proprietary and confidential information of Intel Corporation, its
// suppliers, and licensors, and is protected by worldwide copyright and trade
// secret laws and treaty provisions. No part of the Material may be used,
// copied, reproduced, modified, published, uploaded, posted, transmitted,
// distributed, or disclosed in any way without Intel's prior express written
// permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or
// delivery of the Materials, either expressly, by implication, inducement,
// estoppel or otherwise. Any license under such intellectual property rights
// must be express and approved by Intel in writing.
//
// Unless otherwise agreed by Intel in writing, you may not remove or alter
// this notice or any other notice embedded in Materials by Intel or Intel's
// suppliers or licensors in any way.
//
//////////////////////////////////////////////////////////////////////////////
#pragma once

#include <map>
#include <memory>
#include <mutex>
#include <string>
#include <sstream>

#include <Foundation/Error/Error.h>

namespace CommonUtils
{
    // An abstract base class for loading and caching device-specific data
    template<typename DataType>
    class DeviceDataCache
    {
    public:

        // The look up key used to determine which data to find
        class DataKey
        {
        public:
            // Constructs a data key for the specified device
            DataKey(OpenIPC_DeviceId deviceId);

            // Gets the device ID of the key
            OpenIPC_DeviceId GetDeviceId() const;

            // Adds a descriminating property to the key
            void AddDeviceProperty(const std::string& propertyName, const std::string& propertyValue);

            // Gets the value of a descriminating property of the key
            std::string GetDeviceProperty(const std::string& propertyName) const;

            std::string AsKey() const;

            bool operator<(const DataKey& other) const;
            bool operator==(const DataKey& other) const;

        private:
            OpenIPC_DeviceId _deviceId;
            std::map<std::string, std::string> _deviceProperties;
        };

        // Loads and caches the data for the given device
        OpenIPC_Error LoadDataForDevice(OpenIPC_DeviceId deviceId);

        // Gets the cached data for the given device or loads/caches the data
        // if it was not yet loaded
        OpenIPC_Error GetOrLoadDataForDevice(OpenIPC_DeviceId deviceId, std::shared_ptr<DataType>& data);

    protected:
        virtual DataKey CreateDataKey(OpenIPC_DeviceId deviceId) = 0;
        virtual OpenIPC_Error LoadData(const DataKey& dataKey, std::shared_ptr<DataType>& data) = 0;

    private:
        OpenIPC_Error _EnsureDataLoaded(const DataKey& dataKey);

        std::mutex _mutex;
        std::map<DataKey, std::shared_ptr<DataType>> _cachedData;
    };

    template <typename DataType>
    DeviceDataCache<DataType>::DataKey::DataKey(OpenIPC_DeviceId deviceId) :
        _deviceId(deviceId)
    {
    }

    template <typename DataType>
    OpenIPC_DeviceId DeviceDataCache<DataType>::DataKey::GetDeviceId() const
    {
        return _deviceId;
    }

    template <typename DataType>
    void DeviceDataCache<DataType>::DataKey::AddDeviceProperty(const std::string& propertyName, const std::string& propertyValue)
    {
        _deviceProperties[propertyName] = propertyValue;
    }

    template <typename DataType>
    std::string DeviceDataCache<DataType>::DataKey::GetDeviceProperty(const std::string& propertyName) const
    {
        auto it = _deviceProperties.find(propertyName);
        if (it != _deviceProperties.end())
        {
            return it->second;
        }
        else
        {
            return "";
        }
    }

    template <typename DataType>
    std::string DeviceDataCache<DataType>::DataKey::AsKey() const
    {
        std::stringstream ss;
        for (auto& pair : _deviceProperties)
        {
            ss << pair.second << "|";
        }
        return ss.str();
    }

    template <typename DataType>
    bool DeviceDataCache<DataType>::DataKey::operator<(const DataKey& other) const
    {
        return AsKey() < other.AsKey();
    }

    template <typename DataType>
    bool DeviceDataCache<DataType>::DataKey::operator==(const DataKey& other) const
    {
        return AsKey() == other.AsKey();
    }

    template <typename DataType>
    OpenIPC_Error DeviceDataCache<DataType>::LoadDataForDevice(OpenIPC_DeviceId deviceId)
    {
        auto dataKey = CreateDataKey(deviceId);
        std::lock_guard<std::mutex> lock(_mutex);

        return _EnsureDataLoaded(dataKey);
    }

    template <typename DataType>
    OpenIPC_Error DeviceDataCache<DataType>::GetOrLoadDataForDevice(OpenIPC_DeviceId deviceId, std::shared_ptr<DataType>& data)
    {
        auto dataKey = CreateDataKey(deviceId);
        std::lock_guard<std::mutex> lock(_mutex);

        OpenIPC_Error error = _EnsureDataLoaded(dataKey);
        if (OpenIPC_PASS(error))
        {
            data = _cachedData[dataKey];
        }

        return error;
    }

    template <typename DataType>
    OpenIPC_Error DeviceDataCache<DataType>::_EnsureDataLoaded(const DataKey& dataKey)
    {
        OpenIPC_Error error = OpenIPC_Error_No_Error;

        // Skip loading data if it is already cached
        auto it = _cachedData.find(dataKey);
        if (it != _cachedData.end())
        {
            return error;
        }

        std::shared_ptr<DataType> data;
        error = LoadData(dataKey, data);
        if (OpenIPC_PASS(error))
        {
            _cachedData[dataKey] = data;
        }

        return error;
    }
}
