/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////

#pragma once

#include <thread>
#include <mutex>

#ifdef HOST_WINDOWS

#include <windows.h>

    class HardwareBreakpointProtect
    {
        enum class HardwareBreakpointType { Execute = 0, Write = 1, Reserved = 2, ReadWrite = 3 };
        enum class Size { Byte = 0, Word = 1, Dword = 3, Qword = 2 }; // dword and qword are not backwords, dword is 0b11, and qword is 0b10

        HANDLE _targetThread{ 0 };
        intptr_t _targetAddress{ 0 };
        HardwareBreakpointType _type{ HardwareBreakpointType::Write };
        Size _size{ Size::Qword };
        int _registerUsed{ 0 };
        bool _isInstalled{ false };
        std::mutex _lock{};

    public:
        template<typename T>
        explicit HardwareBreakpointProtect(const T& target)
        {
            const size_t byteCount = sizeof(T);
            if (byteCount > 4)
            {
                _size = Size::Qword;
            }
            else if (byteCount > 2)
            {
                _size = Size::Dword;
            }
            else if (byteCount > 1)
            {
                _size = Size::Word;
            }
            else
            {
                _size = Size::Byte;
            }
            const auto pseudoThreadHandle = GetCurrentThread();
            const auto procHandle = GetCurrentProcess();

            DuplicateHandle(procHandle, pseudoThreadHandle, procHandle, &_targetThread, 0, 1, DUPLICATE_SAME_ACCESS);

            _targetAddress = reinterpret_cast<intptr_t>(&target);
            _SetHardwareBreakpoint();
        }

        HardwareBreakpointProtect(const HardwareBreakpointProtect& other)     = delete;
        HardwareBreakpointProtect(HardwareBreakpointProtect&& other) noexcept = delete;
        HardwareBreakpointProtect& operator=(const HardwareBreakpointProtect& other)     = delete;
        HardwareBreakpointProtect& operator=(HardwareBreakpointProtect&& other) noexcept = delete;

        ~HardwareBreakpointProtect()
        {
            _ClearHardwareBreakpoint();
            CloseHandle(_targetThread);
        }

    private:
        template<typename T>
        T SetBits(T source, int lowBit, int bits, int newValue)
        {
            T mask = (1 << bits) - 1;
            return (source & ~(mask << lowBit)) | (newValue << lowBit);
        }

        void _SetHardwareBreakpoint();
        void _ClearHardwareBreakpoint();
        void _InstallHardwareBreakpointThread();
        void _UninstallHardwareBreakpointThread();
    };

#else

    class HardwareBreakpointProtect
    {
    public:
        template<typename T>
        explicit HardwareBreakpointProtect(const T&)
        {
        }

        HardwareBreakpointProtect(const HardwareBreakpointProtect&)     = delete;
        HardwareBreakpointProtect(HardwareBreakpointProtect&&) noexcept = delete;
        HardwareBreakpointProtect& operator=(const HardwareBreakpointProtect&)     = delete;
        HardwareBreakpointProtect& operator=(HardwareBreakpointProtect&&) noexcept = delete;

        ~HardwareBreakpointProtect() = default;
    };
#endif
