/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#pragma once

#include <atomic>
#include <chrono>
#include <condition_variable>
#include <functional>
#include <map>
#include <memory>
#include <mutex>
#include <thread>
#include <vector>

namespace CommonUtils
{
    ///
    /// @brief A handle to a polling thread.
    typedef size_t PollingThreadHandle;

    ///
    /// @brief The action a polling performs on each interval.
    typedef std::function<void()> PollingThreadAction;

    ///
    /// @brief Describes the details of how to create a polling thread.
    struct PollingThreadDescriptor
    {
        ///
        /// @brief The action invoked on each iteration.
        PollingThreadAction action;

        ///
        /// @brief The minimum duration of time to wait between the beginning
        ///        of the last iteration and the beginning next iteration.
        std::chrono::milliseconds startInterval;

        ///
        /// @brief The minimum duration of time to wait between the end
        ///        of the last iteration and the beginning next iteration.
        std::chrono::milliseconds endInterval;
    };

    ///
    /// @brief Provides the functionality to spawn threads that poll at a
    ///        specified interval.
    class PollingThreadPool
    {
    public:
        PollingThreadPool();

        ///
        /// @brief Kills all active polling threads.
        ~PollingThreadPool();

        ///
        /// @brief Spawns a new polling thread.
        ///
        /// @param descriptor
        ///     The descriptor used to spawn the polling thread.
        ///
        /// @returns A handle to the newly spawned polling thread.
        PollingThreadHandle Spawn(const PollingThreadDescriptor& descriptor);

        ///
        /// @brief Kills a spawned polling thread.
        ///
        /// @param handle The handle of the polling thread to kill.
        void Kill(PollingThreadHandle handle);

        ///
        /// @brief Kills all spawned polling threads.
        void KillAll();

        ///
        /// @brief Suspends a polling thread from performing its action.
        ///
        /// @param handle The handle of the polling thread to suspend.
        void Suspend(PollingThreadHandle handle);

        ///
        /// @brief Resumes a suspended polling thread.
        ///
        /// @param handle The handle of the polling thread to resume.
        void Resume(PollingThreadHandle handle);

    private:

        // Uncopyable
        PollingThreadPool(const PollingThreadPool&) = delete;
        PollingThreadPool& operator=(const PollingThreadPool&) = delete;

        struct ThreadData
        {
            ThreadData(const PollingThreadDescriptor& descriptor) :
                action(descriptor.action),
                startToStartInterval(descriptor.startInterval),
                endToStartInterval(descriptor.endInterval),
                stop(false),
                suspend(false)
            {

            }

            const PollingThreadAction action;

            const std::chrono::milliseconds startToStartInterval;
            const std::chrono::milliseconds endToStartInterval;

            bool stop;
            bool suspend;

            std::thread thread;

            // protects stop and suspend members
            std::mutex mutex;
            std::condition_variable stopOrSuspendCondition;
        };

        ThreadData* _LookUpThreadData(PollingThreadHandle handle);
        void _PollAction(ThreadData& threadData);

        PollingThreadHandle _nextHandle;
        std::map<PollingThreadHandle, std::unique_ptr<ThreadData>> _handleToData;
    };

}
