/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#pragma once

#include <string>

namespace CommonUtils
{
    ///
    /// @brief Safely copy a string into a destination buffer.
    ///
    /// This helper function copies a string into a destination buffer,
    /// making sure the string is zero-terminated and doesn't overflow the
    /// destination buffer.
    ///
    /// @par Assumptions:
    ///   The @a destBufferMaxSize is really the size of the @a destBuffer
    ///   buffer.
    ///
    /// @par Side Effects:
    ///   The destination buffer is altered to contain the source string.
    ///
    /// @param destinationBuffer
    ///     The buffer in which to copy the string.
    /// @param destinationBufferSize
    ///     The maximum number of characters, including zero-terminator, that
    ///     can fit into the destination buffer.
    /// @param sourceBuffer
    ///     The source string to copy into the destination buffer.
    ///
    /// @return (int):
    ///     Returns 0 for success; otherwise, returns an error code.
    int SafeStringCopy(char* destinationBuffer, size_t destinationBufferSize, const char* sourceBuffer);

    /// @copydoc SafeStringCopy()
    int SafeStringCopy(char* destinationBuffer, size_t destinationBufferSize, const std::string& sourceBuffer);

    /// @brief A platform-independent "safe" version of sprintf(), that protects
    ///        against buffer overruns.
    ///
    /// This function is a "safe" implementation of the standard C Library
    /// sprintf() function.  This version prevents buffer overruns.  This
    /// function also guarantees a zero terminator is added even if it
    /// truncates the buffer.
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   The specified buffer is modified.
    ///
    /// @param[out] destinationBuffer
    ///   A buffer in which to put the formatted text.
    /// @param[in]  destinationBufferSize
    ///   The maximum number of characters that can fit in the buffer, including
    ///   the zero terminator.
    /// @param[in] format
    ///   The format string that specifies the formatted output.  The remaining
    ///   parameters, if any, are arguments to the format string.  This is the same
    ///   format string as passed to sprintf().
    ///
    ///
    /// @return (int): The number of characters written to the buffer, not
    ///                including the zero terminator.  If the buffer is not large
    ///                enough, returns -1.  The string is always zero-terminated.
    //////////////////////////////////////////////////////////////////////////////
    int SafeStringFormat(char* destinationBuffer, size_t destinationBufferSize, const char* format, ...);

    /// @brief A platform-independent "safe" version of sprintf(), that protects
    ///        against buffer overruns.
    ///
    /// This function is a "safe" implementation of the standard C Library
    /// sprintf() function.  This version prevents buffer overruns.  This
    /// function also guarantees a zero terminator is added even if it
    /// truncates the buffer.
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   The specified buffer is modified.
    ///
    /// @param[out] destinationBuffer
    ///   A buffer in which to put the formatted text.
    /// @param[in] format
    ///   The format string that specifies the formatted output.  The remaining
    ///   parameters, if any, are arguments to the format string.  This is the same
    ///   format string as passed to sprintf().
    /// @param[in] args
    ///   The arguments to apply to the string format
    ///
    ///
    /// @return (int): The number of characters written to the buffer, not
    ///                including the zero terminator.  If the buffer is not large
    ///                enough, returns -1.  The string is always zero-terminated.
    //////////////////////////////////////////////////////////////////////////////
    template<size_t N, class... Args>
    int SafeStringFormat(char (&destinationBuffer)[N], const char* format, Args&&... args)
    {
        return SafeStringFormat(destinationBuffer, N, format, std::forward<Args>(args)...);
    }

    /// @brief A platform-independent "safe" version of strcat(), that protects
    ///        against buffer overruns.
    ///
    /// This function is a "safe" implementation of the standard C Library
    /// strcat() function.  This version prevents buffer overruns.  This
    /// function also guarantees a zero terminator is added even if it
    /// truncates the buffer.
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   The specified buffer is modified.
    ///
    /// @param[out] destinationBuffer
    ///   A buffer in which to put the formatted text.
    /// @param[in]  destinationBufferSize
    ///   The maximum number of characters that can fit in the buffer, including
    ///   the zero terminator.
    /// @param[in] sourceBuffer
    ///   The source string to append to the destinationBuffer.
        ///
    /// @return (int): The number of characters written to the buffer, not
    ///                including the zero terminator.  If the buffer is not large
    ///                enough, returns -1.  The string is always zero-terminated.
    //////////////////////////////////////////////////////////////////////////////
    int SafeStringConcatenate(char* destinationBuffer, size_t destinationBufferSize, const char* sourceBuffer);

    /// @brief A platform-independent "safe" version of strcat(), that protects
    ///        against buffer overruns.
    ///
    /// This function is a "safe" implementation of the standard C Library
    /// strcat() function.  This version prevents buffer overruns.  This
    /// function also guarantees a zero terminator is added even if it
    /// truncates the buffer.
    ///
    /// @par Assumptions:
    ///   None.
    ///
    /// @par Side Effects:
    ///   The specified buffer is modified.
    ///
    /// @param[out] destinationBuffer
    ///   A buffer in which to put the formatted text.
    /// @param[in] sourceBuffer
    ///   The source string to append to the destinationBuffer.
    ///
    /// @return (int): The number of characters written to the buffer, not
    ///                including the zero terminator.  If the buffer is not large
    ///                enough, returns -1.  The string is always zero-terminated.
    //////////////////////////////////////////////////////////////////////////////
    template<size_t N>
    int SafeStringConcatenate(char(&destinationBuffer)[N], const char* sourceBuffer)
    {
        return SafeStringConcatenate(destinationBuffer, N, sourceBuffer);
    }

}
