﻿/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
/// @file
///
/// @brief Contains the error handling library code.
///
//////////////////////////////////////////////////////////////////////////////
#pragma once

#include <map>
#include <mutex>
#include <stack>
#include <string>
#include <vector>

#include <Foundation/Error/Error.h>

///
/// @brief An instance of an error.
///
/// @remarks Contains information such as the error code, message, and source
/// file context.
class ErrorImpl
{
public:
    ErrorImpl();

    ///
    /// @brief Constructs an error.
    ///
    /// @param errorCode
    ///     The IDL error code of the error.
    /// @param message
    ///     The error message (if empty then the default error message is used).
    /// @param sourceFile
    ///     The path to the source file which posted the error.
    /// @param sourceLineNumber
    ///     The number of the line in the source file where the error was
    ///     posted.
    ErrorImpl(OpenIPC_Error errorCode, const char* message, const char* sourceFile, unsigned sourceLineNumber);

    ///
    /// @brief Returns the IDL error code of the error.
    OpenIPC_Error GetErrorCode() const;

    ///
    /// @brief Returns the error message.
    ///
    /// @remarks If no message was set when the error was posted then the
    /// default error message for the error code is used.  If the error has an
    /// inner error bound to it then the message of the inner error is appended
    /// to the error message.
    std::string GetErrorMessage() const;

    ///
    /// @brief Returns whether the error has source file context information.
    bool HasSourceContext() const;

    ///
    /// @brief Returns the path to the source file which posted the error.
    const std::string& GetSourceFile() const;

    ///
    /// @brief Returns the line in the source file which posted the error.
    unsigned GetSourceLineNumber() const;

private:
    OpenIPC_Error _errorCode;
    std::string _message;
    std::string _sourceFile;
    unsigned _sourceLineNumber;
};

///
/// @brief Maps error codes to their default messages.
class ErrorMessages
{
public:

    ///
    /// @brief Sets the default error message for an error code.
    ///
    /// @param error
    ///     The error code.
    /// @param message
    ///     The error message.
    static void SetDefaultMessage(OpenIPC_Error error, const char* message);

    ///
    /// @brief Gets the default error message for an error code.
    ///
    /// @param error
    ///     The error code.
    ///
    /// @returns The default error message.
    static const char* GetDefaultMessage(OpenIPC_Error error);
};

///
/// @brief A self-registering listener of all posted errors on the current thread.
class ErrorContextImpl
{
public:

    ///
    /// @brief Constructs the error context and begins listening to all posted
    /// errors on the current thread.
    ErrorContextImpl();

    ///
    /// @brief Stops listening to all posted errors on the current thread.
    ~ErrorContextImpl();

    ///
    /// @brief Posts an error to all active contexts on the current thread.
    ///
    /// @param errorCode
    ///     The IDL error code of the error.
    /// @param message
    ///     The error message (if empty then the default error message is used).
    /// @param sourceFile
    ///     The path to the source file which posted the error.
    /// @param sourceLineNumber
    ///     The number of the line in the source file where the error was
    ///     posted.
    static void PostError(OpenIPC_Error errorCode, const char* message = "", const char* sourceFile = "", unsigned sourceLineNumber = 0);

    ///
    /// @brief Gets the full message containing all errors that occurred during
    /// lifetime of the context.
    std::string GetErrorMessage() const;

    void AddError(const ErrorImpl& error);
    const std::vector<ErrorImpl>& GetErrors() const;

    ErrorImpl GetLastError() const;

private:
    std::vector<ErrorImpl> _errors;

};
