﻿/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////

#include <stdafx.h>

#include "JtagShiftBuilder.h"


void JtagTdoHandler::ExtractFromTdoBuffer(BitData* reg, const BitData* tdo, uint32_t offset)
{
	// copy out the register value
	if (reg->bitsize != _dataBits)
	{
		BitData_Resize(reg, _dataBits);
	}
	BitData_Copy(tdo, _preBits + offset, reg, 0, _dataBits);
}

bool JtagTdoHandler::IsVerificationKeyValid(const BitData* tdo, uint32_t offset) const
{
    return (_verificationKey != 0) &&
        (BitData_GetUInt32(tdo, _postBits + _dataBits + _preBits + offset) == _verificationKey);
}

void JtagBypassBuilder::AddDevice(uint32_t irlen, uint32_t offset)
{
	_totalIrLength += irlen;
	JtagDevice jtagDevice;
	jtagDevice.irlength = irlen;
	_deviceList.insert(_deviceList.begin() + offset, jtagDevice);
	_resyncOffsets(offset);
}

void JtagBypassBuilder::RemoveDevice(uint32_t offset)
{
	_totalIrLength -= _deviceList[offset].irlength;
	_deviceList.erase(_deviceList.begin() + offset);
	if (offset > 0)
	{
		_resyncOffsets(offset);
	}
}

void JtagBypassBuilder::_resyncOffsets(uint32_t uptodevice)
{
	uint32_t iroffset = ((uptodevice + 1) == _deviceList.size()) ? 0 : _deviceList[uptodevice + 1].iroffset;
	for (uint32_t idx = uptodevice; idx < _deviceList.size(); idx--)
	{
		_deviceList[idx].iroffset = iroffset;
		iroffset += _deviceList[idx].irlength;
	}
}

void JtagBypassBuilder::ClearDevices()
{
	_totalIrLength = 0;
	_deviceList.clear();
}

uint32_t JtagBypassBuilder::GetDeviceCount()
{
	return (uint32_t)_deviceList.size();
}

uint32_t JtagBypassBuilder::GetTotalIrLength()
{
	return _totalIrLength;
}

void JtagBypassBuilder::GetPrePostBitCounts(uint32_t& msbs, uint32_t& lsbs, bool irNotDrScan, uint32_t devidx) const
{
	lsbs = irNotDrScan ? _irLowDeracers : _drLowDeracers;
	msbs = irNotDrScan ? _irHighDeracers : _drHighDeracers;
	if (devidx != uint32_t(-1))
	{
		// This is an IR scan, so count the IR lengths (for the devices before and after) and add in the pre/post padding
		if (irNotDrScan)
		{
			const JtagDevice& jtagDevice = _deviceList[devidx];
			lsbs += jtagDevice.iroffset;
			msbs += _totalIrLength - (jtagDevice.iroffset + jtagDevice.irlength);
		}
		// this is a DR scan, so count the bypass bits (# devices before and after) and add in the pre/post padding
		else
		{
			lsbs += (uint32_t)_deviceList.size() - (devidx + 1);
			msbs += devidx;
		}
	}
}

void JtagBypassBuilder::GenerateTdiBuffer(bool irNotDrScan, BitData* tdi, const BitData* reg, uint32_t reglen, uint32_t devidx, uint32_t verificationKey)
{
	uint32_t preBits;
	uint32_t postBits;
	GetPrePostBitCounts(postBits, preBits, irNotDrScan, devidx);

	const uint32_t localshiftlen = (reglen != 0) ? reglen : (reg != nullptr) ? (uint32_t)reg->bitsize : 0; // TBD: coming out of here with a length of 0 is illegal

	const uint32_t verificationkeysize = (verificationKey != 0) ? 32 : 0;

	// resize our TDI buffer to the final shift length
	const uint32_t tdisize = verificationkeysize + localshiftlen + preBits + postBits;
	BitData_Resize(tdi, tdisize);

	// set the Bypass padding value as the default value of the buffer
	if ((preBits + postBits) != 0)
	{
		memset(tdi->buffer, (irNotDrScan || _DrPadValue) ? -1 : 0, BitsToUint8s(tdisize));
	}

	// add in the shift verification key
	if (verificationkeysize != 0)
	{
		BitData_SetUInt32(tdi, 0, verificationKey);
	}

	// add in the register value
	if (reg != nullptr)
	{
		if (localshiftlen > reg->bitsize)
		{
			BitData_Copy(reg, 0, tdi, verificationkeysize + preBits, reg->bitsize);
			BitData_InitValue(tdi, verificationkeysize + preBits + reg->bitsize, localshiftlen - reg->bitsize, 0);
		}
		else
		{
			BitData_Copy(reg, 0, tdi, verificationkeysize + preBits, localshiftlen);
		}
	}
	else
	{
		BitData_InitValue(tdi, verificationkeysize + preBits, localshiftlen, 0);
	}
}


void JtagBypassBuilder::GetTdoHandlerForField(JtagTdoHandler& tdoHandler, bool irNotDrScan, uint32_t reglen, uint32_t regFieldStart, uint32_t regFieldLength, uint32_t devidx, uint32_t verificationKey, bool ignorePrePost)
{
	assert(reglen >= regFieldLength + regFieldStart);
    uint32_t preBits = 0;
    uint32_t postBits = 0;
    if (!ignorePrePost)
    {
        GetPrePostBitCounts(postBits, preBits, irNotDrScan, devidx);
    }
	tdoHandler._preBits = preBits + regFieldStart;
	tdoHandler._dataBits = regFieldLength;
	tdoHandler._postBits = postBits + (reglen - (regFieldStart + regFieldLength));
	tdoHandler._verificationKey = verificationKey;
}

void JtagBypassBuilder::GetTdoHandler(JtagTdoHandler& tdoHandler, bool irNotDrScan, uint32_t reglen, uint32_t devidx, uint32_t verificationKey)
{
	uint32_t preBits;
	uint32_t postBits;
	GetPrePostBitCounts(postBits, preBits, irNotDrScan, devidx);
	tdoHandler._preBits = preBits;
	tdoHandler._dataBits = reglen;
	tdoHandler._postBits = postBits;
	tdoHandler._verificationKey = verificationKey;
}

void JtagBypassBuilder::AddToDrLowDeracers(uint32_t value)
{
    _drLowDeracers += value;
}

void JtagBypassBuilder::AddToDrHighDeracers(uint32_t value)
{
    _drHighDeracers += value;
}


uint32_t JtagBypassBuilder::GetDrLowDeracers() const
{
	return _drLowDeracers;
}

uint32_t JtagBypassBuilder::GetDrHighDeracers() const
{
	return _drHighDeracers;
}

void JtagShift(BitData* shift, uint32_t count, uint32_t offset, const uint8_t* tdi, uint8_t* tdo)
{
	BitData tdiBd = BitData_CreateLocalFromBuffer(count, count, (void*)tdi);
	BitData tdoBd = BitData_CreateLocalFromBuffer(count, count, (void*)tdo);

	if (count > shift->bitsize)
	{
		if (tdo != nullptr)
		{
			BitData_Copy(shift, 0, &tdoBd, offset, shift->bitsize);
			if (tdi != nullptr)
			{
				BitData_Copy(&tdiBd, offset, &tdoBd, offset + shift->bitsize, count - shift->bitsize);
			}
			else
			{
				BitData_InitValue(&tdoBd, offset + shift->bitsize, count - shift->bitsize, 0);
			}
		}
		if (tdi != nullptr)
		{
			BitData_Copy(&tdiBd, offset + (count - shift->bitsize), shift, 0, shift->bitsize);
		}
		else
		{
			BitData_InitValue(shift, 0, shift->bitsize, 0);
		}
	}
	else if (count < shift->bitsize)
	{
		if (tdo != nullptr)
		{
			BitData_Copy(shift, 0, &tdoBd, offset, count);
		}
		BitData_Copy(shift, count, shift, 0, shift->bitsize - count);
		if (tdi != nullptr)
		{
			BitData_Copy(&tdiBd, offset, shift, shift->bitsize - count, count);
		}
		else
		{
			BitData_InitValue(shift, shift->bitsize - count, count, 0);
		}
	}
	else
	{
		if (tdo != nullptr)
		{
			BitData_Copy(shift, 0, &tdoBd, offset, shift->bitsize);
		}
		if (tdi != nullptr)
		{
			BitData_Copy(&tdiBd, offset, shift, 0, shift->bitsize);
		}
		else
		{
			BitData_InitValue(shift, 0, shift->bitsize, 0);
		}
	}
}

