﻿/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#pragma once

#include <map>
#include <vector>
#include <string>

#include <Foundation/Types.h>
#include <Foundation/BitData/BitData.h>
#include <Foundation/BitData/OpenIPC_BitData.h>

#include "TapStateMachine.h"
#include "JtagShiftBuilder.h"

class JtagStubTapRegister
{
protected:
	BitData* _captureValue;
	BitData* _updateValue;
	const std::string _name;

public:
	JtagStubTapRegister(const char* regname, BitData* initvalue = nullptr, bool rdonly = false, bool rdwrsame = true);
	virtual ~JtagStubTapRegister() {}

	void SetAttributes(uint32_t length, BitData* initvalue = nullptr, bool rdonly = false, bool rdwrsame = true);
	virtual uint32_t GetLength(){return (uint32_t)_captureValue->bitsize;}
	virtual void GetCapture(BitData* dest, uint32_t& offset);
	virtual void SetUpdate(const BitData* src, uint32_t& offset);
	void GetUpdate(BitData* dest, uint32_t& offset);

protected:
	JtagStubTapRegister& operator=( const JtagStubTapRegister&) = delete;
};

class JtagStubTapController
{
private:
	const uint32_t _irTapReset;
	JtagStubTapRegister* _instructionReg;
	std::map<uint32_t, JtagStubTapRegister*> _tapRegisters;
	std::map<uint32_t, bool> _tapRegistersReadonly;
	std::map<uint32_t, std::string> _tapRegisterNames;
	const std::string _name;

public:
	JtagStubTapController(const char* name, uint32_t irlen, uint32_t ircaptureval, uint32_t tapresetir = -1);
    ~JtagStubTapController();

    JtagStubTapController(const JtagStubTapController& other) = delete;
    JtagStubTapController(JtagStubTapController&& other) noexcept = delete;
    JtagStubTapController& operator=(const JtagStubTapController& other) = delete;
    JtagStubTapController& operator=(JtagStubTapController&& other) noexcept = delete;

	JtagStubTapRegister* GetIr() {return _instructionReg;}
	JtagStubTapRegister* GetDr();
	void AddInstruction(uint32_t irValue, const char* instructionName = nullptr, JtagStubTapRegister* reg=nullptr, bool readonly=false);
	void TapReset();
	uint32_t GetShiftLength(bool bIrScan);
	const char* GetName(){return _name.c_str();}

};

class JtagStubTapChain
{
protected:
	TapStateMachine _tapStateMachine;
	std::vector<JtagStubTapController*> _tapControllers;
	OpenIPC_BitData _virtualIr;
    OpenIPC_BitData _virtualDr;
	uint32_t _shiftIrClks;
	uint32_t _shiftDrClks;

public:
	JtagStubTapChain();
	virtual ~JtagStubTapChain() {}

	void AddTapController(JtagStubTapController* tapcontroller, uint32_t position = -1);
	void RemoveTapControllers(uint32_t position = -1, uint32_t count = -1);
	void TapReset();

	virtual	uint32_t GetTotalShiftLength(bool bIrScan);
	virtual	void GetTotalCapture(bool bIrScan, BitData* dest, uint32_t& offset);
	virtual	void SetTotalUpdate(bool bIrScan, const BitData* src, uint32_t& offset);

	void GotoState(TapStateMachine::JtagStateEncode state, uint32_t count);
	void Shift(TapStateMachine::JtagStateEncode state, uint32_t count, const uint8_t* writebuffer, uint8_t* readbuffer);
	void DrivePins(uint32_t count, const uint8_t* tms, const uint8_t* tdi, uint8_t* tdo);
	TapStateMachine::JtagStateEncode GetCurrentState(){return _tapStateMachine.GetState();}
};

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// Intel(c) TAPLink extension classes
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

class JtagStubTapChainTapLink : public JtagStubTapChain
{
protected:
	uint32_t _irLowDeracers;
	uint32_t _irHighDeracers;
	uint32_t _drLowDeracers;
	uint32_t _drHighDeracers;

public:
	JtagStubTapChainTapLink(JtagStubTapController* parentTapController, uint32_t irRegIr, uint32_t drRegIr, uint32_t irLowDeracers, uint32_t irHighDeracers, uint32_t drLowDeracers, uint32_t drHighDeracers);

	virtual	uint32_t GetTotalShiftLength(bool bIrScan);
	virtual	void GetTotalCapture(bool bIrScan, BitData* dest, uint32_t& offset);
	virtual	void SetTotalUpdate(bool bIrScan, const BitData* src, uint32_t& offset);

protected:
	JtagStubTapChainTapLink& operator=( const JtagStubTapChainTapLink&) = delete;
};

class JtagStubTapRegisterTapLink : public JtagStubTapRegister
{
protected:
	JtagStubTapChainTapLink* const _tapLinkChain;
	bool _isIrRegister;

public:
	JtagStubTapRegisterTapLink(const char* regname, bool isIrRegister, JtagStubTapChainTapLink* tapLinkChain) : JtagStubTapRegister(regname), _tapLinkChain(tapLinkChain), _isIrRegister(isIrRegister) {}
	virtual uint32_t GetLength();
	virtual void GetCapture(BitData* dest, uint32_t& offset);
	virtual void SetUpdate(const BitData* src, uint32_t& offset);

protected:
	JtagStubTapRegisterTapLink& operator=( const JtagStubTapRegisterTapLink&) = delete;
};



