/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#include "stdafx.h"
#include "Public/DataStore.h"

#include <PushNewOverride.h>
#include <boost/algorithm/string.hpp>
#include <PopNewOverride.h>

using namespace boost::algorithm;

DataQueryResult::DataQueryResult(DataElement::Array&& dataElements) :
    _dataElements(dataElements)
{
}

DataQueryResult::DataQueryResult(const std::string& errorMessage) :
    _errorMessage(errorMessage)
{
}

size_t DataQueryResult::DataElementCount() const
{
    return _dataElements.size();
}

DataElement* DataQueryResult::FirstDataElement() const
{
    DataElement* dataElement = nullptr;

    auto it = _dataElements.begin();
    if (it != _dataElements.end())
    {
        dataElement = *it;
    }

    return dataElement;
}

const DataElement::Array& DataQueryResult::DataElements() const
{
    return _dataElements;
}

bool DataQueryResult::ErrorOccurred() const
{
    return !_errorMessage.empty();
}

std::string DataQueryResult::ErrorMessage() const
{
    return _errorMessage;
}

DataElement::Set WhereAttributeHasNode::Execute(DataIndexNode& index) const
{
    DataElement::Set results;

    // Get the index node for elements that have an attribute of the given name
    DataIndexNode* attributeIndex = index.Find(_name);
    if (attributeIndex)
    {
        // Get the index node for elements with an attribute containing the
        // given value
        DataIndexNode* valueIndex = attributeIndex->Find(_value);
        if (valueIndex)
        {
            results = valueIndex->Elements();
        }

        // Include wildcard matches
        DataIndexNode* wildCardIndex = attributeIndex->Find("*");
        if (wildCardIndex)
        {
            const DataElement::Set& wildCardResults = wildCardIndex->Elements();
            results.insert(wildCardResults.begin(), wildCardResults.end());
        }
    }

    return results;
}

bool WhereAttributeHasNode::MatchesElement(const DataElement& element) const
{
    bool match = false;

    const DataAttribute* attribute = DataQuery::FindAttribute(element, _name);
    if (attribute)
    {
        match = DataQuery::AttributeHasValue(*attribute, _value);
    }
    else if (!attribute && _value.empty())
    {
        match = true;
    }

    return match;
}

WhereAttributeHasNode::WhereAttributeHasNode(const std::string& name, const std::string& value) :
    _name(name),
    _value(value)
{
}

SelectElementsNode& SelectElementsNode::WhereAttributeHas(const std::string& name, const std::string& value)
{
    std::shared_ptr<WhereAttributeHasNode> node(new WhereAttributeHasNode(name, value));
    _whereAttributeHasNodes.push_back(node);
    return *this;
}

DataElement::Set SelectElementsNode::Execute(DataIndexNode& index) const
{
    DataElement::Set results;

    // Get the index node for documents of the given tye
    DataIndexNode* typeIndex = index.Find(_documentType);
    if (typeIndex)
    {
        // Get the index node for elements of the given name
        DataIndexNode* elementIndex = typeIndex->Find(_elementName);
        if (elementIndex)
        {
            // If there are no discriminators then include all of the elements
            // on the element index node
            if (_whereAttributeHasNodes.empty())
            {
                results = elementIndex->Elements();
            }
            else
            {
                // For each child discriminator
                bool firstNode = true;
                for (auto& node : _whereAttributeHasNodes)
                {
                    if (firstNode)
                    {
                        // If this is the first discriminator then use its
                        // results; following discriminators will futher
                        // filter down these results
                        results = node->Execute(*elementIndex);
                        firstNode = false;
                    }
                    else
                    {
                        // For each of the existing results, build a new set
                        // of filtered results for the current discriminator
                        DataElement::Set filteredResults;
                        for (auto& result : results)
                        {
                            if (node->MatchesElement(*result))
                            {
                                filteredResults.insert(result);
                            }
                        }

                        // The filtered results are now the current results
                        results = std::move(filteredResults);
                    }
                }
            }
        }
    }

    return results;
}

SelectElementsNode::SelectElementsNode(const std::string& documentType, const std::string& elementName) :
    _documentType(documentType),
    _elementName(elementName)
{
}

SelectElementsNode& DataQuery::SelectElements(const std::string& documentType, const std::string& elementName)
{
    std::shared_ptr<SelectElementsNode> node(new SelectElementsNode(documentType, elementName));
    _selectElementsNodes.push_back(node);
    return *node;
}

DataElement::Set DataQuery::Execute(DataIndexNode& index) const
{
    DataElement::Set results;

    // For each of the child nodes
    for (auto& node : _selectElementsNodes)
    {
        // Execute the query node and keep the results
        const DataElement::Set& nodeResults = node->Execute(index);

        // Insert all of the results (duplicates will be ignored because we
        // are using a set)
        results.insert(nodeResults.begin(), nodeResults.end());
    }

    return results;
}

std::vector<std::string> DataQuery::ExtractAttributeValues(const DataAttribute& attribute)
{
    std::vector<std::string> values;
    split(values, attribute.Value(), is_any_of(","), token_compress_on);
    for (std::string& value : values)
    {
        trim(value);
    }
    return values;
}

bool DataQuery::AttributeHasValue(const DataAttribute& attribute, const std::string& attributeValue)
{
    bool hasValue = false;

    auto values = ExtractAttributeValues(attribute);
    for (const std::string& value : values)
    {
        if (iequals(value, attributeValue) || value == "*")
        {
            hasValue = true;
            break;
        }
    }

    if (values.empty() && attributeValue.empty())
    {
        hasValue = true;
    }

    return hasValue;
}

const DataAttribute* DataQuery::FindAttribute(const DataElement& element, const std::string& attributeName)
{
    const DataAttribute* attribute = element.FirstAttribute();
    while (attribute)
    {
        if (iequals(attribute->Name(), attributeName))
        {
            break;
        }
        attribute = attribute->NextSibling();
    }
    return attribute;
}
