/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#include "stdafx.h"
#include "Public/DataStore.h"

#include <Foundation/Logging/OpenIPC_Logger.h>
#include <ManipulatePath.h>

#if CONFIG_INCLUDE_STRUCTUREDDATA_ENCRYPTION_SUPPORT
#include <Obfuscate.h>
#include <Aes.h>
#endif

#include <PushNewOverride.h>
#include <boost/algorithm/string.hpp>
#include <boost/filesystem.hpp>
#include <PopNewOverride.h>

static std::unique_ptr<OpenIPC_Logger> _logger(OpenIPC_Logger::GetLogger(OpenIPC_LoggingLoggerId_DataStore));

DataStore* DataStore::Get()
{
    static DataStore dataStore;
    return &dataStore;
}

DataStore::DataStore() :
    _rootIndexNode(*this),
    _schemaValidationEnabled(true)
{
    boost::filesystem::path::imbue(std::locale());
}

void DataStore::IncludeDirectory(const std::string& directory, bool useIndexFiles, bool recursive)
{
    std::vector<DataStoreException> exceptions;
    IncludeDirectory(directory, useIndexFiles, recursive, exceptions);
}

void DataStore::IncludeDirectory(const std::string& directory, bool useIndexFiles, bool recursive, std::vector<DataStoreException>& exceptions)
{
    std::lock_guard<std::recursive_mutex> lock(_mutex);

    // "Normalize" the full path to the directory
    boost::filesystem::path fullDirectoryPath(directory);

    if (!OpenIPC_PASS(CommonUtils::SanitizePath(fullDirectoryPath)))
    {
        exceptions.push_back(DataStoreException("Failed to sanitize the given directory path: '" + fullDirectoryPath.string() + "'"));
        return;
    }

    fullDirectoryPath = boost::filesystem::canonical(fullDirectoryPath);
    _directories.insert(fullDirectoryPath.string());

    OpenIPC_LOG(_logger, OpenIPC_LoggingSeverityLevel_Info, "Including directory '" << fullDirectoryPath.string() << "'");

    // Make sure the directory exists
    if (!boost::filesystem::exists(fullDirectoryPath))
    {
        OpenIPC_LOG(_logger, OpenIPC_LoggingSeverityLevel_Warning, "Directory '" << fullDirectoryPath.string() << "' does not exist and could not be included");
        return;
    }

    // A list of the paths of all non-index XML files in the directory
	std::vector<std::tuple<std::string, bool>> xmlFilePaths;

    // For each file in the path recursively
    auto it = boost::filesystem::recursive_directory_iterator(fullDirectoryPath);
    boost::filesystem::recursive_directory_iterator end;
    for (; it != end; ++it)
    {
        const boost::filesystem::path path = it->path().lexically_normal();

        if (!recursive)
        {
            //We need to do this with every iteration
            it.no_push(true);
        }

        std::string fileName = path.filename().string();
		if (boost::iequals(fileName, "index.bin") || boost::iequals(fileName, "index.xml"))
		{
			try
            {
                if (useIndexFiles)
                {
					bool encrypted = boost::iequals(fileName, "index.bin");
					_AddIndex(path.string(), encrypted);
                }
            }
            catch (BinLoadException& exception)
            {
                OpenIPC_LOG(_logger, OpenIPC_LoggingSeverityLevel_Error, exception.Message());
                exceptions.push_back(exception);
            }
            catch (std::exception& exception)
            {
                OpenIPC_LOG(_logger, OpenIPC_LoggingSeverityLevel_Error, exception.what());
                exceptions.push_back(DataStoreException(exception.what()));
            }

			continue;
		}

		bool encrypted = false;

        // Skip non-XML files
        std::string extension = path.extension().string();
        if (!boost::iequals(extension, ".xml"))
        {
#if CONFIG_INCLUDE_STRUCTUREDDATA_ENCRYPTION_SUPPORT
			std::string deobfuscatedName = Obfuscate::ObfuscateOrDeobfuscateString(path.filename().string());

			if (!boost::iequals(boost::filesystem::path(deobfuscatedName).extension().string(), ".xml"))
			{
				continue;
			}
			else
			{
				encrypted = true;
			}
#else
            continue;
#endif
        }

        // Remember the file to load it later if it was never indexed
		xmlFilePaths.push_back(std::tuple<std::string, bool>(path.string(), encrypted));
    }

    // For each non-index XML file in the directory
    for (const std::tuple<std::string, bool>& tuple : xmlFilePaths)
    {
        // Add the document; the document will be skipped if it was already
        // indexed in an index file
        AddDocument(std::get<0>(tuple), std::get<1>(tuple), true, exceptions);
    }
}

DataQueryResult DataStore::Find(const DataQuery& query)
{
    std::lock_guard<std::recursive_mutex> lock(_mutex);

    try
    {
        // Execute the query starting at the root index node
        DataElement::Set results = query.Execute(_rootIndexNode);

        for (DataElement* element : results)
        {
            //If element->Document() is new in index
            if (element->Document()->NewToIndex())
            {
                //Mark as old so we don't send multiple notifications for the same document.
                element->Document()->MarkAsOld();

                //Notify user that we we are using definitions from a file
                //that is in the install that is newer than the index
                const std::string message = "Detected relevant file modified (or added) after OpenIPC was installed: " + element->Document()->FilePath();
                OpenIPC_LOG(_logger, OpenIPC_LoggingSeverityLevel_Warning, message);
                if (_notificationCallback)
                {
                    _notificationCallback(message);
                }
            }
        }

        // Build a sorted array of the results
        DataElement::Array resultsArray(results.begin(), results.end());
        std::sort(resultsArray.begin(), resultsArray.end(), [](const DataElement* left, const DataElement* right)
            {
                // If the elements are in the same document then sort by element position
                if (left->Document() == right->Document())
                {
                    return left->Position() < right->Position();
                }

                // Otherwise, sort by the documents' file paths
                else
                {
                    return left->Document()->FilePath() < right->Document()->FilePath();
                }
            }
        );

        // Return the results
        return DataQueryResult(std::move(resultsArray));
    }
    catch (DataStoreException& exception)
    {
        // If an exception occurred, return the result as an error message
        return DataQueryResult(exception.Message());
    }
}

bool DataStore::IsSchemaValidationEnabled() const
{
    return _schemaValidationEnabled;
}

void DataStore::SetSchemaValidationEnabled(bool enabled)
{
    _schemaValidationEnabled = enabled;
}

void DataStore::SetNotificationCallback(NotificationCallback&& callback)
{
    _notificationCallback = std::move(callback);
}

const std::set<std::string> DataStore::_willNotNotifyIfChanged
{
    "OpenIpcConfig.xml",
    "Logging.xml",
};

std::shared_ptr<DataDocument> DataStore::_LoadXmlDocument(const std::string& filePath, bool decrypt, bool useSchema, const std::string& schemaFilePath, bool alreadyIndexed)
{
    // Load the file
    auto document = std::make_shared<DataDocument>();

	if (decrypt)
	{
#if CONFIG_INCLUDE_STRUCTUREDDATA_ENCRYPTION_SUPPORT
		Aes aes(FILE_KEYSEED, FILE_KEYSEED_LENGTH);

		boost::filesystem::ifstream infile(filePath, std::ios::binary | std::ios::ate);

		// get length of file
		std::streamoff length = infile.tellg();

		// go back to the start
		infile.seekg(0, infile.beg);

		// read the file
		std::vector<uint8_t> buffer((unsigned int)length);

		infile.read(reinterpret_cast<char*>(&buffer[0]), length);

		infile.close();

		std::vector<uint8_t> decryptedFileData = aes.Decrypt(buffer);

		std::stringstream decryptedStream(std::stringstream::in | std::stringstream::out | std::stringstream::binary);
		decryptedStream.write(reinterpret_cast<char*>(&decryptedFileData[0]), decryptedFileData.size());

		document->LoadBinStream(decryptedStream, !alreadyIndexed);
#endif
	}
    else if (!schemaFilePath.empty() && useSchema)
    {
        document->LoadXmlFile(filePath, schemaFilePath, !alreadyIndexed);
    }
    else
	{
		document->LoadXmlFile(filePath, useSchema, !alreadyIndexed);
	}

	return document;
}

void DataStore::_AddIndex(const std::string& indexFile, bool decrypt)
{
    boost::filesystem::path filePath(indexFile);
    OpenIPC_LOG(_logger, OpenIPC_LoggingSeverityLevel_Info, "Loading index file '" << filePath << "'");

    // Get the time that the index file was last modified
    std::time_t indexFileLastModified = boost::filesystem::last_write_time(filePath);

    // Get the path to the parent directory of the index file
    auto fileDirectoryPath = filePath.parent_path().lexically_normal();

    // Load the index file
    std::shared_ptr<DataDocument> indexDocument = _LoadXmlDocument(filePath.string(), decrypt, false, "", true);

    // Get the <Index> element
    DataElement* indexElement = indexDocument->RootElement();
    if (indexElement)
    {
        // For each <File> element
        DataElement* fileElement = indexElement->FirstChild();
        while (fileElement)
        {
            // Get the "Path" and "RootElement" attributes
            DataAttribute* pathAttribute = fileElement->FindAttribute("Path");
            DataAttribute* rootElementAttribute = fileElement->FindAttribute("RootElement");
            if (pathAttribute && rootElementAttribute)
            {
                // Get the "normalized" the full path to the file
                const boost::filesystem::path fullFilePath = fileDirectoryPath / pathAttribute->Value();
                if (boost::filesystem::exists(fullFilePath))
                {
                    const std::time_t fileLastModified = boost::filesystem::last_write_time(fullFilePath);

                    if (fileLastModified > indexFileLastModified)
                    {
                        OpenIPC_LOG(_logger, OpenIPC_LoggingSeverityLevel_Info, "File '" << fullFilePath << "' has been modified since it was indexed; discarding index for the file");
                        fileElement = fileElement->NextSibling();
                        continue;
                    }
                }

                OpenIPC_LOG(_logger, OpenIPC_LoggingSeverityLevel_Debug, "Indexing file '" << fullFilePath << "'");

                // Remember that this file is indexed
                const std::string fullFilePathString = fullFilePath.string();
                _indexedDocuments.insert(fullFilePathString);

                // Find or create the index node for the root element
                DataIndexNode& typeIndex = _rootIndexNode.FindOrCreate(rootElementAttribute->Value());

                // For each <Element> element
                DataElement* elementElement = fileElement->FirstChild();
                while (elementElement)
                {
                    // Get the "Name" attribute
                    DataAttribute* nameAttribute = elementElement->FindAttribute("Name");
                    if (nameAttribute)
                    {
                        // Find or create the index node for elements with of
                        // this name
                        DataIndexNode& elementIndex = typeIndex.FindOrCreate(nameAttribute->Value());
                        elementIndex.AddDocumentPath(fullFilePathString);

                        // For each <Attribute> element
                        DataElement* attributeElement = elementElement->FirstChild();
                        while (attributeElement)
                        {
                            // Get the "Name" and "Value" attributes
                            DataAttribute* attributeNameAttribute = attributeElement->FindAttribute("Name");
                            DataAttribute* attributeValueAttribute = attributeElement->FindAttribute("Value");
                            if (attributeNameAttribute && attributeValueAttribute)
                            {
                                // Find or create the index node for elements with
                                // attributes of this name
                                DataIndexNode& attributeIndex = elementIndex.FindOrCreate(attributeNameAttribute->Value());

                                // Add this file to the index node
                                attributeIndex.AddDocumentPath(fullFilePathString);

                                // For each value in the attribute's value
                                for (std::string& value : DataQuery::ExtractAttributeValues(*attributeValueAttribute))
                                {
                                    // Find or create the index node for elements
                                    // with attributes of this value
                                    DataIndexNode& valueIndex = attributeIndex.FindOrCreate(value);

                                    // Add this file to the index node
                                    valueIndex.AddDocumentPath(fullFilePathString);
                                }
                            }
                            attributeElement = attributeElement->NextSibling();
                        }
                    }
                    elementElement = elementElement->NextSibling();
                }
            }
            fileElement = fileElement->NextSibling();
        }
    }
}

void DataStore::AddDocument(const std::string& filePath, bool decrypt, bool skipIfIndexed, std::vector<DataStoreException>& exceptions)
{
    AddDocument(filePath, decrypt, skipIfIndexed, "", exceptions);
}

void DataStore::AddDocument(const std::string& filePath, bool decrypt, bool skipIfIndexed, const std::string& schemaFilePath, std::vector<DataStoreException>& exceptions)
{
    // Do not add the document if it is already indexed
    bool alreadyIndexed = _indexedDocuments.find(filePath) != _indexedDocuments.end();
    if (skipIfIndexed && alreadyIndexed)
    {
        return;
    }

    // Do not add the document if it is already loaded
    if (_loadedDocuments.find(filePath) != _loadedDocuments.end())
    {
        return;
    }
    _loadedDocuments.insert(filePath);

    boost::filesystem::path path(filePath);
    if (!OpenIPC_PASS(CommonUtils::SanitizePath(path)))
    {
        exceptions.push_back(DataStoreException("Failed to sanitize the given document path: '" + path.string() + "'"));
        return;
    }

    // Check if the file exists
    if (!boost::filesystem::exists(path))
    {
        const std::string message = "Detected relevant file deleted after OpenIPC was installed: " + path.string();
        OpenIPC_LOG(_logger, OpenIPC_LoggingSeverityLevel_Warning, message);
        if (_notificationCallback)
        {
            _notificationCallback(message);
        }
        return;
    }

    OpenIPC_LOG(_logger, OpenIPC_LoggingSeverityLevel_Debug, "Loading file '" << path << "'");

    if (_willNotNotifyIfChanged.find(path.filename().string()) != _willNotNotifyIfChanged.end())
    {
        alreadyIndexed = true;
    }

    std::shared_ptr<DataDocument> document;
    try
    {
        document = _LoadXmlDocument(path.string(), decrypt, _schemaValidationEnabled, schemaFilePath, alreadyIndexed);
    }
    catch (DataStoreException& exception)
    {
        OpenIPC_LOG(_logger, OpenIPC_LoggingSeverityLevel_Error, exception.Message());
        exceptions.push_back(exception);
    }
    catch (std::exception& exception)
    {
        OpenIPC_LOG(_logger, OpenIPC_LoggingSeverityLevel_Error, exception.what());
        exceptions.push_back(DataStoreException(exception.what()));
    }

    if (document)
    {
        _documents.push_back(document);

        // Get the root element
        DataElement* rootElement = document->RootElement();
        if (rootElement)
        {
            // Find or create the index node for the root element
            DataIndexNode& typeIndex = _rootIndexNode.FindOrCreate(rootElement->Name());

            DataElement* element = rootElement->FirstChild();
            while (element)
            {
                // Find or create the index node for elements of this name
                DataIndexNode& elementIndex = typeIndex.FindOrCreate(element->Name());

                // Add the element to the index node
                elementIndex.AddElement(*element);

                DataAttribute* attribute = element->FirstAttribute();
                while (attribute)
                {
                    // Find or create the index node for elements with attributes
                    // of this name
                    DataIndexNode& attributeIndex = elementIndex.FindOrCreate(attribute->Name());

                    // Add the element to the index node
                    attributeIndex.AddElement(*element);

                    // For each value in the attribute's value
                    for (std::string& value : DataQuery::ExtractAttributeValues(*attribute))
                    {
                        // Find or create the index node for elements with
                        // attributes of this value
                        DataIndexNode& valueIndex = attributeIndex.FindOrCreate(value);

                        // Add the element to the index node
                        valueIndex.AddElement(*element);
                    }

                    attribute = attribute->NextSibling();
                }

                element = element->NextSibling();
            }
        }
    }
}
