/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#pragma once

class DataStore;

///
/// @brief An node of a data index.
///
/// @note Contains both pointers to data elements of documents that have
/// been loaded and the file paths of documents which need to be loaded before
/// the index node is complete.
class DataIndexNode
{
public:
    DataIndexNode(DataStore& dataStore);

    ///
    /// @brief Adds the path of a data document which must be loaded for the
    /// index node to be complete.
    ///
    /// @param filePath
    ///     The full path of the document.
    void AddDocumentPath(const std::string& filePath);

    ///
    /// @brief Adds the data element of an already loaded document which
    /// belongs to this index node.
    ///
    /// @param element
    ///     The data element to add.
    void AddElement(DataElement& element);

    ///
    /// @brief Returns the elements belonging to the index node.
    ///
    /// @note May cause documents to be loaded in order to first complete the
    /// index node.
    ///
    /// @throws XmlLoadException If any malformed XML is loaded.
    const DataElement::Set& Elements();

    ///
    /// @brief Finds the child index node keyed from the given value.
    ///
    /// @param value
    ///     The value to find the child index node for (case-insensitive).
    ///     Depending on the node, the value can be an element name, an
    ///     attribute name, or an attribute value.  The meaning of the value
    ///     is dependent on how the index tree was constructed.
    ///
    /// @returns A pointer to the resulting index node; NULL if no child index
    /// node exists for the given value.
    DataIndexNode* Find(const std::string& value);

    ///
    /// @brief Creates or finds the child index node keyed from the given
    /// value.
    ///
    /// @param value
    ///     The value to find or create the child index node for (case-
    ///     insensitive).  Depending on the node, the value can be an element
    ///     name, an attribute name, or an attribute value.  The meaning of
    ///     the value is dependent on how the index tree was constructed.
    ///
    /// @returns The new or existing index node.
    DataIndexNode& FindOrCreate(const std::string& value);

    ///
    /// @brief Returns the key values for the index node.
    std::vector<std::string> KeyValues() const;

private:

    // A case-insensitive less-than comparison function to be used with std::map
    struct CaseInsensitiveLess :
        std::binary_function<std::string, std::string, bool>
    {
        struct LowerCaseCompare :
            public std::binary_function<unsigned char, unsigned char, bool>
        {
            bool operator()(const unsigned char& lhs, const unsigned char& rhs) const
            {
                return tolower(lhs) < tolower(rhs);
            }
        };

        bool operator()(const std::string & lhs, const std::string & rhs) const
        {
            return std::lexicographical_compare(lhs.begin(), lhs.end(), rhs.begin(), rhs.end(), LowerCaseCompare());
        }
    };

    DataStore* _dataStore;
    std::set<std::string> _documentPaths;
    DataElement::Set _elements;

    // Maps case-insensitive values to the corresponding child index node (see
    // the "value" parameter in Find() and FindOrCreate())
    std::map<std::string, std::shared_ptr<DataIndexNode>, CaseInsensitiveLess> _nodes;
};
