/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#pragma once

#ifdef _MSC_VER
// Disable the DLL interface warning for STL usage
#pragma warning( push )
#pragma warning( disable : 4251 )
#endif

///
/// @brief The result of a data store query.
class STRUCTUREDDATA_API DataQueryResult
{
public:
    DataQueryResult(DataElement::Array&& dataElements);
    DataQueryResult(const std::string& errorMessage);

    ///
    /// @brief Returns the number of resulting data elements were found.
    size_t DataElementCount() const;

    ///
    /// @brief Returns the first resulting data element (NULL if no data
    /// elements were found).
    DataElement* FirstDataElement() const;

    ///
    /// @brief Returns the resulting data elements that were found.
    ///
    /// @note The elements are sorted by the file path of the document they
    /// appear in.  Elements within the same document are sorted by their
    /// position in the document.
    const DataElement::Array& DataElements() const;

    ///
    /// @brief Returns whether malformed XML was encountered during the query.
    bool ErrorOccurred() const;

    ///
    /// @brief Returns the error message.
    std::string ErrorMessage() const;

private:
    DataElement::Array _dataElements;
    std::string _errorMessage;
};

///
/// @brief A node in a data query.
class STRUCTUREDDATA_API QueryNode
{
public:
    virtual ~QueryNode() { }

    ///
    /// @brief Executes the query on a data index node, returning the results
    /// of the query.
    ///
    /// @param index
    ///     The node in the data index to operate from.
    virtual DataElement::Set Execute(DataIndexNode& index) const = 0;
};

///
/// @brief A data query node for descriminating elements based on attribute
/// values.
///
/// @note This node can only be a leaf so it will never have any children.
class STRUCTUREDDATA_API WhereAttributeHasNode :
    public QueryNode
{
    friend class SelectElementsNode;
public:
    DataElement::Set Execute(DataIndexNode& index) const override;

    ///
    /// @brief Returns whether an element meets the criteria of this
    /// discriminator.
    ///
    /// @param element
    ///     The element.
    bool MatchesElement(const DataElement& element) const;

private:
    WhereAttributeHasNode(const std::string& name, const std::string& value);

    std::string _name;
    std::string _value;
};

///
/// @brief A data query node for selecting elements based on their name.
///
/// @note This node can have one or more WhereAttributeHasNode's as children.
/// The results from these child nodes will be combined using "and" semantics.
class STRUCTUREDDATA_API SelectElementsNode :
    public QueryNode
{
    friend class DataQuery;
public:

    ///
    /// @brief Adds a discriminator for the selected elements to have
    /// attributes containing a specific value, returning the parent element
    /// selection node.
    ///
    /// @param name
    ///     The attribute name the selected elements must have (case-
    ///     insensitive).
    /// @param value
    ///     The attribute value the selected elements attribute must contain in
    ///     its comma-separated value (case-insensitive).
    SelectElementsNode& WhereAttributeHas(const std::string& name, const std::string& value);

    DataElement::Set Execute(DataIndexNode& index) const override;

private:
    SelectElementsNode(const std::string& documentType, const std::string& elementName);

    std::string _documentType;
    std::string _elementName;

    std::vector<std::shared_ptr<WhereAttributeHasNode>> _whereAttributeHasNodes;
};

///
/// @brief A query to be applied to a data store, resulting in zero or more
/// data elements.
///
/// @note Data queries are built up in a tree-like fashion.
class STRUCTUREDDATA_API DataQuery
{
public:

    ///
    /// @brief Adds elements of the given name to be included in the query
    /// results, returning the resulting data query node.
    ///
    /// @param documentType
    ///     The document type to include in the query results (case-
    ///     insensitive).
    /// @param elementName
    ///     The element name to include in the query results (case-
    ///     insensitive).
    SelectElementsNode& SelectElements(const std::string& documentType, const std::string& elementName);

    ///
    /// @brief Executes the query on a data index node, returning the results
    /// of the query.
    ///
    /// @param index
    ///     The node in the data index to operate from.
    DataElement::Set Execute(DataIndexNode& index) const;

    ///
    /// @brief Returns the values of an attribute (extracted as comma-
    /// separated values).
    ///
    /// @param attribute
    ///     The attribute to extract the values of.
    static std::vector<std::string> ExtractAttributeValues(const DataAttribute& attribute);

    ///
    /// @brief Returns whether an attribute's extracted values contains the
    /// given value (case-insensitive).
    ///
    /// @param attribute
    ///     The attribute to check the value for.
    /// @param value
    ///     The value to check.
    static bool AttributeHasValue(const DataAttribute& attribute, const std::string& value);

    ///
    /// @brief Finds an attribute from an element by name (case-insensitive).
    ///
    /// @param element
    ///     The element to find the attribute of.
    /// @param attributeName
    ///     The attribute name to look for.
    ///
    /// @returns A pointer to the found attribute; NULL if no attribute with
    /// the given name was found.
    static const DataAttribute* FindAttribute(const DataElement& element, const std::string& attributeName);

private:
    std::vector<std::shared_ptr<SelectElementsNode>> _selectElementsNodes;
};

#ifdef _MSC_VER
#pragma warning( pop )
#endif
