/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#include "stdafx.h"
#include "StructuredData.h"
#include "Xerces.h"

#include <ManipulatePath.h>

#include <iostream>

DataDocument::DataDocument() :
    _xmlParser(new XercesParser()),
    _newToIndex(false)
{
    boost::filesystem::path::imbue(std::locale());
}

DataDocument::DataDocument(const std::string& rootElementName, const std::string& nameSpaceUri) :
    _xmlParser(new XercesParser()),
    _newToIndex(false)
{
    boost::filesystem::path::imbue(std::locale());
    _CreateRootElement(rootElementName, nameSpaceUri);
}

DataDocument::DataDocument(const DataElement& rootElement) :
    _xmlParser(new XercesParser()),
    _newToIndex(false)
{
    _CreateRootElement("", "");
    _rootElement->Copy(&rootElement);
}

DataDocument::DataDocument(const DataDocument& document) :
	_filePath(document._filePath),
    _xmlParser(new XercesParser()),
    _newToIndex(document._newToIndex)
{
    boost::filesystem::path::imbue(std::locale());
    _Copy(document);
}

DataDocument::DataDocument(DataDocument&& document) :
    _rootElement(std::move(document._rootElement)),
	_filePath(std::move(document._filePath)),
    _xmlParser(new XercesParser()),
    _newToIndex(document._newToIndex)
{
    boost::filesystem::path::imbue(std::locale());
	_rootElement->_SetDocument(this);
    document._rootElement.reset();
    document._newToIndex = false;
}

DataDocument::~DataDocument()
{
}

DataElement* DataDocument::RootElement()
{
    return _rootElement.get();
}

const DataElement* DataDocument::RootElement() const
{
    return _rootElement.get();
}

void DataDocument::LoadXmlFile(const std::string& filePath, bool enableSchemaValidation, bool newToIndex)
{
	_filePath = filePath;
    _CreateRootElement("Root", "");
    _xmlParser->LoadXmlFile(this, filePath, enableSchemaValidation);
    _newToIndex = newToIndex;
}

void DataDocument::LoadXmlFile(const std::string& filePath, const std::string& schemaFilePath, bool newToIndex)
{
    _filePath = filePath;
    _CreateRootElement("Root", "");

    XercesGrammarPool grammarPool;
    grammarPool.LoadSchemaFile(schemaFilePath);
    _xmlParser->LoadXmlFile(this, filePath, grammarPool);
    _newToIndex = newToIndex;
}


void DataDocument::LoadXmlString(const std::string& xml, bool newToIndex)
{
	_filePath = "";
    _CreateRootElement("Root", "");
    _xmlParser->LoadXmlString(this, xml);
    _newToIndex = newToIndex;
}

void DataDocument::SaveXmlFile(const std::string& filePath) const
{
    _xmlParser->SaveXmlFile(this, filePath);
}

std::string DataDocument::SaveXmlString() const
{
    return _xmlParser->SaveXmlString(this);
}

std::string DataDocument::_GetStringTableValue(std::vector<std::string>& stringTable, uint32_t index)
{
	if (index >= stringTable.size())
	{
		throw BinLoadException("Index out of range");
	}

	return stringTable[index];
}

uint32_t DataDocument::_GetStringTableIndex(std::vector<std::string>& stringTable, std::string value)
{
	for (size_t i = 0; i < stringTable.size(); ++i)
	{
		if (stringTable[i] == value)
		{
			return static_cast<uint32_t>(i);
		}
	}

	stringTable.push_back(value);
	return static_cast<uint32_t>(stringTable.size() - 1);
}

void DataDocument::_SaveBinElement(const DataElement* element, std::stringstream& stream, std::vector<std::string>& stringTable) const
{
	uint32_t index = _GetStringTableIndex(stringTable, element->Name());
	stream.write(reinterpret_cast<char*>(&index), sizeof(index));

	bool writeNamespace = !element->NameSpaceUri().empty();
	bool writeText = !element->Text().empty();
	bool writeAttributes = element->_attributes.size() > 0;
	bool writeElements = element->_elements.size() > 0;

	char flags = 0;
	if (writeNamespace)
	{
		flags |= 0x01;
	}

	if (writeText)
	{
		flags |= 0x02;
	}

	if (writeAttributes)
	{
		flags |= 0x04;
	}

	if (writeElements)
	{
		flags |= 0x08;
	}

	stream.write(reinterpret_cast<char*>(&flags), sizeof(char));

	if (writeNamespace)
	{
		uint32_t nameSpaceUriIndex = _GetStringTableIndex(stringTable, element->NameSpaceUri());
		stream.write(reinterpret_cast<char*>(&nameSpaceUriIndex), sizeof(nameSpaceUriIndex));
	}

	if (writeText)
	{
        uint32_t textIndex = _GetStringTableIndex(stringTable, element->Text());
		stream.write(reinterpret_cast<char*>(&textIndex), sizeof(textIndex));
	}

	if (writeAttributes)
	{
        uint32_t attributeCount = (int)element->_attributes.size();
		stream.write(reinterpret_cast<char*>(&attributeCount), sizeof(attributeCount));

		for (uint32_t i = 0; i < attributeCount; ++i)
		{
            uint32_t attributeNameIndex = _GetStringTableIndex(stringTable, element->_attributes[i]->Name());
			stream.write(reinterpret_cast<char*>(&attributeNameIndex), sizeof(attributeNameIndex));

            uint32_t attributeValueIndex = _GetStringTableIndex(stringTable, element->_attributes[i]->Value());
			stream.write(reinterpret_cast<char*>(&attributeValueIndex), sizeof(attributeValueIndex));
		}
	}

	if (writeElements)
	{
        uint32_t elementCount = static_cast<uint32_t>(element->_elements.size());
		stream.write(reinterpret_cast<char*>(&elementCount), sizeof(elementCount));

		for (uint32_t i = 0; i < elementCount; ++i)
		{
			_SaveBinElement(element->_elements[i], stream, stringTable);
		}
	}
}

void DataDocument::_LoadBinElement(DataElement* parent, std::basic_istream<char>& stream, std::vector<std::string>& stringTable)
{
	uint32_t index = 0;
	stream.read(reinterpret_cast<char*>(&index), sizeof(index));
	std::string name = _GetStringTableValue(stringTable, index);

	char flags = 0;
	stream.read(reinterpret_cast<char*>(&flags), sizeof(char));

	bool readNamespace = (flags & 0x01) != 0;
	bool readText = (flags & 0x02) != 0;
	bool readAttributes = (flags & 0x04) != 0;
	bool readElements = (flags & 0x08) != 0;

	std::string nameSpaceUri;
	if (readNamespace)
	{
		uint32_t nameSpaceUriIndex = 0;
		stream.read(reinterpret_cast<char*>(&nameSpaceUriIndex), sizeof(nameSpaceUriIndex));
		nameSpaceUri = _GetStringTableValue(stringTable, nameSpaceUriIndex);
	}

	DataElement* element = nullptr;
	if (parent == nullptr)
	{
		_CreateRootElement(name, nameSpaceUri);
		element = RootElement();
	}
	else
	{
		element = parent->CreateChild(name, nameSpaceUri);
	}

	if (readText)
	{
		uint32_t textIndex = 0;
		stream.read(reinterpret_cast<char*>(&textIndex), sizeof(textIndex));
		std::string text = _GetStringTableValue(stringTable, textIndex);

		element->SetText(text);
	}

	if (readAttributes)
	{
		uint32_t attributeCount = 0;
		stream.read(reinterpret_cast<char*>(&attributeCount), sizeof(attributeCount));

		for (uint32_t i = 0; i < attributeCount; ++i)
		{
			uint32_t attributeNameIndex = 0;
			stream.read(reinterpret_cast<char*>(&attributeNameIndex), sizeof(attributeNameIndex));
			std::string attributeName = _GetStringTableValue(stringTable, attributeNameIndex);

            uint32_t attributeValueIndex = 0;
			stream.read(reinterpret_cast<char*>(&attributeValueIndex), sizeof(attributeValueIndex));
			std::string attributeValue = _GetStringTableValue(stringTable, attributeValueIndex);

			element->CreateAttribute(attributeName, attributeValue);
		}
	}

	if (readElements)
	{
		uint32_t elementCount = 0;
		stream.read(reinterpret_cast<char*>(&elementCount), sizeof(elementCount));
		for (uint32_t i = 0; i < elementCount; ++i)
		{
			_LoadBinElement(element, stream, stringTable);
		}
	}
}

void DataDocument::LoadBinFile(const std::string& filePath, bool newToIndex)
{
    boost::filesystem::path path(filePath);
    if (!OpenIPC_PASS(CommonUtils::SanitizePath(path)))
    {
        throw BinLoadException("Failed sanitize the given binary file path: '" + path.string() + "'");
    }
	boost::filesystem::ifstream file(path, std::ifstream::binary);

	if (file)
	{
		LoadBinStream(file);

		file.close();

		_filePath = filePath;
        _newToIndex = newToIndex;
	}
	else
	{
		throw BinLoadException("Unable to open binary file: '" + path.string() + "'");
	}
}

void DataDocument::LoadBinStream(std::basic_istream<char>& stream, bool newToIndex)
{
	_filePath = "";

	std::vector<std::string> stringTable;

    uint64_t signature = 0;
    stream.read(reinterpret_cast<char*>(&signature), sizeof(signature));

    if (stream.fail())
    {
        throw BinLoadException("Failed to read the file signature");
    }
    else if (signature != BIN_FILE_SIGNATURE)
    {
        throw BinLoadException("The file signature does not match. This generally indicates a collateral decryption failure.");
    }

	uint32_t stringCount = 0;
	stream.read(reinterpret_cast<char*>(&stringCount), sizeof(stringCount));

	for (uint32_t i = 0; i < stringCount; ++i)
	{
		uint32_t size = 0;
		stream.read(reinterpret_cast<char*>(&size), sizeof(size));

		std::vector<char> buffer(size + 1, '\0');
		stream.read(buffer.data(), size);
		stringTable.push_back(buffer.data());
	}

	_LoadBinElement(nullptr, stream, stringTable);
    _newToIndex = newToIndex;
}

void DataDocument::SaveBinStream(std::basic_ostream<char>& stream) const
{
	std::stringstream elementStream (std::stringstream::in | std::stringstream::out | std::stringstream::binary);

	std::vector<std::string> stringTable;

	_SaveBinElement(RootElement(), elementStream, stringTable);

    uint64_t signature = BIN_FILE_SIGNATURE;
    stream.write(reinterpret_cast<char*>(&signature), sizeof(signature));

	uint32_t stringCount = static_cast<uint32_t>(stringTable.size());
	stream.write(reinterpret_cast<char*>(&stringCount), sizeof(stringCount));

	for (uint32_t i = 0; i < stringCount; ++i)
	{
		uint32_t size = static_cast<uint32_t>(stringTable[i].size());
		stream.write(reinterpret_cast<char*>(&size), sizeof(size));
		stream << stringTable[i];
	}

	// get length of string data:
    elementStream.seekg (0, elementStream.end);
    size_t length = static_cast<size_t>(elementStream.tellg());
    elementStream.seekg (0, elementStream.beg);

	std::vector<char> buffer(length+1);
	elementStream.read(buffer.data(), length);

	stream.write(buffer.data(), length);
}

void DataDocument::SaveBinFile(const std::string& filePath)
{
    boost::filesystem::path path(filePath);
    if (!OpenIPC_PASS(CommonUtils::SanitizePath(path)))
    {
        throw BinSaveException("Failed sanitize the given binary file path: '" + path.string() + "'");
    }
	boost::filesystem::ofstream file(path, std::ofstream::binary);

	if (file)
	{
		SaveBinStream(file);

		file.close();
	}
	else
	{
        throw BinSaveException("Unable to open binary file: '" + path.string() + "'");
	}
}

const std::string& DataDocument::FilePath() const
{
	return _filePath;
}

DataDocument& DataDocument::operator=(const DataDocument& document)
{
    _Copy(document);
    return *this;
}

DataDocument& DataDocument::operator=(DataDocument&& document)
{
    _rootElement = document._rootElement;
    document._rootElement.reset();

    if (_rootElement)
    {
        _rootElement->_SetDocument(this);
    }

	_processingInstructions = std::move(document._processingInstructions);
	_filePath = std::move(document._filePath);
    _xmlParser = std::move(document._xmlParser);
    _newToIndex = document._newToIndex;

    return *this;
}

void DataDocument::_CreateRootElement(const std::string& name, const std::string& nameSpaceUri)
{
    _rootElement.reset(new DataElement(name, nameSpaceUri, nullptr, 0, this));
}

void DataDocument::_Copy(const DataDocument& document)
{
    _filePath = document.FilePath();
    _rootElement.reset();

    const DataElement* root = document.RootElement();
    if (root)
    {
        // Copy the entire tree
        _CreateRootElement(root->Name(), root->NameSpaceUri());
        _rootElement->Copy(root);
		_rootElement->_SetDocument(this);
    }
}

DataProcessingInstruction* DataDocument::CreateDataProcessingInstruction(const std::string& target, const std::string& data)
{
	DataProcessingInstruction* ret = new DataProcessingInstruction(target, data, _processingInstructions.size(), this);

	_processingInstructions.push_back(std::shared_ptr<DataProcessingInstruction>(ret));

	return ret;
}

DataProcessingInstruction* DataDocument::FirstProcessingInstruction() const
{
	if (_processingInstructions.size() == 0)
	{
		return nullptr;
	}

	return _processingInstructions[0].get();
}

const std::vector<std::shared_ptr<DataProcessingInstruction>>& DataDocument::GetProcessingInstructions() const
{
    return _processingInstructions;
}

bool DataDocument::NewToIndex() const
{
    return _newToIndex;
}
void DataDocument::MarkAsOld()
{
    _newToIndex = false;
}
