/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#pragma once

#include <Frequency.h>

class DataElement;

///
/// @brief An attribute of an DataElement consisting of a key and a value.
class STRUCTUREDDATA_API DataAttribute :
    public DataNode
{
    friend class DataElement;
public:

    ///
    /// @brief A table mapping attribute parameter names to their corresponding
    /// values.
    typedef std::map<std::string, std::string> ParameterValues;

    ///
    /// @brief Gets the value of the attribute.
    ///
    /// @returns The attribute value.
    const std::string& Value() const;

    ///
    /// @brief Changes the value of the attribute.
    ///
    /// @param value
    ///     The new attribute value.
    void SetValue(const std::string& value);

    ///
    /// @brief Converts the value to a bool.
    ///
    /// @param defaultValue
    ///     The value to return if the attribute is not a valid boolean.
    ///
    /// @returns The converted value if the attribute is a valid boolean;
    ///          otherwise returns the specified default value.
    bool ToBool(bool defaultValue = false) const;

    ///
    /// @brief Converts the value to an 8-bit signed integer.
    ///
    /// @param defaultValue
    ///     The value to return if the attribute is not a valid integer.
    ///
    /// @returns The converted value if the attribute is a valid integer;
    ///          otherwise returns the specified default value.
    int8_t ToInt8(int8_t defaultValue = 0) const;

    ///
    /// @brief Converts the value to an 16-bit signed integer.
    ///
    /// @param defaultValue
    ///     The value to return if the attribute is not a valid integer.
    ///
    /// @returns The converted value if the attribute is a valid integer;
    ///          otherwise returns the specified default value.
    int16_t ToInt16(int16_t defaultValue = 0) const;

    ///
    /// @brief Converts the value to an 32-bit signed integer.
    ///
    /// @param defaultValue
    ///     The value to return if the attribute is not a valid integer.
    ///
    /// @returns The converted value if the attribute is a valid integer;
    ///          otherwise returns the specified default value.
    int32_t ToInt32(int32_t defaultValue = 0) const;

    ///
    /// @brief Converts the value to an 64-bit signed integer.
    ///
    /// @param defaultValue
    ///     The value to return if the attribute is not a valid integer.
    ///
    /// @returns The converted value if the attribute is a valid integer;
    ///          otherwise returns the specified default value.
    int64_t ToInt64(int64_t defaultValue = 0) const;

    ///
    /// @brief Converts the value to an 8-bit unsigned integer.
    ///
    /// @param defaultValue
    ///     The value to return if the attribute is not a valid integer.
    ///
    /// @returns The converted value if the attribute is a valid integer;
    ///          otherwise returns the specified default value.
    uint8_t ToUInt8(uint8_t defaultValue = 0) const;

    ///
    /// @brief Converts the value to an 16-bit unsigned integer.
    ///
    /// @param defaultValue
    ///     The value to return if the attribute is not a valid integer.
    ///
    /// @returns The converted value if the attribute is a valid integer;
    ///          otherwise returns the specified default value.
    uint16_t ToUInt16(uint16_t defaultValue = 0) const;

    ///
    /// @brief Converts the value to an 32-bit unsigned integer.
    ///
    /// @param defaultValue
    ///     The value to return if the attribute is not a valid integer.
    ///
    /// @returns The converted value if the attribute is a valid integer;
    ///          otherwise returns the specified default value.
    uint32_t ToUInt32(uint32_t defaultValue = 0) const;

    ///
    /// @brief Converts the value to an 64-bit unsigned integer.
    ///
    /// @param defaultValue
    ///     The value to return if the attribute is not a valid integer.
    ///
    /// @returns The converted value if the attribute is a valid integer;
    ///          otherwise returns the specified default value.
    uint64_t ToUInt64(uint64_t defaultValue = 0) const;

    ///
    /// @brief Converts the value to a Frequency object
    ///
    /// @param defaultValue
    ///     The value to return if the attribute is not a valid Frequency.
    ///
    /// @returns The converted value if the attribute is a valid Frequency;
    ///          otherwise returns the specified default value.
    Frequency ToFrequency(Frequency defaultValue = Frequency(0)) const;


    ///
    /// @brief Converts the value to a std::chrono::duration object.
    ///        This expects values like 10ms, 7s, or 123us
    ///
    /// @param defaultValue
    ///     The value to return if the attribute is not a valid duration.
    ///
    /// @returns The converted value if the attribute is a valid Frequency;
    template<typename T>
    T ToDuration(T defaultValue = T(0)) const
    {
        try
        {
            std::size_t pos;
            unsigned long long value = std::stoull(_value, &pos);
            std::string unit = _value.substr(pos);
            
            if (unit == "ns")
            {
                return std::chrono::duration_cast<T>(std::chrono::nanoseconds(value));
            }
            if (unit == "us")
            {
                return std::chrono::duration_cast<T>(std::chrono::microseconds(value));
            }
            if(unit == "ms")
            {
                return std::chrono::duration_cast<T>(std::chrono::milliseconds(value));
            }
            if (unit == "s")
            {
                return std::chrono::duration_cast<T>(std::chrono::seconds(value));
            }
            if (unit == "min")
            {
                return std::chrono::duration_cast<T>(std::chrono::minutes(value));
            }
            if (unit == "h")
            {
                return std::chrono::duration_cast<T>(std::chrono::hours(value));
            }
        }
        catch (const std::exception&) {}

        return defaultValue;
    }

    ///
    /// @brief Converts the value to a comma-separated list of strings.
    std::vector<std::string> ToCommaSeparatedList() const;

    ///
    /// @brief Gets the next sibling of the attribute.
    ///
    /// @returns The next sibling of the attribute; null if the attribute has
    /// no sibling.
    DataAttribute* NextSibling() const;

    /// @param parameterName
    ///     The name of the parameter.
    /// @param parameterValue
    ///     The value of the parameter.
    void ResolveParameter(const std::string& parameterName, const std::string& parameterValue);

private:
    DataAttribute(const std::string& name, const std::string& value, DataElement* parent, size_t position);

    template <typename T>
    T _ParseInt(T defaultValue, bool checkMaxBound) const
    {
        try
        {
            const long long value = std::stoll(_value);
            const T min = std::numeric_limits<T>::min();
            const T max = std::numeric_limits<T>::max();
            if ((!checkMaxBound || value <= static_cast<long long>(max)) &&
                value >= static_cast<long long>(min))
            {
                return static_cast<T>(value);
            }
        }
        catch (const std::exception&) { }

        return defaultValue;
    }

    // Case-insensitive string compare
    static bool _CaseInsensitiveCompare(const std::string& left, const std::string& right);
    static bool _ComparePredicate(unsigned char a, unsigned char b);

    // Make non-copyable
    DataAttribute(const DataAttribute&);
    DataAttribute& operator=(const DataAttribute&);

    DataElement* _parent;
    size_t _position;

    std::string _value;
};
