/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#pragma once

class DataDocument;

///
/// @brief An element in a document.
class STRUCTUREDDATA_API DataElement :
    public DataNode
{
    friend class DataDocument;
    friend class DataAttribute;
public:

    ///
    /// @brief An array of element pointers.
    typedef std::vector<DataElement*> Array;

    ///
    /// @brief A set of element pointers.
    typedef std::set<DataElement*> Set;

    ///
    /// @brief A predicate function given an element.
    typedef std::function<bool(const DataElement*)> Predicate;

    ///
    /// @brief Returns the parent element (null if the element has no parent).
    DataElement* Parent() const;

    ///
    /// @brief Creates an attribute.
    ///
    /// @note If an attribute of the same name already exists then the
    /// existing attribute is set to the new value and is returned.
    ///
    /// @param name
    ///     The name of the attribute.
    /// @param value
    ///     The value of the attribute.
    ///
    /// @returns The new or existing attribute.
    DataAttribute* CreateAttribute(const std::string& name, const std::string& value);

    ///
    /// @brief Creates a child element.
    ///
    /// @param name
    ///     The name of the child element.
    /// @param nameSpaceUri
    ///     The URI of the namespace for the child element.
    ///
    /// @returns The name of the child element.
    DataElement* CreateChild(const std::string& name, const std::string& nameSpaceUri = std::string());

    ///
    /// @brief Gets the first attribute of the element.
    ///
    /// @returns The first attribute of the element; null if the element has
    /// no attributes.
    DataAttribute* FirstAttribute() const;


    ///
    /// @brief Gets all the attributes of the element.
    ///
    /// @returns The attributes of the element
    std::vector<DataAttribute*> GetAttributes() const;

    ///
    /// @brief Finds an attribute with a specific name.
    ///
    /// @param name
    ///     The name of the attribute.
    ///
    /// @returns The found attribute; null if no attribute was found.
    DataAttribute* FindAttribute(const std::string& name) const;

    ///
    /// @brief Returns whether the element has an attribute with a specific
    /// value.
    ///
    /// @param name
    ///     The name of the attribute.
    /// @param value
    ///     The value of the attribute.
    ///
    /// @returns True if the element has an attribute of the given name with
    /// the given attribute; false if the element does not have an attribute of
    /// the given name or the attribute does not have the given value.
    bool HasAttribute(const std::string& name, const std::string& value) const;

    ///
    /// @brief Gets the first child element.
    ///
    /// @returns The first child element; null if the element has no children.
    DataElement* FirstChild() const;

    ///
    /// @brief Gets the next sibling of the element.
    ///
    /// @returns The next sibling of the element; null if the element has no
    /// sibling.
    DataElement* NextSibling() const;

    ///
    /// @brief Searches all immediate children for the first element with a
    /// specific name.
    ///
    /// @param name
    ///     The element name to search for.
    ///
    /// @returns The found child element; null if no element was found.
    DataElement* FindChildByName(const std::string& name) const;

    ///
    /// @brief Searches all immediate children for the first element with an
    /// attribute of a specific value.
    ///
    /// @param name
    ///     The attribute name to search for.
    /// @param value
    ///     The attribute value to search for.
    ///
    /// @returns The found child element; null if no element was found.
    DataElement* FindChildByAttribute(const std::string& name, const std::string& value) const;

    ///
    /// @brief Searches all immediate children for the elements with a specific
    /// name.
    ///
    /// @param name
    ///     The element name to search for.
    ///
    /// @returns The found child elements.
    DataElement::Array FindChildrenByName(const std::string& name) const;

    ///
    /// @brief Searches all immediate children for the elements with an
    /// attribute of a specific value.
    ///
    /// @param name
    ///     The attribute name to search for.
    /// @param value
    ///     The attribute value to search for.
    ///
    /// @returns The found child elements.
    DataElement::Array FindChildrenByAttribute(const std::string& name, const std::string& value) const;

    ///
    /// @brief Searches all immediate children for first elements satisfying
    /// the predicate.
    ///
    /// @param predicate
    ///     The search predicate.
    ///
    /// @returns The found child element; null if no element was found.
    DataElement* FindChild(DataElement::Predicate predicate) const;

    ///
    /// @brief Searches all immediate children for the elements satisfying the
    /// predicate.
    ///
    /// @param predicate
    ///     The search predicate.
    ///
    /// @returns The found child elements.
    DataElement::Array FindChildren(DataElement::Predicate predicate) const;

    ///
    /// @brief Searches depth-first for the first descendant element with a
    /// specific name.
    ///
    /// @param name
    ///     The element name to search for.
    ///
    /// @returns The found element; null if no element was found.
    DataElement* FindDescendantByName(const std::string& name) const;

    ///
    /// @brief Searches depth-first for the first descendant element with an
    /// attribute of a specific value.
    ///
    /// @param name
    ///     The attribute name to search for.
    /// @param value
    ///     The attribute value to search for.
    ///
    /// @returns The found element; null if no element was found.
    DataElement* FindDescendantByAttribute(const std::string& name, const std::string& value) const;

    ///
    /// @brief Searches depth-first for the descendant elements with a specific
    /// name.
    ///
    /// @param name
    ///     The element name to search for.
    ///
    /// @returns The found elements.
    DataElement::Array FindDescendantsByName(const std::string& name) const;

    ///
    /// @brief Searches depth-first for the descendant elements with an
    /// attribute of a specific value.
    ///
    /// @param name
    ///     The attribute name to search for.
    /// @param value
    ///     The attribute value to search for.
    ///
    /// @returns The found elements.
    DataElement::Array FindDescendantsByAttribute(const std::string& name, const std::string& value) const;

    ///
    /// @brief Searches depth-first for the first descendant element satisfying
    /// the predicate.
    ///
    /// @param predicate
    ///     The search predicate.
    ///
    /// @returns The found element; null if no element was found.
    DataElement* FindDescendant(DataElement::Predicate predicate) const;

    ///
    /// @brief Searches depth-first for the descendant elements satisfying
    /// the predicate.
    ///
    /// @param predicate
    ///     The search predicate.
    ///
    /// @returns The found elements.
    DataElement::Array FindDescendants(DataElement::Predicate predicate) const;

    ///
    /// @brief Searches for the first ancestor element with a specific name.
    ///
    /// @param name
    ///     The element name to search for.
    ///
    /// @returns The found element; null if no element was found.
    DataElement* FindAncestorByName(const std::string& name) const;

    ///
    /// @brief Searches for the first ancestor element with an
    /// attribute of a specific value.
    ///
    /// @param name
    ///     The attribute name to search for.
    /// @param value
    ///     The attribute value to search for.
    ///
    /// @returns The found element; null if no element was found.
    DataElement* FindAncestorByAttribute(const std::string& name, const std::string& value) const;

    ///
    /// @brief Searches for the first ancestor element satisfying the
    /// predicate.
    ///
    /// @param predicate
    ///     The search predicate.
    ///
    /// @returns The found element; null if no element was found.
    DataElement* FindAncestor(DataElement::Predicate predicate) const;

    ///
    /// @brief Returns the text of the element.
    const std::string& Text() const;

    ///
    /// @brief Changes the text of the element.
    ///
    /// @param text
    ///     The new text.
    void SetText(const std::string& text);

    ///
    /// @brief Changes the name of the element.
    ///
    /// @param name
    ///     The new element name.
    void SetName(const std::string& name);

    ///
    /// @brief Gets the position of the element within its generation.
    size_t Position() const;

	///
	/// @brief Gets the document that the element belongs to.
	DataDocument* Document() const;

	///
	/// @brief Gets the line number that the element was loaded from.
	unsigned LineNumber() const;

	///
	/// @brief Sets the line number that the element was loaded from.
	///
	/// @param lineNumber
	///		The line number that the element was loaded from.
	void SetLineNumber(unsigned lineNumber);

    ///
    /// @brief Gets the namespace URI of the element (empty if no namespace is used).
    const std::string& NameSpaceUri() const;

    ///
    /// @brief Copies the name, attributes, and children from another element.
    ///
    /// @note Any existing attributes/children are removed.
    ///
    /// @param element
    ///     The element to copy from.
    void Copy(const DataElement* element);

    ///
    /// @brief Recursively resolves attribute parameter placeholders using the
    /// provided parameter look-up table.
    ///
    /// @param parameterValues
    ///     The mapping of parameter names to their values.
    void ResolveAttributeParameters(const DataAttribute::ParameterValues& parameterValues);

    ///
    /// @brief Returns the element and all descendant elements as an XML string.
    std::string ToXmlString() const;

private:
	void _SetDocument(DataDocument* document);

    DataElement(const std::string& name, const std::string& nameSpaceUri, DataElement* parent, size_t position, DataDocument* document);

    // Make non-copyable
    DataElement(const DataElement&);
    DataElement& operator=(const DataElement&);

	DataDocument* _document;
	unsigned _lineNumber;

    DataElement* _parent;
    size_t _position;

    std::string _text;
    std::string _nameSpaceUri;

    std::vector<std::shared_ptr<DataNode>> _children;

    std::map<std::string, DataAttribute*> _attributeMap;
    std::vector<DataAttribute*> _attributes;
    std::vector<DataElement*> _elements;
};
