/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////
#pragma once

#include "StructuredData.h"

/*!
\brief Represents a pool of grammars (i.e. a collection of schemas)
*/
class GrammarPool
{
public:
    GrammarPool() {}
    virtual ~GrammarPool() {}

    /*!
    \brief Recursively loads all schema files under the given directory into the grammar pool.

    \param [in] directoryPath The path to the directory to traverse

    \throws XmlLoadException with a description of the error
    */
    virtual void LoadSchemaDirectory(const std::string& directoryPath) = 0;

    /*!
    \brief Loads the given schema file into the grammar pool.

    \param [in] filePath The path to the schema file to load

    \throws XmlLoadException with a description of the error
    */
    virtual void LoadSchemaFile(const std::string& filePath) = 0;
};

class XmlParser
{
public:
    XmlParser() {}
    virtual ~XmlParser() {}

    /*!
    \brief Loads an XML document from a file.

    \param [out] doc      The DataDocument object that will be populated with the elements parsed from the file
    \param [in]  filePath The path to the XML file to load
    \param [in]  enableSchemaValidation If true, the XML parser will validate the XML against any schemas it references

    \throws XmlLoadException with a description of the error
    */
    virtual void LoadXmlFile(DataDocument* doc, const std::string& filePath, bool enableSchemaValidation) const = 0;

    /*!
    \brief Loads an XML document from a file.

    \param [out] doc      The DataDocument object that will be populated with the elements parsed from the file
    \param [in]  filePath The path to the XML file to load
    \param [in]  grammarPool The pool of grammars to be used when validating the XML. Any schema files that are referenced
                             by the XML file will NOT be used

    \throws XmlLoadException with a description of the error
    */
    virtual void LoadXmlFile(DataDocument* doc, const std::string& filePath, const GrammarPool& grammarPool) const = 0;

    /*!
    \brief Loads an XML document from a string.

    \param [out] doc The DataDocument object that will be populated with the elements parsed from the string
    \param [in]  xml The XML data to parse

    \throws XmlLoadException with a description of the error
    */
    virtual void LoadXmlString(DataDocument* doc, const std::string& xml) const = 0;

    /*!
    \brief Saves an XML document to a file.

    \param [in] doc      The DataDocument object that contains the XML elements
    \param [in] filePath The path to the resulting XML file

    \throws XmlSaveException with a description of the error
    */
    virtual void SaveXmlFile(const DataDocument* doc, const std::string& filePath) const = 0;

    /*!
    \brief Saves an XML document to a string.

    \param [in] doc      The DataDocument object that contains the XML elements

    \throws XmlSaveException with a description of the error
    */
    virtual std::string SaveXmlString(const DataDocument* doc) const = 0;
};
