#pragma once
//////////////////////////////////////////////////////////////////////////////
//
//                      INTEL CONFIDENTIAL
//       Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written
// permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or
// delivery of the Materials, either expressly, by implication, inducement,
// estoppel or otherwise. Any license under such intellectual property rights
// must be express and approved by Intel in writing.
//
// Unless otherwise agreed by Intel in writing, you may not remove or alter
// this notice or any other notice embedded in Materials by Intel or Intel'ss
// suppliers or licensors in any way.
//
//////////////////////////////////////////////////////////////////////////////
///  @file
///
///  @brief Contains methods for obtaining instances of the Probe interfaces
///
///  For additional information on obtaining and using instances, see @ref probeusage.
///
//////////////////////////////////////////////////////////////////////////////

#if !defined(__ACTION_OPERATIONS_H__)
#define __ACTION_OPERATIONS_H__

/**
    The model of an action is a capability that is advertised by a plugin.
	This capability is specific to a probe/interface within a plugin and can be run at any time.
	Parameters are not directly supported, but are indirectly supported with the Device*Config operations.
	After an action is run, any messages can be accessed (though these are not designed to be machie parsable).

    The lifetime for any of the returned PPI_Action pointers is for the lifetime of an interface. Basically, the
    actions pointer must be valid until an interface is deinitialized. Aditionally, after any interface initialization, a client
    must call PPI_DeviceListActions to determine the valid actions for that interface.
*/

typedef struct PPI_Action PPI_Action;

/**
* @brief PPI_DeviceListActions provides a mechanism to list all of the actions that a particular device has avaliable.
*
* @param [in] deviceID The device id to query.
* @param [in] actionsLength The number of entries in the @a actions buffer
* @param [out] actions The produced list of actions. This can be NULL. Only the buffer's memory is owned by the client, everythigng else is owned by the plugin and has a valid lifetime until the interface is deinitialized.
* @param [out] numberOfActions The total number of actions for this device. This will always be written and cannot be NULL.
* @return OpenIPC_Error
* @retval OpenIPC_Error_No_Error The out parameters were populated correctly
*/
PPI_FUNCTION(PPI_DeviceListActions, OpenIPC_Error, OpenIPC_DeviceId deviceID,
             uint32_t actionsLength, PPI_Action** actions, uint32_t* numberOfActions);

/**
* @brief PPI_ActionGetName provides a mechanism to determine the name of a particular action (provided by the plugin).
*
* @param [in] action The action to query.
* @param [in] valueLength The number of characters in the @a value buffer.
* @param [out] value The buffer to be in the description.
*
* @return OpenIPC_Error
* @retval OpenIPC_Error_No_Error The out parameters were populated correctly
*/
PPI_FUNCTION(PPI_ActionGetName, OpenIPC_Error, const PPI_Action* action,
             uint32_t valueLength, PPI_char* value);

/**
* @brief PPI_ActionGetDescription provides a mechanism to determine the description of a particular action (provided by the plugin).
*
* @param [in] action The action to query.
* @param [in] valueLength The number of characters in the @a value buffer.
* @param [out] value The buffer to be in the description.
*
* @return OpenIPC_Error
* @retval OpenIPC_Error_No_Error The out parameters were populated correctly
*/
PPI_FUNCTION(PPI_ActionGetDescription, OpenIPC_Error, const PPI_Action* action,
             uint32_t valueLength, PPI_char* value);

/*
* @brief PPI_ActionSetArgument provides a mechanism to set the parameter of an action. By default, this is set to a NULL pointer.
*
* @param [in] action The action to set the parametr.
* @param [in] value The null-terminated buffer.
*
* @return OpenIPC_Error
* @retval OpenIPC_Error_No_Error The out parameters were populated correctly
*/
PPI_FUNCTION(PPI_ActionSetArgument, OpenIPC_Error, PPI_Action* action, const PPI_char* value)
/**
* @brief PPI_DeviceActionRun runs an action
*
* @param [in] deviceID The device id to query.
* @param [in] action The action to run on a device.
* @param [in] messageLength The length of the buffer pointed to by message in bytes.
* @param [out] message The buffer to be filled in with any messages generated by the action.
*
* @return OpenIPC_Error
* @retval OpenIPC_Error_No_Error The out parameters were populated correctly and the action ran successfully
*/
PPI_FUNCTION(PPI_DeviceActionRun, OpenIPC_Error, OpenIPC_DeviceId deviceID,
             const PPI_Action* action, uint32_t messageLength, PPI_char* message);

/**
* @brief PPI_ActionIsVisible Queries an action to see if it should be visible to clients
*
* @param [in] action The action to query.
* @param [out] isVisible Is true iff the action should be listed for clients.
*
* @return OpenIPC_Error
* @retval OpenIPC_Error_No_Error The out parameters were populated correctly and the action ran successfully
*/
PPI_FUNCTION(PPI_ActionIsVisible, OpenIPC_Error, const PPI_Action* action,
	PPI_bool* isVisible);
#endif // __ACTION_OPERATIONS_H__
