//////////////////////////////////////////////////////////////////////////////
//
//                      INTEL CONFIDENTIAL
//       Copyright 2014 Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its
// suppliers. Title to the Material remains with Intel Corporation, its
// suppliers, or licensors. The Material contains trade secrets and
// proprietary and confidential information of Intel Corporation, its
// suppliers, and licensors, and is protected by worldwide copyright and trade
// secret laws and treaty provisions. No part of the Material may be used,
// copied, reproduced, modified, published, uploaded, posted, transmitted,
// distributed, or disclosed in any way without Intel's prior express written
// permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or
// delivery of the Materials, either expressly, by implication, inducement,
// estoppel or otherwise. Any license under such intellectual property rights
// must be express and approved by Intel in writing.
//
// Unless otherwise agreed by Intel in writing, you may not remove or alter
// this notice or any other notice embedded in Materials by Intel or Intel's
// suppliers or licensors in any way.
//
//////////////////////////////////////////////////////////////////////////////
///  @file
///
///  @brief Contains methods for obtaining instances of the Probe interfaces
///
///  For additional information on obtaining and using instances, see @ref probeusage.
///
//////////////////////////////////////////////////////////////////////////////

#pragma once

#include "JtagEncoding.h"	// for JtagStateEncode
#include "ProbeTypes.h"
#include "TapStateMachineEncode.h"
#include "PluginLogger.h"
#include "PluginError.h"
#include "PluginNotifications.h"
#include "TriggerResponse.h"
#include "PinOperations.h"
#include "JTAGPaddingOperations.h"
#include "ActionOperations.h"
#include "StateportOperations.h"

#include "JTAGPinsBasedOperations.h"
#include "JtagStateBasedOperations.h"

#include <Components/Common/CommonErrors.h>
#include <Foundation/Logging/LoggingMacros.h>
#include <Foundation/Error/Error.h>
#include <Foundation/Types.h>
#include <ExceptionBridge.h>
#include <TimeScaleBase.h>

#include <map>
#include <set>
#include <vector>
#include <assert.h>
#include <mutex>
#include <memory>
#include <cstdlib>
#include <unordered_set>
#include <regex>

// In order to make the overrides/new dll functions work, we need to make a special definition of this type
typedef void* PPI_ProbeBundleHandle;

/// @brief the white box view of a trigger/response; the client is blissfully unaware of this
struct PPI_InterfaceTriggerResponse
{
private:
    bool triggerOnAssert;
    PPI_Pins_TypeEncode trigger;
    bool respondWithAssert;
    PPI_Pins_TypeEncode response;
    bool observable; /// Indicates if this trigger is observable
    bool observed; /// Indicates if this trigger is currently being observed
    bool isSet;
    uint32_t identifier;
public:
    /// @brief A constructor to save silly bugs later!
    PPI_InterfaceTriggerResponse(uint32_t identifier, bool triggerOnAssert,
                                        PPI_Pins_TypeEncode trigger, bool respondWithAssert,
                                        PPI_Pins_TypeEncode response, bool observable) :
    triggerOnAssert(triggerOnAssert),
        trigger(trigger),
        respondWithAssert(respondWithAssert),
        response(response),
        observable(observable),
        observed(false),
        isSet(false),
        identifier(identifier)
    {
    }

    virtual ~PPI_InterfaceTriggerResponse(){}

    // Getters so that we can override this in plugins if needed

    virtual OpenIPC_Error GetObservable(bool& observable)  const;
    virtual OpenIPC_Error GetObserved(bool& observed)  const;
    virtual OpenIPC_Error SetObserved(bool observed);
    virtual OpenIPC_Error GetIsEnabled(bool& enabled)  const;
    virtual OpenIPC_Error SetIsEnabled(bool enabled);

    virtual OpenIPC_Error GetTriggerInfo(PPI_Pins_TypeEncode& pin, bool& triggerOnAssert) const;
    virtual OpenIPC_Error GetResponseInfo(PPI_Pins_TypeEncode& pin, bool& respondWithAssert)  const;
    virtual OpenIPC_Error GetIdentifier(uint32_t& identifier) const;
};

/// @brief the white box view of an action; this will likely need to be subclassed per project for utility
struct PPI_Action {
protected:
    std::string _name;
    std::string _description;
	std::string _input;
	bool _isVisible;

    static void _ParseParameter(std::string input, size_t count, std::string& param);
    static void _ParseParameter(std::string input, size_t count, uint32_t& param);
    template<typename T, typename... Args>
    static void _IterateParameters(std::sregex_token_iterator iter, size_t count, T& param, Args&... rest)
    {
        static std::sregex_token_iterator end;
        if (iter == end)
        {
            PPI_EXCEPT_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, OpenIPC_Error_Invalid_Argument, "Could not parse parameter " << count << " of the input, because there are only " << (count - 1) << " parameters.");
        }
        _ParseParameter(*iter, count, param);
        _IterateParameters(++iter, count + 1, rest...);
    }
    inline static void _IterateParameters(std::sregex_token_iterator iter, size_t count)
    {
        static std::sregex_token_iterator end;
        if (iter != end)
        {
            PPI_EXCEPT_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, OpenIPC_Error_Invalid_Argument, "Found parameter " << count << " of the input (\"" << *iter << "\"), but there should only be " << (count - 1) << " parameters.");
        }
    }
public:
    PPI_Action(const std::string& name, const std::string& description, bool isVisible);
    const std::string& Name() const;
    const std::string& GetDescription() const;
    virtual std::string Run() const;
	virtual void SetParameter(const std::string& input);
	bool IsVisible() const;

    template<typename ...Args>
    void ParseParameters(Args&... args) const
    {
        //static std::regex delimiter("^\\s*|\\s*,\\s*|\\s*$");
        static std::regex delimiter(",");
        _IterateParameters(std::sregex_token_iterator{_input.begin(), _input.end(), delimiter, -1}, 1, args...);
    }
	virtual ~PPI_Action() {}
};

struct PPI_InterfaceStatePortOperation
{
	virtual ~PPI_InterfaceStatePortOperation()
	{
	}

	virtual OpenIPC_Error SetParameter(const PPI_InterfaceStatePortParameter* parameter, const uint8_t* value, uint32_t size) = 0;
	virtual OpenIPC_Error SetWriteValue(const uint8_t* buffer, uint32_t size) = 0;
	virtual OpenIPC_Error SetReadBuffer(uint8_t* buffer, uint32_t size) = 0;
	virtual OpenIPC_Error SetErrorBuffer(ProbeStatePortError* errorBuffer) = 0;
};

struct PPI_InterfaceStatePortDefinition
{
	std::string name;
	std::string description;
	PPI_StatePort_OPERATION_TYPE_ET capabilities;
	uint32_t accessSize;
	std::vector<PPI_InterfaceStatePortParameter const*> parameters;

	inline PPI_InterfaceStatePortDefinition(const std::string& name, const std::string& description, PPI_StatePort_OPERATION_TYPE_ET capabilities, uint32_t accessSize, std::vector<PPI_InterfaceStatePortParameter const*>&& parameters)
		:name(name)
		,description(description)
		,capabilities(capabilities)
		,accessSize(accessSize)
		,parameters(std::move(parameters))
	{
	}

	virtual ~PPI_InterfaceStatePortDefinition()
	{
	}

	virtual OpenIPC_Error ErrorGetName(ProbeStatePortError error, char* buffer, uint32_t bufferSize, uint32_t* realSize) const = 0;
	virtual OpenIPC_Error ErrorGetDescription(ProbeStatePortError error, char* buffer, uint32_t bufferSize, uint32_t* realSize) const = 0;
	virtual OpenIPC_Error ErrorGetOpenIpcError(ProbeStatePortError error, OpenIPC_Error* openIpcError) const = 0;
};

struct PPI_InterfaceStatePortParameter
{
	uint32_t index;
	std::string name;
	std::string description;
	uint32_t size;
	bool optional;
	uint8_t const* pDefaultValue;
};

/**
* @brief The maximum number of JTAG chains.
*/
#define MAX_JTAG_CHAINS 64
#define MAX_INTERFACE_CHAINS  32

/// @brief constant string for the triggerScanExitOnTimeout interface parameter
const std::string TriggerScanExitOnTimeout = "Jtag.TriggerScanExitOnTimeout";


struct DeviceSetting
{
    DeviceSetting() = default;
    explicit DeviceSetting(std::string value) : Value(std::move(value)) {}
    DeviceSetting(std::string value, std::string description) : Value(std::move(value)), Description(std::move(description)) {}
    DeviceSetting(std::string value, std::string description, std::function<bool(const DeviceSetting&, const std::string&)> onSet) : Value(std::move(value)), Description(std::move(description)), OnSet(std::move(onSet)) {}
    DeviceSetting(std::string value, std::string description, std::unordered_set<std::string> options) : Value(std::move(value)), Description(std::move(description)), Options(std::move(options)) {}
    DeviceSetting(std::string value, std::string description, std::unordered_set<std::string> options, std::function<bool(const DeviceSetting&, const std::string&)> onSet) : Value(std::move(value)), Description(std::move(description)), Options(std::move(options)), OnSet(std::move(onSet)) {}
    std::string Value;
    std::string Description;
    std::unordered_set<std::string> Options;

    // When provided, this should be called before the value is set, and if it returns false, the value should not be set
    std::function<bool(const DeviceSetting&, const std::string&)> OnSet;

    static const std::unordered_set<std::string> BoolOptions;
};


/**
* @brief Class for storing/retrieving Inteface settings key/value pairs
*/
class DeviceSettings
{
    typedef std::map<std::string, DeviceSetting> Map;//! Typedef for the map we will be using
    typedef std::vector<std::unique_ptr<std::string>> Keys;
    Keys keys;
    Map _map;//! The map that stores all the settings key/value pairs.
    mutable std::recursive_mutex lock; //! A lock to ensure that we don't have any race conditions accessing the data

    void AddKeyIfNotExist(const std::string& key);
    void RemoveKeyIfExists(const std::string& key);

public:
    /**
    * If value is not empty, it is converted to bool and returned.
    * Valid values are 'true', 'True', 'TRUE', and '1' for true, or 'false', 'False', 'FALSE', and '0' for false.
    * If value is invalid, std::invalid_argument is thrown.
    */
    static bool ParseBool(std::string const& value);
    /**
    * If value is not empty, it is converted to double and returned.
    * Valid values are any base 10 floating point value with no whitespace.
    * If value is invalid, std::invalid_argument or std::out_of_range is thrown.
    */
    static double ParseDouble(std::string const& value);
    /**
    * If value is not empty, it is converted to int and returned.
    * Valid values are any base 10 integer value with no whitespace.
    * If value is invalid, std::invalid_argument or std::out_of_range is thrown.
    */
    static int ParseInt(std::string const& value);
    /**
    * If value is not empty, it is converted to a uint32_t and returned.
    * Valid values are any positive base 10 integer value with no whitespace.
    * If value is invalid, std::invalid_argument or std::out_of_range is thrown.
    */
    static uint32_t ParseUint32(std::string const& value);
    /**
    * If value is not empty, it is split (whitespace delimited) into a list of strings, which is returned.
    */
    static std::vector<std::string> ParseList(std::string const& value);
    /**
    * If value is not empty, it is converted to type T and assigned to dest, and true is returned.
    * If an std::invalid_argument or std::out_of_range was thrown during the conversion, it will be caught.
    * In this case dest is left untouched, and false is returned.
    */
    template<typename T>
    static bool TryParseT(T(*convert)(std::string const&), std::string const& value, T& dest)
    {
        if (convert == nullptr)
        {
            return false;
        }
        try
        {
            dest = convert(value);
            return true;
        }
        catch (std::invalid_argument)
        {
            return false;
        }
        catch (std::out_of_range)
        {
            return false;
        }
    }

    /**
    * If value is not empty, it is converted to bool and assigned to dest, and true is returned.
    * Valid values are 'true', 'True', 'TRUE', and '1' for true, or 'false', 'False', 'FALSE', and '0' for false.
    * If value is invalid, dest is left untouched, and false is returned.
    */
    static bool TryParseBool(std::string const& value, bool& dest);
    /**
    * If value is not empty, it is converted to double and assigned to dest, and true is returned.
    * Valid values are any base 10 floating point value with no whitespace.
    * If value is invalid, dest is left untouched, and false is returned.
    */
    static bool TryParseDouble(std::string const& value, double& dest);
    /**
    * If value is not empty, it is converted to int and assigned to dest, and true is returned.
    * Valid values are any base 10 integer value with no whitespace.
    * If value is invalid, dest is left untouched, and false is returned.
    */
    static bool TryParseInt(std::string const& value, int& dest);
    /**
    * If value is not empty, it is converted to a uint32_t and assigned to dest, and true is returned.
    * Valid values are any positive base 10 integer value with no whitespace.
    * If value is invalid, dest is left untouched, and false is returned.
    */
    static bool TryParseUint32(std::string const& value, uint32_t& dest);
    /**
    * If value is not empty, it is split (whitespace delimited) into a list of strings, which is assigned to dest.
    */
    static bool TryParseList(std::string const& value, std::vector<std::string>& dest);

    template<typename T>
    static bool TryParseListT(T(*convert)(std::string const&), std::string const& value, std::vector<T>& dest)
    {
        if (convert == nullptr)
        {
            return false;
        }
        std::vector<std::string> values = ParseList(value);
        std::vector<T> temp(values.size());
        for (size_t i = 0; i < values.size(); ++i)
        {
            if (!TryParseT<T>(convert, values[i], temp[i]))
            {
                return false;
            }
        }
        dest = std::move(temp);
        return true;
    }

    static std::string const EmptyValue;

    /**
    * Calls begin on the underlying map.
    */
    Keys::iterator keyBegin();

    /**
    * Calls end on the underlying map.
    */
    Keys::iterator keyEnd();

    /**
    * Calls begin on the underlying map.
    */
    Keys::const_iterator keyBegin() const;

    /**
    * Calls cbegin on the underlying map.
    */
    Keys::const_iterator keyCbegin() const;

    /**
    * Calls end on the underlying map.
    */
    Keys::const_iterator keyEnd() const;

    /**
    * Calls cend on the underlying map.
    */
    Keys::const_iterator keyCend() const;


	/**
	* Just default constructs the underlying map, nothing special.
	*/
	DeviceSettings();

	/**
	* Loads values from begin to end into the map, overwriting any duplicate keys.
	*/
	void Load(Map::const_iterator const& begin, Map::const_iterator const& end);

	/**
	* Sets an entry to a new value, or creates a new entry if one didn't exist.
	*/
	void Set(std::string const& key, std::string value);

    /**
    * Sets an entry with a new setting, or creates a new entry if one didn't exist.
    */
    void Add(std::string const& key, DeviceSetting value);

	/**
	* Removes an entry, if it exists.
	*/
	void Remove(std::string const& key);

	/**
	* If key is in the map, it's value is assigned to dest, and true is returned.
	* If key isn't in the map, dest is left untouched, and false is returned.
	*/
	bool TryGetString(std::string const& key, std::string& dest) const;
	/**
	* If key is in the map, it's value is converted to bool and assigned to dest, and true is returned.
	* Valid values are 'true', 'True', 'TRUE', and '1' for true, or 'false', 'False', 'FALSE', and '0' for false.
	* If value is invalid, dest is left untouched, and false is returned.
	* If key isn't in the map, dest is left untouched, and false is returned.
	*/
	bool TryGetBool(std::string const& key, bool& dest) const;
	/**
	* If key is in the map, it's value is converted to double and assigned to dest, and true is returned.
	* Valid values are any base 10 floating point value with no whitespace.
	* If value is invalid, dest is left untouched, and false is returned.
	* If key isn't in the map, dest is left untouched, and false is returned.
	*/
	bool TryGetDouble(std::string const& key, double& dest) const;
	/**
	* If key is in the map, it's value is converted to int and assigned to dest, and true is returned.
	* Valid values are any base 10 integer value with no whitespace.
	* If value is invalid, dest is left untouched, and false is returned.
	* If key isn't in the map, dest is left untouched, and false is returned.
	*/
	bool TryGetInt(std::string const& key, int& dest) const;
	/**
	* If key is in the map, it's value is converted to a uint32_t and assigned to dest, and true is returned.
	* Valid values are any positive base 10 integer value with no whitespace.
	* If value is invalid, dest is left untouched, and false is returned.
	* If key isn't in the map, dest is left untouched, and false is returned.
	*/
	bool TryGetUint32(std::string const& key, uint32_t& dest) const;

    /**
    * If key is in the map, it's value is split (whitespace delimited) into a list of strings, which is assigned to dest.
    * If key isn't in the map, dest is left untouched, and false is returned.
    */
    bool TryGetStringList(std::string const& key, std::vector<std::string>& dest) const;
    /**
    * If key is in the map, it's value is split (whitespace delimited) into a list of strings,
    * which are converted to bool and assigned to dest, and true is returned.
    * Valid values are 'true', 'True', 'TRUE', and '1' for true, or 'false', 'False', 'FALSE', and '0' for false.
    * If value is invalid, dest is left untouched, and false is returned.
    * If key isn't in the map, dest is left untouched, and false is returned.
    */
    bool TryGetBoolList(std::string const& key, std::vector<bool>& dest) const;
    /**
    * If key is in the map, it's value is split (whitespace delimited) into a list of strings,
    * which are converted to double and assigned to dest, and true is returned.
    * Valid values are any base 10 floating point value with no whitespace.
    * If value is invalid, dest is left untouched, and false is returned.
    * If key isn't in the map, dest is left untouched, and false is returned.
    */
    bool TryGetDoubleList(std::string const& key, std::vector<double>& dest) const;
    /**
    * If key is in the map, it's value is split (whitespace delimited) into a list of strings,
    * which are converted to int and assigned to dest, and true is returned.
    * Valid values are any base 10 integer value with no whitespace.
    * If value is invalid, dest is left untouched, and false is returned.
    * If key isn't in the map, dest is left untouched, and false is returned.
    */
    bool TryGetIntList(std::string const& key, std::vector<int>& dest) const;
    /**
    * If key is in the map, it's value is split (whitespace delimited) into a list of strings,
    * which are converted to a uint32_t and assigned to dest, and true is returned.
    * Valid values are any positive base 10 integer value with no whitespace.
    * If value is invalid, dest is left untouched, and false is returned.
    * If key isn't in the map, dest is left untouched, and false is returned.
    */
    bool TryGetUint32List(std::string const& key, std::vector<uint32_t>& dest) const;

	/**
	* If key is in the map, it's value is returned.
	* If key isn't in the map, an empty string is returned.
	*/
	const std::string& GetString(std::string const& key) const;
	/**
	* If key is in the map, it's value is converted to bool, and that is returned.
	* Valid values are 'true', 'True', 'TRUE', and '1' for true, or 'false', 'False', 'FALSE', and '0' for false.
	* If value is invalid, false is returned.
	* If key isn't in the map, false is returned.
	*/
	bool GetBool(std::string const& key, bool defaultValue=false) const;
	/**
	* If key is in the map, it's value is converted to double, and that is returned.
	* Valid values are any base 10 floating point value with no whitespace.
	* If value is invalid, 0.0 is returned.
	* If key isn't in the map, 0.0 is returned.
	*/
	double GetDouble(std::string const& key, double defaultValue=0.0) const;
	/**
	* If key is in the map, it's value is converted to int, and that is returned.
	* Valid values are any base 10 integer value with no whitespace.
	* If value is invalid, 0 is returned.
	* If key isn't in the map, 0 is returned.
	*/
	int GetInt(std::string const& key, int defaultValue=0) const;
	/**
	* If key is in the map, it's value is converted to a uint32_t, and that is returned.
	* Valid values are any positive base 10 integer value with no whitespace.
	* If value is invalid, 0 is returned.
	* If key isn't in the map, 0 is returned.
	*/
	uint32_t GetUint32(std::string const& key, uint32_t defaultValue=0) const;
    /**
    * If key is in the map, it's value is split (whitespace delimited) into a list of strings, which is returned.
    * If key isn't in the map, an empty list is returned.
    */
    std::vector<std::string> GetStringList(std::string const& key, std::vector<std::string> defaultValue = {}) const;
    /**
    * If key is in the map, it's value is split (whitespace delimited) into a list of strings, which are converted to bool, and that is returned.
    * Valid values are 'true', 'True', 'TRUE', and '1' for true, or 'false', 'False', 'FALSE', and '0' for false.
    * If value is invalid, false is returned.
    * If key isn't in the map, false is returned.
    */
    std::vector<bool> GetBoolList(std::string const& key, std::vector<bool> defaultValue = {}) const;
    /**
    * If key is in the map, it's value is split (whitespace delimited) into a list of strings, which are converted to double, and that is returned.
    * Valid values are any base 10 floating point value with no whitespace.
    * If value is invalid, 0.0 is returned.
    * If key isn't in the map, 0.0 is returned.
    */
    std::vector<double> GetDoubleList(std::string const& key, std::vector<double> defaultValue = {}) const;
    /**
    * If key is in the map, it's value is split (whitespace delimited) into a list of strings, which are converted to int, and that is returned.
    * Valid values are any base 10 integer value with no whitespace.
    * If value is invalid, 0 is returned.
    * If key isn't in the map, 0 is returned.
    */
    std::vector<int> GetIntList(std::string const& key, std::vector<int> defaultValue = {}) const;
    /**
    * If key is in the map, it's value is split (whitespace delimited) into a list of strings, which are converted to a uint32_t, and that is returned.
    * Valid values are any positive base 10 integer value with no whitespace.
    * If value is invalid, 0 is returned.
    * If key isn't in the map, 0 is returned.
    */
    std::vector<uint32_t> GetUint32List(std::string const& key, std::vector<uint32_t> defaultValue = {}) const;

	/**
	* Calls operator[] on the underlying map.
	*/
	std::string& operator[](std::string const& key);

	/**
	* Calls find on the underlying map.
	*/
	Map::iterator find(std::string const& key);

	/**
	* Calls erase on the underlying map.
	*/
	void erase(std::string const& key);

    /**
    * Calls clear on the underlying map.
    */
    void clear();

	/**
	* Calls size on the underlying map.
	*/
	size_t size();

	/**
	* Calls begin on the underlying map.
	*/
	Map::iterator begin();

	/**
	* Calls end on the underlying map.
	*/
	Map::iterator end();

	/**
	* Calls begin on the underlying map.
	*/
	Map::const_iterator begin() const;

	/**
	* Calls cbegin on the underlying map.
	*/
	Map::const_iterator cbegin() const;

	/**
	* Calls end on the underlying map.
	*/
	Map::const_iterator end() const;

	/**
	* Calls cend on the underlying map.
	*/
	Map::const_iterator cend() const;
};



class InterfaceInstance;
struct IInterfaceBundle
{
	OpenIPC_DeviceId const interfaceDeviceId;
	IInterfaceBundle(OpenIPC_DeviceId interfaceRefId);
	virtual ~IInterfaceBundle();

	// Returns true iff this == &rhs
	bool operator==(IInterfaceBundle const& rhs);
	// Returns true iff this != &rhs
	bool operator!=(IInterfaceBundle const& rhs);

	virtual OpenIPC_Error Append(IInterfaceBundle& source, uint8_t* outputBuffer, uint32_t outputBufferSize) = 0;
	static IInterfaceBundle& Default;
	static IInterfaceBundle& ExecuteImmediately;
};

class ProbePluginBundle
{
private:
	std::unique_ptr<IInterfaceBundle> _interfaceBundle;
public:
	ProbePluginBundle();
	virtual ~ProbePluginBundle();
	virtual OpenIPC_Error Clear();
	virtual OpenIPC_Error Append(ProbePluginBundle& source, uint8_t* outputBuffer, uint32_t outputBufferSize);
	IInterfaceBundle& InterfaceBundle();
	void InterfaceBundle(std::unique_ptr<IInterfaceBundle>&& bundle);
};

enum class InitializationState
{
    Uninitialized, Initalizing, Initialized
};

/**
* @brief Class to hold information regarding an interface.
*/
class InterfaceInstance
{
public:
	const uint32_t	refid;					//! plugin's unique ID of the interface
	const PPI_EInterfaceType	interfaceType;	//! JTAG, I2C, etc.
	const uint32_t	instanceid;				//! generic instance ID to distinguish between say JTAG TCK0 vs TCK1, or perhaps external vs internal I2C
	OpenIPC_DeviceId		deviceID;		//! the assigned logical ID for this interface (-1 before init)
	bool					lock;			//! if a lock (hold) needs to be asserted with the probe for this interface
	class ProbeInstance*	probe;			//! back-reference for probe that owns interface
	//std::map<std::string, std::string> settings;
    InitializationState initializationState;
	DeviceSettings settings;
    std::vector<std::unique_ptr<PPI_Action>> actions;

	virtual void SettingsUpdated() {}		//! Function to indicate that the interface settings have been updated; this is for common items like leadingTCKs for all JTAG interfaces
    mutable std::recursive_mutex _instanceMutex;      //! Lock on accessing data members

	virtual OpenIPC_Error OperationCancel();

	virtual std::unique_ptr<IInterfaceBundle> BundleAllocate();
	virtual OpenIPC_Error BundleExecute(IInterfaceBundle& bundle, bool keepLock);
protected:
	InterfaceInstance(uint32_t interface_refid, PPI_EInterfaceType interface_type, uint32_t instance_id);
    virtual ~InterfaceInstance();
private:
	// Copy constructor and assignment operator are private
	// as we don't allow classes derived from this class to
	// copied.
	InterfaceInstance(const InterfaceInstance& e);
	InterfaceInstance& operator=(const InterfaceInstance& e);
};

/**
* @brief Class to hold information regarding a JTAG interface.
*/
class InterfaceInstanceJtag : public InterfaceInstance
{
public:

	bool					jtagActive;		// TBD: should this be generalized for all interface types?
	JtagStateEncode			lastjtagstate;	// the memory of the current TSM state of the JTAG chain
	PPI_InterfaceJTAGCapabilities	capabilities;			// a bit vector of capabilities supported (just set to zero for now)
	uint32_t irPaddingNearTDI, irPaddingNearTDO, drPaddingNearTDI, drPaddingNearTDO;
    PPI_bool drValueConstantOne;
	// TBD: add JTAG config knobs to the settings map

	InterfaceInstanceJtag(uint32_t interface_refid, uint32_t instance_id, PPI_InterfaceJTAGCapabilities caps) :
		InterfaceInstance(interface_refid, PPI_interfaceTypeJtag, instance_id), jtagActive(false), lastjtagstate(JtagStateEncode::JtagTLR), capabilities(caps),
        irPaddingNearTDI(0), irPaddingNearTDO(0), drPaddingNearTDI(0),drPaddingNearTDO(0), drValueConstantOne(0){
			settings.Set("Jtag.LeadingTCKs", "10");
			settings.Set("Jtag.TrailingTCKs", "10");
			settings.Set("Jtag.TriggerScanExitOnTimeout", "false");
	}
	virtual void SettingsUpdated() override;

    virtual std::unique_ptr<IInterfaceBundle> GetNewBundle(void);
};

/**
* @brief Class to hold information regarding an I2C interface.
*/
class InterfaceInstanceI2c : public InterfaceInstance
{
public:

	// TBD: add I2C config knobs here
	PPI_InterfaceI2CCapabilities	capabilities;
	InterfaceInstanceI2c(uint32_t interface_refid, uint32_t instance_id, PPI_InterfaceI2CCapabilities caps) :
		InterfaceInstance(interface_refid, PPI_interfaceTypeI2c, instance_id), capabilities(caps){}
};

/**
* @brief Class to hold information regarding a pins interface.
*/
class InterfaceInstancePins : public InterfaceInstance
{
public:

	// TBD: add pin/hook config knobs here
	PPI_InterfacePinsCapabilities	capabilities;
	InterfaceInstancePins(uint32_t interface_refid, uint32_t instance_id, PPI_InterfacePinsCapabilities caps) :
		InterfaceInstance(interface_refid, PPI_interfaceTypePins, instance_id), capabilities(caps){
            std::shared_ptr<PPI_InterfaceTriggerResponse> testTriggerResponse(new PPI_InterfaceTriggerResponse((uint32_t)(-1),false,PPI_PINS_NO_ACTION,false,PPI_PINS_NO_ACTION,false));
            this->_triggerResponsesSupported.push_back(testTriggerResponse);
            // This is purely for testing; this trigger has no way to occur and the response cannot cause an action
    }

    /// @note These functions for getters/setters are avaliable to be overridded by subclasses if this state
    ///       is owned by something else within a particular probe implementation.
    virtual OpenIPC_Error ListSupportedTriggerResponses(uint32_t length, PPI_InterfaceTriggerResponse** responses,  uint32_t* triggerResponsesPopulated) const;
    virtual OpenIPC_Error ListObservableTriggerResponses(uint32_t length, PPI_InterfaceTriggerResponse** responses, uint32_t* triggerResponsesPopulated) const;
    virtual OpenIPC_Error TriggerResponseIsSet(const PPI_InterfaceTriggerResponse* triggerResponse, PPI_bool* triggerResponseSet, PPI_TriggerResponseSetOptions* optionsSet) const;
    virtual OpenIPC_Error TriggerResponseSet(PPI_InterfaceTriggerResponse* triggerResponse, PPI_bool triggerResponseSet, const PPI_TriggerResponseSetOptions* optionsSet);

protected:
    std::vector<std::shared_ptr<PPI_InterfaceTriggerResponse>> _triggerResponsesSupported;  //! a list of the supported trigger responses; actionable == true for all items in here
};

/**
* @brief Class to hold information regarding an trace interface.
*/
class InterfaceInstancePort : public InterfaceInstance
{
public:
	bool _isWindowOpen;

	PPI_Trace_PortAccessMode _accessMode;
	PPI_InterfaceTraceCapabilities capabilities;

	InterfaceInstancePort(uint32_t interface_refid, PPI_EInterfaceType interface_type, uint32_t instance_id, PPI_InterfaceTraceCapabilities caps) :
		InterfaceInstance(interface_refid, interface_type, instance_id),
		_isWindowOpen(false),
		_accessMode(PPI_Trace_PortAccessRead),
		capabilities(caps)
	{
	}
};

/**
* @brief Class to hold information regarding a stateport interface.
*/
class InterfaceInstanceStatePort : public InterfaceInstance
{
protected:
	std::vector<PPI_InterfaceStatePortDefinition const*> definitions;
public:
	InterfaceInstanceStatePort(uint32_t interface_refid, uint32_t instance_id, std::vector<PPI_InterfaceStatePortDefinition const*>&& definitions);
	virtual ~InterfaceInstanceStatePort();


	virtual OpenIPC_Error AllocateOperation(const PPI_InterfaceStatePortDefinition* definition, PPI_StatePort_OPERATION_TYPE_ET operationType, PPI_InterfaceStatePortOperation** operation, const PPI_StatePortAllocateOperation_Options* options) = 0;
	virtual OpenIPC_Error FreeOperation(PPI_InterfaceStatePortOperation** operation) = 0;
	virtual OpenIPC_Error AppendOperation(IInterfaceBundle& bundle, PPI_InterfaceStatePortOperation** operationPointer, const PPI_StatePortAppendOperation_Options* options) = 0;
	virtual OpenIPC_Error GetDefinitions(const PPI_InterfaceStatePortDefinition** definitions, uint32_t definitionsSize, uint32_t* numberOfDefinitions);
    virtual OpenIPC_Error GetValueChangedEventDescription(PPI_InterfaceStatePortValueChangedEventDescription const** eventDescriptionPtr) = 0;
};

/**
* @brief A class to hold the information regarding a single probe.
*/
class ProbeInstance
{
	std::map<uint32_t, std::weak_ptr<InterfaceInstance>> _refid2interface;

	// Mutex used to protect members for multiple simultaneous access by different threads
	std::mutex _refid2interfaceMutex;
public:
	const uint32_t	refid;
	const uint32_t	typid;
	std::string	typestring;

	OpenIPC_DeviceId		deviceID;
    InitializationState initializationState;
    DeviceSettings settings; // Probe specific settings
    std::vector<std::unique_ptr<PPI_Action>> actions;
	ProbeInstance(uint32_t probe_refid, std::string typestring, uint32_t typid = 0);

	virtual ~ProbeInstance(){};
	virtual OpenIPC_Error AddInterface(std::weak_ptr<InterfaceInstance> probeinterface);
    //By default, this returns OpenIPC_Error_Not_Implemented.  If your probe plugin implements its own _GetProbeInterfaces, then you don't need to implement GetInterfaces on the probe instance.
    virtual OpenIPC_Error GetInterfaces(std::vector<std::weak_ptr<InterfaceInstance>>& interfaces);
	virtual OpenIPC_Error GetInterfaceRefids(std::vector<uint32_t>& refids);
	virtual OpenIPC_Error GetInterfaceFromRefid(uint32_t interfacerefid, std::weak_ptr<InterfaceInstance>& probeinterface);
    virtual void SettingsUpdated() {}		//! Function to indicate that the probe settings have been updated; this is for common items like leadingTCKs for all JTAG interfaces
};

class ProbePlugin_TimeScaler : public TimeScalerBase
{
    DeviceSettings& _settings;
public:
    ProbePlugin_TimeScaler(DeviceSettings& settings)
        :_settings(settings)
    {
    }
private:
    uint32_t GetFactorPercentage() const override
    {
        uint32_t scaleFactor = 100;
        _settings.TryGetUint32("TimeScale", scaleFactor);
        if (scaleFactor > 1000000)
        {
            // If the scale factor is too large (1 second would be over 2 hours)
            //   the setting is probaly wrong, so just use realtime
            return 100;
        }
        return scaleFactor;
    }
};

/**
* @brief a class to hold information regarding a plugin.
*/
class CProbePlugin
{
public:
	virtual OpenIPC_Error PluginGetInfo(PPI_PluginApiVersion clientInterfaceVersion, PPI_PluginInfo *info);
	virtual OpenIPC_Error PluginInitialize(PPI_RefId pluginid, const char* vendorinit);
	virtual OpenIPC_Error PluginDeinitialize(void);
	virtual OpenIPC_Error PluginGetProbeTypes(uint32_t maxProbeTypes, PPI_char const** probeTypes, uint32_t* probeTypeCount);
	virtual OpenIPC_Error PluginCreateStaticProbe(const PPI_char probeType[PPI_MAX_PROBE_TYPE_LEN], PPI_RefId* refId);

	virtual OpenIPC_Error ProbeGetRefIds(uint32_t maxIds, PPI_RefId* refIds, uint32_t* probecount);
	virtual OpenIPC_Error ProbeGetInfo(PPI_RefId refId, PPI_ProbeInfo* info);
    virtual OpenIPC_Error ProbeBeginInitialization(PPI_RefId probe_refid, OpenIPC_DeviceId probeID);
    virtual OpenIPC_Error ProbeSpecifyVendorInitString(OpenIPC_DeviceId probeID, const PPI_char* vendorinit);
    virtual OpenIPC_Error ProbeFinishInitialization(OpenIPC_DeviceId probeID);
	virtual OpenIPC_Error ProbeInitialize(PPI_RefId probe_refid, OpenIPC_DeviceId probeID, const char* vendorinit);
	virtual OpenIPC_Error ProbeDeInitialize(OpenIPC_DeviceId probeID);

	virtual OpenIPC_Error InterfaceGetRefIds(OpenIPC_DeviceId probeID, uint8_t maxIds, PPI_RefId* refIds, uint32_t* interfacecount);
	virtual OpenIPC_Error InterfaceGetType(OpenIPC_DeviceId probeID, PPI_RefId interface_refid, PPI_EInterfaceType* interface_type);
	virtual OpenIPC_Error InterfaceGetInfoJTAG(OpenIPC_DeviceId probeID, PPI_RefId interface_refid, PPI_InterfaceJTAGCapabilities* capabilities);
	virtual OpenIPC_Error InterfaceGetInfoPins(OpenIPC_DeviceId probeID, PPI_RefId interface_refid, PPI_InterfacePinsCapabilities* capabilities);
	virtual OpenIPC_Error InterfaceGetInfoI2C(OpenIPC_DeviceId probeID, PPI_RefId interface_refid, PPI_InterfaceI2CCapabilities* capabilities);
    virtual OpenIPC_Error InterfaceBeginInitialization(OpenIPC_DeviceId probeID, PPI_RefId interface_refid, OpenIPC_DeviceId interfaceID);
    virtual OpenIPC_Error InterfaceSpecifyVendorInitString(OpenIPC_DeviceId interfaceID, const PPI_char* vendorinit);
    virtual OpenIPC_Error InterfaceFinishInitialization(OpenIPC_DeviceId interfaceID);
	virtual OpenIPC_Error InterfaceInitialize(OpenIPC_DeviceId probeID, uint32_t interface_refid, OpenIPC_DeviceId interfaceID, const char* vendorinit);
	virtual OpenIPC_Error InterfaceDeInitialize(OpenIPC_DeviceId interfaceID);
	virtual OpenIPC_Error InterfaceScan(OpenIPC_DeviceId interfaceID, const uint32_t* input, uint32_t inputdwords, uint32_t* output, uint32_t maxoutputdwords, uint32_t* outputdwords);
	virtual OpenIPC_Error InterfaceListConfigString(OpenIPC_DeviceId interfaceID, uint32_t configTypeLength, char const ** configType, uint32_t* numberOfConfigTypes);
	virtual OpenIPC_Error InterfaceOperationCancel(OpenIPC_DeviceId interfaceID);
	virtual OpenIPC_Error InterfaceSetConfig(OpenIPC_DeviceId interfaceID, const char* configType, const PPI_char value[PPI_MAX_INFO_LEN]);
	virtual OpenIPC_Error InterfaceGetConfig(OpenIPC_DeviceId interfaceID, const char* configType, PPI_char value[PPI_MAX_INFO_LEN]);
	virtual OpenIPC_Error InterfacePortRead(OpenIPC_DeviceId interfaceID, uint8_t* output, uint32_t maxoutputbytes, uint32_t* outputbytes);
	virtual OpenIPC_Error InterfacePortWrite(OpenIPC_DeviceId interfaceID, const uint8_t* input, uint32_t maxinputbytes, uint32_t* inputbytes);
	virtual OpenIPC_Error InterfacePortOpenWindow(OpenIPC_DeviceId interfaceID, PPI_Trace_PortAccessMode accessMode);
	virtual OpenIPC_Error InterfacePortCloseWindow(OpenIPC_DeviceId interfaceID);
	virtual OpenIPC_Error InterfacePortIsReadDataAvailable(OpenIPC_DeviceId interfaceID, bool* isDataAvailable);
	virtual OpenIPC_Error InterfacePortIsWindowOpen(OpenIPC_DeviceId interfaceID, bool* isWindowOpen);

	virtual OpenIPC_Error PluginRegisterEventHandler(PluginEventCallbackHandler eventHandlerFunction);
	virtual OpenIPC_Error PluginRegisterNotificationHandler(PluginNotificationCallbackHandler notificationHandlerFunction);
    virtual OpenIPC_Error PluginSetLogEventHandler(PluginLogCallbackHandler logHandlerFunction);

    ///@brief Should be called to post any new error messages by framework/plugin-specific code.
	virtual OpenIPC_Error PostError(OpenIPC_Error errorNumber, const char* errorMessage);
	virtual const char* GetLastErrorMessage(OpenIPC_Error errorNumber);

    /// @brief This framework implementation simply appends this device to a set of locked interfaces, this should be overriden by clients (though this implementation should be called  at the end of the override to setup the convention for other methods to work properly unless overriding all of the related methods!).
    virtual OpenIPC_Error LockTargetInterface(OpenIPC_DeviceId deviceInterface);
    /// @brief This framework implementation uses the convention established earlier; it is unlikely that this will need to be modified.
    virtual OpenIPC_Error ListLockedInterfaces(uint32_t maxNumberOfInterfaces, OpenIPC_DeviceId* interfaces, uint32_t* numberOfInterfaces);
    /// @brief this framework implementation is *safe*, e.g., we assume that locking 1 interface locks the entire probe API; if this is not the case, an implementer should override this method.
    virtual OpenIPC_Error InterfaceListLockInterfacePeers( OpenIPC_DeviceId interfaceID, uint32_t peerInterfacesLength,OpenIPC_DeviceId * peerInterfaces, uint32_t* numberOfPeerInterfaces);
	/// @brief This framework implementation creates a simple wrapper bundle that interfaces can fill with whatever they need.  If you override this to allocate some type that doesn't inherit from ProbePluginBundle, you must override BundleFree, BundleExecute, BundleClear, and BundleAppend.
	virtual PPI_ProbeBundleHandle BundleAllocate();
	/// @brief This framework implementation just deletes the bundle as if it were a ProbePluginBundle
	virtual OpenIPC_Error BundleFree(PPI_ProbeBundleHandle* handle);
    /// @brief The framework implementation will handle PPI_PROBE_LOCK_HOLD and PPI_PROBE_LOCK_RELEASE by calling _ProbeLockHold and _ProbeLockRelease. Other handles will be treated like a ProbePluginBundle* and used to invoke Execute
    virtual OpenIPC_Error BundleExecute(PPI_ProbeBundleHandle handle, OpenIPC_DeviceId deviceInterface, PPI_bool keepLock);
    /// @brief The framework implementation for multi-chain BundleExecute
    virtual OpenIPC_Error BundleExecuteMultiChain(PPI_ProbeBundleHandle handle, OpenIPC_DeviceId* deviceInterfaces, uint32_t deviceInterfacesLength, PPI_bool keepLock);
	/// @brief This framework implementation just treats handle like a ProbePluginBundle* and invokes Clear
	virtual OpenIPC_Error BundleClear(PPI_ProbeBundleHandle handle);
	/// @brief This framework implementation just treats handle like a ProbePluginBundle* and invokes Append
	virtual OpenIPC_Error BundleAppend(PPI_ProbeBundleHandle destHandle, PPI_ProbeBundleHandle sourceHandle, uint8_t* outputBuffer, uint32_t outputBufferSize);
    /// @brief The framework implementation sets the log filtering variable
    virtual OpenIPC_Error PluginSetNotificationRequiredForLogCallBack(PPI_NotificationLevel severity);
    /// @brief The framework implementation sets the notification filtering variable
    virtual OpenIPC_Error PluginSetNotificationRequiredForNotificationCallBack(PPI_NotificationLevel severity);

    virtual OpenIPC_Error PPI_TriggerGetTrigger(const PPI_InterfaceTriggerResponse* triggerResponse, PPI_Pins_TypeEncode* pin, PPI_bool* onAssert, uint32_t* identifier) const;

    virtual OpenIPC_Error PPI_TriggerGetResponse(const PPI_InterfaceTriggerResponse* triggerResponse, PPI_Pins_TypeEncode* pin, PPI_bool* respondAsAssert, uint32_t* identifier) const;

    virtual OpenIPC_Error PPI_InterfaceListTriggerResponses(OpenIPC_DeviceId device, uint32_t triggerResponseSupportedLength, PPI_InterfaceTriggerResponse** triggerResponseSupported, uint32_t* triggerResponsePopulated) const;

    virtual OpenIPC_Error PPI_InterfaceListObservableTriggerResponses(OpenIPC_DeviceId device, uint32_t triggerResponseSupportedLength, PPI_InterfaceTriggerResponse** triggerResponseSupported, uint32_t* triggerResponsePopulated) const;

    virtual OpenIPC_Error PPI_InterfaceTriggerResponseIsSet(OpenIPC_DeviceId device, const PPI_InterfaceTriggerResponse* triggerResponse, PPI_bool* triggerResponseSet, PPI_TriggerResponseSetOptions* optionsSet);

    virtual OpenIPC_Error PPI_InterfaceTriggerResponseSet(OpenIPC_DeviceId device, PPI_InterfaceTriggerResponse* triggerResponse, PPI_bool triggerResponseSet, const PPI_TriggerResponseSetOptions* optionsSet) const;

    virtual OpenIPC_Error PPI_InterfacePinsQueryAliases(OpenIPC_DeviceId device, PPI_Pins_TypeEncode pin,
        uint32_t pinListLength, PPI_Pins_TypeEncode* pinList, uint32_t* numberOfPinsWritten) const;

    virtual OpenIPC_Error PPI_InterfacePinsListDrivablePins(
        OpenIPC_DeviceId device, uint32_t pinsListLength,
        PPI_Pins_TypeEncode* pinsList, uint32_t* numberOfPinsPopulated) const;

    virtual OpenIPC_Error PPI_InterfacePinsListReadablePins(
        OpenIPC_DeviceId device, uint32_t pinsListLength,
        PPI_Pins_TypeEncode* pinsList, uint32_t* numberOfPinsPopulated) const;

    virtual OpenIPC_Error GetInterfaceJTAGPadding(OpenIPC_DeviceId device,
             uint32_t* irPaddingNearTDI,
             uint32_t* irPaddingNearTDO,
             uint32_t* drPaddingNearTDI,
             uint32_t* drPaddingNearTDO,
             PPI_bool* drValueConstantOne
             );
    virtual OpenIPC_Error SetInterfaceJTAGPadding( OpenIPC_DeviceId device,
             uint32_t irPaddingNearTDI,
             uint32_t irPaddingNearTDO,
             uint32_t drPaddingNearTDI,
             uint32_t drPaddingNearTDO,
             PPI_bool drValueConstantOne
             );

    virtual OpenIPC_Error PPI_PluginSetStreamLogEventHandler(PluginStreamLogCallbackHandler logCallBackFunction);
    /// @brief This function is likely to be overridden in plugins that use anything other than the defautl logging stream
    virtual OpenIPC_Error PPI_PluginUsesLoggerStream( PPI_Stream stream, PPI_bool* usesLoggerStream);
    virtual OpenIPC_Error PPI_PluginSetNotificationRequiredForStreamLogCallBack( PPI_Stream stream, PPI_NotificationLevel level);

    virtual OpenIPC_Error PPI_PluginEventGenerated(OpenIPC_DeviceId probe,PPI_PluginEvent pluginEvent, PPI_bool* pluginGeneratesEvent) const;

    virtual OpenIPC_Error PPI_DeviceSetConfig(OpenIPC_DeviceId deviceID,const PPI_char* configType,const PPI_char value[PPI_MAX_INFO_LEN]);
    virtual OpenIPC_Error PPI_DeviceGetConfigDescription(OpenIPC_DeviceId deviceID, const PPI_char* configType, PPI_char description[PPI_MAX_INFO_LEN]);
    virtual OpenIPC_Error PPI_DeviceListConfigValues(OpenIPC_DeviceId deviceID, const PPI_char* configType, uint32_t valuesLength, PPI_char const** values, uint32_t* numberOfValues);
    virtual OpenIPC_Error PPI_DeviceGetConfig(OpenIPC_DeviceId deviceID,const PPI_char* configType,PPI_char value[PPI_MAX_INFO_LEN]);
    virtual OpenIPC_Error PPI_DeviceListConfigString(OpenIPC_DeviceId deviceID,uint32_t configTypeLength,PPI_char const ** configType, uint32_t* numberOfConfigTypes);

    virtual OpenIPC_Error PPI_DeviceListActions(OpenIPC_DeviceId deviceID, uint32_t actionsLength, PPI_Action** actions, uint32_t* numberOfActions);
    virtual OpenIPC_Error PPI_ActionGetName (const PPI_Action* action,uint32_t valueLength, PPI_char* value);
    virtual OpenIPC_Error PPI_ActionGetDescription(const PPI_Action* action, uint32_t valueLength, PPI_char* value);
    virtual OpenIPC_Error PPI_DeviceActionRun(OpenIPC_DeviceId deviceID,const PPI_Action* action, uint32_t messageLength, PPI_char* message);
	virtual OpenIPC_Error PPI_ActionSetArgument(PPI_Action* action, const PPI_char* value);
	virtual OpenIPC_Error PPI_ActionIsVisible(const PPI_Action* action, PPI_bool* isVisible) const;

	virtual OpenIPC_Error PPI_StatePortAllocateOperation(OpenIPC_DeviceId deviceId, const PPI_InterfaceStatePortDefinition* definition, PPI_StatePort_OPERATION_TYPE_ET operationType, PPI_InterfaceStatePortOperation** operation, const PPI_StatePortAllocateOperation_Options* options);
	virtual OpenIPC_Error PPI_StatePortFreeOperation(OpenIPC_DeviceId deviceId, PPI_InterfaceStatePortOperation** operation);
	virtual OpenIPC_Error PPI_StatePortAppendOperation(OpenIPC_DeviceId deviceId, PPI_ProbeBundleHandle bundle, PPI_InterfaceStatePortOperation** operationPointer, const PPI_StatePortAppendOperation_Options* options);
	virtual OpenIPC_Error PPI_StatePortOperationSetParameter(PPI_InterfaceStatePortOperation* operation, const PPI_InterfaceStatePortParameter* parameter, const uint8_t* value, uint32_t size);
	virtual OpenIPC_Error PPI_StatePortOperationSetWriteValue(PPI_InterfaceStatePortOperation* operation, const uint8_t* buffer, uint32_t size);
	virtual OpenIPC_Error PPI_StatePortOperationSetReadBuffer(PPI_InterfaceStatePortOperation* operation, uint8_t* buffer, uint32_t size);
	virtual OpenIPC_Error PPI_StatePortOperationSetErrorBuffer(PPI_InterfaceStatePortOperation* operation, ProbeStatePortError* errorBuffer);
	virtual OpenIPC_Error PPI_StatePortGetDefinitions(OpenIPC_DeviceId deviceId, const PPI_InterfaceStatePortDefinition** definitions, uint32_t definitionsSize, uint32_t* numberOfDefinitions);
	virtual OpenIPC_Error PPI_StatePortGetName(const PPI_InterfaceStatePortDefinition* definition, char* buffer, uint32_t bufferSize, uint32_t* realSize);
	virtual OpenIPC_Error PPI_StatePortGetDescription(const PPI_InterfaceStatePortDefinition* definition, char* buffer, uint32_t bufferSize, uint32_t* realSize);
	virtual OpenIPC_Error PPI_StatePortGetCapabilites(const PPI_InterfaceStatePortDefinition* definition, PPI_StatePort_OPERATION_TYPE_ET* capabilities);
	virtual OpenIPC_Error PPI_StatePortGetAccessSize(const PPI_InterfaceStatePortDefinition* definition, uint32_t* size);
	virtual OpenIPC_Error PPI_StatePortGetParameters(const PPI_InterfaceStatePortDefinition* definition, const PPI_InterfaceStatePortParameter** parameters, uint32_t length, uint32_t* realLength);
	virtual OpenIPC_Error PPI_StatePortParameterGetName(const PPI_InterfaceStatePortParameter* parameter, char* buffer, uint32_t bufferSize, uint32_t* realSize);
	virtual OpenIPC_Error PPI_StatePortParameterGetDescription(const PPI_InterfaceStatePortParameter* parameter, char* buffer, uint32_t bufferSize, uint32_t* realSize);
	virtual OpenIPC_Error PPI_StatePortParameterGetSize(const PPI_InterfaceStatePortParameter* parameter, uint32_t* size);
	virtual OpenIPC_Error PPI_StatePortParameterGetDefaultValue(const PPI_InterfaceStatePortParameter* parameter, uint8_t* buffer, uint32_t size);
	virtual OpenIPC_Error PPI_StatePortErrorGetName(const PPI_InterfaceStatePortDefinition* definition, ProbeStatePortError error, char* buffer, uint32_t bufferSize, uint32_t* realSize);
	virtual OpenIPC_Error PPI_StatePortErrorGetDescription(const PPI_InterfaceStatePortDefinition* definition, ProbeStatePortError error, char* buffer, uint32_t bufferSize, uint32_t* realSize);
	virtual OpenIPC_Error PPI_StatePortErrorGetOpenIpcError(const PPI_InterfaceStatePortDefinition* definition, ProbeStatePortError error, OpenIPC_Error* openIpcError);
    virtual OpenIPC_Error PPI_StatePortEventGetValueChangedEventDescription(OpenIPC_DeviceId deviceId, PPI_InterfaceStatePortValueChangedEventDescription const** eventDescriptionPtr);

    virtual OpenIPC_Error PPI_JTAG_GetCurrentBundlePadding(PPI_ProbeBundleHandle bundle, int32_t* irPaddingNearTDI, int32_t* irPaddingNearTDO, int32_t* drPaddingNearTDI, int32_t* drPaddingNearTDO, PPI_bool* drValueConstantOne);
    virtual OpenIPC_Error PPI_JTAG_UpdateBundlePadding (PPI_ProbeBundleHandle bundle, int32_t irPaddingNearTDI, int32_t irPaddingNearTDO, int32_t drPaddingNearTDI, int32_t drPaddingNearTDO, PPI_bool drValueConstantOne);

    virtual OpenIPC_Error PPI_JTAG_PinsShift(PPI_ProbeBundleHandle handle, uint32_t shiftLengthBits, const uint8_t* const tdi, const uint8_t* const tms, uint8_t* tdo, const PPI_JTAG_PinsOptions* const options);
    virtual OpenIPC_Error PPI_JTAG_GoToState(PPI_ProbeBundleHandle handle,JtagStateEncode gotoState,uint32_t numberOfClocksInState,const PPI_JTAG_StateGotoOptions* const options);
    virtual OpenIPC_Error PPI_JTAG_StateIRShift(PPI_ProbeBundleHandle handle, uint32_t shiftLengthBits, const uint8_t* const inBits, uint8_t* outBits, const PPI_JTAG_StateShiftOptions* const options);
    virtual OpenIPC_Error PPI_JTAG_StateDRShift(PPI_ProbeBundleHandle handle, uint32_t shiftLengthBits, const uint8_t* const inBits, uint8_t* outBits, const PPI_JTAG_StateShiftOptions* const options);

    virtual OpenIPC_Error PPI_Bundle_InterfaceSetConfig(PPI_ProbeBundleHandle handle, const char* configType, const char* value);

    virtual OpenIPC_Error PPI_MemoryRead(PPI_ProbeBundleHandle bundle, uint64_t address, uint8_t* buffer, uint32_t* size);
    virtual OpenIPC_Error PPI_MemoryWrite(PPI_ProbeBundleHandle bundle, uint64_t address, const uint8_t* buffer, uint32_t* size);

protected:

	const char* const _pluginname;
	bool const _supportBundles;

	uint32_t _plugin_refid;

	// callback functions
	PluginEventCallbackHandler _eventHandlerFunction;

    // Note: we should migrate to using shared pointers here

	// physical enumeration of probes
	std::map<uint32_t, std::weak_ptr<ProbeInstance>> _refid2ProbeInstance;

	// logical mapping of DeviceIds for fast access
	std::map<OpenIPC_DeviceId, std::weak_ptr<ProbeInstance>> _deviceid2ProbeInstance;
	std::map<OpenIPC_DeviceId, std::weak_ptr<InterfaceInstance>> _deviceid2probeinterface;

    // set to keep track of the current locked interfaces as well as the current target
    std::set<OpenIPC_DeviceId> _lockedInterfaces;
    OpenIPC_DeviceId _currentLockedInterface;
    bool _inLock;

    // The plugin settings for device set/get config
    DeviceSettings _settings;
    std::vector<std::unique_ptr<PPI_Action>> _actions;
    ProbePlugin_TimeScaler _timeScaler {_settings};

	// Mutex used to protect members for multiple simultaneous access by different threads (mutable, since it doesn't contribute to the state)
	mutable std::recursive_mutex _pluginMutex;

    // Factory method to get the right bundle type (should be overridden)
    virtual std::unique_ptr<IInterfaceBundle> CreateInterfaceBundle(PPI_EInterfaceType interfaceType);

    /// Return OpenIPC_Error_No_Error if the interfaceBundle has been populated properly
    OpenIPC_Error GetInterfaceBundle(PPI_EInterfaceType interfaceType, PPI_ProbeBundleHandle handle, IInterfaceBundle*& interfaceBundle);

    virtual OpenIPC_Error _ProbeBeginInitialization(PPI_RefId probe_refid, OpenIPC_DeviceId probeID);
    virtual OpenIPC_Error _ProbeSpecifyVendorInitString(OpenIPC_DeviceId probeID, const PPI_char* vendorinit);
    virtual OpenIPC_Error _ProbeFinishInitialization(OpenIPC_DeviceId probeID);

    // JTAG operations to be subclassed
    virtual OpenIPC_Error _AppendJtagPinsOperationToBundle(IInterfaceBundle& handle,
        uint32_t shiftLengthBits, const uint8_t* const tdi,
        const uint8_t* const tms, uint8_t* tdo,
        const PPI_JTAG_PinsOptions* const options);
    virtual OpenIPC_Error _AppendJtagGoToStateOperationToBundle(IInterfaceBundle& handle,
        JtagStateEncode gotoState,
        uint32_t numberOfClocksInState,
        const PPI_JTAG_StateGotoOptions* const options);
    virtual OpenIPC_Error _AppendJtagStateIRShiftOperationToBundle(IInterfaceBundle& handle,
        uint32_t shiftLengthBits,
        const uint8_t* const inBits,
        uint8_t* outBits,
        const PPI_JTAG_StateShiftOptions* const options);
    virtual OpenIPC_Error _AppendJtagStateDRShiftOperationToBundle(IInterfaceBundle& handle,
        uint32_t shiftLengthBits,
        const uint8_t* const inBits,
        uint8_t* outBits,
        const PPI_JTAG_StateShiftOptions* const options);
    virtual OpenIPC_Error _AppendJtagSetConfigToBundle(IInterfaceBundle& interfaceBundle,
        const char* configType,
        const char* value);

    virtual bool _eventCanBeReliedOn(const std::shared_ptr<ProbeInstance>& probe, PPI_PluginEvent pluginEvent) const;
    virtual OpenIPC_Error _LookupPinAliases(PPI_Pins_TypeEncode pin,std::vector<PPI_Pins_TypeEncode>& pinAliases) const;
    virtual OpenIPC_Error _GetDrivablePins(std::vector<PPI_Pins_TypeEncode>& pins) const;
    virtual OpenIPC_Error _GetReadablePins(std::vector<PPI_Pins_TypeEncode>& pinAliases) const;
	virtual OpenIPC_Error _GetProbes(std::vector<std::weak_ptr<ProbeInstance>>& probes, const char* vendorinit); // this is called during PlugIn init to get the initial enumeration of probes
	//virtual OpenIPC_Error _AddProbe(ProbeInstance*); // this is called by a plug event to add a new probe
	//virtual OpenIPC_Error _RemoveProbe(ProbeInstance*); // this is called by an unplug event to remove an existing probe
    //By default, this calls probe->GetInterfaces.  If your probe instance implements GetInterfaces, then you don't need to implement _GetProbeInterfaces on your probe plugin.
	virtual OpenIPC_Error _GetProbeInterfaces(std::shared_ptr<ProbeInstance> probe, std::vector<std::weak_ptr<InterfaceInstance>>& probeinterfaces); // this is called during probe init to get the initial enumeration of interfaces

    template <typename P>
    OpenIPC_Error _LookupProbe(OpenIPC_DeviceId probeID, std::shared_ptr<P> &probeInstance) const
    {
        std::lock_guard<std::recursive_mutex> lock(_pluginMutex);
        auto openIPCError = OpenIPC_Error_No_Error;
        auto iter = _deviceid2ProbeInstance.find(probeID);
        if (iter != _deviceid2ProbeInstance.end())
        {
            probeInstance = std::dynamic_pointer_cast<P>(iter->second.lock());
        }
        else
        {
            openIPCError = OpenIPC_Error_Invalid_Device_ID;
			PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Unable to find a probe with the given probeID: " << LOG_UINT32_HEX(probeID));
        }
        if (OpenIPC_PASS(openIPCError) && !probeInstance)
        {
            openIPCError = OpenIPC_Error_Invalid_Device_ID;
            PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "The given probeID doesn't correspond to the expected type: " << LOG_UINT32_HEX(probeID));
        }
        return openIPCError;
    }

    template <typename I>
    OpenIPC_Error _LookupInterface(OpenIPC_DeviceId interfaceID, std::shared_ptr<I> &interfaceInstance) const
    {
        std::lock_guard<std::recursive_mutex> lock(_pluginMutex);
        auto openIPCError = OpenIPC_Error_No_Error;
        auto iter = _deviceid2probeinterface.find(interfaceID);
        if (iter != _deviceid2probeinterface.end())
        {
            interfaceInstance = std::dynamic_pointer_cast<I>(iter->second.lock());
        }
        else
        {
            openIPCError = OpenIPC_Error_Invalid_Device_ID;
            PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "Unable to find an interface with the given interfaceID: " << LOG_UINT32_HEX(interfaceID));
        }
        if (OpenIPC_PASS(openIPCError) && !interfaceInstance)
        {
            openIPCError = OpenIPC_Error_Invalid_Device_ID;
            PPI_ERROR_WITH_MESSAGE(OpenIPC_INVALID_DEVICE_ID, openIPCError, "The given interfaceID: " << LOG_UINT32_HEX(interfaceID) << "doesn't correspond to the expected type");
        }
        return openIPCError;
    }

    /// @ Method for an implementation to fill in for multi chain bundle execute; the vector must have at least 1 interface and the ProbeBundle is not a release nor hold.
    virtual OpenIPC_Error _BundleExecuteMultiChain(ProbePluginBundle* bundle, std::vector<std::shared_ptr<InterfaceInstance>>& interfaces, PPI_bool keepLock);
	virtual OpenIPC_Error _JtagScan(InterfaceInstance& interfaceInstance, const uint32_t* input, uint32_t inputdwords, uint32_t* output, uint32_t maxoutputdwords, uint32_t* outputdwords);
	virtual OpenIPC_Error _I2cScan(InterfaceInstance& interfaceInstance, const uint32_t* input, uint32_t inputdwords, uint32_t* output, uint32_t maxoutputdwords, uint32_t* outputdwords);
	virtual OpenIPC_Error _PinsScan(InterfaceInstance& interfaceInstance, const uint32_t* input, uint32_t inputdwords, uint32_t* output, uint32_t maxoutputdwords, uint32_t* outputdwords);
	virtual OpenIPC_Error _PortRead(InterfaceInstance& interfaceInstance, uint8_t* output, uint32_t maxoutputbytes, uint32_t* outputbytes);
	virtual OpenIPC_Error _PortWrite(InterfaceInstance& interfaceInstance, const uint8_t* input, uint32_t maxinputbytes, uint32_t* inputbytes);
	virtual OpenIPC_Error _PortOpenWindow(InterfaceInstance& interfaceInstance, PPI_Trace_PortAccessMode accessMode);
	virtual OpenIPC_Error _PortCloseWindow(InterfaceInstance& interfaceInstance);
	virtual OpenIPC_Error _PortIsReadDataAvailable(InterfaceInstance& interfaceInstance, bool* isDataAvailable);
	virtual OpenIPC_Error _PortIsWindowOpen(InterfaceInstance& interfaceInstance, bool* isWindowOpen);
	virtual OpenIPC_Error _ProbeLockHold(OpenIPC_DeviceId deviceInterface);
	virtual OpenIPC_Error _ProbeLockRelease(OpenIPC_DeviceId deviceInterface);

	virtual void _SendEvent(OpenIPC_DeviceId deviceId, PPI_PluginEvent pluginEvent, uint64_t value);
	virtual void _SendNotification(OpenIPC_DeviceId deviceId, PPI_NotificationLevel level, OpenIPC_Error error, const char* message);

    virtual void SettingsUpdated() {}		//! Function to indicate that the probe settings have been updated; this is for common items like leadingTCKs for all JTAG interfaces

	explicit CProbePlugin(const char* plugin_name, bool supportBundles = false);

    virtual ~CProbePlugin() noexcept { }

private:
    OpenIPC_Error _GetDeviceSettings(OpenIPC_DeviceId deviceID, const PPI_char* configType, DeviceSetting*& deviceSettings);
};
