/////////////////////////<Source Code Embedded Notices>/////////////////////////
//
// INTEL CONFIDENTIAL
// Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or delivery
// of the Materials, either expressly, by implication, inducement, estoppel or
// otherwise. Any license under such intellectual property rights must be
// express and approved by Intel in writing.
//
/////////////////////////<Source Code Embedded Notices>/////////////////////////

#pragma once

#ifdef OpenIPC_LOG
#error LoggingMacros.h was included before PluginLogger.h. The "standard" logging macros are to be removed and routed via PPI so PluginLogger.h needs to be included first.
#endif

#include <Foundation/Logging/LoggingMacros.h>
#include <map>
#include <mutex>
#include <unordered_map>

#undef OpenIPC_LOG
#undef OpenIPC_LOG_FUNC
#undef OpenIPC_LOG_USAGE

#define PPI_LOG(DEVICE, LEVEL, FUNC) \
    if (PluginLogger::GetInstance().IsLogEnabled(LEVEL)) \
    { \
        PluginLogger::GetInstance().Log(DEVICE, LEVEL, [&](std::basic_ostream<char>& stream) { stream << FUNC; }); \
    }

#define PPI_LOG_FUNC(DEVICE, LEVEL, FUNC) \
    if (PluginLogger::GetInstance().IsLogEnabled(LEVEL)) \
    { \
        PluginLogger::GetInstance().Log(DEVICE, LEVEL, FUNC); \
    }

#define PPI_LOG_STREAM(DEVICE, STREAM, LEVEL, FUNC) \
    if (PluginLogger::GetInstance().IsLogEnabled(LEVEL, STREAM)) \
    { \
        PluginLogger::GetInstance().Log(DEVICE, STREAM, LEVEL, [&](std::basic_ostream<char>& stream) { stream << FUNC; }); \
    }

#define PPI_LOG_STREAM_FUNC(DEVICE, STREAM, LEVEL, FUNC) \
    if (PluginLogger::GetInstance().IsLogEnabled(LEVEL, STREAM)) \
    { \
        PluginLogger::GetInstance().Log(DEVICE, STREAM, LEVEL, FUNC); \
    }

#define OpenIPC_LOG(LOGGER, LEVEL, FUNC) PPI_LOG(OpenIPC_INVALID_DEVICE_ID, LEVEL, FUNC)
#define OpenIPC_LOG_FUNC(LOGGER, LEVEL, FUNC) PPI_LOG_FUNC(OpenIPC_INVALID_DEVICE_ID, LEVEL, FUNC)

#include "ProbeTypes.h"

#include <sstream>

// For legacy support we need to define OpenIPC_Logger
class OpenIPC_Logger
{
public:
    OpenIPC_Logger(void) {}
    ///
    /// @brief Returns an instance of a logger
    static OpenIPC_Logger* GetLogger(OpenIPC_LoggingLoggerId /*loggerId*/)
    {
        return new OpenIPC_Logger();
    };
};

class PluginLogger
{
private:

    PPI_NotificationLevel _default;
    std::mutex _mutex;

    PluginLogCallbackHandler _logHandlerFunction;
    PluginStreamLogCallbackHandler _logStreamHandlerFunction;

    std::unordered_map<PPI_Stream, PPI_NotificationLevel> _streamToLevelMap;

    PPI_NotificationLevel _getCurrentLevel(PPI_Stream stream)
    {
        std::lock_guard<std::mutex> lock(_mutex);

        auto sIter = _streamToLevelMap.find(stream);
        if (sIter != _streamToLevelMap.end())
        {
            return sIter->second;
        }
        else
        {
            return _default;
        }
    }

    void _setCurrentLevel(PPI_Stream stream, PPI_NotificationLevel level)
    {
        std::lock_guard<std::mutex> lock(_mutex);

        _streamToLevelMap[stream] = level;
    }

public:
    PluginLogger() :
        _default(PPI_noNotification),
        _logHandlerFunction(nullptr),
        _logStreamHandlerFunction(nullptr),
        _streamToLevelMap()
    {
    }

    OpenIPC_Error Handler(PluginLogCallbackHandler logHandlerFunction);

    OpenIPC_Error Handler(PluginStreamLogCallbackHandler logHandlerFunction);

    void Level(PPI_NotificationLevel level, PPI_Stream stream = PPI_INTERNAL_STREAM)
    {
        _setCurrentLevel(stream, level);
    }

    template <typename TFunc>
    void Log(OpenIPC_DeviceId deviceId, PPI_NotificationLevel level, const TFunc& fn)
    {
        if (IsLogEnabled(level))
        {
            std::stringstream ss;
            fn(ss);

            if (_logStreamHandlerFunction != nullptr)
            {
                _logStreamHandlerFunction(deviceId, PPI_INTERNAL_STREAM,level,ss.str().c_str());
            }
            else if (_logHandlerFunction != nullptr)
            {
                _logHandlerFunction(deviceId, level, ss.str().c_str());
            }

        }
    }

    template <typename TFunc>
    void Log(OpenIPC_DeviceId deviceId, PPI_Stream stream,PPI_NotificationLevel level, const TFunc& fn)
    {
        if (IsLogEnabled(level,stream))
        {
            std::stringstream ss;
            fn(ss);
            if (_logStreamHandlerFunction != nullptr)
            {
                _logStreamHandlerFunction(deviceId, stream, level, ss.str().c_str());
            }
        }
    }

    bool IsLogEnabled(PPI_NotificationLevel level, PPI_Stream stream = PPI_INTERNAL_STREAM)
    {
        return level >= _getCurrentLevel(stream) && _logHandlerFunction != nullptr;
    }

    void Clear()
    {
        _default = PPI_noNotification;
        _logHandlerFunction = nullptr;
        _logStreamHandlerFunction = nullptr;
        _streamToLevelMap.clear();
    }

    static PluginLogger& GetInstance()
    {
        static PluginLogger _instance;
        return _instance;
    }
};
