#pragma once
//////////////////////////////////////////////////////////////////////////////
//
//                      INTEL CONFIDENTIAL
//       Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written
// permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or
// delivery of the Materials, either expressly, by implication, inducement,
// estoppel or otherwise. Any license under such intellectual property rights
// must be express and approved by Intel in writing.
//
// Unless otherwise agreed by Intel in writing, you may not remove or alter
// this notice or any other notice embedded in Materials by Intel or Intel'ss
// suppliers or licensors in any way.
//
//////////////////////////////////////////////////////////////////////////////
///  @file
///
///  @brief Contains Functions for creating/manipulating the slot instances.
///
///  For additional information on obtaining and using instances, see @ref probeusage.
///
//////////////////////////////////////////////////////////////////////////////

#ifndef __STATE_PORT_OPERATIONS_H
#define __STATE_PORT_OPERATIONS_H__

#include "BundleOperations.h"

/// @brief The opaque type for denoting StatePort operation
///
///  This struct holds the type of operation being performed,
///  the values of any parameters, and any other information that
///  might be necessary to perform the operation.
///
///  This is treated as an opaque data type, i.e., a client cannot know how it is implemented internally,
///  only observe what happens when interacting via API functions.
///
typedef struct PPI_InterfaceStatePortOperation PPI_InterfaceStatePortOperation;

static PPI_InterfaceStatePortOperation* const INVALID_OPERATION = nullptr;

/// @brief Opaque type for the definition of a state port operation
/// This includes the fields, outputs, etc. of a particular stateport
typedef struct PPI_InterfaceStatePortDefinition PPI_InterfaceStatePortDefinition;


/** @brief StatePort Operation Types
*   This is a flags enum, where each operation type has just one bit set, which isn't shared with any other operation type.
*/
DECLARE_ENUM(PPI_StatePort_OPERATION_TYPE_ET) {
      StatePort_OPERATION_NONE =  0x00,
      StatePort_OPERATION_READ =  0x01,
      StatePort_OPERATION_WRITE = 0x02,
};


/// Typedef for the error code from stateport operations.
typedef uint32_t ProbeStatePortError;
/// Definition for an "ok" operation error
const ProbeStatePortError PROBE_STATE_PORT_ERROR_OK = 0;
/// Definition for an invalid operation error
const ProbeStatePortError PROBE_STATE_PORT_ERROR_INVALID = (ProbeStatePortError)(-1);

/// @brief The opaque type for denoting a StatePort operation parameter
///
///  This struct holds the parameter name, description, and size.
///  This struct also knows what operation type it is for.
///
///  This is treated as an opaque data type, i.e., a client cannot know how it is implemented internally,
///  only observe what happens when interacting via API functions.
///
typedef struct PPI_InterfaceStatePortParameter PPI_InterfaceStatePortParameter;


/* Functions for interacting with StatePort operations. */


/** PPI_StatePortAllocateOperation optional arguments.*/
typedef struct {
      //Empty now, but left for future expansion.
} PPI_StatePortAllocateOperation_Options;

/** @brief This function allocates a StatePort Operation.
*
* Every PPI_InterfaceStatePortOperation object must be freed with a call to PPI_StatePortFreeOperation.
*
* @param [in] deviceID The device id of the StatePort instance.
* @param [in] definition The definition of the stateport.
* @param [in] operationType The type of operation you will perform.  Can only be one type.
* @param [out] operation Populated with a pointer to the newly allocated PPI_InterfaceStatePortOperation
* @param [in] options Additional options (NULL for default).  Currently empty, but may be extended later.
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Invalid_Device_ID if deviceId is invalid
* @retval OpenIPC_Error_Null_Pointer if definition is null
* @retval OpenIPC_Error_Invalid_Parameter if operationType is not valid
* @retval OpenIPC_Error_Null_Pointer if operation is null
* @retval OpenIPC_Error_Internal_Error if unable to allocate the memory.
*/
PPI_FUNCTION(PPI_StatePortAllocateOperation, OpenIPC_Error,
             OpenIPC_DeviceId deviceId,
             const PPI_InterfaceStatePortDefinition* definition,
             PPI_StatePort_OPERATION_TYPE_ET operationType,
             PPI_InterfaceStatePortOperation** operation,
             const PPI_StatePortAllocateOperation_Options* options);

/** @brief This function frees a PPI_InterfaceStatePortOperation that was allocated with PPI_StatePortAllocateOperation.
*
* @param [in] deviceID The device id of the StatePort instance.
* @param [inout] operation The PPI_InterfaceStatePortOperation to be freed.  After the procedure completes *operation == INVALID_OPERATION
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Invalid_Device_ID if deviceId is invalid
* @retval OpenIPC_Error_Null_Pointer if operation is null
*/
PPI_FUNCTION(PPI_StatePortFreeOperation, OpenIPC_Error,
             OpenIPC_DeviceId deviceId,
             PPI_InterfaceStatePortOperation** operation);

/** PPI_StatePortSubmitOperation optional arguments.*/
typedef struct {
      //Empty now, but left for future expansion.
} PPI_StatePortAppendOperation_Options;

/** @brief This function appends a StatePort operation to a bundle.
*
* @param [in] deviceID The device id of the StatePort instance.
* @param [in] bundle The bundle you are submitting this operation to.
* @param [in] operation The StatePort Operation to want to submit. After the procedure completes, *operationPointer == INVALID_OPERATION
* @param [in] options Additional options (NULL for default).  Currently empty, but may be extended later.
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Invalid_Device_ID if deviceId is invalid
* @retval OpenIPC_Error_Invalid_Parameter if bundle is not valid
* @retval OpenIPC_Error_Null_Pointer if operation is null
*/
PPI_FUNCTION(PPI_StatePortAppendOperation, OpenIPC_Error,
             OpenIPC_DeviceId deviceId,
             PPI_ProbeBundleHandle bundle,
             PPI_InterfaceStatePortOperation** operationPointer,
             const PPI_StatePortAppendOperation_Options* options);

/** @brief This function sets a parameter on an operation
*
* @param [in] operation The StatePort Operation to want to set the parameter on.
* @param [in] parameter The parameter that you want to set on the operation.
* @param [in] value A byte buffer for the value you want to set.
* @param [in] size The size of value in bytes.
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Null_Pointer if operation is null
* @retval OpenIPC_Error_Null_Pointer if parameter is null
* @retval OpenIPC_Error_Null_Pointer if value is null
* @retval OpenIPC_Error_Invalid_Parameter if size is too large
*/
PPI_FUNCTION(PPI_StatePortOperationSetParameter, OpenIPC_Error,
             PPI_InterfaceStatePortOperation* operation,
             const PPI_InterfaceStatePortParameter* parameter,
             const uint8_t* value,
             uint32_t size);

/** @brief This function sets a write value for an operation.
*
* @param [in] operation The StatePort Operation to want to set the write value for.
* @param [in] buffer A byte buffer that holds the write value.
* @param [in] size The size of buffer in bytes.
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Null_Pointer if operation is null
* @retval OpenIPC_Error_Null_Pointer if buffer is null
* @retval OpenIPC_Error_Invalid_Parameter if size is too large
* @retval OpenIPC_Error_Not_Supported if the stateport that created this operation doesn't support writes
*/
PPI_FUNCTION(PPI_StatePortOperationSetWriteValue, OpenIPC_Error,
             PPI_InterfaceStatePortOperation* operation,
             const uint8_t* buffer,
             uint32_t size);

/** @brief This function sets a read buffer for an operation.
*
* @param [in] operation The StatePort Operation to want to set the read buffer for.
* @param [out] buffer A byte buffer that will be filled with the return value.
* @param [in] size The size of buffer in bytes.  Must be large enough to hold the read value of the operation.
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Null_Pointer if operation is null
* @retval OpenIPC_Error_Null_Pointer if buffer is null
* @retval OpenIPC_Error_Invalid_Parameter if size is too small
* @retval OpenIPC_Error_Not_Supported if the stateport that created this operation doesn't support reads
*/
PPI_FUNCTION(PPI_StatePortOperationSetReadBuffer, OpenIPC_Error,
             PPI_InterfaceStatePortOperation* operation,
             uint8_t* buffer,
             uint32_t size);

/** @brief This function sets an error buffer for an operation.
*
* @param [in] operation The StatePort Operation to want to set the error buffer for.
* @param [out] errorBuffer Will be set to the error value.
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Null_Pointer if operation is null
* @retval OpenIPC_Error_Null_Pointer if errorBuffer is null
*/
PPI_FUNCTION(PPI_StatePortOperationSetErrorBuffer, OpenIPC_Error,
             PPI_InterfaceStatePortOperation* operation,
             ProbeStatePortError* errorBuffer);


/* End of functions for interacting with StatePort operation*/

/** @brief Gets the Operation definitions for a particular stateport.
*
* @param [in] deviceID The device id of the StatePort instance to query.
* @param [in] definitionsSize The number of entries in the @a definitions buffer. If this is 0, then the definitions pointer can be NULL.
* @param [out] definitions The produced list of definitions. This can be NULL. Only the buffer's memory is owned by the client, everythigng else is owned by the plugin and has a valid lifetime until the interface is deinitialized.
* @param [out] numberOfDefinitions The total number of definitions for this device. This will always be written and cannot be NULL.
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Invalid_Device_ID if deviceId is invalid
* @retval OpenIPC_Error_Null_Pointer if definitions is null and definitionsSize is non-zero
*/
PPI_FUNCTION(PPI_StatePortGetDefinitions, OpenIPC_Error,
             OpenIPC_DeviceId deviceId,
             const PPI_InterfaceStatePortDefinition** definitions,
             uint32_t definitionsSize,
             uint32_t* numberOfDefinitions);


/** @brief This function gets a string for the StatePort name.
*
* @param [in] definition The definition of the stateport.
* @param [out] buffer A byte buffer that will be filled with the name (if not null).  buffer will not be filled beyond bufferSize.
* @param [in] bufferSize The size of buffer in bytes.  May be 0.
* @param [out] realSize If not null, will be populated with the size of the name in bytes.
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Null_Pointer if definition is null
* @retval OpenIPC_Error_Null_Pointer if buffer is null and bufferSize is non-zero
*/
PPI_FUNCTION(PPI_StatePortGetName, OpenIPC_Error,
             const PPI_InterfaceStatePortDefinition* definition,
             char* buffer,
             uint32_t bufferSize,
             uint32_t* realSize);

/** @brief This function gets a string for the StatePort description.
*
* @param [in] definition The definition of the stateport.
* @param [out] buffer A byte buffer that will be filled with the description (if not null).  buffer will not be filled beyond bufferSize.
* @param [in] bufferSize The size of buffer in bytes.  May be 0.
* @param [out] realSize If not null, will be populated with the size of the description in bytes.
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Null_Pointer if definition is null
* @retval OpenIPC_Error_Null_Pointer if buffer is null and bufferSize is non-zero
*/
PPI_FUNCTION(PPI_StatePortGetDescription, OpenIPC_Error,
             const PPI_InterfaceStatePortDefinition* definition,
             char* buffer,
             uint32_t bufferSize,
             uint32_t* realSize);

/** @brief This function gets a PPI_StatePort_OPERATION_ET for all the operation types this StatePort supports.
*
* @param [in] definition The definition of the operation.
* @param [out] capabilities All operation types that this StatePort supports (and only those that this StatePort supports) will be set.
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Null_Pointer if definition is null
* @retval OpenIPC_Error_Null_Pointer if capabilities is null
*/
PPI_FUNCTION(PPI_StatePortGetCapabilites, OpenIPC_Error,
             const PPI_InterfaceStatePortDefinition* definition,
             PPI_StatePort_OPERATION_TYPE_ET* capabilities);

/** @brief This function gets the access size (in bits) for an operation type on this StatePort.
*
* @param [in] definition The definition of the stateport.
* @param [in] operationType The operation type you want the access size of.
* @param [out] size Will be set to the access size (in bits) of the specified operation type.
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Null_Pointer if definition is null
* @retval OpenIPC_Error_Null_Pointer if size is null
*/
PPI_FUNCTION(PPI_StatePortGetAccessSize, OpenIPC_Error,
	const PPI_InterfaceStatePortDefinition* definition,
	uint32_t* size);

/** @brief This function gets the parameters for an operation type.
*
* @param [in] definition The definition of the stateport.
* @param [out] parameters An array of PPI_InterfaceStatePortParameter*, which will be populated with the parameters for the given operation type.
* @param [in] length The length of parameters (counted in number of items as an array of PPI_InterfaceStatePortParameter*).  May be 0.
* @param [out] realLength If not null, will be populated with the number of parameters for this operation.
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Null_Pointer if definition is null
* @retval OpenIPC_Error_Null_Pointer if parameters is null and length is non-zero
*/
PPI_FUNCTION(PPI_StatePortGetParameters, OpenIPC_Error,
             const PPI_InterfaceStatePortDefinition* definition,
             const PPI_InterfaceStatePortParameter** parameters,
             uint32_t length,
             uint32_t* realLength);

/** @brief This function gets the name of a parameter.
*
* @param [in] parameter The parameter that you will get the name of.
* @param [out] buffer A byte buffer that will be filled with the name (if not null).  buffer will not be filled beyond bufferSize.
* @param [in] bufferSize The size of buffer in bytes.  May be 0.
* @param [out] realSize If not null, will be populated with the size of the name in bytes.
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Null_Pointer if parameter is null
* @retval OpenIPC_Error_Null_Pointer if buffer is null and bufferSize is non-zero
*/
PPI_FUNCTION(PPI_StatePortParameterGetName, OpenIPC_Error,
             const PPI_InterfaceStatePortParameter* parameter,
             char* buffer,
             uint32_t bufferSize,
             uint32_t* realSize);

/** @brief This function gets the description of a parameter.
*
* @param [in] parameter The parameter that you will get the description of.
* @param [out] buffer A byte buffer that will be filled with the description (if not null).  buffer will not be filled beyond bufferSize.
* @param [in] bufferSize The size of buffer in bytes.  May be 0.
* @param [out] realSize If not null, will be populated with the size of the description in bytes.
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Null_Pointer if parameter is null
* @retval OpenIPC_Error_Null_Pointer if buffer is null and bufferSize is non-zero
*/
PPI_FUNCTION(PPI_StatePortParameterGetDescription, OpenIPC_Error,
             const PPI_InterfaceStatePortParameter* parameter,
             char* buffer,
             uint32_t bufferSize,
             uint32_t* realSize);


/** @brief This function gets the size of a parameter in bits.
*
* @param [in] parameter The parameter that you will get the size of.
* @param [out] size The size of the parameter in bits.
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Null_Pointer if parameter is null
* @retval OpenIPC_Error_Null_Pointer if size is null
*/
PPI_FUNCTION(PPI_StatePortParameterGetSize, OpenIPC_Error,
             const PPI_InterfaceStatePortParameter* parameter,
             uint32_t* size);

/** @brief This function gets the default value of a parameter.
*
* @param [in] parameter The parameter that you will get the default value of.
* @param [out] buffer Populated with the default value of the parameter.
* @param [in] size The size of buffer in bytes. Must be large enough to hold the value (same as the parameter size).
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Null_Pointer if parameter is null
* @retval OpenIPC_Error_Not_Supported if parameter doesn't support a default value
* @retval OpenIPC_Error_Null_Pointer if buffer is null
* @retval OpenIPC_Error_Invalid_Argument if size is too small
*/
PPI_FUNCTION(PPI_StatePortParameterGetDefaultValue, OpenIPC_Error,
             const PPI_InterfaceStatePortParameter* parameter,
             uint8_t* buffer,
             uint32_t size);


/** @brief This function gets the name of an error.
*
* @param [in] definition The definition of the stateport.
* @param [in] error The error that you will get the name of.
* @param [out] buffer A byte buffer that will be filled with the name (if not null).  buffer will not be filled beyond bufferSize.
* @param [in] bufferSize The size of buffer in bytes.  May be 0.
* @param [out] realSize If not null, will be populated with the size of the name in bytes.
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Null_Pointer if definition is null
* @retval OpenIPC_Error_Invalid_Argument if error is not valid
* @retval OpenIPC_Error_Null_Pointer if buffer is null and bufferSize is non-zero
*/
PPI_FUNCTION(PPI_StatePortErrorGetName, OpenIPC_Error,
             const PPI_InterfaceStatePortDefinition* definition,
             ProbeStatePortError error,
             char* buffer,
             uint32_t bufferSize,
             uint32_t* realSize);

/** @brief This function gets the description of an error.
*
* @param [in] definition The definition of the stateport.
* @param [in] error The error that you will get the description of.
* @param [out] buffer A byte buffer that will be filled with the description (if not null).  buffer will not be filled beyond bufferSize.
* @param [in] bufferSize The size of buffer in bytes.  May be 0.
* @param [out] realSize If not null, will be populated with the size of the description in bytes.
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Null_Pointer if definition is null
* @retval OpenIPC_Error_Invalid_Argument if error is not valid
* @retval OpenIPC_Error_Null_Pointer if buffer is null and bufferSize is non-zero
*/
PPI_FUNCTION(PPI_StatePortErrorGetDescription, OpenIPC_Error,
             const PPI_InterfaceStatePortDefinition* definition,
             ProbeStatePortError error,
             char* buffer,
             uint32_t bufferSize,
             uint32_t* realSize);

/** @brief This function gets the success of an error.
*
* @param [in] definition The definition of the stateport.
* @param [in] error The error that you will get the success of.
* @param [out] openIpcError If error indicates success (PROBE_STATE_PORT_ERROR_OK), openIpcError is set to OpenIPC_Error_No_Error.  Otherwise, it's set to an applicable error (OpenIPC_Error_Internal_Error by default).
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Null_Pointer if definition is null
* @retval OpenIPC_Error_Invalid_Argument if error is not valid
* @retval OpenIPC_Error_Null_Pointer if openIpcError is null
*/
PPI_FUNCTION(PPI_StatePortErrorGetOpenIpcError, OpenIPC_Error,
             const PPI_InterfaceStatePortDefinition* definition,
             ProbeStatePortError error,
             OpenIPC_Error* openIpcError);

/// @brief Describes the value of a parameter for a StatePort Event
///
///  This struct holds the Parameter, and a byte buffer storing
///  the value of the Parameter for this event.
///
typedef struct PPI_InterfaceStatePortEventParameterDescription
{
    PPI_InterfaceStatePortParameter const* Parameter;
    uint32_t ValueBufferByteSize;
    uint8_t const* ValueBuffer;
} PPI_InterfaceStatePortEventParameterDescription;

/// @brief Describes a StatePort Event indicating that the value of a StatePort has changed.
///
///  This struct holds the Definition for the StatePort whos value changed, along with the
///  Parameter values used to invoke that value change, and (of course) the new value.
///  The new value is stored in a byte buffer.
///
typedef struct PPI_InterfaceStatePortValueChangedEventDescription
{
    PPI_InterfaceStatePortDefinition const* Definition;
    uint32_t ParameterArrayLength;
    PPI_InterfaceStatePortEventParameterDescription const* ParameterArray;
    uint32_t NewValueBufferByteSize;
    uint8_t const* NewValueBuffer;
} PPI_InterfaceStatePortValueChangedEventDescription;

/** @brief This function gets the description of an event.
*
* This funtion must be called from within an event handler that is handling a PPI_statePortValueChanged
* event. Calling this function in any other context will return an error.
*
* If this function is successful, eventDescriptionPtr will point to a PPI_InterfaceStatePortValueChangedEventDescription
* instance that is populated with values for the event you are processing. This instance is only valid
* while you are in the event handler. If you need any value from the event description, you will have
* to copy them before exiting the event handler.
*
* @param [in] deviceId The interface that the event orgininated from.
* @param [out] eventDescriptionPtr Will be set to reference a PPI_InterfaceStatePortEventDescription for the event.
* @return (OpenIPC_Error)
* @retval OpenIPC_Error_Invalid_Device_ID if deviceId is invalid
* @retval OpenIPC_Error_Operation_Not_Supported if this is called from a thread that isn't servicing a PPI_statePortValueChanged event.
* @retval OpenIPC_Error_Null_Pointer if eventDescriptionPtr is null
*/
PPI_FUNCTION(PPI_StatePortEventGetValueChangedEventDescription, OpenIPC_Error,
    OpenIPC_DeviceId deviceId,
    PPI_InterfaceStatePortValueChangedEventDescription const** eventDescriptionPtr);

#endif //STATE_PORT_OPERATIONS_H
