#pragma once
//////////////////////////////////////////////////////////////////////////////
//
//                      INTEL CONFIDENTIAL
//       Copyright (C) Intel Corporation All Rights Reserved.
//
// The source code contained or described herein and all documents related to
// the source code ("Material") are owned by Intel Corporation or its suppliers
// or licensors. Title to the Material remains with Intel Corporation or its
// suppliers and licensors. The Material contains trade secrets and proprietary
// and confidential information of Intel or its suppliers and licensors. The
// Material is protected by worldwide copyright and trade secret laws and
// treaty provisions. No part of the Material may be used, copied, reproduced,
// modified, published, uploaded, posted, transmitted, distributed, or disclosed
// in any way without Intel's prior express written
// permission.
//
// No license under any patent, copyright, trade secret or other intellectual
// property right is granted to or conferred upon you by disclosure or
// delivery of the Materials, either expressly, by implication, inducement,
// estoppel or otherwise. Any license under such intellectual property rights
// must be express and approved by Intel in writing.
//
// Unless otherwise agreed by Intel in writing, you may not remove or alter
// this notice or any other notice embedded in Materials by Intel or Intel'ss
// suppliers or licensors in any way.
//
//////////////////////////////////////////////////////////////////////////////
///  @file
///
///  @brief Contains Methods for bulk interface operations.
///
///  For additional information on obtaining and using instances, see @ref probeusage.
///
//////////////////////////////////////////////////////////////////////////////

#ifndef __TRACE_OPERATIONS_H__
#define __TRACE_OPERATIONS_H__
#include <ProbePlugin.h>

//////////////////////////////////////////////////////////////////////////
//  Function: InterfaceTraceOpen
///
/// @brief Opens a trace connection on the specified interface.
///
///     InterfaceTraceOpen opens a trace connection on the specified interface.
///     The trace model assumes that the specific configuration of the trace endpoint is
///     done by the initialization ( @a InterfaceInitialization ) and interface configuration
///     ( @a InterfaceSetConfig ).
///
/// @param[in] interfaceID	
///     The unique identification number of the interface, set during @a InterfaceInitialize.
///
/// @param[in] accessMode
///     The settign read or write to open the trace window for.
///
/// @return (IPC_ErrorCode): A code indicating success or failure.
/// @retval IPC_Error_No_Error
///     Indicates success.
///
//////////////////////////////////////////////////////////////////////////
PPI_FUNCTION(PPI_InterfaceTraceOpenWindow, OpenIPC_Error, 
			 OpenIPC_DeviceId interfaceID,
			 PPI_Trace_PortAccessMode accessMode);

//////////////////////////////////////////////////////////////////////////
//  Function: InterfaceTraceClose
///
/// @brief Closes a trace connection on the specified interface.
///
///     InterfaceTraceClose closes a trace connection on the specified interface
///     which terminates all data transfer through the interface.
///
/// @param[in] interfaceID	
///     The unique identification number of the interface, set during @a InterfaceInitialize.
///
/// @return (IPC_ErrorCode): A code indicating success or failure.
/// @retval IPC_Error_No_Error
///     Indicates success.
///
//////////////////////////////////////////////////////////////////////////
PPI_FUNCTION(PPI_InterfaceTraceCloseWindow, OpenIPC_Error, 
			 OpenIPC_DeviceId interfaceID);

//////////////////////////////////////////////////////////////////////////
//  Function: InterfaceTraceIsDataAvailable
///
/// @brief Returns true if data is available on the interface.
///
/// @param[in] interfaceID	
///     The unique identification number of the interface, set during @a InterfaceInitialize.
/// @param[out] isDataAvailable
///     Returns true if data is available for reading
///
/// @return (IPC_ErrorCode): A code indicating success or failure.
/// @retval IPC_Error_No_Error
///     Indicates success.
///
//////////////////////////////////////////////////////////////////////////
PPI_FUNCTION( PPI_InterfaceTraceIsReadDataAvailable, OpenIPC_Error, 
			 OpenIPC_DeviceId interfaceID, 
			 PPI_bool* isDataAvailable);

//////////////////////////////////////////////////////////////////////////
//  Function: InterfaceTraceIsWindowOpen
///
/// @brief Returns true if the trace window is open.
///
/// @param[in] interfaceID	
///     The unique identification number of the interface, set during @a InterfaceInitialize.
/// @param[out] isWindowOpen
///     Returns true if the window is open
/// @return (IPC_ErrorCode): A code indicating success or failure.
/// @retval IPC_Error_No_Error
///     Indicates success.
///
//////////////////////////////////////////////////////////////////////////
PPI_FUNCTION(PPI_InterfaceTraceIsWindowOpen, OpenIPC_Error, 
			 OpenIPC_DeviceId interfaceID, 
			 PPI_bool* isWindowOpen);

//////////////////////////////////////////////////////////////////////////
//  Function: InterfaceTraceRead
///
/// @brief InterfaceTraceRead provides the ability to read blocks of data from a previously configured trace source.
///
///     InterfaceTraceRead allows a caller to request blocks of data by providing a buffer and requesting a maximum number of bytes
///
/// @param[in] interfaceID	
///     The unique identification number of the interface, set during @a InterfaceInitialize.
/// @param[out] output
///     An output buffer that is allocated by the caller
/// @param[in] maxOutputBytes
///     The maximum number of bytes that can be safely written to @a output buffer.
/// @param[out] outputBytes
///     The actual number of bytes written to @a output.
///
/// @return (IPC_ErrorCode): A code indicating success or failure.
/// @retval IPC_Error_No_Error
///     Indicates success.
/// @return OpenIPC_Error_Invalid_Device_ID
///     The probe reference id @a probeID is not valid
///
//////////////////////////////////////////////////////////////////////////
PPI_FUNCTION(PPI_InterfaceTracePortRead, OpenIPC_Error, 
			 OpenIPC_DeviceId interfaceID, 
			 uint8_t* output, 
			 uint32_t maxOutputBytes,
			 uint32_t* outputBytes);

//////////////////////////////////////////////////////////////////////////
//  Function: InterfaceTraceRead
///
/// @brief InterfaceTraceRead provides the ability to read blocks of data from a previously configured trace source.
///
///     InterfaceTraceRead allows a caller to request blocks of data by providing a buffer and requesting a maximum number of bytes
///
/// @param[in] interfaceID	
///     The unique identification number of the interface, set during @a InterfaceInitialize.
/// @param[in] input
///     An in buffer that is allocated by the caller
/// @param[in] maxInputBytes
///     The maximum number of bytes that can be safely read from @a input buffer.
/// @param[out] inputBytes
///     The actual number of bytes written to @a output.
///
/// @return (IPC_ErrorCode): A code indicating success or failure.
/// @retval IPC_Error_No_Error
///     Indicates success.
/// @return OpenIPC_Error_Invalid_Device_ID
///     The probe reference id @a probeID is not valid
///
//////////////////////////////////////////////////////////////////////////
PPI_FUNCTION(PPI_InterfaceTracePortWrite, OpenIPC_Error, 
			 OpenIPC_DeviceId interfaceID, 
			 const uint8_t* input, 
			 uint32_t maxInputBytes,
			 uint32_t* inputBytes);

#endif